#!/usr/bin/perl
#********************************************************************************
# IBM Storage Protect
# 
# name: uniquify_data_files.pl
#
# desc: The uniquify_data_files.pl utility is used to make the test files in the benchdata folder
#       unique again so that the next iteration by sp_load_test will not have a 100% 
#       deduplicaiton result.  By default, 128 test files will be updated.  The
#       objcount parameter can be used to change a different number of test files.
#
#
#       The benchdata folder must exist and already be populated with the test
#       files using create_data_files.pl before using uniquify_data_files.pl.  On Windows, this script depends
#       on crerand.exe and wdeedee.exe existing in the same directory.
#
# usage:  perl uniquify_data_files.pl [objcount]
#
# Notice: This program is provided as a tool intended for use by IBM Internal, 
#         IBM Business Partners, and IBM Customers. This program is provided as is, 
#         without support, and without warranty of any kind expressed or implied.
#
# (C) Copyright International Business Machines Corp. 2013, 2018
#********************************************************************************

if (@ARGV > 1)
{
  print "USAGE: perl uniquify_data_files.pl [objcount]\n";
  print "  Defaults:  objcount will default to 128\n";
  exit;
}

$arg = shift(@ARGV);
if ($arg =~ m/\d+/)
{
  $objcount = $arg;
}
else
{
  $objcount = 128;
}

if (! -d "benchdata")
{
  print "ERROR: The benchdata folder must exist in the current directory\n";
  print "  Have you run create_data_files.pl yet?\n";
  exit;
}

if ($^O =~ m/^MSWin32/)
{
  $SS = "\\";
  $platform = "WIN32";
  if (! -e "crerand.exe")
  {
    print "ERROR: Cannot locate the required file crerande.exe\n";
    exit;
  }
}
else
{
  $SS = "/";
}

$blockSize = 32768;
$pctChange = 75;

for ($i=1; $i<=$objcount; $i++)
{
  if (! -e "benchdata${SS}bench${i}")
  {
    print "ERROR: Cannot locate the file benchdata${SS}bench${i}\n";
    print "Did you run create_data_files.pl yet using the same objcount?\n";
  }
  else
  {
    # Determine file size and # of blocks to change

    $filename = "benchdata${SS}bench${i}";
    $size = -s $filename;
    $blockCount = int($size / $blockSize);
    $changeCount = int($blockCount * ($pctChange/100));

    # Generate a random file to use as change contents
    if ($platform eq "WIN32")
    {
      $sizeKB = int($size / 1024) + 1000;
      $seed = int (rand $size) +1;
      `..\\bin\\crerand.exe update_rand_$sizeKB $sizeKB $seed`;
      $ddPfx = "..\\bin\\wdeedee.exe if=update_rand_$sizeKB";
    }
    else
    {
      $ddPfx = "dd if=/dev/urandom";
    }

    # Loop through the file and replace one byte in each block with a new random byte
    for ($j =0; $j <= $changeCount; $j++)
    {
      $offset = int (rand $blockSize) +1;
      $seekLoc = $blockSize * $j + $offset;

      if ($platform eq "WIN32")
      {
        $cmd = "$ddPfx of=$filename conv=notrunc seek=$seekLoc skip=$seekLoc bs=1 obs=1 count=1 2>&1";
      }
      else # on UNIX skipping from the input file is not necessary and impacts performance
      {
        $cmd = "$ddPfx of=$filename conv=notrunc seek=$seekLoc bs=1 obs=1 count=1 2>&1";
      }
      $out = `$cmd`;
      if ($? != 0)
      {
        print "ERROR writing change number: $i\n";
        print "command = $cmd\n$out\n";
        die;
      }
     }

     if ($platform eq "WIN32")
     {
       unlink "update_rand_$sizeKB";
     }
  }
}
