#!/bin/bash

#------------------------------------------------------------------------------
#  Name: monthly_backup.sh
#
#  Desc:  Adjust SAP HANA configuration to use a dedicated profile for the 
#         database backup with IBM Storage Protect for ERP, run the backup and 
#         revert the configuration changes.
#
#  Usage: You must modify the values in the configuration section below before
#         executing this script to match the actual environment. This script 
#         must be run as SAP HANA database instance user.
#
#         <path to script>/monthly_backup.sh
#
# Notice: This program is provided as a tool intended for use by IBM Internal,
#         IBM Business Partners, and IBM Customers. This program is provided as is,
#         without support, and without warranty of any kind expressed or implied.
#
# (C) Copyright International Business Machines Corp. 2021
#------------------------------------------------------------------------------

##########################################
# configuration section
# adjust values to the actual environment

# the tenant database subject of this backup operation - default tenant
TENANT=${SAPSYSTEMNAME}

# hdb keystore entry with appropriate privileges to alter system configuration
# and create backups
KEY_STORE_ENTRY=TSM

# tag which uniquely identifies the backup
BACKUP_TAG="monthly_backup_"`date +%Y_%m_%d`

# addititonal comment that is shown in the backup catalog
COMMENT="monthly backup"

# the profile to be used to store the monthly backups
MONTHLY_PROFILE=/usr/sap/${SAPSYSTEMNAME}/SYS/global/hdb/opt/hdbconfig/init${SAPSYSTEMNAME}_monthly.utl 

# the profile to be used for the regular backups
ORIGINAL_PROFILE=/usr/sap/${SAPSYSTEMNAME}/SYS/global/hdb/opt/hdbconfig/'init$(SAPSYSTEMNAME).utl'

# end of configuration section
##########################################

if [ -z "${SAPSYSTEMNAME}" ]; then
	echo "Environment variable SAPSYSTEMNAME is not set. Make sure to run this command as database instance user."
	exit 2;
fi

if [ ! -f ${MONTHLY_PROFILE} ]; then
	echo "Profile for monthly backups ${MONTHLY_PROFILE} not found."
	exit 2;
fi

echo ""
echo "Create backup with special profile ${MONTHLY_PROFILE} using backup tag ${BACKUP_TAG}"

# show current config
echo ""
echo "Original layers and values for parameter 'data_backup_parameter_file'"
hdbsql -U ${KEY_STORE_ENTRY} -d ${TENANT} -x "select layer_name,value from m_inifile_contents where key like 'data_backup_parameter_file'" 
echo ""

echo "Adjust config to use monthly profile ${MONTHLY_PROFILE}"
hdbsql -U ${KEY_STORE_ENTRY} -d ${TENANT} "alter system alter configuration ('global.ini', 'DATABASE') set ('backup', 'data_backup_parameter_file') = '${MONTHLY_PROFILE}' with reconfigure"

if [ $? -ne 0 ]; then
	echo "Error while changing configuration. Aborting"
	exit 2;
fi

echo "Running the backup ..."
hdbsql -U ${KEY_STORE_ENTRY} -d ${TENANT} "backup data using backint ('${BACKUP_TAG}') comment '${COMMENT} - tag: ${BACKUP_TAG}'"

echo "Restore original profile"
hdbsql -U ${KEY_STORE_ENTRY} -d ${TENANT} "alter system alter configuration ('global.ini', 'DATABASE') SET ('backup', 'data_backup_parameter_file') = '${ORIGINAL_PROFILE}' with reconfigure"

# check config
echo "Layers and values for parameter 'data_backup_parameter_file' after backup"
hdbsql -U ${KEY_STORE_ENTRY} -d ${TENANT} -x "select layer_name,value from m_inifile_contents where key like 'data_backup_parameter_file'" 
echo ""
echo "Make sure the original profile settings have been restored correctly"

