# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA, 5900-AMG
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

import numpy as np
import pandas as pd
import pytest

import pmlib
from pmlib import api
from pmlib.anomaly_detection import AnomalyDetectionEstimator
from pmlib.transformer import IdentifyPreFailureWindow, TransformNotNaToEvent


class MyAnomalyDetectionEstimator(AnomalyDetectionEstimator):
    def get_stages(self):
        from sklearn.ensemble import IsolationForest
        from sklearn.preprocessing import StandardScaler, RobustScaler, MinMaxScaler
        from srom.anomaly_detection.generalized_anomaly_model import GeneralizedAnomalyModel
        from srom.utils.no_op import NoOp

        return [
            [
                ('skipscaling', NoOp()), 
                ('standardscaler', StandardScaler()),
                ('robustscaler', RobustScaler()), 
                ('minmaxscaling', MinMaxScaler())
            ],
            [
                # Rule/Density based Anomaly Models
                ('isolationforest', GeneralizedAnomalyModel(base_learner=IsolationForest(), predict_function='decision_function', score_sign=-1)), 

            ]
        ]


class MyAnomalyDetectionAssetGroupPipeline(pmlib.AnomalyDetectionAssetGroupPipeline):
    def prepare_execute(self, pipeline, model_config):
        estimator = MyAnomalyDetectionEstimator(**model_config)
        pipeline.add_stage(estimator)
        if len(model_config['features_for_training']) > 0:
            estimator.add_training_preprocessor(TransformNotNaToEvent(model_config['features_for_training'][0]))
            estimator.add_training_preprocessor(IdentifyPreFailureWindow(model_config['features_for_training'][0], pre_failure_window_size=model_config.get('pre_failure_window_size', 20), pre_failure_failure_size=model_config.get('pre_failure_failure_size', 10)))


def test(asset_group_id, iot_type, deviceid_prefix, assetid_prefix, siteid='BEDFORD'):
    group = MyAnomalyDetectionAssetGroupPipeline(
                asset_group_id=asset_group_id, 
                model_pipeline={
                    'features': ['%s:axlevibration' % iot_type, '%s:axlemomentum' % iot_type],
                    'features_for_training': [':faildate'],
                    'predictions': ['anomaly_score_custom', 'anomaly_threshold_custom'],
                    'srom_training_options': {
                        'exectype': 'single_node_random_search'
                    },
                },
                incremental_summary=False,
                asset_device_mappings={
                    '%s%s-____-%s' % (assetid_prefix, i, siteid): ['%s:%s_%s' % (iot_type, deviceid_prefix, i)] for i in range(1, 6)
                },
            )
    df = group.execute(start_ts='2019-07-11', end_ts='2019-07-15')

    print(pmlib.log_df_info(df, head=0))

    group.register()

    resp = group.enable(enabled=True)
    if resp is not None:
        print(resp.text)


if __name__ == '__main__':
    test(asset_group_id='abcd5', iot_type='BrakeSensor1030', assetid_prefix='BRAKE1030', siteid='BEDFORD', deviceid_prefix='Brake1030')

