# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA, 5900-AMG
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

import numpy as np
import pandas as pd

import pmlib
from pmlib import api
from pmlib.util import current_directory


def print_resp(resp, msg=''):
    '''Quick method for printing out response content.
    '''
    print(str(resp))
    if resp is not None:
        print(resp.text)
    else:
        raise RuntimeError(msg)


def test(asset_group_id, iot_type, deviceid_prefix, assetid_prefix, siteid='BEDFORD', use_wiotp=False, local_asset_group=True):
    assetid_prefix = assetid_prefix.upper()

    pmlib.setup_logging()
    api.init_environ()

    db = api._get_db()
    db_schema = None

    current_directory = pmlib.util.current_directory(__file__)
    print(current_directory)

    df = pd.read_csv('%s/../pmlib/tests/trainbrake_asset_attributes.csv' % current_directory)
    df['asset'] = df['asset'].str.replace('TRAINBRAKE', assetid_prefix)
    df['site'] = siteid
    print(df.head())
    if local_asset_group:
        api.set_asset_cache_dimension(df, siteid_column='site', assetid_column='asset', db=db, db_schema=db_schema, delete_df_asset_first=True)
    else:
        resp = pmlib.create_assets(df, siteid_column='site', assetid_column='asset')
        print_resp(resp, 'create_assets')

        resp = pmlib.setup_asset_attributes(df=df, siteid_column='site', assetid_column='asset', attribute_columns=['installdate','estendoflife'], parse_dates=['installdate','estendoflife'])
        print_resp(resp, 'setup_asset_attributes')
    
    df = pd.read_csv('%s/../pmlib/tests/trainbrake_asset_faildates.csv' % current_directory)
    df['asset'] = df['asset'].str.replace('TRAINBRAKE', assetid_prefix)
    df['site'] = siteid
    print(df.head())
    if local_asset_group:
        api.set_asset_cache(df, siteid_column='site', assetid_column='asset', faildate_column='faildate', db=db, db_schema=db_schema, delete_df_asset_first=True)
    else:
        resp = pmlib.import_asset_failure_history(df, siteid_column='site', assetid_column='asset', faildate_column='faildate')
        print_resp(resp, 'import_asset_failure_history')
    
    df = pd.read_csv('%s/../pmlib/tests/trainbrake_device_data.csv' % current_directory)
    df['DEVICEID'] = df['DEVICEID'].str.replace('TrainBrake', deviceid_prefix)
    print(df.head())

    rename_columns = {
        'TRAINBRAKESIMULATION_AXLEMOMENTUM': 'axlemomentum',
        'TRAINBRAKESIMULATION_AXLEVIBRATION': 'axlevibration',
        'TRAINBRAKESIMULATION_HUMIDITY': 'humidity', 
        'TRAINBRAKESIMULATION_LOAD': 'load',
        'TRAINBRAKESIMULATION_NUMBEROFPASSENGERS': 'passengers', 
        'TRAINBRAKESIMULATION_OBSCOND': 'obscond',
        'TRAINBRAKESIMULATION_RUNHOURS': 'runhours',
        'TRAINBRAKESIMULATION_TEMP': 'temp',
        'TRAINBRAKESIMULATION_WHEELSPEED': 'wheelspeed',
    }

    t = pmlib.setup_iot_type(
        name=iot_type, 
        df=df, 
        columns=list(rename_columns.keys()), 
        deviceid_column='DEVICEID',
        timestamp_column='RCV_TIMESTAMP_UTC', 
        timestamp_in_payload=False,
        rename_columns=rename_columns,
        write='deletefirst',
        use_wiotp=use_wiotp,
        import_only=True,
        db=db,
        db_schema=db_schema)

    df= pd.read_csv('%s/../pmlib/tests/trainbrake_mappings.csv' % current_directory)
    df['asset'] = df['asset'].str.replace('TRAINBRAKE', assetid_prefix)
    df['site'] = siteid
    df['devicetype'] = iot_type
    df['deviceid'] = df['deviceid'].str.replace('TrainBrake', deviceid_prefix)
    print(df.head())

    pmlib.set_asset_device_mappings(df=df, db=db, db_schema=db_schema, delete_df_asset_first=True, drop_table_first=False)

    if local_asset_group:
        df = df.groupby(['site', 'asset']).first().reset_index()[['site', 'asset']]
        pmlib.set_asset_group_members(asset_group_id, df, siteid_column='site', assetid_column='asset', db=db, db_schema=db_schema)


if __name__ == '__main__':
    test(asset_group_id='abcd', iot_type='BrakeSensor1205', deviceid_prefix='Brake1205', assetid_prefix='BRAKE1205', use_wiotp=False)

