# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA, 5900-AMG
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

import numpy as np
import pandas as pd
import pytest

from sqlalchemy.exc import OperationalError

from .. import api
from ..util import current_directory, log_df_info, setup_logging


def test_resampling_methods_and_outputs(iot_type='abcdsensor'):
    if iot_type is None:
        import math
        import random
        iot_type = 'abcdsensor_%05d' % math.floor(random.random() * 10**5)

    setup_logging()
    api.init_environ()
    
    features = ['temp', 'load']
    num_rows = 100
    df = pd.DataFrame(columns=features, data=[[i for f in features] for i in range(num_rows)])
    df['timestamp'] = pd.date_range(start='2019-05-03 07:10:23', periods=len(df), freq='30s')
    df['deviceid'] = np.where(df['timestamp'].dt.second == 23, 'D1', 'D2') # 2 devices

    print(log_df_info(df, head=-1))

    try:
        db = api._get_db()
        db_schema = None

        api.setup_iot_type(iot_type, df, columns=features, deviceid_column='deviceid', timestamp_column='timestamp', timestamp_in_payload=False, parse_dates=None, write='deletefirst', use_wiotp=False, import_only=False, db=db, db_schema=db_schema)

        entity_type = api.get_entity_type(iot_type, db=db)

        with pytest.raises(ValueError):
            df_agg = api.get_entity_type_data(entity_type=None)
        with pytest.raises(TypeError):
            df_agg = api.get_entity_type_data(entity_type=db)

        # whether names in agg_methoeds are valid
        df_agg = api.get_entity_type_data(entity_type, time_grain='day')
        assert 'temp' in df_agg.columns
        with pytest.raises(ValueError):
            df_agg = api.get_entity_type_data(entity_type, time_grain='day', agg_outputs={'temp': 'temp_mean'})
        df_agg = api.get_entity_type_data(entity_type, time_grain='day', agg_methods={'temp': ['mean']})
        assert 'temp_mean' in df_agg.columns
        df_agg = api.get_entity_type_data(entity_type, time_grain='day', agg_methods={'load': ['mean']})
        assert 'load_mean' in df_agg.columns
        df_agg = api.get_entity_type_data(entity_type, data_items=['temp', 'load'], time_grain='day', agg_methods={'temp': 'mean', 'load': 'mean'})
        assert 'temp_mean' in df_agg.columns
        assert 'load_mean' in df_agg.columns
        df_agg = api.get_entity_type_data(entity_type, data_items=['load'], time_grain='day')
        assert 'temp' not in df_agg.columns
        assert 'load' in df_agg.columns
        with pytest.raises(ValueError):
            df_agg = api.get_entity_type_data(entity_type, data_items=['load'], time_grain='day', agg_outputs={'load': 'load_mean'})
        # if agg_methods include some columns that are not in columns (when given), they are automatically added
        df_agg = api.get_entity_type_data(entity_type, data_items=['load'], time_grain='day', agg_methods={'temp': ['mean']})
        assert 'temp_mean' in df_agg.columns
        assert 'load' in df_agg.columns
        # Key Error when agg_methods include some columns that are not available from the entity type
        with pytest.raises(KeyError):
            df_agg = api.get_entity_type_data(entity_type, time_grain='day', agg_methods={'t': ['mean']})

        for time_grain in [entity_type._timestamp, 'day', 'S', 'H', 'T', 'month', 'year']:
            # normal condition and methods
            df_agg = api.get_entity_type_data(entity_type, time_grain=time_grain, agg_methods={'temp': ['count', 'max', 'mean', 'min', 'sum']})
            # whether outputs are automatically generated correctly
            assert all([col in df_agg.columns for col in ['temp_count', 'temp_max', 'temp_mean', 'temp_min', 'temp_sum']])

            # invalid method 'abc'
            with pytest.raises(ValueError):
                df_agg = api.get_entity_type_data(entity_type, time_grain=time_grain, agg_methods={'temp': ['count', 'max', 'mean', 'min', 'sum'], 'load': ['abc']})
            # invalid method 'var' (not supported)
            with pytest.raises(ValueError):
                df_agg = api.get_entity_type_data(entity_type, time_grain=time_grain, agg_methods={'temp': ['var']})
            # invalid method 'median' (not supported)
            with pytest.raises(ValueError):
                df_agg = api.get_entity_type_data(entity_type, time_grain=time_grain, agg_methods={'temp': ['median']})
            if db.db_type == 'sqlite':
                # unsupported method 'std' for sqlite
                with pytest.raises(ValueError):
                    df_agg = api.get_entity_type_data(entity_type, time_grain=time_grain, agg_methods={'temp': 'std'})
            else:
                # method 'std' are supported by db2/postgresql
                df_agg = api.get_entity_type_data(entity_type, time_grain=time_grain, agg_methods={'temp': 'std'})
                assert 'temp_std' in df_agg.columns

        # 'first' and 'last'

        if db.db_type == 'sqlite':
            # for sqlite, 'first' and 'last' only works for time grain: timestamp and 'day', but not all other time_grain
            for time_grain in [entity_type._timestamp, 'day']:
                for method in ['first', 'last']:
                    df_agg = api.get_entity_type_data(entity_type, time_grain=time_grain, agg_methods={'temp': method})
                    assert ('temp_%s' % method) in df_agg.columns
            for time_grain in ['S', 'T', '1T', '1min', '2T', '2min', 'H', '1H', '2H', '2D', 'week', 'W', 'month', 'MS', '2MS', 'year', 'YS', '2YS']:
                for method in ['first', 'last']:
                    with pytest.raises(ValueError):
                        df_agg = api.get_entity_type_data(entity_type, time_grain=time_grain, agg_methods={'temp': method})

            # TODO testing iotfunctions directly for issues, remove these once they are fixed
            # 'first'/'last' cannot work with Pandas aggregation
            for time_grain in ['S', '2T', '2D', 'W', '2MS', '2YS']:
                for method in ['first', 'last']:
                    print('time_grain=%s, method=%s' % (time_grain, method))
                    with pytest.raises(KeyError):
                        entity_type._pre_aggregate_time_grain = time_grain
                        entity_type._pre_agg_rules = {'temp': [method]}
                        entity_type._pre_agg_outputs = {}
                        df_agg = entity_type.get_data()
            # sqlite does not support most direct SQL approach, only supporting 'day'
            # 'nmin'/'nH' are in affect same all using db query directly (n is ignored for now)
            for time_grain in ['1min', '2min', '1H', '2H', 'week', 'month', 'year']:
                for method in ['first', 'last']:
                    print('time_grain=%s, method=%s' % (time_grain, method))
                    with pytest.raises(OperationalError):
                        entity_type._pre_aggregate_time_grain = time_grain
                        entity_type._pre_agg_rules = {'temp': [method]}
                        entity_type._pre_agg_outputs = {}
                        df_agg = entity_type.get_data()
        else:
            # for other dbs, Pandas agg would cause error (iotfunctiosn defects which could and should be fixed)
            for time_grain in [entity_type._timestamp, 'T', '1T', '1min', 'H', '1H', 'day', 'week', 'month', 'MS', 'year', 'YS']:
                for method in ['first', 'last']:
                    df_agg = api.get_entity_type_data(entity_type, time_grain=time_grain, agg_methods={'temp': method})
                    assert ('temp_%s' % method) in df_agg.columns
            for time_grain in ['S', '2T', '2min', '2H', '2D', 'W', '2MS', '2YS']:
                for method in ['first', 'last']:
                    with pytest.raises(ValueError):
                        df_agg = api.get_entity_type_data(entity_type, time_grain=time_grain, agg_methods={'temp': method})

            # TODO testing iotfunctions directly for issues, remove these once they are fixed
            # 'first'/'last' cannot work with Pandas aggregation
            for time_grain in ['S', '2T', '2D', 'W', 'MS', '2MS', 'YS', '2YS']:
                for method in ['first', 'last']:
                    print('time_grain=%s, method=%s' % (time_grain, method))
                    with pytest.raises(KeyError):
                        entity_type._pre_aggregate_time_grain = time_grain
                        entity_type._pre_agg_rules = {'temp': [method]}
                        entity_type._pre_agg_outputs = {}
                        df_agg = entity_type.get_data()
            # 'nmin'/'nH' are in affect same all using db query directly (n is ignored for now)
            for time_grain in ['1min', '2min', '1H', '2H', 'day', 'week', 'month', 'year']:
                for method in ['first', 'last']:
                    print('time_grain=%s, method=%s' % (time_grain, method))
                    entity_type._pre_aggregate_time_grain = time_grain
                    entity_type._pre_agg_rules = {'temp': [method]}
                    entity_type._pre_agg_outputs = {}
                    df_agg = entity_type.get_data()

        # outputs

        # outputs are generated correctly when not given
        df_agg = api.get_entity_type_data(entity_type, time_grain='day', agg_methods={'temp': ['max', 'mean', 'min']})
        assert all([col in df_agg.columns for col in ['temp_max', 'temp_mean', 'temp_min']])
        df_agg = api.get_entity_type_data(entity_type, time_grain='day', agg_methods={'temp': ['max', 'mean', 'min']}, agg_outputs={'temp': None})
        assert all([col in df_agg.columns for col in ['temp_max', 'temp_mean', 'temp_min']])
        df_agg = api.get_entity_type_data(entity_type, time_grain='day', agg_methods={'temp': ['max', 'mean', 'min']}, agg_outputs={'temp': []})
        assert all([col in df_agg.columns for col in ['temp_max', 'temp_mean', 'temp_min']])
        # when outputs are given to a name, all of its variants (by methods) must be given
        with pytest.raises(ValueError):
            df_agg = api.get_entity_type_data(entity_type, time_grain='day', agg_methods={'temp': ['max', 'mean', 'min']}, agg_outputs={'temp': ['temp_max', 'temp_mean']})
        # when outputs are given to a name, as long as all of its variants (by methods) are given, extra outpus are ignored (by sequence)
        df_agg = api.get_entity_type_data(entity_type, time_grain='day', agg_methods={'temp': ['max', 'mean', 'min']}, agg_outputs={'temp': ['max_temp', 'mean_temp', 'min_temp', 'extra_temp']})
        assert all([col in df_agg.columns for col in ['max_temp', 'mean_temp', 'min_temp']])
    except:
        raise
    finally:
        api.delete_iot_type(iot_type, use_wiotp=False, db=db, db_schema=db_schema)


def test_get_resampled_data(iot_type='abcdsensor'):
    if iot_type is None:
        import math
        import random
        iot_type = 'abcdsensor_%05d' % math.floor(random.random() * 10**5)

    setup_logging()
    api.init_environ()

    features = ['temp', 'load']
    num_rows = 100
    df = pd.DataFrame(columns=features, data=[[i for f in features] for i in range(num_rows)])
    df['timestamp'] = pd.date_range(start='2019-05-03 07:10:23', periods=len(df), freq='30s')
    df['deviceid'] = np.where(df['timestamp'].dt.second == 23, 'D1', 'D2') # 2 devices

    print(log_df_info(df, head=-1))

    try:
        db = api._get_db()
        db_schema = None

        api.setup_iot_type(iot_type, df, columns=features, deviceid_column='deviceid', timestamp_column='timestamp', timestamp_in_payload=False, parse_dates=None, write='deletefirst', use_wiotp=False, import_only=False, db=db, db_schema=db_schema)

        entity_type = api.get_entity_type(iot_type, db=db)

        # group by timestamp directly, can be used to eliminate duplicate records at the same time

        df_agg = api.get_entity_type_data(entity_type, time_grain=entity_type._timestamp, agg_methods={'temp': ['last'], 'load': ['last']}, agg_outputs={'temp': ['temp'], 'load': ['load']}).sort_index(level=[1, 0])

        print(log_df_info(df_agg, head=-1))

        assert all([col in df_agg.columns for col in ['temp', 'load']])
        assert len(df_agg) == 100
        assert df_agg.index.values[0][1] == pd.Timestamp('2019-05-03 07:10:23')
        assert df_agg.index.values[1][1] == pd.Timestamp('2019-05-03 07:10:53')

        df_agg = api.get_entity_type_data(entity_type, time_grain=entity_type._timestamp, agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']}).sort_index(level=[1, 0])

        print(log_df_info(df_agg, head=-1))

        assert all([col in df_agg.columns for col in ['temp_min', 'temp_max', 'temp_mean', 'load']])
        assert len(df_agg) == 100
        assert df_agg.index.values[0][1] == pd.Timestamp('2019-05-03 07:10:23')
        assert df_agg.index.values[1][1] == pd.Timestamp('2019-05-03 07:10:53')

        # minutely slot

        for minutely in ['min', '1min', 'MIN', 'Min', 't', 'T', '1t', '1T']:
            df_agg = api.get_entity_type_data(entity_type, time_grain=minutely, agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']}).sort_index(level=[1, 0])

            print(log_df_info(df_agg, head=-1))

            assert all([col in df_agg.columns for col in ['temp_min', 'temp_max', 'temp_mean', 'load']])
            assert len(df_agg) == 100
            assert df_agg.index.values[0][1] == pd.Timestamp('2019-05-03 07:10:00')
            assert df_agg.index.values[1][1] == pd.Timestamp('2019-05-03 07:10:00')
            assert df_agg.index.values[2][1] == pd.Timestamp('2019-05-03 07:11:00')
            assert df_agg.index.values[3][1] == pd.Timestamp('2019-05-03 07:11:00')
            assert df_agg.iloc[98]['deviceid'] == 'D1'
            assert df_agg.iloc[98]['temp_min'] == 98
            assert df_agg.iloc[98]['temp_max'] == 98
            assert df_agg.iloc[98]['temp_mean'] == 98
            assert df_agg.iloc[98]['load'] == 98
            assert df_agg.iloc[99]['deviceid'] == 'D2'
            assert df_agg.iloc[99]['temp_min'] == 99
            assert df_agg.iloc[99]['temp_max'] == 99
            assert df_agg.iloc[99]['temp_mean'] == 99
            assert df_agg.iloc[99]['load'] == 99

        for minutely in ['5T']:
            df_agg = api.get_entity_type_data(entity_type, time_grain=minutely, agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']}).sort_index(level=[1, 0])

            print(log_df_info(df_agg, head=-1))

            assert all([col in df_agg.columns for col in ['temp_min', 'temp_max', 'temp_mean', 'load']])
            assert len(df_agg) == 20
            assert df_agg.index.values[0][1] == pd.Timestamp('2019-05-03 07:10:00')
            assert df_agg.index.values[1][1] == pd.Timestamp('2019-05-03 07:10:00')
            assert df_agg.index.values[2][1] == pd.Timestamp('2019-05-03 07:15:00')
            assert df_agg.index.values[3][1] == pd.Timestamp('2019-05-03 07:15:00')
            assert df_agg.iloc[18]['deviceid'] == 'D1'
            assert df_agg.iloc[18]['temp_min'] == 90
            assert df_agg.iloc[18]['temp_max'] == 98
            assert df_agg.iloc[18]['temp_mean'] == 94
            assert df_agg.iloc[18]['load'] == 94
            assert df_agg.iloc[19]['deviceid'] == 'D2'
            assert df_agg.iloc[19]['temp_min'] == 91
            assert df_agg.iloc[19]['temp_max'] == 99
            assert df_agg.iloc[19]['temp_mean'] == 95
            assert df_agg.iloc[19]['load'] == 95

        # hourly slot

        for hourly in ['1h', '1H']:
            df_agg = api.get_entity_type_data(entity_type, time_grain=hourly, agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']}).sort_index(level=[1, 0])

            print(log_df_info(df_agg, head=-1))

            assert all([col in df_agg.columns for col in ['temp_min', 'temp_max', 'temp_mean', 'load']])
            assert len(df_agg) == 2
            assert df_agg.index.values[0][1] == pd.Timestamp('2019-05-03 07:00:00')
            assert df_agg.index.values[1][1] == pd.Timestamp('2019-05-03 07:00:00')
            assert df_agg.iloc[0]['deviceid'] == 'D1'
            assert df_agg.iloc[0]['temp_min'] == 0
            assert df_agg.iloc[0]['temp_max'] == 98
            assert df_agg.iloc[0]['temp_mean'] == 49
            assert df_agg.iloc[0]['load'] == 49
            assert df_agg.iloc[1]['deviceid'] == 'D2'
            assert df_agg.iloc[1]['temp_min'] == 1
            assert df_agg.iloc[1]['temp_max'] == 99
            assert df_agg.iloc[1]['temp_mean'] == 50
            assert df_agg.iloc[1]['load'] == 50

        for bihourly in ['2h', '2H']:
            df_agg = api.get_entity_type_data(entity_type, time_grain=bihourly, agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']}).sort_index(level=[1, 0])

            print(log_df_info(df_agg, head=-1))

            assert all([col in df_agg.columns for col in ['temp_min', 'temp_max', 'temp_mean', 'load']])
            assert len(df_agg) == 2
            assert df_agg.index.values[0][1] == pd.Timestamp('2019-05-03 06:00:00')
            assert df_agg.index.values[1][1] == pd.Timestamp('2019-05-03 06:00:00')
            assert df_agg.iloc[0]['deviceid'] == 'D1'
            assert df_agg.iloc[0]['temp_min'] == 0
            assert df_agg.iloc[0]['temp_max'] == 98
            assert df_agg.iloc[0]['temp_mean'] == 49
            assert df_agg.iloc[0]['load'] == 49
            assert df_agg.iloc[1]['deviceid'] == 'D2'
            assert df_agg.iloc[1]['temp_min'] == 1
            assert df_agg.iloc[1]['temp_max'] == 99
            assert df_agg.iloc[1]['temp_mean'] == 50
            assert df_agg.iloc[1]['load'] == 50

        # daily slot

        for daily in ['day', 'D', '1D']:
            df_agg = api.get_entity_type_data(entity_type, time_grain=daily, agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']}).sort_index(level=[1, 0])

            print(log_df_info(df_agg, head=-1))

            assert all([col in df_agg.columns for col in ['temp_min', 'temp_max', 'temp_mean', 'load']])
            assert len(df_agg) == 2
            assert df_agg.index.values[0][1] == pd.Timestamp('2019-05-03 00:00:00')
            assert df_agg.index.values[1][1] == pd.Timestamp('2019-05-03 00:00:00')
            assert df_agg.iloc[0]['deviceid'] == 'D1'
            assert df_agg.iloc[0]['temp_min'] == 0
            assert df_agg.iloc[0]['temp_max'] == 98
            assert df_agg.iloc[0]['temp_mean'] == 49
            assert df_agg.iloc[0]['load'] == 49
            assert df_agg.iloc[1]['deviceid'] == 'D2'
            assert df_agg.iloc[1]['temp_min'] == 1
            assert df_agg.iloc[1]['temp_max'] == 99
            assert df_agg.iloc[1]['temp_mean'] == 50
            assert df_agg.iloc[1]['load'] == 50

        # weekly slot

        # using direct SQL approach, 'week' does indeed go back to start of week, but Pandas approach falls on end of week
        # sqlite can only use Pandas way, hence the branching logic below
        df_agg = api.get_entity_type_data(entity_type, time_grain='week', agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']}).sort_index(level=[1, 0])

        print(log_df_info(df_agg, head=-1))

        assert all([col in df_agg.columns for col in ['temp_min', 'temp_max', 'temp_mean', 'load']])
        assert len(df_agg) == 2
        if entity_type.db.db_type == 'sqlite':
            assert df_agg.index.values[0][1] == pd.Timestamp('2019-05-05 00:00:00')
            assert df_agg.index.values[1][1] == pd.Timestamp('2019-05-05 00:00:00')
        else:
            assert df_agg.index.values[0][1] == pd.Timestamp('2019-04-29 00:00:00')
            assert df_agg.index.values[1][1] == pd.Timestamp('2019-04-29 00:00:00')
        assert df_agg.iloc[0]['deviceid'] == 'D1'
        assert df_agg.iloc[0]['temp_min'] == 0
        assert df_agg.iloc[0]['temp_max'] == 98
        assert df_agg.iloc[0]['temp_mean'] == 49
        assert df_agg.iloc[0]['load'] == 49
        assert df_agg.iloc[1]['deviceid'] == 'D2'
        assert df_agg.iloc[1]['temp_min'] == 1
        assert df_agg.iloc[1]['temp_max'] == 99
        assert df_agg.iloc[1]['temp_mean'] == 50
        assert df_agg.iloc[1]['load'] == 50

        for weekly in ['W-SUN', 'W']:
            print('weekly=%s' % weekly)

            df_agg = api.get_entity_type_data(entity_type, time_grain=weekly, agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']}).sort_index(level=[1, 0])

            print(log_df_info(df_agg, head=-1))

            assert all([col in df_agg.columns for col in ['temp_min', 'temp_max', 'temp_mean', 'load']])
            assert len(df_agg) == 2
            assert df_agg.index.values[0][1] == pd.Timestamp('2019-05-05 00:00:00')
            assert df_agg.index.values[1][1] == pd.Timestamp('2019-05-05 00:00:00')
            assert df_agg.iloc[0]['deviceid'] == 'D1'
            assert df_agg.iloc[0]['temp_min'] == 0
            assert df_agg.iloc[0]['temp_max'] == 98
            assert df_agg.iloc[0]['temp_mean'] == 49
            assert df_agg.iloc[0]['load'] == 49
            assert df_agg.iloc[1]['deviceid'] == 'D2'
            assert df_agg.iloc[1]['temp_min'] == 1
            assert df_agg.iloc[1]['temp_max'] == 99
            assert df_agg.iloc[1]['temp_mean'] == 50
            assert df_agg.iloc[1]['load'] == 50

        # monthly slot

        for monthly in ['month', 'MS', '1MS']:
            df_agg = api.get_entity_type_data(entity_type, time_grain=monthly, agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']}).sort_index(level=[1, 0])

            print(log_df_info(df_agg, head=-1))

            assert all([col in df_agg.columns for col in ['temp_min', 'temp_max', 'temp_mean', 'load']])
            assert len(df_agg) == 2
            assert df_agg.index.values[0][1] == pd.Timestamp('2019-05-01 00:00:00')
            assert df_agg.index.values[1][1] == pd.Timestamp('2019-05-01 00:00:00')
            assert df_agg.iloc[0]['deviceid'] == 'D1'
            assert df_agg.iloc[0]['temp_min'] == 0
            assert df_agg.iloc[0]['temp_max'] == 98
            assert df_agg.iloc[0]['temp_mean'] == 49
            assert df_agg.iloc[0]['load'] == 49
            assert df_agg.iloc[1]['deviceid'] == 'D2'
            assert df_agg.iloc[1]['temp_min'] == 1
            assert df_agg.iloc[1]['temp_max'] == 99
            assert df_agg.iloc[1]['temp_mean'] == 50
            assert df_agg.iloc[1]['load'] == 50

        # yearly slot

        for yearly in ['year', 'YS', '1YS', 'AS', '1AS']:
            df_agg = api.get_entity_type_data(entity_type, time_grain=yearly, agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']}).sort_index(level=[1, 0])

            print(log_df_info(df_agg, head=-1))

            assert all([col in df_agg.columns for col in ['temp_min', 'temp_max', 'temp_mean', 'load']])
            assert len(df_agg) == 2
            assert df_agg.index.values[0][1] == pd.Timestamp('2019-01-01 00:00:00')
            assert df_agg.index.values[1][1] == pd.Timestamp('2019-01-01 00:00:00')
            assert df_agg.iloc[0]['deviceid'] == 'D1'
            assert df_agg.iloc[0]['temp_min'] == 0
            assert df_agg.iloc[0]['temp_max'] == 98
            assert df_agg.iloc[0]['temp_mean'] == 49
            assert df_agg.iloc[0]['load'] == 49
            assert df_agg.iloc[1]['deviceid'] == 'D2'
            assert df_agg.iloc[1]['temp_min'] == 1
            assert df_agg.iloc[1]['temp_max'] == 99
            assert df_agg.iloc[1]['temp_mean'] == 50
            assert df_agg.iloc[1]['load'] == 50

        # invalid time_grain

        with pytest.raises(ValueError):
            df_agg = api.get_entity_type_data(entity_type, time_grain='abc', agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']})

        with pytest.raises(ValueError):
            df_agg = api.get_entity_type_data(entity_type, time_grain='id', agg_methods={'temp': ['min', 'max', 'mean']}, agg_outputs={'temp': ['temp_min', 'temp_max', 'temp_mean']})
    except:
        raise
    finally:
        api.delete_iot_type(iot_type, use_wiotp=False, db=db, db_schema=db_schema)


if __name__ == '__main__':
    test_resampling_methods_and_outputs()
