# -*- coding: utf-8 -*-
"""
Created on Thu Apr  1 21:22:04 2021

@author: Bryan Tsai, Kewei Yang, AmareshRajasekharan
"""

# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA, 5900-AMG
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

"""This module contains wrapper functions for interacting with Maximo APM APIs.
"""

__all__ = [
    'set_asset_group_members',
    'delete_asset_group_members',
    'create_assets',
    'delete_assets',
    'setup_asset_attributes',
    'import_asset_failure_history',
    'delete_asset_failure_history',
    'set_asset_device_mappings',
    'delete_asset_device_mappings',
    'setup_iot_type',
    'delete_iot_type',
    'setup_iot_type_v2',
    'delete_iot_type_v2',
    # 'set_asset_cache',
    # 'delete_asset_cache',
    # 'set_asset_cache_dimension',
    # 'delete_asset_cache_dimension',
]

import json
import datetime as dt
import json
import logging
import math
import os
import random
import re
from venv import create


import time
from datetime import datetime
from datetime import timedelta

import numpy as np
import pandas as pd
from pandas.api.types import is_bool_dtype, is_integer_dtype, is_float_dtype, is_string_dtype, is_datetime64_any_dtype
from pandas.tseries.frequencies import to_offset
from pandas.tseries.offsets import Day, Hour, Minute, MonthBegin, Second, YearBegin
import requests
from requests_futures.sessions import FuturesSession
from sqlalchemy import and_, create_engine, func, select, Boolean, Column, DateTime, Float, Index, Integer, MetaData, SmallInteger, String, Table, text
from sqlalchemy.sql.sqltypes import TIMESTAMP, VARCHAR, FLOAT, INTEGER
from sqlalchemy.orm.session import sessionmaker

import ibm_db
from iotfunctions.db import BaseTable, Database
import iotfunctions.db as db_module
from iotfunctions.dbtables import DBModelStore
from iotfunctions.metadata import EntityType, ServerEntityType
from iotfunctions.util import resample

from . import api
from .util import _mkdirp, api_request, log_df_info, mask_credential, mask_credential_in_dict, get_as_schema, get_table_name


from .cache_loader import AssetCacheRefresher
from .loader import AssetLoader
#from .pipeline import _AssetGroupEntityType

#kewei define _AssetGroupEntityType here to avoid recursive import.

class _AssetGroupEntityType(EntityType):
    def __init__(self,
                 asset_group_id,
                 db,
                 **kwargs):
        super().__init__(('APM_%s' % asset_group_id).lower(), db, **kwargs)

    def get_data(self, start_ts=None, end_ts=None, entities=None, columns=None):
        return None


"""
Constants
"""

asset_cache_entity_type = 'ASSET_CACHE'
asset_cache_entity_type_table_prefix = 'APM'
asset_cache_entity_type_timestamp = 'evt_timestamp'  # change from event_timestamp to evt_timestamp
asset_cache_table_name = 'apm_asset_cache'
asset_cache_dimension_table_name = 'apm_asset_cache_dimension2'
asset_group_table_name = 'apm_asset_groups'
asset_device_mappings_table_name = 'apm_asset_devices'
asset_device_attribute_mappings_table_name = 'apm_asset_device_attributes'
default_asset_group_column = 'assetgroup'
default_site_column = 'site'
default_asset_column = 'asset'
default_assetid_column = 'assetid'
default_devicetype_column = 'devicetype'
default_faildate_column = 'faildate'
default_failurecode_column = 'failurecode'
default_problemcode_column = 'problemcode'
default_causecode_column = 'causecode'
default_remedycode_column = 'remedycode'
default_deviceid_column = 'deviceid'
default_attribute_column = 'attribute'
default_timestamp_column = 'rcv_timestamp_utc'

APM_ID = None
APM_API_BASEURL = None
APM_API_KEY = None
MAXIMO_BASEURL = None
MAXIMO_API_CONTEXT = None
MAXIMO_LINKED = None
API_BASEURL = None
API_KEY = None
API_TOKEN = None
TENANT_ID = None
DB_CERTIFICATE_FILE = None

logger = logging.getLogger(__name__)

"""
Local mode.
"""

def is_local_mode():
    """Check whether local mode is currently enabled."""
    local_mode = os.environ.get('LOCAL_MODE', None)
    if isinstance(local_mode, str) and local_mode.lower() in ('true', 'yes', '1'):
        local_mode = True
    else:
        local_mode = False
    return local_mode


def set_local_mode():
    """Enabble local mode.

    Local mode is mainly for unit testing purpose, which has the following characteristics:

    * A local sqlite database is created, if not already, and used, instead of the remote data lake.
    * No API call can be made, since all credentials are not initialized.
    * No entity type, therefore no validation of entity type metadata are performed. Any use of any data item is legal.
    * A few APIs for checking things are short circuited, like whether there's asset device mappings defined, return negative immediately.
    """
    os.environ['LOCAL_MODE'] = 'True'


class _LocalDatabaseMetadataDict:
    def copy(self):
        return self

    def __setitem__(self, key, value):
        pass

    def __contains__(self, key):
        return True

    def __getitem__(self, key):
        return {
            'name': key,
            'metricTableName': 'iot_%s' % str(key).lower(),
            'metricTimestampColumn': default_timestamp_column,
            'schemaName': None,
            'dimensionTableName': None,
            # 'dataItemDto': [],
        }


class _LocalCosClient():
    def __init__(self, root_path='apm/pmi'):
        self.root_path = root_path
        try:
            _mkdirp(self.root_path)
        except:
            pass

    def cos_get(self, key, bucket=None, binary=False):
        with open('/'.join([self.root_path, key]), mode='r%s' % ('b' if binary else '')) as file:
            return file.read()

    def cos_put(self, key, payload, bucket=None, binary=False, serialize=True):
        try:
            _mkdirp('/'.join([self.root_path, key]))
        except:
            pass
        with open('/'.join([self.root_path, key]), mode='w%s' % ('b' if binary else '')) as file:
            file.write(payload)

    def cos_delete(self, key, bucket=None):
        os.remove('/'.join([self.root_path, key]))


class _LocalDatabase(Database):
    def __init__(self, tenant_id=None, url=None):
        self.function_catalog = {}
        self.tenant_id = tenant_id
        self.write_chunk_size = 100
        if url is None:
            url = 'sqlite:///pmi_test.sqlite'
        self.db_type = url.split('://')[0].split('+')[0]
        self.connection = create_engine(url)
        self.Session = sessionmaker(bind=self.connection)
        self.session = None
        self.metadata = MetaData(self.connection)
        self.entity_type_metadata = _LocalDatabaseMetadataDict()
        self.credentials = {'tenant_id': self.tenant_id, 'config': {'bos_runtime_bucket': None}}
        self.cos_client = _LocalCosClient()


def _get_db(tenant_id=None, echo=False, asset_group_id=None):
    logger.debug('Request made to retrieve DB...')
    if is_local_mode():
        db = None
        try:
            # local mode uses a local sqlite db created on-the-fly
            db = _LocalDatabase(tenant_id=tenant_id if tenant_id is not None else 'abcd', url=os.environ.get('LOCAL_MODE_DB', None))
        except:
            pass
        finally:
            return db

    if tenant_id is not None:
        db = Database(tenant_id=tenant_id, echo=echo)
        logger.debug('Successfully retrieved database with tenant ID: %s', tenant_id)
        return db

    init_environ()

    tmp = os.environ.get('DB_TYPE')
    logger.debug('Database Type: %s', tmp)

    db_type_lowercase = tmp.lower()

    if tmp is not None and db_type_lowercase == 'db2':
        connection_string_from_env = os.environ.get('DB_CONNECTION_STRING')
        try:
            if connection_string_from_env.endswith(';'):
                ev = dict(item.split("=",1) for item in connection_string_from_env[:-1].split(";"))
            else:
                ev = dict(item.split("=",1) for item in connection_string_from_env.split(";"))
        except Exception:
            raise ValueError('Connection string \'%s\' is incorrect. Expected format for DB2 is '
                                'DATABASE=xxx;HOSTNAME=xxx;PORT=xxx;UID=xxx;PWD=xxx[;SECURITY=xxx]' % connection_string_from_env)

    if 'TENANT_ID' in os.environ:
        if tmp is not None and tmp.lower() == 'db2':
            logger.debug('DB2 Connection String: %s', connection_string_from_env)
            if (connection_string_from_env.find('/var/www/as-pipeline/db2_certificate.pem') == -1 ) and (connection_string_from_env.find('/project_data/data_asset/db2_certificate.pem') == -1):
                if ( connection_string_from_env.find('/opt/ibm/test/src/predict/db2_certificate.pem') != -1):
                    pass;  # Need to add the db2_certificate.pem in FVT
                else:
                    connection_string_from_env = connection_string_from_env +';SSLServerCertificate=/var/www/as-pipeline/db2_certificate.pem;'
                
            else:
                logger.debug('use /project_data/data_asset/db2_certificate.pem in CP4D')
            logger.debug('before ibm_db.connect DB2 Connection String: %s', connection_string_from_env)
            db_connection = ibm_db.connect(connection_string_from_env,'','')
            schema = ev['UID']
            if 'AS_SCHEMA' in os.environ:
                schema = os.environ['AS_SCHEMA']
            model_store = DBModelStore(os.getenv('TENANT_ID'), None, schema, db_connection, db_type_lowercase)
            db = Database(tenant_id=os.getenv('TENANT_ID'), echo=echo, model_store=model_store)

            if asset_group_id is not None and asset_group_id in db.entity_type_metadata.keys():
                db.model_store.entity_type_id = db.entity_type_metadata[asset_group_id].get('entityTypeId', None)
            #else:
            #   # entity_type_id can never be null, this code is for integration testing only
            #    db.model_store.entity_type_id ='9999'
                # logger.debug('we can not get db.model_store.entity_type_id from entity_type_metadata. entity_type_id=%s', db.model_store.entity_type_id)
            logger.debug('Retrieved db.model_store.entity_type_id=%s', db.model_store.entity_type_id)
            
            logger.debug('Successfully retrieved DB2 database')
            return db
        else:
            db = Database(tenant_id=os.getenv('TENANT_ID'), echo=echo)
            logger.debug('Successfully retrieved database')
            return db
    else:
        raise RuntimeError('environment variable TENANT_ID must be set')


"""
Asset related functions.
"""

def is_maximo_linked():
    """Is there a Maximo instance currently linked?

    Notes
    -----
    Normally, Maximo APM PMI always links to a Maximo instance. In some cases, like early phase PoC, a special
    provisioning might be made without a Maximo linked to speed up the initial model training experiement. This
    method provides a way to check this condition.
    """
    if MAXIMO_LINKED is None:
        init_environ()

    return MAXIMO_LINKED is True


def set_asset_group_members(asset_group_id, df, siteid_column=default_site_column, assetid_column=default_asset_column, db=None, db_schema=None, drop_table_first=False):
    """Set asset group members from the given dataframe.

    The dataframe given is expected to have two columns: one for asset ID and the other for site ID. All assets
    in the dataframe are set to be part of the given asset_group_id.

    This function only works for local asset groups, that is, IDs not existing in Maximo. This is meant for quick
    trial or PoC style work so you don't need to manage Maximo side of work first. The transition from local asset
    group to actual remote one is transparent, as long as you create the asset group in Maximo (with the same ID),
    the local group definition no longer works and is overrides by the remote group's members.

    Parameters
    ----------
    asset_group_id : str
        The asset group ID.
    df : DataFrame
        The dataframe to get asset device mapping, assumed to have 4 mandatory columns.
    siteid_column : str, optional
        The column name in dataframe df representing the asset's site ID. Default is 'site'.
    assetid_column : str, optional
        The column name in dataframe df representing the asset's ID. Default is 'asset'.
    db_schema : str, optional
        The schema to be used. Default is None.
    drop_table_first : bool, optional
        Whether to drop asset device mapping table first. Default is False.
    """
    logger.debug('begin of set_asset_group_members')

    db_schema = get_as_schema(db_schema)
    if asset_group_id is None:
        raise ValueError('missing parameter asset_group_id=%s' % asset_group_id)
    if df is None:
        raise ValueError('missing parameter df=%s' % df)

    if get_maximo_asset_device_mappings(asset_group_id) is not None:
        raise RuntimeError('this function can only set local asset group, but asset_group_id=%s represents a remote asset group, please use another one instead' % asset_group_id)

    if db is None:
        db = _get_db()

    if drop_table_first:
        db.drop_table(table_name=asset_group_table_name, schema=db_schema)
        db.metadata = MetaData(db.engine)

    table = _get_asset_group_table(db=db, db_schema=db_schema)

    df = df.rename(columns={
        siteid_column: default_site_column,
        assetid_column: default_asset_column,
    })

    # make both siteid and assetnum all upper case since Maximo does that
    if not df.empty:
        df[default_site_column] = df[default_site_column].str.upper()
        df[default_asset_column] = df[default_asset_column].str.upper()

    df[default_asset_group_column] = asset_group_id
    df[default_assetid_column] = df[default_asset_column] + '-____-' + df[default_site_column]

    db.start_session()
    try:
        with db.engine.connect() as conn:
            conn.execute(table.delete().where(table.c[default_asset_group_column] == asset_group_id))

        logger.debug('in set_asset_group_members _write_dataframe ')

        _write_dataframe(df=df, table_name=asset_group_table_name, db=db, db_schema=db_schema)
    except:
        db.session.rollback()
        raise
    finally:
        db.commit()


def delete_asset_group_members(asset_group_id, db=None, db_schema=None):
    """Delete all asset group members of the given asset_group_id.

    This function only works for local asset groups, that is, IDs not existing in Maximo. This is meant for quick
    trial or PoC style work so you don't need to manage Maximo side of work first. The transition from local asset
    group to actual remote one is transparent, as long as you create the asset group in Maximo (with the same ID),
    the local group definition no longer works and is overrides by the remote group's members.

    Parameters
    ----------
    asset_group_id : str
        The asset group ID.
    """

    db_schema = get_as_schema(db_schema)
    return set_asset_group_members(asset_group_id=asset_group_id, df=pd.DataFrame(data=[], columns=[default_site_column, default_asset_column]), db=db, db_schema=db_schema)


def create_assets(df, siteid_column, assetid_column):
    """Create the assets found in the given dataframe.

    The given dataframe is expected to have one column representing asset ID and another for site ID.

    Parameters
    ----------
    df : DataFrame
        The dataframe to get assets to create.
    siteid_column : str
        The column name in dataframe df representing the asset's site ID.
    assetid_column : str
        The column name in dataframe df representing the asset ID.
    """
    logger.info('Creating assets...')
    resp = setup_asset_attributes(df[[siteid_column, assetid_column]], siteid_column=siteid_column, assetid_column=assetid_column)
    logger.info('Asset creation complete.')

    return resp


def delete_assets(df, siteid_column, assetid_column):
    """Delete the assets in the given dataframe.

    The given dataframe is expected to have one column representing asset ID and another for site ID.

    Parameters
    ----------
    df : DataFrame
        The dataframe to get assets to delete.
    siteid_column : str
        The column name in dataframe df representing the asset's site ID.
    assetid_column : str
        The column name in dataframe df representing the asset ID.
    """

    if assetid_column is None or not isinstance(assetid_column, str) or len(assetid_column) == 0:
        raise ValueError('invalid parameter assetid_column=%s, must be a non-empty string' % assetid_column)
    if siteid_column is None or not isinstance(siteid_column, str) or len(siteid_column) == 0:
        raise ValueError('invalid parameter siteid_column=%s, must be a non-empty string' % siteid_column)

    # make both siteid and assetnum all upper case since Maximo does that
    if not df.empty:
        df = df.copy()
        df[siteid_column] = df[siteid_column].str.upper()
        df[assetid_column] = df[assetid_column].str.upper()

    asset_key_columns = [siteid_column, assetid_column]
    df_assets = df.groupby(asset_key_columns).size().reset_index()[asset_key_columns]

    assets = dict()
    for row in df_assets.itertuples():
        asset_id = '%s-____-%s' % (getattr(row, assetid_column), getattr(row, siteid_column))
        assets.setdefault(asset_id, dict())

    return delete_maximo_assets(assets)


def setup_asset_attributes(df, siteid_column, assetid_column, attribute_columns=None, parse_dates=None, rename_columns=None):
    """Setup asset attribute(s) from the given dataframe.

    Asset attributes are taken from the dataframe's columns, one column per attribute. The given dataframe is
    expected to have one column representing asset ID and another for site ID.

    If no sepcific attribute columns are specified, then all columns except siteid_column and assetid_column are used
    as asset attributes. Note that if there are multiple rows in the dataframe for the same asset, the last
    row would be used for that asset.

    If assets are not already created, this function also creates the assets.

    Parameters
    ----------
    df : DataFrame
        The dataframe to get asset attributes, one column per attribute.
    siteid_column : str
        The column name in dataframe df representing the asset's site ID.
    assetid_column : str
        The column name in dataframe df representing the asset ID.
    attriute_columns : list of str, optional
        The list of column names containing asset attributes, one per
        attribute. If not given or an empty list, all except assetid_column and siteid_column are treated as
        asset attribute columns.
    parse_dates : list of str, optional
        The column names in dataframe df to be converted to datetime by
        pd.to_datetime. It is expected to be in ISO 8601 format (ex. '2019-07-29 01:59:00-05:00'). It can include
        timezone info, otherwise is in UTC when no timezone info is present.
    rename_columns : dict of (str, str), optional
        The keys of the dict are the dataframe df column names while the values are
        the actual asset attribute names to be set.
    """

    if assetid_column is None or not isinstance(assetid_column, str) or len(assetid_column) == 0:
        raise ValueError('invalid parameter assetid_column=%s, must be a non-empty string' % assetid_column)
    if siteid_column is None or not isinstance(siteid_column, str) or len(siteid_column) == 0:
        raise ValueError('invalid parameter siteid_column=%s, must be a non-empty string' % siteid_column)
    if attribute_columns is None:
        attribute_columns = []
    if not isinstance(attribute_columns, list) or not all([isinstance(attr, str) for attr in attribute_columns]):
        raise ValueError('invalid parameter attribute_columns=%s, must be a list of string' % attribute_columns)
    if parse_dates is not None and (not isinstance(parse_dates, list) or not all([isinstance(attr, str) for attr in parse_dates])):
        raise ValueError('invalid parameter parse_dates=%s, must be a list of string' % str(parse_dates))
    if rename_columns is None:
        rename_columns = dict()
    if rename_columns is not None and not isinstance(rename_columns, dict):
        raise ValueError('invalid parameter rename_columns=%s' % str(rename_columns))

    # if attribute_columns given, select only the subset of dataframe columns
    if attribute_columns is not None:
        attribute_columns = list({col for col in attribute_columns if col not in [assetid_column, siteid_column]})
        df = df[[assetid_column, siteid_column] + attribute_columns]

    if rename_columns is not None and len(rename_columns) > 0:
        df = df.rename(columns=rename_columns)
        if assetid_column in rename_columns:
            assetid_column = rename_columns[assetid_column]
        if siteid_column in rename_columns:
            siteid_column = rename_columns[siteid_column]
        if parse_dates is not None:
            parse_dates = [rename_columns[col] if col in rename_columns else col for col in parse_dates]

    if parse_dates is not None:
        for col in parse_dates:
            if col in df.columns:
                # currently both installdate and estendoflife are just "date" without time, so reset to day boundary
                df[col] = pd.to_datetime(df[col], utc=True).dt.floor('d')

    # make both siteid and assetnum all upper case since Maximo does that
    if not df.empty:
        df = df.copy()
        df[siteid_column] = df[siteid_column].str.upper()
        df[assetid_column] = df[assetid_column].str.upper()

    assets = dict()
    for row in df.itertuples():
        asset_id = '%s-____-%s' % (getattr(row, assetid_column), getattr(row, siteid_column))
        asset_attributes = assets.setdefault(asset_id, dict())
        for col, dtype in df.dtypes.to_dict().items():
            if col not in {assetid_column, siteid_column}:
                val = getattr(row, col)
                if pd.notna(val):
                    if is_datetime64_any_dtype(dtype):
                        # because of the "date" only nature, timezone conversion cannot work properly with Maximo
                        # we just strip out the timezone info when interchanging with Maximo
                        asset_attributes[col] = val.strftime('%Y-%m-%dT%H:%M:%S')
                    else:
                        asset_attributes[col] = val

    return create_maximo_assets(assets)


def get_asset_attributes(assets, data_items, df_id_column):
    """Get asset attributes as a dataframe.

    Parameters
    ----------
    assets : `list` of `dict`
        The list of assets of which attributes to be retrieved. Assets are identified by a dict,
        with two keys: 'assetNum' and 'siteId'. Example, [{'assetNum':'pump1','siteId':'BEDFORD'}, ...]
    data_items : `list` of `str`
        The list of asset attributes to be retrieved.
    df_id_column : `str`
        The name of the asset ID column to be used in the returned dataframe. Note that if this name
        collides with any asset attribute name, the asset attribute will not be retrieved.

    Returns
    -------
    DataFrame
        A dataframe with one column per data item, and with each asset having one row. Column df_id_column
        in the returned dataframe contains the 'full' asset ID whic is in the form of '<asset_id>-____-<site_id>'.
    """

    if assets is None or not isinstance(assets, list) or len(assets) == 0:
        raise ValueError('invalid parameter assets=%s' % assets)
    if not all([isinstance(asset, dict) and 'assetNum' in asset and 'siteId' in asset for asset in assets]):
        raise ValueError('invalid parameter assets=%s' % assets)

    if data_items is None or len(data_items) == 0:
        raise ValueError('invalid parameter data_items=%s' % data_items)

    if df_id_column is None or len(df_id_column) == 0:
        raise ValueError('invalid parameter df_id_column=%s' % df_id_column)

    # TODO how can we validate data_items whether they exist in Maximo?

    asset_metadata_list = []
    for asset_id, asset_meta in get_maximo_asset_metadata(assets, data_items).items():
        asset_meta = asset_meta.copy()
        asset_meta.update({
            df_id_column: asset_id,
        })
        # currently supported attribtues are all just "date" without time, Maximo does some
        # time stipping which makes timezone conversion impossible. so we always just ignore
        # timezone when dealing with such "date" only datetime attributes.
        for date_field in ['ahcapmnextduedate', 'estendoflife', 'expectedlifedate', 'installdate', 'warrantyexpdate']:
            if date_field in asset_meta:
                asset_meta[date_field] = pd.to_datetime(asset_meta[date_field]).tz_localize(None)
        for date_field in ['ahcalastcompdate', 'changedate', 'lastauditdate', 'lastahscoredate', 'pmnextfaildt', 'pluscduedate', 'statusdate']:
            if date_field in asset_meta:
                asset_meta[date_field] = pd.to_datetime(asset_meta[date_field])
        asset_metadata_list.append(asset_meta)

    if len(asset_metadata_list) == 0:
        return None

    df = pd.DataFrame(asset_metadata_list)

    # for missing data items, add NA columns for them
    for name in data_items:
        if name not in df.columns:
            df[name] = None

    df = df[[df_id_column] + data_items]

    return df


def import_asset_failure_history(df, siteid_column, assetid_column, faildate_column, failurecode_column=None, problemcode_column=None, default_failurecode='PUMPS', default_problemcode='STOPPED', wonum_prefix='APM'):
    """Import asset failure history from the given dataframe.

    Asset failure history are composed of all the asset's associated work orders having a failure date and a problem code.

    This function creates work orders with randomly generated work order number, prefixed by wonum_prefix. A failure work
    order's failure code and problem code can be from the given dataframe, or when not given, can have a default code used.

    Parameters
    ----------
    df : DataFrame
        The dataframe to get asset failure history.
    siteid_column : str
        The column name in dataframe df representing the asset's site ID.
    assetid_column : str
        The column name in dataframe df representing the asset ID.
    faildate_column : str
        The column name in dataframe df representing the asset's failure date(s). It is
        expected to be in ISO 8601 format (ex. '2019-07-29 01:59:00-05:00'). It can includes timezone info,
        otherwise is in UTC when no timezone info is present.
    failurecode_column : str, optional
        The column name in dataframe df representing the failure code. When not
        given, all the failurecode uses the default_failurecode. Default is None.
    problemcode_column : str, optional
        The column name in dataframe df representing the problem code. When not
        given, all the problemcode uses the default_problemcode. Default is None.
    default_failurecode : str, optional
        The default failurecode to use for the created asset failure work orders when
        either failurecode_column is not given or when no value found in the column. Default is 'PUMPS'.
    default_problemcode : str, optional
        The default problemcode to use for the created asset failure work orders when
        either failurecode_column is not given or when no value found in the column. Default is 'STOPPED'.
    wonum_prefix : str, optional
        The prefix for the asset failure work order numbers (wonum). The lenght of the wonum
        used is 10 and a random number is used for the rest of the string (excluding the wonum_prefix part). Default
        is 'APM'.
    """

    logger.info('Importing asset failure history...')

    if assetid_column is None or not isinstance(assetid_column, str) or len(assetid_column) == 0:
        raise ValueError('invalid parameter assetid_column=%s, must be a non-empty string' % assetid_column)
    if siteid_column is None or not isinstance(siteid_column, str) or len(siteid_column) == 0:
        raise ValueError('invalid parameter siteid_column=%s, must be a non-empty string' % siteid_column)
    if faildate_column is None or not isinstance(faildate_column, str) or len(faildate_column) == 0:
        raise ValueError('invalid parameter faildate_column=%s, must be a non-empty string' % faildate_column)

    # drop rows without faildate
    df = df.dropna(subset=[faildate_column])

    asset_failure_history = {}
    for row in df.itertuples():
        asset_id = '%s-____-%s' % (getattr(row, assetid_column).upper(), getattr(row, siteid_column).upper())
        failure_event = {
            'faildate': pd.to_datetime(getattr(row, faildate_column), utc=True).strftime('%Y-%m-%dT%H:%M:%S+00:00'),
            'failurecode': default_failurecode if failurecode_column is None else getattr(row, failurecode_column, default_failurecode),
            'problemcode': default_problemcode if problemcode_column is None else getattr(row, problemcode_column, default_problemcode),
            #'causecode': default_causecode if causecode_column is None else getattr(row, causecode_column, default_causecode),
        }
        asset_failure_history.setdefault(asset_id, []).append(failure_event)

    resp = create_maximo_asset_failure_history(asset_failure_history, default_failurecode=default_failurecode, default_problemcode=default_problemcode, wonum_prefix=wonum_prefix)

    if resp is not None and resp.status_code<300:
        logger.debug('Importing asset failure history response: status_code=%s, response_text=%s', resp.status_code, resp.text)
        logger.info('Succesfully finished execution of importing asset failure history')

    logger.info('Finished importing asset failure history.')

    return resp


def delete_asset_failure_history(df, siteid_column, assetid_column, wonum_prefix='APM'):
    """Delete the complete failure history of the assets from the given dataframe.

    The given dataframe is expected to have one column representing asset ID and another for site ID.

    Note that this is bulk deletion of all assets' failure work orders with work order number prefixed with
    the given wonum_prefix.

    Parameters
    ----------
    df : DataFrame
        The dataframe to get assets to delete.
    siteid_column : str
        The column name in dataframe df representing the asset's site ID.
    assetid_column : str
        The column name in dataframe df representing the asset ID.
    wonum_prefix : str, optional
        The work order number prefix used for search asset failure work
        orders to delete. Default is 'APM'.
    """
    logger.info('Deleting asset failure history...')

    if assetid_column is None or not isinstance(assetid_column, str) or len(assetid_column) == 0:
        raise ValueError('invalid parameter assetid_column=%s, must be a non-empty string' % assetid_column)
    if siteid_column is None or not isinstance(siteid_column, str) or len(siteid_column) == 0:
        raise ValueError('invalid parameter siteid_column=%s, must be a non-empty string' % siteid_column)

    asset_failure_history = {}
    for row in df.itertuples():
        asset_id = '%s-____-%s' % (getattr(row, assetid_column).upper(), getattr(row, siteid_column).upper())
        asset_failure_history.setdefault(asset_id, []).append({})

    resp = delete_maximo_asset_failure_history(asset_failure_history, wonum_prefix=wonum_prefix)
    if resp is not None:
        logger.info('Finished deleting asset failure history: status_code=%s, response_text=%s',  resp.status_code, resp.text)

    return resp


def get_asset_failure_history(assets, data_items, df_id_column, df_timestamp_name, start_ts=None, end_ts=None, separate_asset_site_columns=False, async_mode=True):
    """Get asset failure history as a dataframe.

    Parameters
    ----------
    assets : `list` of `dict`
        The list of assets of which attributes to be retrieved. Assets are identified by a dict,
        with two keys: 'assetNum' and 'siteId'. Example, [{'assetNum':'pump1','siteId':'BEDFORD'}, ...]
    data_items : `list` of `str`
        The list of asset failure event attributes to be retrieved. Return all asset failure event attributes
        available if given None or empty list.
    df_id_column : str
        The name of the asset ID column to be used in the returned dataframe.
    df_timestamp_name : str
        The name of the timestamp column to be used in the returned dataframe.
    start_ts : str, optional
        The start of the range of the history to be retrieved, inclusive. Default is None, meaning all the way
        back the earliest record. It is given in the format like '2019-01-31 06:51:34.234561', with time portion
        optional.
    end_ts : str, optional
        The end of the range of the history to be retrieved, exclusive. Default is None, meaning all the way
        to the latest record. It is given in the format like '2019-01-31 06:51:34.234561', with time portion
        optional.
    separate_asset_site_columns : bool, optional
        Whether to include 2 extra separate columns, one for short asset ID and the other for site ID, in the
        returned dataframe. Default is False.

    Returns
    -------
    DataFrame
        A dataframe with column df_id_column and column df_timestamp_name for asset "full" ID and timestamp of
        asset failure history. The "full" asset ID whic is in the form of "<asset_id>-____-<site_id>". If
        separate_asset_site_columns is True, additional 2 columns are added, one for "short" asset ID and the
        other for site ID.
    """

    if assets is None or not isinstance(assets, list) or len(assets) == 0:
        raise ValueError('invalid parameter assets=%s' % assets)
    if not all([isinstance(asset, dict) and 'assetNum' in asset and 'siteId' in asset for asset in assets]):
        raise ValueError('invalid parameter assets=%s' % assets)

    if data_items is None or len(data_items) == 0:
        data_items = None

    if df_id_column is None or len(df_id_column) == 0:
        raise ValueError('invalid parameter df_id_column=%s' % df_id_column)

    if df_timestamp_name is None or len(df_timestamp_name) == 0:
        raise ValueError('invalid parameter df_timestamp_name=%s' % df_timestamp_name)

    session = FuturesSession() if async_mode else None
    failure_history_requests = []
    failure_history_base = []
    for asset in assets:
        asset["siteId"] = asset["siteId"].upper()
        asset["assetNum"] = asset["assetNum"].upper()

        body = dict()
        body["assetNum"] = asset["assetNum"]
        body["siteId"] = asset["siteId"]
        base_dict = {
            df_id_column: asset["assetNum"] + '-____-' + asset['siteId']
        }
        if separate_asset_site_columns:
            base_dict[default_site_column] = asset['siteId']
            base_dict[default_asset_column] = asset["assetNum"]
        failure_history_base.append(base_dict)
        failure_history_requests.append(get_maximo_failure_history_data(body, session=session))

    failure_columns = {default_faildate_column, default_failurecode_column, default_problemcode_column}
    failurecode_present = False
    failure_history = []
    for idx, future in enumerate(failure_history_requests):
        response = future.result() if async_mode else future
        if response is None or response.status_code not in (requests.codes.ok, requests.codes.created, requests.codes.accepted, requests.codes.no_content):
            logger.warning('error getting failure history for %s', failure_history_base[idx][df_id_column])
            continue

        logger.debug('response=%s', response)

        for one_failure_history_record in response.json()["pastFailure"]:
            logger.debug('Received asset failure record: %s', one_failure_history_record)
            fh = failure_history_base[idx].copy()

            try:
                fh[default_faildate_column] = one_failure_history_record['date']
            except:
                logger.debug('WARNING: the faildate is null. You need to set classcode and problemcode of work order in the Maximo Manage.')
                continue
            
            if one_failure_history_record.get('failurecode', None) is not None:
                fh[default_failurecode_column] = one_failure_history_record['failurecode']
                #failure_history.append(fh)
                failurecode_present = True

            try:
                full_problem_code = one_failure_history_record['classcode']+'/' +one_failure_history_record['problemcode']
            except:
                logger.warning('The classcode and problemcode of failure history of work order must be set in the Maximo Manage but they were not for this record: %s', one_failure_history_record)
                pass

            fh[default_problemcode_column] = full_problem_code
            try:
                full_cause_code=one_failure_history_record['classcode']+'/' +one_failure_history_record['problemcode']+'/'+ one_failure_history_record['causecode']
                logger.debug('Computed full_cause_code=%s', full_cause_code)
                fh[default_causecode_column] = full_cause_code
                failure_columns.add(default_causecode_column)
            except:
                pass

            try:
                full_remedy_code=one_failure_history_record['classcode']+'/' +one_failure_history_record['problemcode']+'/'+ one_failure_history_record['causecode'] + '/' + one_failure_history_record['remedycode']
                logger.debug('Computed full_remedy_code=%s', full_remedy_code)
                fh[default_remedycode_column] = full_remedy_code
                failure_columns.add(default_remedycode_column)
            except:
                pass

            failure_history.append(fh)


            """ if one_failure_history_record.get('problemcode', None) is not None:
                fh[default_problemcode_column] = one_failure_history_record['problemcode']
                failure_history.append(fh)


            if one_failure_history_record.get('causecode', None) is not None:
                fh[default_causecode_column] = one_failure_history_record['causecode']
                failure_history.append(fh)
                logger.debug('failure_history = %s', failure_history)
            else:
                logger.debug('do not have causecode') """


            #we only append it if there is data in the failure history
            #failure_history.append(fh)


    if not failurecode_present:
        failure_columns.remove(default_failurecode_column)



    # filter columns not requested
    if data_items is not None:
        failure_columns = failure_columns & set(data_items)
    logger.debug('Found these failure columns: %s', failure_columns)

    id_columns = [df_id_column, default_site_column, default_asset_column] if separate_asset_site_columns else [df_id_column]
    index_columns = id_columns + [df_timestamp_name]

    if len(failure_history) > 0:
        df = pd.DataFrame(failure_history)

        # Maximo returns datetime in local timezone, we need to first convert it to UTC then convert it back to naive local time
        df[default_faildate_column] = pd.to_datetime(df[default_faildate_column], utc=True).dt.tz_localize(None)
        df[df_timestamp_name] = df[default_faildate_column]

        df = df[index_columns + list(failure_columns)]
    else:
        df = pd.DataFrame(columns=index_columns+list(failure_columns))

    # filter by time range
    if start_ts is not None or end_ts is not None:
        logger.debug('filtering asset time-series data: start_ts=%s, end_ts=%s', start_ts, end_ts)
        if start_ts is not None and end_ts is not None:
            df = df[(df[df_timestamp_name] >= pd.to_datetime(start_ts)) & (df[df_timestamp_name] < pd.to_datetime(end_ts))]
        elif end_ts is not None:
            df = df[(df[df_timestamp_name] < pd.to_datetime(end_ts))]
        else:
            df = df[(df[df_timestamp_name] >= pd.to_datetime(start_ts))]

    df.dropna(inplace=True)
    logger.debug('Retrieved asset failure history. Final df=%s', log_df_info(df, head=0, logger=logger, log_level=logging.DEBUG))

    return df


def get_asset_corrective_maintenance_history(assets, data_items, df_id_column, df_timestamp_name, start_ts=None, end_ts=None, separate_asset_site_columns=False, async_mode=True,cm_code='CM'):
    """Get asset failure history as a dataframe.

    Parameters
    ----------
    assets : `list` of `dict`
        The list of assets of which attributes to be retrieved. Assets are identified by a dict,
        with two keys: 'assetNum' and 'siteId'. Example, [{'assetNum':'pump1','siteId':'BEDFORD'}, ...]
    data_items : `list` of `str`
        The list of asset failure event attributes to be retrieved. Return all asset failure event attributes
        available if given None or empty list.
    df_id_column : str
        The name of the asset ID column to be used in the returned dataframe.
    df_timestamp_name : str
        The name of the timestamp column to be used in the returned dataframe.
    start_ts : str, optional
        The start of the range of the history to be retrieved, inclusive. Default is None, meaning all the way
        back the earliest record. It is given in the format like '2019-01-31 06:51:34.234561', with time portion
        optional.
    end_ts : str, optional
        The end of the range of the history to be retrieved, exclusive. Default is None, meaning all the way
        to the latest record. It is given in the format like '2019-01-31 06:51:34.234561', with time portion
        optional.
    separate_asset_site_columns : bool, optional
        Whether to include 2 extra separate columns, one for short asset ID and the other for site ID, in the
        returned dataframe. Default is False.

    Returns
    -------
    DataFrame
        A dataframe with column df_id_column and column df_timestamp_name for asset "full" ID and timestamp of
        asset failure history. The "full" asset ID whic is in the form of "<asset_id>-____-<site_id>". If
        separate_asset_site_columns is True, additional 2 columns are added, one for "short" asset ID and the
        other for site ID.
    """

    if assets is None or not isinstance(assets, list) or len(assets) == 0:
        raise ValueError('invalid parameter assets=%s' % assets)
    if not all([isinstance(asset, dict) and 'assetNum' in asset and 'siteId' in asset for asset in assets]):
        raise ValueError('invalid parameter assets=%s' % assets)

    if data_items is None or len(data_items) == 0:
        data_items = None

    if df_id_column is None or len(df_id_column) == 0:
        raise ValueError('invalid parameter df_id_column=%s' % df_id_column)

    if df_timestamp_name is None or len(df_timestamp_name) == 0:
        raise ValueError('invalid parameter df_timestamp_name=%s' % df_timestamp_name)

    session = FuturesSession() if async_mode else None
    failure_history_requests = []
    failure_history_base = []
    for asset in assets:
        asset["siteId"] = asset["siteId"].upper()
        asset["assetNum"] = asset["assetNum"].upper()

        body = dict()
        body["assetNum"] = asset["assetNum"]
        body["siteId"] = asset["siteId"]
        base_dict = {
            df_id_column: asset["assetNum"] + '-____-' + asset['siteId']
        }
        if separate_asset_site_columns:
            base_dict[default_site_column] = asset['siteId']
            base_dict[default_asset_column] = asset["assetNum"]
        failure_history_base.append(base_dict)
        failure_history_requests.append(get_maximo_corrective_maintenance_data(body, session=session,cm_code=cm_code))

    failure_columns = {default_faildate_column, default_failurecode_column, default_problemcode_column}
    failurecode_present = False
    failure_history = []

    logger.debug('Iterating over failure history requests...')
    for idx, future in enumerate(failure_history_requests):
        response = future.result() if async_mode else future
        if response is None or response.status_code not in (requests.codes.ok, requests.codes.created, requests.codes.accepted, requests.codes.no_content):
            logger.warning('error getting failure history for %s', failure_history_base[idx][df_id_column])
            continue

        logger.debug('Current failure history request: response=%s', response)

        logger.debug('Now iterating over failure history records...')
        for one_failure_history_record in response.json()["pastFailure"]:
            logger.debug('one_failure_history_record=%s', one_failure_history_record)
            fh = failure_history_base[idx].copy()
            #fh[default_faildate_column] = one_failure_history_record['date']
            try:
                fh[default_faildate_column] = one_failure_history_record['date']
            except:
                logger.debug('WARNING: the faildate is null. You need to set classcode and problemcode of work order in the Maximo Manage.')
                continue
            
            if one_failure_history_record.get('failurecode', None) is not None:
                fh[default_failurecode_column] = one_failure_history_record['failurecode']
                #failure_history.append(fh)
                failurecode_present = True

            try:
                full_problem_code = one_failure_history_record['classcode']+'/' +one_failure_history_record.get('problemcode','CM')
            except:
                logger.debug('WARNING: the classcode and problemcode of failure history of work order must be set in the Maximo Manage.')
                pass

            fh[default_problemcode_column] = full_problem_code
            try:
                full_cause_code=one_failure_history_record['classcode']+'/' +one_failure_history_record['problemcode']+'/'+ one_failure_history_record['causecode']
                logger.debug('Failure cause code: full_cause_code=%s', full_cause_code)
                fh[default_causecode_column] = full_cause_code
                failure_columns.add(default_causecode_column)
            except:
                pass

            try:
                full_remedy_code=one_failure_history_record['classcode']+'/' +one_failure_history_record['problemcode']+'/'+ one_failure_history_record['causecode'] + '/' + one_failure_history_record['remedycode']
                logger.debug('Failure remedy code: full_remedy_code=%s', full_remedy_code)
                fh[default_remedycode_column] = full_remedy_code
                failure_columns.add(default_remedycode_column)
            except:
                pass

            failure_history.append(fh)


            """ if one_failure_history_record.get('problemcode', None) is not None:
                fh[default_problemcode_column] = one_failure_history_record['problemcode']
                failure_history.append(fh)


            if one_failure_history_record.get('causecode', None) is not None:
                fh[default_causecode_column] = one_failure_history_record['causecode']
                failure_history.append(fh)
                logger.debug('failure_history = %s', failure_history)
            else:
                logger.debug('do not have causecode') """


            #we only append it if there is data in the failure history
            #failure_history.append(fh)


    if not failurecode_present:
        failure_columns.remove(default_failurecode_column)



    # filter columns not requested
    logger.debug('Requested failure event attributes: data_item=%s', data_items)
    logger.debug('Failure columns to check: failure_columns=%s', failure_columns)
    if data_items is not None:
        failure_columns = failure_columns & set(data_items)

    id_columns = [df_id_column, default_site_column, default_asset_column] if separate_asset_site_columns else [df_id_column]
    index_columns = id_columns + [df_timestamp_name]

    if len(failure_history) > 0:
        df = pd.DataFrame(failure_history)

        # Maximo returns datetime in local timezone, we need to first convert it to UTC then convert it back to naive local time
        df[default_faildate_column] = pd.to_datetime(df[default_faildate_column], utc=True).dt.tz_localize(None)
        df[df_timestamp_name] = df[default_faildate_column]

        df = df[index_columns + list(failure_columns)]
    else:
        df = pd.DataFrame(columns=index_columns+list(failure_columns))

    # filter by time range
    if start_ts is not None or end_ts is not None:
        logger.debug('filtering asset time-series data: (start_ts=%s, end_ts=%s)', start_ts, end_ts)
        if start_ts is not None and end_ts is not None:
            df = df[(df[df_timestamp_name] >= pd.to_datetime(start_ts)) & (df[df_timestamp_name] < pd.to_datetime(end_ts))]
        elif end_ts is not None:
            df = df[(df[df_timestamp_name] < pd.to_datetime(end_ts))]
        else:
            df = df[(df[df_timestamp_name] >= pd.to_datetime(start_ts))]

    logger.debug('Asset failure history before call to dropna: df=%s', log_df_info(df, head=66, logger=logger, log_level=logging.DEBUG))
    logger.debug('Final failure columns: failure_columns=%s', failure_columns)


    df.dropna(inplace=True)
    logger.debug('Final asset failure history DF: df=%s', log_df_info(df, head=66, logger=logger, log_level=logging.DEBUG))

    return df


def set_asset_device_mappings(df, siteid_column=default_site_column, assetid_column=default_asset_column, devicetype_column=default_devicetype_column, deviceid_column=default_deviceid_column, db=None, db_schema=None, delete_df_asset_first=True, drop_table_first=False):
    """Set asset device mappings from the given dataframe.

    The asset device mappings set by this method is for PMI usage only. At runtime, in additional to PMI's
    asset device mappings, those from AHI side is also loaded to override PMI's. Think of PMI's mappings
    as the (device) type level applicable to all type's attributes while AHI's allows finer grained control.

    Typically, AHI side's mappings are meant for AHI's own usage, like for IOT meter reading value. PMI
    side's is purely for PMI's asset group model pipeline usage, never visible to AHI. However, PMI also
    loads AHI side's mapping to create a composite view of the two sides.

    There are options for controlling how to deal with conflicts, when some assets from the given dataframe
    conflict with existing mappings. You can choose to drop the whole table first, or delete all data first,
    or delete only mappings of the assets appear in the given dataframe, or simply appending (when all three
    options are False).

    Parameters
    ----------
    df : DataFrame
        The dataframe to get asset device mapping, assumed to have 4 mandatory columns.
    siteid_column : str, optional
        The column name in dataframe df representing the asset's site ID. Default is 'site'.
    assetid_column : str, optional
        The column name in dataframe df representing the asset's ID. Default is 'asset'.
    devicetype_column : str, optional
        The column name in dataframe df representing the device' type. Default is 'devicetype'.
    deviceid_column : str, optional
        The column name in dataframe df representing the device's ID. Default is 'deviceid'.
    db_schema : str, optional
        The schema to be used. Default is None.
    delete_df_asset_first : bool, optional
        Whether to delete all data of the assets appearing in the dataframe df first. Default is True.
    drop_table_first : bool, optional
        Whether to drop asset device mapping table first. Default is False.
    """
    logger.info('Setting asset device mappings...')

    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    cleanup_df_assets = False
    if drop_table_first:
        db.drop_table(table_name=asset_device_mappings_table_name, schema=db_schema)
        db.metadata = MetaData(db.engine)
    elif delete_df_asset_first:
        cleanup_df_assets = True

    table = _get_asset_device_mapping_table(db=db, db_schema=db_schema)

    df = df.rename(columns={
        siteid_column: default_site_column,
        assetid_column: default_asset_column,
        devicetype_column: default_devicetype_column,
        deviceid_column: default_deviceid_column,
    })

    # make both siteid and assetnum all upper case since Maximo does that
    if not df.empty:
        df[default_site_column] = df[default_site_column].str.upper()
        df[default_asset_column] = df[default_asset_column].str.upper()

    key_columns = [default_site_column, default_asset_column, default_devicetype_column, default_deviceid_column]
    df = df.groupby(key_columns).size().reset_index()[key_columns]

    db.start_session()
    try:
        if cleanup_df_assets:
            asset_key_columns = [default_site_column, default_asset_column]
            df_assets = df.groupby(asset_key_columns).size().reset_index()[asset_key_columns]
            for row in df_assets.itertuples():
                row = row._asdict()
                with db.engine.connect() as conn: 
                    conn.execute(table.delete().where(and_(table.c[default_site_column] == row[default_site_column], table.c[default_asset_column] == row[default_asset_column])))
                    conn.commit()

        _write_dataframe(df=df, table_name=asset_device_mappings_table_name, db=db, db_schema=db_schema)
        logger.info('Succesfully finished execution of setting asset device mappings')
    except:
        db.session.rollback()
        raise
    finally:
        db.commit()

    logger.info('Finished setting asset device mappings.')


def delete_asset_device_mappings(df, siteid_column=default_site_column, assetid_column=default_asset_column, db=None, db_schema=None):
    """Delete asset device mappings from the given dataframe.

    It deletes all the mappings of the assets appera in the given dataframe.

    Parameters
    ----------
    df : DataFrame
        The dataframe to get asset device mapping, assumed to have 4 mandatory columns.
    siteid_column : str, optional
        The column name in dataframe df representing the asset's site ID. Default is 'site'.
    assetid_column : str, optional
        The column name in dataframe df representing the asset's ID. Default is 'asset'.
    db_schema : str, optional
        The schema to be used. Default is None.
    """
    logger.info('Deleting asset device mappings...')

    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    table = _get_asset_device_mapping_table(db=db, db_schema=db_schema)

    # make both siteid and assetnum all upper case since Maximo does that
    if not df.empty:
        df = df.copy()
        df[siteid_column] = df[siteid_column].str.upper()
        df[assetid_column] = df[assetid_column].str.upper()

    asset_key_columns = [siteid_column, assetid_column]
    df_assets = df.groupby(asset_key_columns).size().reset_index()[asset_key_columns]

    db.start_session()
    try:
        for row in df_assets.itertuples():
            row = row._asdict()
            with db.engine.connect() as conn: 
                conn.execute(table.delete().where(and_(table.c[default_site_column] == row[siteid_column], table.c[default_asset_column] == row[assetid_column])))
                conn.commit()
    except:
        db.session.rollback()
        raise
    finally:
        db.commit()

    logger.info('Finished deleting asset device mappings.')

"""
Device related functions.
"""

def setup_iot_type(name, df, columns=None, deviceid_column=None, timestamp_column=None, timestamp_in_payload=False, parse_dates=None, rename_columns=None, write=None, use_wiotp=True, import_only=False, db=None, db_schema=None):
    """Create an IOT type for its IOT devices to be able to send events to WIOTP and then be saved in the data lake.

    It takes care of the creation of WIOTP device type and registration of devices on WIOTP. It also takes care of
    setting up the physical and logical interfaces and their mapping, based on the given datafame schema. Optionally,
    the dataframe's content can be bulk imported into the data lake directly.

    It is meant to be used to create from scratch, so if some type of the given name already exists, you will have
    to remove it first before using this method.

    Note that all column names given by several parameters are made all lower case first, including even df's columns.
    This is to avoid the trouble with sqlalchemy handling mixed-case column names. So, don't worry about case mis-match,
    everything about column names in this method is case insensitive.

    Parameters
    ----------
    name : str
        The name of the IOT type to create.
    df : DataFrame
        The data frame from which to infer the schema and optionally to bulk import data.
    columns : list of str, optoinal
        The list of columns in dataframe df to be used/selected. If not given, all
        columns of dataframe df is used.
    deviceid_column : str
        The column name in dataframe df representing the IOT device ID. It is assumed the whole
        dataframe is of the same type, hence no need for a type column. It is not required to be in columns.
    timestamp_column : str
        The column name in dataframe df representing the time-series time base. This column
        is converted to datetime by pd.to_datetime first. It is not required to be add to columns. This column is
        expected to be in ISO 8601 format (ex. '2019-07-29 01:59:00-05:00'). It can include timezone info, otherwise
        is assumed to be in UTC when no timezone info is present.
    timestamp_in_payload : bool, optional
        If True, the Watson IOT Platform's event receiving time is used as the
        time base for all time-series data processing. If False, timestamp_column from the event payload would be
        used as the time base instead. This is useful when you want to use the event originating time on device
        instead of Watson IOT Platform's event receiving time as the time base for time-series data handling.
        Note in the later case, you have to make sure that the IOT events included a datetime attribute
        named by the given timestamp_column, after renamed if specified, must be in the format
        '2019-07-31T01:35:59.123456+00:00', otherwise Watson IOT Platform cannot recognize it and would rejects
        such events. Default is False.
    parse_dates : list of str, optional
        The column names in dataframe df to be converted to datetime by
        pd.to_datetime. It is expected to be in ISO 8601 format (ex. '2019-07-29 01:59:00-05:00'). It can include
        timezone info, otherwise is assumed to be in UTC when no timezone info is present.
    rename_columns : dict of (str, str), optional
        The keys of the dict are the dataframe df column names while the values are
        the names to be used to create the IOT type.
    write : {'deletefirst', 'dropfirst', 'append'}, optional
        Whether to bulk write the dataframe df to the data lake. Default is None, not writing.
    use_wiotp : bool, optional
        Whether to register to Watson IOT Platform as device type. Default is True.
    import_only : bool, optional
        Whether only importing the given dataframe into data lake without any IOT type
        operation, assuming all has been setup already. Default is False.
    """
    logger.info('Creating IOT type...')

    db_schema = get_as_schema(db_schema)
    if name is None or not isinstance(name, str) or len(name) == 0:
        raise ValueError('invalid parameter name=%s, must be a non-empty string' % name)

    if not import_only and not is_local_mode():
        if use_wiotp and check_wiotp_device_type(name):
            raise RuntimeError('device_type=%s already exists, delete it first before calling this method' % name)
        if _check_entity_type(name):
            raise RuntimeError('entity_type=%s already exists, delete it first before calling this method' % name)

    if df is None or not isinstance(df, pd.DataFrame) or df.columns.empty:
        raise ValueError('invalid parameter df, must be a non-empty DataFrame')

    # first lower case all column names, it's complicated with sqlalchemy and database, we just take
    # a simple approach, data items are all lower case (case-insensitive)
    df.columns = map(str.lower, df.columns)

    if columns is not None and (not isinstance(columns, list) or not all([isinstance(col, str) for col in columns])):
        raise ValueError('invalid parameter columns=%s' % str(columns))

    # lower case all column names
    if columns is not None:
        columns = map(str.lower, columns)

    if deviceid_column is None or not isinstance(deviceid_column, str) or len(deviceid_column) == 0:
        raise ValueError('invalid parameter deviceid_column=%s, must be a non-empty string' % str(deviceid_column))

    # lower case all column names
    deviceid_column = deviceid_column.lower()

    if deviceid_column not in df.columns:
        raise ValueError('invalid parameter deviceid_column=%s, must be in df.columns' % str(deviceid_column))

    if timestamp_column is None or not isinstance(timestamp_column, str) or len(timestamp_column) == 0:
        raise ValueError('invalid parameter timestamp_column=%s' % str(timestamp_column))

    # lower case all column names
    timestamp_column = timestamp_column.lower()

    if timestamp_column not in df.columns:
        raise ValueError('invalid parameter timestamp_column=%s, must be in df.columns' % str(timestamp_column))

    if parse_dates is not None and (not isinstance(parse_dates, list) or not all([isinstance(attr, str) for attr in parse_dates])):
        raise ValueError('invalid parameter parse_dates=%s, must be a list of string' % str(parse_dates))
    if rename_columns is None:
        rename_columns = dict()
    if not isinstance(rename_columns, dict):
        raise ValueError('invalid parameter rename_columns=%s' % str(rename_columns))
    if write is not None and write.lower() not in ['deletefirst', 'dropfirst', 'append']:
        raise ValueError('invalid parameter write=%s, can only be one of [deletefirst, dropfirst, append] (case-insensitive)' % str(write))
    if import_only and write is None:
        raise ValueError('invalid parameter import_only=%s, write=%s cannot be None when it is True' % (import_only, write))

    # lower case all column names
    if parse_dates is not None:
        parse_dates = map(str.lower, parse_dates)
    rename_columns = {k.lower(): v.lower() for k, v in rename_columns.items()}

    # if columns given, select only the subset of columns
    if columns is not None:
        columns = set(columns)
        # make sure deviceid or timestamp column are selected as well
        columns.add(deviceid_column)
        columns.add(timestamp_column)
        columns = list(columns)
        df = df[columns]

    if deviceid_column is not None:
        rename_columns[deviceid_column] = default_deviceid_column

    if timestamp_in_payload != True:
        if timestamp_column in rename_columns and rename_columns[timestamp_column] != default_timestamp_column:
            raise ValueError('invalid renamed timestamp column "%s", when timestamp_in_payload is False, renamed timestamp column must be "%s". please remove the rename rule for the timestamp column.' % (rename_columns[timestamp_column], default_timestamp_column))
        else:
            # if using WIOTP receiving time, the time base column has to be renamed to the default one so
            # we can import dataframe data into the table
            rename_columns[timestamp_column] = default_timestamp_column

    if rename_columns is not None and len(rename_columns) > 0:
        df = df.rename(columns=rename_columns)
        if timestamp_column in rename_columns:
            timestamp_column = rename_columns[timestamp_column]
        if parse_dates is not None:
            parse_dates = [rename_columns[col] if col in rename_columns else col for col in parse_dates]

    if timestamp_in_payload == True and timestamp_column == default_timestamp_column:
        raise ValueError('invalid timestamp column "%s", after renamed if specified, when timestamp_in_payload is True, it must not be "%s"' % (str(timestamp_column), default_timestamp_column))

    df[timestamp_column] = pd.to_datetime(df[timestamp_column], utc=True)

    # a few columns are by default taken care of by iotfunctions hence we don't pass them
    # note that the default timestamp is not one of them
    system_columns = [default_deviceid_column]

    if timestamp_in_payload == True:
        # if using a timestamp column in payload, this column is set as AS metrics timestamp column for the
        # entity type, but we still add column default_timestamp_column for creating AS table/entity-type
        # (when WIOTP-ICS creation failed) and populate it with custom timestamp for WIOTP # receiving time
        df[default_timestamp_column] = df[timestamp_column]

    if parse_dates is not None:
        for col in parse_dates:
            if col in df.columns:
                df[col] = pd.to_datetime(df[col], utc=True)

    columns = [Column(col, DateTime() if is_datetime64_any_dtype(dtype) else
                    (Float() if is_float_dtype(dtype) else
                        (Float() if is_integer_dtype(dtype) else String(64))))
                for col, dtype in df.dtypes.to_dict().items() if col not in system_columns]

    options = {
        'columns': columns,
        'db': db,
        'db_schema': db_schema,
        'timestamp_column': timestamp_column,
        'write': write is not None,
        'df': df if write is not None else None,
        'delete_table_first': write is not None and write.lower() == 'deletefirst',
        'drop_table_first': write is not None and write.lower() == 'dropfirst',
        'use_wiotp': use_wiotp if not import_only else False,
        'devices': list(df[default_deviceid_column].unique())
    }

    resp = create_entity_type(name, **options)
    logger.info('Finished creating IOT type.')
    return resp


def setup_iot_type_v2(name, df, columns=None, deviceid_column=None, timestamp_column=None, timestamp_in_payload=False, parse_dates=None, rename_columns=None, write=None, use_wiotp=True, import_only=False, db=None, db_schema=None,batch_size=10):
    """Create an IOT type for its IOT devices to be able to send events to WIOTP and then be saved in the data lake.

    It takes care of the creation of WIOTP device type and registration of devices on WIOTP. It also takes care of
    setting up the physical and logical interfaces and their mapping, based on the given datafame schema. Optionally,
    the dataframe's content can be bulk imported into the data lake directly.

    It is meant to be used to create from scratch, so if some type of the given name already exists, you will have
    to remove it first before using this method.

    Note that all column names given by several parameters are made all lower case first, including even df's columns.
    This is to avoid the trouble with sqlalchemy handling mixed-case column names. So, don't worry about case mis-match,
    everything about column names in this method is case insensitive.

    Parameters
    ----------
    name : str
        The name of the IOT type to create.
    df : DataFrame
        The data frame from which to infer the schema and optionally to bulk import data.
    columns : list of str, optoinal
        The list of columns in dataframe df to be used/selected. If not given, all
        columns of dataframe df is used.
    deviceid_column : str
        The column name in dataframe df representing the IOT device ID. It is assumed the whole
        dataframe is of the same type, hence no need for a type column. It is not required to be in columns.
    timestamp_column : str
        The column name in dataframe df representing the time-series time base. This column
        is converted to datetime by pd.to_datetime first. It is not required to be add to columns. This column is
        expected to be in ISO 8601 format (ex. '2019-07-29 01:59:00-05:00'). It can include timezone info, otherwise
        is assumed to be in UTC when no timezone info is present.
    timestamp_in_payload : bool, optional
        If True, the Watson IOT Platform's event receiving time is used as the
        time base for all time-series data processing. If False, timestamp_column from the event payload would be
        used as the time base instead. This is useful when you want to use the event originating time on device
        instead of Watson IOT Platform's event receiving time as the time base for time-series data handling.
        Note in the later case, you have to make sure that the IOT events included a datetime attribute
        named by the given timestamp_column, after renamed if specified, must be in the format
        '2019-07-31T01:35:59.123456+00:00', otherwise Watson IOT Platform cannot recognize it and would rejects
        such events. Default is False.
    parse_dates : list of str, optional
        The column names in dataframe df to be converted to datetime by
        pd.to_datetime. It is expected to be in ISO 8601 format (ex. '2019-07-29 01:59:00-05:00'). It can include
        timezone info, otherwise is assumed to be in UTC when no timezone info is present.
    rename_columns : dict of (str, str), optional
        The keys of the dict are the dataframe df column names while the values are
        the names to be used to create the IOT type.
    write : {'deletefirst', 'dropfirst', 'append'}, optional
        Whether to bulk write the dataframe df to the data lake. Default is None, not writing.
    use_wiotp : bool, optional
        Whether to register to Watson IOT Platform as device type. Default is True.
    import_only : bool, optional
        Whether only importing the given dataframe into data lake without any IOT type
        operation, assuming all has been setup already. Default is False.
        set it to be False when you create new V2 device type
        set it to be True when you append record to existing V2 device type
    batch_size : Numeric, optional
        The default value is 10. If it is greater than 50, then it will be reset to be 50. If it causes gateway timeout error from Monitor API, reduce batch_size to 10.
    """

    db_schema = get_as_schema(db_schema)
    if name is None or not isinstance(name, str) or len(name) == 0:
        raise ValueError('invalid parameter name=%s, must be a non-empty string' % name)

    if not import_only and not is_local_mode():
        if use_wiotp and check_wiotp_device_type(name):
            raise RuntimeError('device_type=%s already exists, delete it first before calling this method' % name)
        if _check_entity_type(name):
            raise RuntimeError('entity_type=%s already exists, delete it first before calling this method' % name)

    if df is None or not isinstance(df, pd.DataFrame) or df.columns.empty:
        raise ValueError('invalid parameter df, must be a non-empty DataFrame')

    # first lower case all column names, it's complicated with sqlalchemy and database, we just take
    # a simple approach, data items are all lower case (case-insensitive)
    df.columns = map(str.lower, df.columns)

    if columns is not None and (not isinstance(columns, list) or not all([isinstance(col, str) for col in columns])):
        raise ValueError('invalid parameter columns=%s' % str(columns))

    # lower case all column names
    if columns is not None:
        columns = map(str.lower, columns)

    if deviceid_column is None or not isinstance(deviceid_column, str) or len(deviceid_column) == 0:
        raise ValueError('invalid parameter deviceid_column=%s, must be a non-empty string' % str(deviceid_column))

    # lower case all column names
    deviceid_column = deviceid_column.lower()

    if deviceid_column not in df.columns:
        raise ValueError('invalid parameter deviceid_column=%s, must be in df.columns' % str(deviceid_column))

    if timestamp_column is None or not isinstance(timestamp_column, str) or len(timestamp_column) == 0:
        raise ValueError('invalid parameter timestamp_column=%s' % str(timestamp_column))

    # lower case all column names
    timestamp_column = timestamp_column.lower()

    if timestamp_column not in df.columns:
        raise ValueError('invalid parameter timestamp_column=%s, must be in df.columns' % str(timestamp_column))

    if parse_dates is not None and (not isinstance(parse_dates, list) or not all([isinstance(attr, str) for attr in parse_dates])):
        raise ValueError('invalid parameter parse_dates=%s, must be a list of string' % str(parse_dates))
    if rename_columns is None:
        rename_columns = dict()
    if not isinstance(rename_columns, dict):
        raise ValueError('invalid parameter rename_columns=%s' % str(rename_columns))
    if write is not None and write.lower() not in ['deletefirst', 'dropfirst', 'append']:
        raise ValueError('invalid parameter write=%s, can only be one of [deletefirst, dropfirst, append] (case-insensitive)' % str(write))
    if import_only and write is None:
        raise ValueError('invalid parameter import_only=%s, write=%s cannot be None when it is True' % (import_only, write))

    # lower case all column names
    if parse_dates is not None:
        parse_dates = map(str.lower, parse_dates)
    rename_columns = {k.lower(): v.lower() for k, v in rename_columns.items()}

    # if columns given, select only the subset of columns
    if columns is not None:
        columns = set(columns)
        # make sure deviceid or timestamp column are selected as well
        columns.add(deviceid_column)
        columns.add(timestamp_column)
        columns = list(columns)
        df = df[columns]

    if deviceid_column is not None:
        rename_columns[deviceid_column] = default_deviceid_column

    if timestamp_in_payload != True:
        if timestamp_column in rename_columns and rename_columns[timestamp_column] != default_timestamp_column:
            raise ValueError('invalid renamed timestamp column "%s", when timestamp_in_payload is False, renamed timestamp column must be "%s". please remove the rename rule for the timestamp column.' % (rename_columns[timestamp_column], default_timestamp_column))
        else:
            # if using WIOTP receiving time, the time base column has to be renamed to the default one so
            # we can import dataframe data into the table
            rename_columns[timestamp_column] = default_timestamp_column

    if rename_columns is not None and len(rename_columns) > 0:
        df = df.rename(columns=rename_columns)
        if timestamp_column in rename_columns:
            timestamp_column = rename_columns[timestamp_column]
        if parse_dates is not None:
            parse_dates = [rename_columns[col] if col in rename_columns else col for col in parse_dates]

    if timestamp_in_payload == True and timestamp_column == default_timestamp_column:
        raise ValueError('invalid timestamp column "%s", after renamed if specified, when timestamp_in_payload is True, it must not be "%s"' % (str(timestamp_column), default_timestamp_column))

    df[timestamp_column] = pd.to_datetime(df[timestamp_column], utc=True)

    # a few columns are by default taken care of by iotfunctions hence we don't pass them
    # note that the default timestamp is not one of them
    system_columns = [default_deviceid_column]

    if timestamp_in_payload == True:
        # if using a timestamp column in payload, this column is set as AS metrics timestamp column for the
        # entity type, but we still add column default_timestamp_column for creating AS table/entity-type
        # (when WIOTP-ICS creation failed) and populate it with custom timestamp for WIOTP # receiving time
        df[default_timestamp_column] = df[timestamp_column]

    if parse_dates is not None:
        for col in parse_dates:
            if col in df.columns:
                df[col] = pd.to_datetime(df[col], utc=True)

    columns = [Column(col, DateTime() if is_datetime64_any_dtype(dtype) else
                    (Float() if is_float_dtype(dtype) else
                        (Float() if is_integer_dtype(dtype) else String(64))))
                for col, dtype in df.dtypes.to_dict().items() if col not in system_columns]
    logger.debug('columns=%s', columns)

    options = {
        'columns': columns,
        'db': db,
        'db_schema': db_schema,
        'timestamp_column': timestamp_column,
        'write': write is not None,
        'df': df if write is not None else None,
        'delete_table_first': write is not None and write.lower() == 'deletefirst',
        'drop_table_first': write is not None and write.lower() == 'dropfirst',
        'use_wiotp': use_wiotp if not import_only else False,
        'devices': list(df[default_deviceid_column].unique()),
        'batch_size' : batch_size
    }
    return create_entity_type_v2(name, **options)


def delete_iot_type(name, use_wiotp=True, db=None, db_schema=None):
    """Delete an IOT type.

    It takes care of the deregistration of devices on WIOTP and deletion of WIOTP device type. It also removes
    all the data and artifacts from the data lake belonging to the given IOT type.

    Parameters
    ----------
    name : str
        The name of the IOT type to delete.
    use_wiotp : bool, optional
        Whether to register to Watson IOT Platform as device type. Default is True.
    """

    db_schema = get_as_schema(db_schema)
    if use_wiotp and not is_local_mode():
        resp = delete_wiotp_devices(name)

    resp = _delete_entity_type(name, db=db, db_schema=db_schema)

    return resp


def delete_iot_type_v2(name, use_wiotp=True, db=None, db_schema=None):
    """Delete an IOT type.

    It takes care of the deregistration of devices on WIOTP and deletion of WIOTP device type. It also removes
    all the data and artifacts from the data lake belonging to the given IOT type.

    Parameters
    ----------
    name : str
        The name of the IOT type to delete.
    use_wiotp : bool, optional
        Whether to register to Watson IOT Platform as device type. Default is True.
    """

    db_schema = get_as_schema(db_schema)
    if use_wiotp and not is_local_mode():
        resp = delete_wiotp_devices(name)

    resp = _delete_entity_type_v2(name, db=db, db_schema=db_schema)

    return resp


"""
APM API.
"""

def _call(url, method, json=None, headers=None, **kwargs):
    return api_request(url, method=method, headers=headers, json=json, **kwargs)


def _normalize_apm_api_baseurl():
    if APM_API_BASEURL is None:
        raise RuntimeError('missing mandatory environment variable APM_API_BASEURL')

    apm_api_baseurl = APM_API_BASEURL
    if apm_api_baseurl[-1] == '/':
        apm_api_baseurl = apm_api_baseurl[:-1]

    return '%s/ibm/pmi/service/rest' % apm_api_baseurl


def _call_apm(url_path, method, json=None, headers=None, **kwargs):
    if any([env is None for env in [APM_ID, APM_API_BASEURL, APM_API_KEY]]):
        # if any environment variable is not set, check if we can get from AS constants
        if all([env is not None for env in [API_BASEURL, API_KEY, API_TOKEN, TENANT_ID]]):
            init_environ()
        else:
            raise RuntimeError('missing mandatory environment variable APM_ID, APM_API_BASEURL, APM_API_KEY')

    if APM_ID is None:
        raise RuntimeError('missing mandatory environment variable APM_ID')
    if APM_API_BASEURL is None:
        raise RuntimeError('missing mandatory environment variable APM_API_BASEURL')
    if APM_API_KEY is None:
        raise RuntimeError('missing mandatory environment variable APM_API_KEY')

    url = '%s%s' % (_normalize_apm_api_baseurl(), url_path)

    if headers is None:
        headers = {}

    headers['apmapitoken'] = APM_API_KEY

    ssl_verify = os.environ.get('SSL_VERIFY_APM', 'true')
    if isinstance(ssl_verify, str) and ssl_verify.lower() in ('true', 'yes', '1'):
        ssl_verify = True
    else:
        ssl_verify = False
    if '.svc:' in url:
        ssl_verify = False
    kwargs['ssl_verify'] = ssl_verify

    return _call(url=url, method=method, json=json, headers=headers, **kwargs)


def register_model_template(model_template_body):
    init_environ()
    return _call_apm(url_path='/ds/%s/template?instanceId=%s' % (APM_API_KEY, APM_ID), method='post', json=model_template_body)


def unregister_model_template(model_template_id):
    init_environ()
    return _call_apm(url_path='/ds/%s/template/%s?instanceId=%s' % (APM_API_KEY, model_template_id, APM_ID), method='delete')


def get_model_template():
    init_environ()
    return _call_apm(url_path='/model/template/?instanceId=%s' % (APM_ID), method='get')


def get_catalog_function(function_name=None):
    init_environ()
    if function_name is None:
        return _call_as(url_path='/function?customFunctionsOnly=true', method='get', api_type='catalog')
    else:
        return _call_as(url_path='/function?functionName=%s' % function_name, method='get', api_type='catalog')


def update_catalog_function(function_name, body):
    init_environ()
    return _call_as(url_path='/function/%s' % function_name, method='put', api_type='catalog', json=body)


def register_model_instance(asset_group_id, model_instance_body):
    init_environ()
    # TODO: Health Check API Here
    is_91_instance = str(monitor_health_check())
    return _call_apm(url_path='/ds/%s/%s/model?instanceId=%s&is91Instance=%s' % (APM_API_KEY, asset_group_id, APM_ID, is_91_instance), method='post', json=model_instance_body)


def unregister_model_instance(asset_group_id, model_instance_id, force=True):
    init_environ()
    return _call_apm(url_path='/ds/%s/%s/model/%s?instanceId=%s&force=%s' % (APM_API_KEY, asset_group_id, model_instance_id, APM_ID, 'true' if force is True else 'false'), method='delete')


def enable_model_instance(asset_group_id, model_instance_id, enabled=True, schedule=None, backtrack=None):
    body = {
        'enabled': enabled
    }
    if enabled == True:
        body['schedule'] = schedule
        body['backtrack'] = backtrack

    init_environ()
    # TODO: Health Check API Here
    is_91_instance = str(monitor_health_check())
    return _call_apm(url_path='/assetgroup/%s/model/%s?instanceId=%s&is91Instance=%s' % (asset_group_id, model_instance_id, APM_ID, is_91_instance), method='put', json=body)


def get_model_instance(asset_group_id, model_instance_id=None):
    init_environ()
    resp = _call_apm(url_path='/assetgroup/%s/model/%s?instanceId=%s' % (asset_group_id, model_instance_id if model_instance_id is not None else '', APM_ID), method='get')
    return resp


"""
Maximo API.
"""

def _call_maximo(url_path, method, json=None, headers=None, **kwargs):
    init_environ()

    if APM_API_KEY is None:
        raise RuntimeError('missing mandatory environment variable APM_API_KEY')

    if MAXIMO_BASEURL is None:
        # TODO how to be able to continue when no Maximo is linked?
        raise RuntimeError('missing mandatory environment variable MAXIMO_BASEURL')

    maximo_baseurl = MAXIMO_BASEURL
    if maximo_baseurl[-1] == '/':
        maximo_baseurl = maximo_baseurl[:-1]

    maximo_api_context = MAXIMO_API_CONTEXT
    if maximo_api_context is None or len(maximo_api_context.strip()) == 0:
        maximo_api_context = '/api'

    url = '%s%s%s' % (maximo_baseurl, maximo_api_context, url_path)

    if headers is None:
        headers = {}

    headers['apikey'] = APM_API_KEY

    ssl_verify = os.environ.get('SSL_VERIFY_MAXIMO', None) # for Maximo, disable SSL verify by default
    if isinstance(ssl_verify, str) and ssl_verify.lower() in ('true', 'yes', '1'):
        ssl_verify = True
    else:
        ssl_verify = False
    kwargs['ssl_verify'] = ssl_verify

    return _call(url=url, method=method, json=json, headers=headers, timeout=300, **kwargs)


def get_maximo_asset_group_id(group_label):
    if group_label is None:
        raise ValueError('the given argument group_label is None')

    resp = _call_maximo(url_path='/os/mxapiexpgroup?oslc.select=expgroupname&lean=1&querylocalized=1&oslc.where=grouplabel="%s"' % (group_label), method='get')
    if resp is not None:
        resp = resp.json()
        if 'member' in resp and isinstance(resp['member'], list) and len(resp['member']) > 0 and 'expgroupname' in resp['member'][0]:
            asset_group_id =  str(resp['member'][0]['expgroupname'])
            logger.info('Found asset group ID %s for group label %s', asset_group_id, group_label)
            return asset_group_id

    return None


def get_maximo_asset_metadata(assets, data_items, transform=True):
    # request payload:
    # {
    #     "assets":[{"assetNum":"pump-514-0","siteId":"BEDFORD"},{"assetNum":"pump-514-1","siteId":"BEDFORD"}],
    #     "attributes":["installdate","estendoflife","status","statusdate"]
    # }
    init_environ()

    body = {
        'assets': [{'assetNum':a['assetNum'].upper(),'siteId':a['siteId'].upper()} for a in assets],
        'attributes': data_items
    }
    resp = _call_apm(url_path='/pmiboard/assetmeta?instanceId=%s' % APM_ID, method='post', json=body)

    if resp is not None:
        # response format
        # {
        #     "meta": [
        #         {
        #             "installdate": "2018-01-01T00:00:00-06:00",
        #             "assetNum": "PUMP-514-0",
        #             "siteId": "BEDFORD"
        #         },
        #         {
        #             "installdate": "2018-01-01T00:00:00-06:00",
        #             "assetNum": "PUMP-514-1",
        #             "siteId": "BEDFORD",
        #             "estendoflife": "2018-06-20T00:00:00-05:00"
        #         }
        #     ]
        # }
        if transform is True:
            asset_metadata = {}
            for meta in resp.json()['meta']:
                asset_metadata[meta['assetNum'] + '-____-' + meta['siteId']] = meta
                del meta['assetNum']
                del meta['siteId']
            return asset_metadata
        else:
            return resp.json()['meta']
    else:
        return None


def get_maximo_failure_history_data(body, session=None):
    # /pmiboard/failurebyasset?assetNum=AH000&siteId=BEDFORD&instanceId=08e5ad71
    init_environ()
    return _call_apm(url_path='/pmiboard/failurebyasset?assetNum=%s&siteId=%s&instanceId=%s' % (body['assetNum'].upper(), body['siteId'].upper(), APM_ID), method='get', json=body, session=session)


def get_maximo_corrective_maintenance_data(body, session=None,cm_code='CM'):
    # /pmiboard/cmbyasset?assetNum=AH000&siteId=BEDFORD&instanceId=08e5ad71
    init_environ()
    return _call_apm(url_path='/pmiboard/cmbyasset?assetNum=%s&siteId=%s&instanceId=%s&cmCode=%s' % (body['assetNum'].upper(), body['siteId'].upper(), APM_ID, cm_code), method='get', json=body, session=session)


def get_maximo_asset_device_mappings(asset_group_id):
    if not is_maximo_linked() or is_local_mode():
        return None

    init_environ()
    resp = _call_apm(url_path='/assetgroup/%s/devicemapping?instanceId=%s' % (asset_group_id, APM_ID), method='get')
    if resp is not None:
        logger.debug('Call get_maximo_asset_device_mappings resp: %s', resp.text)

    if resp is not None and asset_group_id in resp.json():
        # response format
        # {
        #     "1001": [
        #         {
        #             "devices": ["RUNHOURS:103","TEMPERATURE:101","VIBRATION:102"],
        #             "assetNum": "AH001",
        #             "siteId": "BEDFORD"
        #         },
        #         {
        #             "devices": ["RUNHOURS:106","TEMPERATURE:104","VIBRATION:105"],
        #             "assetNum": "AH002",
        #             "siteId": "BEDFORD"
        #         }
        #     ]
        # }

        return resp.json()[asset_group_id]
    else:
        return None


def create_maximo_assets(assets):
    if assets is None or len(assets) == 0 or not isinstance(assets, dict):
        raise ValueError('the given argument assets must be a non-empty dict')

    body = []
    attributes = set()
    for asset_id, asset_data in assets.items():
        asset_num, site_id = asset_id.split('-____-')
        asset_data['assetnum'] = asset_num.upper()
        asset_data['siteid'] = site_id.upper()
        asset_data['description'] = asset_num.upper()
        body.append({'_data': asset_data})
        attributes.update(asset_data.keys())

    query = 'oslc.select=%s&oslc.where=assetnum in [%s] and siteid="%s"' % (
        ','.join(attributes),
        ','.join(['"%s"' % item['_data']['assetnum'] for item in body]),
        body[0]['_data']['siteid']
    )

    return _maximo_bulk_create(mxapp='mxapiasset', payload=body, resource_query=query, resource_attributes=attributes, key_attributes=['assetnum', 'siteid'])


def _generate_unique_wonum(wonum_prefix):
    wonum_length_limit = 10
    wonum_length = wonum_length_limit - len(wonum_prefix)
    wonum_str_format = '%0' + str(wonum_length) + 'd'
    return wonum_prefix + (wonum_str_format % math.floor(random.random() * 10**wonum_length))[-1 * wonum_length:]


def create_maximo_asset_failure_history(asset_failure_history, default_failurecode='PUMPS', default_problemcode='STOPPED', wonum_prefix='APM'):
    # "_data": {
    #     "assetnum": asset_id,
    #     "siteid": "BEDFORD",
    #     "wonum": wonum,
    #     "faildate": faildate,
    #     "failurecode": "PUMPS",
    #     "problemcode": "STOPPED"
    # }

    body=[]
    attributes = set()
    for asset_id, failure_history in asset_failure_history.items():
        for failure_event in failure_history:
            asset_num, site_id = asset_id.split('-____-')
            failure_event['assetnum'] = asset_num.upper()
            failure_event['siteid'] = site_id.upper()
            failure_event['wonum'] = _generate_unique_wonum(wonum_prefix=wonum_prefix).upper()
            if 'failurecode' not in failure_event:
                failure_event['failurecode'] = default_failurecode.upper()
            if 'problemcode' not in failure_event:
                failure_event['problemcode'] = default_problemcode.upper()
            body.append({'_data': failure_event})
            attributes.update(failure_event.keys())

    query = 'oslc.select=%s&oslc.where=assetnum in [%s] and wonum=%s%% and siteid="%s"' % (
        ','.join(attributes),
        ','.join({'"%s"' % item['_data']['assetnum'] for item in body}),
        # ','.join({'"%s"' % item['_data']['wonum'] for item in body}),
        ','.join({'"%s"' % wonum_prefix for item in body}),
        body[0]['_data']['siteid']
    )

    return _maximo_bulk_create(mxapp='mxapiwodetail', payload=body, resource_query=query, resource_attributes=attributes, key_attributes=['assetnum', 'siteid', 'wonum'])


def _maximo_bulk_create(mxapp, payload, resource_query, resource_attributes, key_attributes):
    headers = {
        'X-method-override': 'BULK'
    }

    # go ahead to create resources directly
    resp = _call_maximo(url_path='/os/%s?lean=1' % (mxapp), method='post', json=payload, headers=headers)

    # now we deal with already existent resources
    failed_creating = False
    if resp is not None:
        logger.debug('Call Maximo resp: %s', resp.text)

        responses = json.loads(resp.text)
        for response in responses:
            if '_responsemeta' not in response or 'status' not in response['_responsemeta'] or not response['_responsemeta']['status'].startswith('2'):
                failed_creating = True
                break

    if failed_creating:
        logger.info('Failed creating assets because they might already exist, updating existing assets now...')

        logger.debug('resource_query=%s', resource_query)

        query_resp = _call_maximo(url_path='/os/%s?lean=1&%s' % (mxapp, resource_query), method='get')

        if query_resp is not None:
            resources = {tuple([item[key] for key in key_attributes]):item for item in [p['_data'] for p in payload]}
            logger.debug('resources=%s', resources)

            logger.debug('query_responses=%s', query_resp.text)

            query_responses = query_resp.json()
            if len(query_responses['member']) == 0:
                logger.warning('failed finding objects to update, something wrong with creation request, check its response: %s', resp.text)
                return None

            body = []
            # TODO a defect that query_responses does not have 'member'
            for resource_data in query_responses['member']:
                resource_key = tuple([resource_data[key] for key in key_attributes])
                resource_original = resources[resource_key]

                resource = {}
                resource['_data'] = {attr:resource_original.get(attr, None) for attr in resource_attributes if attr in resource_original}
                resource['_meta'] = {
                    'uri': resource_data['href'],
                    'method': 'PATCH',
                    'patchtype': 'MERGE'
                }
                body.append(resource)
            resp = _call_maximo(url_path='/os/%s?lean=1' % (mxapp), method='post', json=body, headers=headers)

    return resp


def delete_maximo_assets(assets):
    logger.info('Deleting Maximo assets...')
    if assets is None or len(assets) == 0 or not isinstance(assets, dict):
        raise ValueError('the given argument assets must be a non-empty dict')

    body = []
    for asset_id, asset_data in assets.items():
        asset_num, site_id = asset_id.split('-____-')
        asset_data['assetnum'] = asset_num.upper()
        asset_data['siteid'] = site_id.upper()
        body.append({'_data': asset_data})

    resp = None

    batch_size = 200
    start = 0
    while start < len(body):
        batch = body[start:start+batch_size]
        start += batch_size

        query = 'oslc.where=assetnum in [%s] and siteid="%s"' % (
            ','.join(['"%s"' % item['_data']['assetnum'] for item in batch]),
            batch[0]['_data']['siteid']
        )

        resp = _maximo_bulk_delete(mxapp='mxapiasset', resource_query=query)
    
    logger.info('Finished deleting Maximo assets.')
    return resp


def delete_maximo_asset_failure_history(asset_failure_history, wonum_prefix='APM'):
    body=[]
    for asset_id, failure_history in asset_failure_history.items():
        asset_num, site_id = asset_id.split('-____-')
        for failure_event in failure_history:
            failure_event['assetnum'] = asset_num.upper()
            failure_event['siteid'] = site_id.upper()
            body.append({'_data': failure_event})
            break # one per asset is enough

    query = 'oslc.where=assetnum in [%s] and wonum="%s%%25" and siteid="%s"' % (
        ','.join(['"%s"' % item['_data']['assetnum'] for item in body]),
        wonum_prefix,
        body[0]['_data']['siteid']
    )

    return _maximo_bulk_delete(mxapp='mxapiwodetail', resource_query=query)


def _maximo_bulk_delete(mxapp, resource_query):
    headers = {
        'X-method-override': 'BULK'
    }

    resp = _call_maximo(url_path='/os/%s?lean=1&%s' % (mxapp, resource_query), method='get')

    if resp is not None:
        query_responses = resp.json()
        body = []
        for resource_data in query_responses['member']:
            body.append({
                '_meta': {
                    'uri': resource_data['href'],
                    'method': 'DELETE'
                }
            })
        resp = _call_maximo(url_path='/os/%s?lean=1' % (mxapp), method='post', json=body, headers=headers)

    return resp

def delete_asset_group_id(asset_group_id):
    if asset_group_id is not None:
        url = os.environ.get('APM_API_BASEURL', None) + '/ibm/pmi/service/rest/assetgroup/' + asset_group_id + '?instanceId=' +os.environ.get('APM_ID', None)
        header={
        'apmapitoken':os.environ.get('APM_API_KEY', None)
        }
        response = requests.delete(url, headers=header,verify=False)
        return response

def get_asset_group_href(grouplabel):
    if grouplabel is None:
        raise ValueError('the given argument grouplabel is None')
    resp = _call_maximo(url_path='/os/MXAPIEXPGROUP?lean=1&addid=1&oslc.where=grouplabel=%22'+grouplabel+'%22&oslc.select=*', method='get')
    if resp is not None:
        data = resp.json()
        logger.debug('resp.json: %s', data)
        if len(data['member']) != 0:
            return (data['member'][0]['href'])
    return 0

def delete_asset_group(grouplabel):
    if grouplabel is None:
        raise ValueError('the given argument grouplabel is None')
    href = get_asset_group_href(grouplabel)
    if href != 0 :
        url_path=href.split("/api")[1]
        logger.debug('url_path to pass to _call_maximo: %s', url_path)
        resp = _call_maximo(url_path=url_path, method='delete')
        logger.debug('response from _call_maximo: %s', resp)

#Create filter and asset group
def create_filter(assetFilter,assetNum,siteId='BEDFORD'):
    querypayload={
        "ispublic":True,
        "clausename":assetFilter
    }
    resp = _call_maximo(url_path='/os/mxapiasset?action=system%3Asavethisquery&oslc.where=siteid=%22'+siteId+'%22%20and%20assetnum=%22'+assetNum+'%25%22&lean=1', method='post',json=querypayload)
    return resp

def create_asset_group(assetGroupLabel,assetFilter):
    group_payload={
        "usewith":"ASSET",
        "intobjectname":"MXAPIASSET",
        "egtype":"QUERY",
        "clausename":assetFilter,
        "usedby":"PMI",
        "grouplabel":assetGroupLabel
    }
    resp = _call_maximo(url_path='/os/MXAPIEXPGROUP?lean=1', method='post',json=group_payload)
    return resp

"""
Watson IOT Platform API.
"""

def _call_wiotp(url_path, method, json=None, headers=None, messageing_api=False, **kwargs):
    init_environ()
    if API_KEY is None or API_TOKEN is None:
        raise RuntimeError('missing mandatory environment variable API_KEY or API_TOKEN')

    ORG_ID = API_KEY.split('-')[1]

    logger.debug('Calling WIOTP...')

    schema = os.environ.get('AS_SCHEMA', None)
    if schema is None:
        if messageing_api:
            url = 'https://%s.messaging.internetofthings.ibmcloud.com/api/v0002%s' % (ORG_ID, url_path)
        else:
            url = 'https://%s.internetofthings.ibmcloud.com/api/v0002%s' % (ORG_ID, url_path)
    else:
        if messageing_api:
            ioturl = API_BASEURL.replace('.api.monitor.', '.messaging.iot.')
        else:
            ioturl = API_BASEURL.replace('.api.monitor.', '.iot.')
        logger.debug('WIOTP Call URL: %s', ioturl)
        url = ioturl + '/api/v0002%s' % (url_path)

    if headers is None:
        headers = {}

    if 'auth' not in kwargs:
        kwargs['auth'] = (API_KEY, API_TOKEN)

    ssl_verify = os.environ.get('SSL_VERIFY_WIOTP', 'true')
    if isinstance(ssl_verify, str) and ssl_verify.lower() in ('true', 'yes', '1'):
        ssl_verify = True
    else:
        ssl_verify = False
    kwargs['ssl_verify'] = ssl_verify

    return _call(url=url, method=method, json=json, headers=headers, **kwargs)


def check_wiotp_device_type(device_type):
    resp = _call_wiotp(url_path='/device/types/%s' % device_type, method='get')
    if resp is not None:
        return True

    return False


def create_wiotp_devices(device_type, device_ids=[]):
    device_type_payload = {
        'id': device_type,
        'description': device_type,
        'classId': 'Device',
        'deviceInfo': {
        }
    }
    devices_payload = [{'typeId': device_type, 'deviceId': device, 'authToken': API_TOKEN} for device in device_ids]

    # create device type
    resp = _call_wiotp(url_path='/device/types', method='post', json=device_type_payload)
    if resp is not None:
        logger.info('created device type %s: status_code=%s, response_text=%s', device_type, resp.status_code, resp.text)

    # register devices
    if len(devices_payload) > 0:
        resp = _call_wiotp(url_path='/bulk/devices/add', method='post', json=devices_payload)
        if resp is not None:
            logger.info('registered devices: status_code=%s, response_text=%s', resp.status_code, resp.text)

    return resp

def create_wiotp_devices_v2(device_type, dataItemDto,device_ids=[],timestamp_column='evt_timestamp'):
    logger.info('start of create_wiotp_devices_v2: device_type=%s', device_type)

    #https://github.ibm.com/wiotp/Maximo-Asset-Monitor/issues/5302
    device_type_payload = {
        "name" : device_type,
        "description": "deviceType_created_by_notebook",
        "metricTimestampColumn":timestamp_column,
        "dataItemDto": dataItemDto,
        "eventDto": [
            {
                "name": "EventA",
                "metricTimestamp": timestamp_column
            }
        ],
        "storageRetentionDays": 10950,
        "derivedMetricRetentionDays": 10950,
        "origin": "Predict-Notebook"
    }
   
    

    devices_payload = [{'typeId': device_type, 'deviceId': device, 'authToken': API_TOKEN} for device in device_ids]

    # create device type
    # TODO: Health Check API Here
    is_91_instance = monitor_health_check()

    monitor_url_path='deviceTypes'
    if (is_91_instance is True):
        monitor_url_path='core/deviceTypes'

    resp = _call_as_v2(url_path=monitor_url_path, method='post', json=device_type_payload)
    if resp is not None  :
        logger.info('created device type %s: status_code=%s, response_text=%s', device_type, resp.status_code, resp.text)
        data = resp.json()
        device_type_uuid= data["uuid"]
        logger.info('device_type_uid =%s', device_type_uuid)

    # register devices
    if len(devices_payload) > 0:
        resp = _call_wiotp(url_path='/bulk/devices/add', method='post', json=devices_payload)
        if resp is not None:
            logger.info('registered devices: status_code=%s, response_text=%s', resp.status_code, resp.text)

    return device_type_uuid

def get_device_uid_map(device_type_uuid, device_id_list ):
    device_uid_map= {}
  
    device_name_array_input = [ {'name': x} for x in device_id_list]


    monitor_url_path='deviceTypes/'+device_type_uuid + "/devices"

    resp = _call_as_v2(url_path=monitor_url_path, method='post', json=device_name_array_input)
    
    '''
    [
    {
        "ddId": "mm_D_UHVtcF9BRk1fVEVTVDM5_c291cmRvdWdoXzAxMA",
        "name": "sourdough_010",
        "origin": null,
        "deviceUID": 58,
        "dimensions": {
            "deviceid": "sourdough_010"
        }
    },
    {
        "ddId": "mm_D_UHVtcF9BRk1fVEVTVDM5_c291cmRvdWdoXzAz",
        "name": "sourdough_03",
        "origin": null,
        "deviceUID": 59,
        "dimensions": {
            "deviceid": "sourdough_03"
        }
    }
]
    '''
    if resp is not None and resp.status_code == 200:
        # resp.text is json array
        json_array = json.loads( resp.text)
        for each_item in json_array:
            device_uid = each_item['deviceUID']
            device_name = each_item['name']
            device_uid_map[device_name] = device_uid
    elif resp.status_code == 504:
        raise RuntimeError('Monitor API server has gateway timeout')

    
    logger.info('get_device_uid_map device_uid_map=%s', device_uid_map)
        
    return device_uid_map




def delete_wiotp_devices(device_type, device_ids=None):
    if device_ids is None:
        device_ids = []
    if not isinstance(device_ids, list) or not all([isinstance(did, str) for did in device_ids]):
        raise ValueError('invalid parameter device_ids=%s, must be a list of string' % str(device_ids))

    if len(device_ids) == 0:
        resp = _call_wiotp(url_path='/device/types/%s/devices' % device_type, method='get')
        if resp is not None:
            devices = resp.json().get('results', [])
            for device in devices:
                device_ids.append(device['deviceId'])

    devices_payload = [{'typeId': device_type, 'deviceId': device} for device in device_ids]

    # remove all devices
    resp = _call_wiotp(url_path='/bulk/devices/remove', method='post', json=devices_payload)
    if resp is not None:
        logger.info('removed devices: status_code=%s, response_text=%s', resp.status_code, resp.text)

    # delete device type
    resp = _call_wiotp(url_path='/device/types/%s' % device_type, method='delete')
    if resp is not None:
        logger.info('deleted device type %s: status_code=%s, response_text=%s', device_type, resp.status_code, resp.text)

    return resp

def delete_wiotp_devices_v2(device_type, device_ids=None):
    if device_ids is None:
        device_ids = []
    if not isinstance(device_ids, list) or not all([isinstance(did, str) for did in device_ids]):
        raise ValueError('invalid parameter device_ids=%s, must be a list of string' % str(device_ids))

    if len(device_ids) == 0:
        resp = _call_wiotp(url_path='/device/types/%s/devices' % device_type, method='get')
        if resp is not None:
            devices = resp.json().get('results', [])
            for device in devices:
                device_ids.append(device['deviceId'])

    devices_payload = [{'typeId': device_type, 'deviceId': device} for device in device_ids]

    # remove all devices
    resp = _call_wiotp(url_path='/bulk/devices/remove', method='post', json=devices_payload)
    if resp is not None:
        logger.info('removed devices: status_code=%s, response_text=%s', resp.status_code, resp.text)

    # delete device type
    resp = _call_wiotp(url_path='/device/types/%s' % device_type, method='delete')
    if resp is not None:
        logger.info('deleted device type %s: status_code=%s, response_text=%s', device_type, resp.status_code, resp.text)

    return resp


def create_wiotp_pi_li_mappings(device_type, device_ids=[], columns=[]):
    event_type = 'sensoreventtype'
    event_id = 'sensorevent'

    # skip the default special column on WIOTP side
    columns = [col for col in columns if col.name.lower() != default_timestamp_column]

    # create physical interface schema
    properties = {}
    for col, col_type in [(col.name, col.type) for col in columns]:
        if isinstance(col_type, Float) or isinstance(col_type, Integer):
            properties[col] = {'type': 'number'}
        elif isinstance(col_type, Boolean):
            properties[col] = {'type': 'boolean'}
        elif isinstance(col_type, String):
            properties[col] = {'type': 'string'}
        elif isinstance(col_type, DateTime):
            properties[col] = {'type': 'string', 'format': 'date-time'}
    payload = json.dumps({
        '$schema': 'http://json-schema.org/draft-04/schema#',
        'type': 'object',
        'title': 'pi-%s' % event_id,
        'description': '',
        'properties': properties
    })
    files = {
        'schemaFile': ('blob.json', payload, 'application/json'),
        'name': 'schemaName',
        'schemaType': 'json-schema',
    }
    resp = _call_wiotp(url_path='/draft/schemas', method='post', files=files)
    if resp is None:
        raise RuntimeError('error creating schema for physical interface')

    physical_schema_id = resp.json().get('id')
    if physical_schema_id is None:
        raise RuntimeError('error creating schema for physical interface')

    logger.info('created schema_id=%s for physical interface', physical_schema_id)

    # create physical interface event type
    resp = _call_wiotp(url_path='/draft/event/types', method='post', json={'name': event_type, 'schemaId': physical_schema_id})
    if resp is None:
        raise RuntimeError('error creating event type for physical interface')

    physical_event_type_id = resp.json().get('id')
    if physical_event_type_id is None:
        raise RuntimeError('error creating event type for physical interface')

    logger.info('created event_type_id=%s using schema_id=%s for physical interface', physical_event_type_id, physical_schema_id)

    # create physical interface
    resp = _call_wiotp(url_path='/draft/physicalinterfaces', method='post', json={'name':'btsensor_PI','description':''})
    if resp is None:
        raise RuntimeError('error creating physical interface')

    physical_interface_id = resp.json().get('id')
    if physical_interface_id is None:
        raise RuntimeError('error creating physical interface')

    logger.info('created physical_interface_id=%s', physical_interface_id)

    # associate physical interface with event type
    resp = _call_wiotp(url_path='/draft/physicalinterfaces/%s/events' % physical_interface_id, method='post', json={'eventId':event_id,'eventTypeId':physical_event_type_id})
    if resp is None:
        raise RuntimeError('error associating physical interface with event type')

    logger.info('associated physical_interface_id=%s with event_type_id=%s', physical_interface_id, physical_event_type_id)

    # associate physical interface with device type
    resp = _call_wiotp(url_path='/draft/device/types/%s/physicalinterface' % device_type, method='post', json={'id':physical_interface_id})
    if resp is None:
        raise RuntimeError('error associating physical interface with device_type=%s' % device_type)

    logger.info('associated physical_interface_id=%s with device_type=%s', physical_interface_id, device_type)

    # create logical interface schema
    payload = json.dumps({
        '$schema': 'http://json-schema.org/draft-04/schema#',
        'type': 'object',
        'title': 'li-%s' % event_id,
        'description': '',
        'additionalProperties': False,
        'properties': properties
    })
    files = {
        'schemaFile': ('blob.json', payload, 'application/json'),
        'name': 'schemaName',
        'schemaType': 'json-schema',
    }
    resp = _call_wiotp(url_path='/draft/schemas', method='post', files=files)
    if resp is None:
        raise RuntimeError('error creating schema for logical interface')

    logical_schema_id = resp.json().get('id')
    if logical_schema_id is None:
        raise RuntimeError('error creating schema for logical interface')

    logger.info('created schema_id=%s for logical interface' % logical_schema_id)

    # create logical interface
    resp = _call_wiotp(url_path='/draft/logicalinterfaces', method='post', json={'name':'btsensor_LI','description':'','schemaId':logical_schema_id})
    if resp is None:
        raise RuntimeError('error creating logical interface')

    logical_interface_id = resp.json().get('id')
    if logical_interface_id is None:
        raise RuntimeError('error creating logical interface')

    logger.info('created logical_interface_id=%s', logical_interface_id)

    # associate logical interface with device type
    resp = _call_wiotp(url_path='/draft/device/types/%s/logicalinterfaces' % device_type, method='post', json={'id':logical_interface_id})
    if resp is None:
        raise RuntimeError('error associating logical interface with device_type=%s' % device_type)

    logger.info('associated logical_interface_id=%s with device_type=%s', logical_interface_id, device_type)

    # create mappings
    property_mappings = {
        event_id: {col: '$event.%s' % col for col in properties}
    }
    resp = _call_wiotp(url_path='/draft/device/types/%s/mappings' % device_type, method='post', json={'logicalInterfaceId':logical_interface_id,'propertyMappings':property_mappings, 'notificationStrategy': 'on-every-event'})
    if resp is None:
        raise RuntimeError('error creating mappings for device_type=%s' % device_type)

    logger.info('created mappings for device_type=%s', device_type)

    # activate config
    resp = _call_wiotp(url_path='/draft/device/types/%s' % device_type, method='patch', json={'operation':'activate-configuration'})
    if resp is None:
        raise RuntimeError('error activating mapping configuration for device_type=%s' % device_type)

    logger.info('activated mapping configuration for device_type=%s', device_type)

    # send some messages to trigger ICS
    if len(device_ids) > 0:
        event_payload = {}
        for col, col_type in [(col.name, col.type) for col in columns]:
            if isinstance(col_type, Float) or isinstance(col_type, Integer):
                event_payload[col] = 0
            elif isinstance(col_type, Boolean):
                event_payload[col] = False
            elif isinstance(col_type, String):
                event_payload[col] = ' '
            elif isinstance(col_type, DateTime):
                event_payload[col] = dt.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%S.%f+00:00')
        for i in range(5):
            _call_wiotp(url_path='/device/types/%s/devices/%s/events/sensorevent' % (device_type, device_ids[0]), method='post', json=event_payload, auth=('use-token-auth', API_TOKEN), messageing_api=True)

        logger.info('sent events to device_type=%s', device_type)

    return resp


"""
Analytic Service API.
"""

def _call_as(url_path, method, api_type, json=None, headers=None, **kwargs):
    # if any environment variable is not set, check if we can get from AS constants
    if any([env is None for env in [API_BASEURL, API_KEY, API_TOKEN, TENANT_ID]]):
        if all([env is not None for env in [APM_ID, APM_API_BASEURL, APM_API_KEY]]):
            init_environ()
        else:
            raise RuntimeError('missing mandatory environment variable API_BASEURL, API_KEY, API_TOKEN, or TENANT_ID')

    tenant_id = os.environ.get('TENANT_ID', None)
    if tenant_id is None:
        raise RuntimeError('missing mandatory environment variable TENANT_ID')

    api_baseurl = API_BASEURL
    if api_baseurl[-1] == '/':
        api_baseurl = api_baseurl[:-1]

    url = '%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path)
    final_headers = {
        'Content-Type': 'application/json',
        'X-api-key' : API_KEY,
        'X-api-token' : API_TOKEN,
        'Cache-Control': 'no-cache',
    }
    if headers is not None:
        final_headers.update(headers)

    ssl_verify = os.environ.get('SSL_VERIFY_AS', 'true')
    if isinstance(ssl_verify, str) and ssl_verify.lower() in ('true', 'yes', '1'):
        ssl_verify = True
    else:
        ssl_verify = False
    kwargs['ssl_verify'] = ssl_verify

    return _call(url=url, method=method, json=json, headers=final_headers, **kwargs)



def _call_as_kpi(url_path, method, api_type, json=None, headers=None, **kwargs):
    # if any environment variable is not set, check if we can get from AS constants
    if any([env is None for env in [API_BASEURL, API_KEY, API_TOKEN, TENANT_ID]]):
        if all([env is not None for env in [APM_ID, APM_API_BASEURL, APM_API_KEY]]):
            init_environ()
        else:
            raise RuntimeError('missing mandatory environment variable API_BASEURL, API_KEY, API_TOKEN, or TENANT_ID')

    tenant_id = os.environ.get('TENANT_ID', None)
    if tenant_id is None:
        raise RuntimeError('missing mandatory environment variable TENANT_ID')

    REST_KPI_URL = os.environ.get('REST_KPI_URL', None)
    if REST_KPI_URL is None:
        api_baseurl = API_BASEURL
    else:
        api_baseurl = REST_KPI_URL


    if api_baseurl[-1] == '/':
        api_baseurl = api_baseurl[:-1]

    url = '%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path)
    final_headers = {
        'Content-Type': 'application/json',
        'X-api-key' : API_KEY,
        'X-api-token' : API_TOKEN,
        'Cache-Control': 'no-cache',
    }
    if headers is not None:
        final_headers.update(headers)

    ssl_verify = os.environ.get('SSL_VERIFY_AS', 'true')
    if isinstance(ssl_verify, str) and ssl_verify.lower() in ('true', 'yes', '1'):
        ssl_verify = True
    else:
        ssl_verify = False
    kwargs['ssl_verify'] = ssl_verify

    return _call(url=url, method=method, json=json, headers=final_headers, **kwargs)



def _call_as_meta(url_path, method, api_type, json=None, headers=None, **kwargs):
    # if any environment variable is not set, check if we can get from AS constants
    if any([env is None for env in [API_BASEURL, API_KEY, API_TOKEN, TENANT_ID]]):
        if all([env is not None for env in [APM_ID, APM_API_BASEURL, APM_API_KEY]]):
            init_environ()
        else:
            raise RuntimeError('missing mandatory environment variable API_BASEURL, API_KEY, API_TOKEN, or TENANT_ID')

    tenant_id = os.environ.get('TENANT_ID', None)
    if tenant_id is None:
        raise RuntimeError('missing mandatory environment variable TENANT_ID')

    REST_METADATA_URL = os.environ.get('REST_METADATA_URL', None)
    if REST_METADATA_URL is None:
        api_baseurl = API_BASEURL
    else:
        api_baseurl = REST_METADATA_URL

    if api_baseurl[-1] == '/':
        api_baseurl = api_baseurl[:-1]

    url = '%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path)
    final_headers = {
        'Content-Type': 'application/json',
        'X-api-key' : API_KEY,
        'X-api-token' : API_TOKEN,
        'Cache-Control': 'no-cache',
    }
    if headers is not None:
        final_headers.update(headers)

    ssl_verify = os.environ.get('SSL_VERIFY_AS', 'true')
    if isinstance(ssl_verify, str) and ssl_verify.lower() in ('true', 'yes', '1'):
        ssl_verify = True
    else:
        ssl_verify = False
    kwargs['ssl_verify'] = ssl_verify

    return _call(url=url, method=method, json=json, headers=final_headers, **kwargs)

def _call_as_v2(url_path, method,  json=None, headers=None, **kwargs):
    logger.debug('Calling Monitor with V2 Protocol.')
    logger.debug('Monitor V2 call JSON=%s', json)
    # if any environment variable is not set, check if we can get from AS constants
    if any([env is None for env in [API_BASEURL, API_KEY, API_TOKEN, TENANT_ID]]):
        if all([env is not None for env in [APM_ID, APM_API_BASEURL, APM_API_KEY]]):
            init_environ()
        else:
            raise RuntimeError('missing mandatory environment variable API_BASEURL, API_KEY, API_TOKEN, or TENANT_ID')

    tenant_id = os.environ.get('TENANT_ID', None)
    if tenant_id is None:
        raise RuntimeError('missing mandatory environment variable TENANT_ID')

    api_baseurl = API_BASEURL
    if api_baseurl[-1] == '/':
        api_baseurl = api_baseurl[:-1]

    url = '%s/api/v2/%s' % (api_baseurl,  url_path)
    logger.debug('Monitor V2 call URL=%s', url)
    final_headers = {
        'Content-Type': 'application/json',
        'X-api-key' : API_KEY,
        'X-api-token' : API_TOKEN,
        'Cache-Control': 'no-cache',
        'tenantId': tenant_id,
        'mam_user_email':'ibm@ibm.com'
    }
    #logger.debug('Monitor V2 call final_headers=%s',final_headers)
    if headers is not None:
        final_headers.update(headers)

    ssl_verify = os.environ.get('SSL_VERIFY_AS', 'true')
    if isinstance(ssl_verify, str) and ssl_verify.lower() in ('true', 'yes', '1'):
        ssl_verify = True
    else:
        ssl_verify = False
    kwargs['ssl_verify'] = ssl_verify

    return _call(url=url, method=method, json=json, headers=final_headers, **kwargs)


#{{url}}/api/v2/core/internal/devices/search
#{{url}}/api/v2/core/internal/devices/search
def _get_asset_device_mapping_from_monitor(input_json):
    # json {'deviceTypeName': 'B2_Boiler_Type', 'siteAssetSearchCriteria': [{'assetName': 'Anu_Asset1_300', 'siteName': 'Anu_Test_Site_300'}]}
    resp = _call_as_v2(url_path='core/internal/devices/search' , method='post',json=input_json)

    #############################
    # WARINING: since there is NO working MONITOR MAS87.7 env , this is the stub, remove it before prediution
    #
    #############################
    #resp =['7001','7002','7003']  # for testing purpose # ROMOVE it from production

    # resp = [
    #         {
    #             "siteName": "Anu_Test_Site_300",
    #             "assetName": "Anu_Asset_73002",
    #             "deviceId": "73002"
    #         },
    #         {
    #         "siteName": "Anu_Test_Site_300",
    #         "assetName": "Anu_Asset_73000",
    #         "deviceId": "73000"
    #         },
    #         {
    #         "siteName": "Anu_Test_Site_300",
    #         "assetName": "Anu_Asset_73001",
    #         "deviceId": "73001"
    #         }
    #         ]
    
    asset_device_mappings=[]

    if resp is None:
        return None

    if resp.status_code == 204:
        return None
    
    # at this point, we should have data
    if resp is not None and resp.status_code== 200:
        data = resp.json()
        for each_element in data:
            #print(each_element)
            asset_device_mappings.append([each_element['siteName'], each_element['assetName'], input_json['deviceTypeName'],each_element['deviceId'] , ''])
        return asset_device_mappings
    else:
        return None


# core/deviceTypes/search?status=ACTIVE&offset=0&limit=100' 

def _get_entity_type_table_name_from_monitor(json_body):
    # json {"search": "KDE_ROBOT_TYPE", }
    resp = _call_as_v2(url_path='core/deviceTypes/search?status=ACTIVE&offset=0&limit=100' , method='post',json=json_body)

    '''
        resp = {'totalCount': 1, 'results': [{'uuid': '69ca2fee-f4b5-4beb-b137-7cf7f01a9c26', 'resourceId': 81, 'ddId': 'mm_D_S0RFX1JPQk9UX1RZUEU', 
        'name': 'KDE_ROBOT_TYPE', 'description': '', 'status': 'ACTIVE', 'logicalInterfaceId': None, 
        'dimensionTableName': 'IOT_DEVICE_TYPE_81_CTG', 
        'metricTableName': 'IOT_DEVICE_TYPE_81',
         'metricTimestampColumn': 'EVT_TIMESTAMP', 'schemaName': 'MAIN_MAM', 'origin': None, 'deviceIdentifier': 'deviceId'}]}
    '''
    
    

    if resp is None:
        return None

    if resp.status_code == 204:
        return None
    
    # at this point, we should have data
    if resp is not None and resp.status_code== 200:
        data = resp.json()
        table_name= data['results'][0]['metricTableName']
        logger.debug('entity type table name: %s', table_name)
        return table_name
        
    return None


def _get_entity_type_uuid_from_monitor(entity_type_name):
    input_json ={"search": entity_type_name }
    resp = _call_as_v2(url_path='core/deviceTypes/search?status=ACTIVE&offset=0&limit=100' , method='post',json=input_json)

    '''
        resp = {'totalCount': 1, 'results': [{'uuid': '69ca2fee-f4b5-4beb-b137-7cf7f01a9c26', 'resourceId': 81, 'ddId': 'mm_D_S0RFX1JPQk9UX1RZUEU', 
        'name': 'KDE_ROBOT_TYPE', 'description': '', 'status': 'ACTIVE', 'logicalInterfaceId': None, 
        'dimensionTableName': 'IOT_DEVICE_TYPE_81_CTG', 
        'metricTableName': 'IOT_DEVICE_TYPE_81',
         'metricTimestampColumn': 'EVT_TIMESTAMP', 'schemaName': 'MAIN_MAM', 'origin': None, 'deviceIdentifier': 'deviceId'}]}
    '''
    
    

    if resp is None:
        return None

    if resp.status_code == 204:
        return None
    
    # at this point, we should have data
    if resp is not None and resp.status_code== 200:
        data = resp.json()
        entity_type_uuid = data['results'][0]['uuid']
        logger.debug('entity type uuid: %s', entity_type_uuid)
        return entity_type_uuid
        
    return None

def _check_entity_type(entity_type):
    # According to Peter Kohlmann , all meta call will use REST_METADATA_URL
    resp = _call_as_meta(url_path='/entityType/%s' % entity_type, method='get', api_type='meta')
    if resp is not None:
        return True

    return False

def get_entity_type_id(entity_type):
    # According to Peter Kohlmann , all meta call will use REST_METADATA_URL
    resp = _call_as_meta(url_path='/entityType/%s' % entity_type, method='get', api_type='meta')
    if resp is not None:
        data = resp.json()
        entityTypeId = data['entityTypeId']
    else:
        entityTypeId = None

    logger.debug('entityTypeId: %s', entityTypeId)
    return entityTypeId


class IotTypeTable(BaseTable):
    def __init__(self, name, db, *args, **kwargs):
        if '_entity_id' in kwargs:
            self._entity_id = kwargs['_entity_id']
        if '_timestamp' in kwargs:
            self._timestamp = kwargs['_timestamp']
        columns = []
        columns.append(Column('devicetype', String(64), nullable=False))
        columns.append(Column(self._entity_id.lower(), String(256), nullable=False))
        columns.append(Column('logicalinterface_id', String(64), nullable=False))
        columns.append(Column('eventtype', String(64)))
        columns.append(Column('format', String(32), nullable=False))
        columns.append(Column(self._timestamp.lower(), DateTime, nullable=False, index=True))
        columns.append(Column('updated_utc', DateTime, nullable=False, server_default=func.current_timestamp()))
        columns.extend(args)
        super().__init__(name, db, *columns, **kwargs)


def register_90_model(e: EntityType, publish_kpis=False, raise_error=False, sample_entity_type=False):
    """Register a model according to MAS90x spec. 91x has upgraded and iotfunctions is not yet fully backward compatible.Hence creating our own method here

    Args:
        e (EntityType): Entity Type to be registered
        publish_kpis (bool, optional): Defines if KPI need to be published. Defaults to False.
        raise_error (bool, optional): Defines if to raise error or not. Defaults to False.
        sample_entity_type (bool, optional): When true calls meta api will additional parameter that skips creation on `rcv_timestamp_utc` column. Defaults to False.
    """
    if e.db is None:
        msg = ('Entity type has no db connection. Local entity types'
                ' may not be registered ')
        raise ValueError(msg)

    if e._timestamp_col is None:
        e._timestamp_col = e._timestamp

    cols = []
    columns = []
    metric_column_names = []
    table = {}
    table['name'] = e.logical_name
    table['metricTableName'] = None
    table['metricTimestampColumn'] = e._timestamp_col
    table['description'] = e.description
    table['origin'] = 'AS_SAMPLE'
    for c in e.db.get_column_names(e.table, schema=e._db_schema):
        cols.append((e.table, c, 'METRIC'))
        metric_column_names.append(c)

    if e._dimension_table is not None:
        table['dimensionTableName'] = None
        for c in e.db.get_column_names(e._dimension_table, schema=e._db_schema):
            if c not in metric_column_names:
                cols.append((e._dimension_table, c, 'DIMENSION'))
    for (table_obj, column_name, col_type) in cols:
        msg = 'found %s column %s' % (col_type, column_name)
        logger.debug(msg)
        # if column_name not in e.get_excluded_cols():
        data_type = table_obj.c[column_name].type
        if isinstance(data_type, (FLOAT, Float, INTEGER, Integer)):
            data_type = 'NUMBER'
        elif db_module.DB2_DOUBLE is not None and isinstance(data_type, db_module.DB2_DOUBLE):
            data_type = 'NUMBER'
        elif isinstance(data_type, (VARCHAR, String)):
            data_type = 'LITERAL'
        elif isinstance(data_type, (TIMESTAMP, DateTime)):
            data_type = 'TIMESTAMP'
        else:
            data_type = str(data_type)
            logger.warning('Unknown datatype %s for column %s' % (data_type, column_name))
        columns.append({'name': column_name, 'type': col_type, 'columnName': column_name, 'columnType': data_type,
                        'tags': None, 'transient': False})
    table['dataItemDto'] = columns
    if e._db_schema is not None:
        table['schemaName'] = e._db_schema
    else:
        try:
            table['schemaName'] = e.db.credentials['db2']['username']
        except KeyError:
            try:
                username = e.db.credentials["postgresql"]['username']
                table["schemaName"] = "public"
            except KeyError:
                raise KeyError('No database credentials found. Unable to register table.')

    response = e.db.http_request(request='POST', object_type='entityType', object_name=None,
                                    payload=table, raise_error=raise_error, sample_entity_type=sample_entity_type)

    response_data = json.loads(response)

    e._entity_type_uuid = response_data.get('uuid')
    e._entity_type_id = response_data.get('resourceId')
    e._metric_table_name = response_data.get('metricTableName')
    e._dimension_table_name = response_data.get('dimensionTableName')

    msg = 'Metadata registered for table %s ' % e.name
    logger.debug(msg)
    if publish_kpis:
        e.publish_kpis(raise_error=raise_error)
        e.db.register_constants(e.ui_constants)

    return response


def create_entity_type(
        entity_type_name,
        columns=[],
        db=None,
        db_schema=None,
        timestamp_column=None,
        entity_type_table_prefix='IOT',
        dimension_table_name=None,
        dimensions=[],
        write=True,
        df=None,
        delete_table_first=True,
        drop_table_first=False,
        use_wiotp=True,
        devices=[]):
    """Create an entity type.

    Note that currently, write-related paramters are only for time-series data, but not for non-time-series dimensions.

    Parameters
    ----------
    entity_type_name : str
        The name of the entity type to create.
    columns (list of sqlalchemy.Column): Optoinal, the list of data item names of the entity type to be created.
        Default is an empty list. Note that there is at least one system default data item representing the device
        ID and another representing the time series base, which do not need to be included in this list.
    timestamp_column : str, optional
        The timestamp data item name of the entity type to be created. An entity type
        has one timestamp typed data item identified as the time base for all time-series data processing. Default
        is 'rcv_timestamp_utc', which is Watson IOT Platform's default providing event receiving time.
    entity_type_table_prefix : str, optional
        The entity type's underlying table name's prefix. An entity type's
        underlying table name is in the format of '<entity_type_table_prefix>_<entity_type_name>'. Default is 'IOT'.
        Note that this is case insensitive, meaning table name is lower case transformed first.
    dimension_table_name : str, optional
        The name of the underlying table for the entity type's dimension data. An
        entity type has two kinds of data: time-series and non-time-series. Non-time-series data are called dimensions,
        which is basically per-device attributes. Default is using the format '<entity_type_table_name>_dimension'.
        Note that this is case insensitive, meaning table name is lower case transformed first.
    dimensions : list of str, optional
        The list of dimension names of the entity type to be created. Default is an
        empty list.
    write : bool, optional
        Whether to write the given dataframe df to the underlying table after the entity type is
        created.
    df : DataFrame, optional
        Dataframe to be written to the entity type's underlying table.
    delete_table_first : bool, optional
        Whether to delete the entity type's underlying table first. Default is True.
    drop_table_first : bool, optional
        Whether to drop the entity type's underlying table first. Default is False.
    use_wiotp : bool, optional
        Whether to create the corresponding Watson IOT Platform device type, interfaces, and
        mappings. Default is True.
    devices : list of str, optional
        The list of device IDs to be registered to Watson IOT Platform. Default is an empty list.
    """
    logger.info('Creating new entity type with name %s...', entity_type_name)

    if db is None:
        db = _get_db()

    # first lower case all column names, it's complicated with sqlalchemy and database, we just take
    # a simple approach, data items are all lower case (case-insensitive)
    if columns is None:
        columns = []
    for col in columns:
        col.name = col.name.lower()

    if timestamp_column is None:
        timestamp_column = default_timestamp_column
    else:
        timestamp_column = timestamp_column.lower()

    table_name = '%s_%s' % (entity_type_table_prefix.lower(), entity_type_name.lower())
    if dimension_table_name is None:
        dimension_table_name = '%s_dimension' % table_name

    # if drop table first, do it before anything else
    if write and db.if_exists(table_name=table_name, schema=db_schema):
        if drop_table_first:
            db.drop_table(table_name=table_name, schema=db_schema)
            db.metadata = MetaData(db.engine)
        elif delete_table_first:
            db.delete_data(table_name=table_name, schema=db_schema)

    wiotp_just_created = False
    if not is_local_mode() and use_wiotp:
        create_wiotp_devices(entity_type_name, devices)
        create_wiotp_pi_li_mappings(entity_type_name, devices, columns)
        wiotp_just_created = True

    if is_local_mode() or not _check_entity_type(entity_type_name):
        # normally, ICS creates AS side entity type fairly quickly, within seconds. but in some error condition,
        # it fails the create entity type and we want to manually create it on AS side so we can continue.
        # however, WIOTP events won't go into data lake until ICS recovers
        logger.warning('entity_type=%s not automatically created from WIOTP device_type=%s, at least not yet, creating it manually for now ...', entity_type_name, entity_type_name)

        table_already_exist = db.if_exists(table_name, db_schema)

        if not table_already_exist:
            # create table first by ourself
            iot_table = IotTypeTable(table_name, db, *columns, _timestamp=timestamp_column, extend_existing=True, schema=db_schema)

            # create timestamp index
            index_name = ('%s-timestamp' % table_name).upper()
            try:
                Index(index_name, iot_table.table.c[timestamp_column]).create(bind=db.engine)
            except Exception as e:
                logger.warning('failed creating index=%s for table %s.%s. Exception: %s', index_name, db_schema, table_name, e)

        # create the entity type object using the already created table
        entity_type = EntityType(
            table_name,
            db,
            **{
                'auto_create_table': False,
                'logical_name': entity_type_name,
                '_timestamp_col': timestamp_column,
                '_db_schema': db_schema,
                '_dimension_table_name': dimension_table_name,
            })

        if dimensions is not None and isinstance(dimensions, list) and len(dimensions) > 0:
            entity_type.make_dimension(entity_type._dimension_table_name.lower(), *dimensions)

        if not is_local_mode():
            # what register() does is to use the table schema loaded to register data items to AS
                # TODO: Health Check API Here
            is_91_instance = monitor_health_check()
            resp = None
            if (is_91_instance is True):
                resp = entity_type.register()
            else:
                resp = register_90_model(entity_type)
            logger.debug('Attempted to register entity type. Results: entity_type=%s, register_response=%s', entity_type.logical_name, resp)
    else:
        logger.info('Entity type was automatically created from WIOTP device type=%s', entity_type_name)
        entity_type = EntityType(
            table_name,
            db,
            **{
                'logical_name': entity_type_name,
                '_timestamp_col': timestamp_column,
                '_db_schema': db_schema,
                '_dimension_table_name': dimension_table_name,
            })

        if wiotp_just_created and timestamp_column != default_timestamp_column:
            # call AS entity type creation API to set the correct time-base data item, if not using the default one
            # by default, when AS entity type is created and custom time-base data item is used, AS randomly chooses
            # one as the time base. if we know what we want to use, we have to call it again to set it correctly
            resp = None
            is_91_instance = monitor_health_check()
            if (is_91_instance is True):
                resp = entity_type.register()
            else:
                resp = register_90_model(entity_type)
            logger.debug('Attempted to register entity type. Results: entity_type=%s, register_response=%s', entity_type.logical_name, resp)

    #MAS8.7 support for Monitor2.0 new table IOT_DEVICE_TYPE_{{entity_type_id}}
    
    json = { 'search': entity_type_name}
    logger.debug('Getting entity type new table name from Monitor. Request body json=%s', json)
    new_table_name = api._get_entity_type_table_name_from_monitor(json)
    logger.debug('Retrieved new table name. new_table_name=%s', new_table_name)

    # write
    if write and df is not None:
        # it appears that sqlalchemy handles mixed-case columns in a bad way, so to make it simpler,
        # we always use lower case column names
        df = df.copy()
        df.columns = map(str.lower, df.columns)

        df['devicetype'] = entity_type_name

        # For Monitor 2.0, the below columns are removed
        #df['logicalinterface_id'] = ''
        #df['eventtype'] = ''
        #df['format'] = ''
        df['updated_utc'] = dt.datetime.utcnow()

        db.start_session()
        try:
            _write_dataframe(df=df, table_name=new_table_name, db=db, db_schema=db_schema)
        except:
            db.session.rollback()
            raise
        finally:
            db.commit()

    logger.info('Successfully created new entity type: entity_type=%s', entity_type.name)
    return entity_type

def mergeTwoDicts(dict1, dict2):
    #print('in Merge')
    res = {**dict1, **dict2}
    #print(res)
    return res

def create_entity_type_v2(
        entity_type_name,
        columns=[],
        db=None,
        db_schema=None,
        timestamp_column=None,
        entity_type_table_prefix='IOT',
        dimension_table_name=None,
        dimensions=[],
        write=True,
        df=None,
        delete_table_first=True,
        drop_table_first=False,
        use_wiotp=True,
        batch_size=25,
        devices=[]
        ):
    """Create an entity type.

    Note that currently, write-related paramters are only for time-series data, but not for non-time-series dimensions.

    Parameters
    ----------
    entity_type_name : str
        The name of the entity type to create.
    columns (list of sqlalchemy.Column): Optoinal, the list of data item names of the entity type to be created.
        Default is an empty list. Note that there is at least one system default data item representing the device
        ID and another representing the time series base, which do not need to be included in this list.
    timestamp_column : str, optional
        The timestamp data item name of the entity type to be created. An entity type
        has one timestamp typed data item identified as the time base for all time-series data processing. Default
        is 'rcv_timestamp_utc', which is Watson IOT Platform's default providing event receiving time.
    entity_type_table_prefix : str, optional
        The entity type's underlying table name's prefix. An entity type's
        underlying table name is in the format of '<entity_type_table_prefix>_<entity_type_name>'. Default is 'IOT'.
        Note that this is case insensitive, meaning table name is lower case transformed first.
    dimension_table_name : str, optional
        The name of the underlying table for the entity type's dimension data. An
        entity type has two kinds of data: time-series and non-time-series. Non-time-series data are called dimensions,
        which is basically per-device attributes. Default is using the format '<entity_type_table_name>_dimension'.
        Note that this is case insensitive, meaning table name is lower case transformed first.
    dimensions : list of str, optional
        The list of dimension names of the entity type to be created. Default is an
        empty list.
    write : bool, optional
        Whether to write the given dataframe df to the underlying table after the entity type is
        created.
    df : DataFrame, optional
        Dataframe to be written to the entity type's underlying table.
    delete_table_first : bool, optional
        Whether to delete the entity type's underlying table first. Default is True.
    drop_table_first : bool, optional
        Whether to drop the entity type's underlying table first. Default is False.
    use_wiotp : bool, optional
        Whether to create the corresponding Watson IOT Platform device type, interfaces, and
        mappings. Default is True.
    devices : list of str, optional
        The list of device IDs to be registered to Watson IOT Platform. Default is an empty list.
    """

    if db is None:
        db = _get_db()

    # first lower case all column names, it's complicated with sqlalchemy and database, we just take
    # a simple approach, data items are all lower case (case-insensitive)
    column_name_list=[]
    if columns is None:
        columns = []
    for col in columns:
        col.name = col.name.lower()
        column_name_list.append(col.name)


    if timestamp_column is None:
        timestamp_column = 'evt_timestamp'
    else:
        timestamp_column = timestamp_column.lower()

    # For MAS 8.8 , table_name will be from the response json
    table_name = '%s_%s' % (entity_type_table_prefix.lower(), entity_type_name.lower())
    if dimension_table_name is None:
        dimension_table_name = '%s_dimension' % table_name

    # if drop table first, do it before anything else
    if write and db.if_exists(table_name=table_name, schema=db_schema):
        if drop_table_first:
            db.drop_table(table_name=table_name, schema=db_schema)
            db.metadata = MetaData(db.engine)
        elif delete_table_first:
            db.delete_data(table_name=table_name, schema=db_schema)

    #columns = options['columns']
    # remove rcv_timestamp_utc
    if 'evt_timestamp' in column_name_list:
        column_name_list.remove('evt_timestamp')  
    if 'rcv_timestamp_utc' in column_name_list:
        column_name_list.remove('rcv_timestamp_utc')  
    # set evt_timestamp to be 
    #timestampDto= {
    #        "name": "evt_timestamp",
    #        "eventName": "EventA",
    #        "columnType": "TIMESTAMP",
    #        "columnName": "evt_timestamp",
    #       "type": "METRIC",
    #        "transient": False
    #    }

    # set evt_timestamp to be 
    timestampDto= {
            "name": timestamp_column,
            "eventName": "EventA",
            "columnType": "TIMESTAMP",
            "columnName": timestamp_column,
            "type": "METRIC",
            "transient": False
        }

    # need to store the map of column name and its type for V2 device type API 
    column_type_map={}
    for col, dtype in df.dtypes.to_dict().items():
        
        if is_float_dtype(dtype) or is_integer_dtype(dtype) :
            column_type_map[col.lower()] = 'NUMBER'
        elif is_string_dtype(dtype):
            column_type_map[col.lower()] = 'LITERAL' 


    dataItemDto=  [{
        "name": column_name.lower(),
        #columnName': column_name.lower(),
        "eventName": "EventA",
        "columnType": column_type_map.get(column_name.lower(), 'NUMBER'),
        "columnName": column_name.lower(),
        "type": "METRIC",
        "transient": False
        } for column_name in column_name_list if column_name != timestamp_column ]
    dataItemDto.append(timestampDto)
    logger.debug('dataItemDto: %s', dataItemDto)

    if not is_local_mode() and use_wiotp:
        device_type_uuid = create_wiotp_devices_v2(entity_type_name, dataItemDto,devices,timestamp_column)

        #No need to call create_wiotp_pi_li_mappings
        #create_wiotp_pi_li_mappings(entity_type_name, devices, columns)

    if is_local_mode() or not _check_entity_type(entity_type_name):
        # normally, ICS creates AS side entity type fairly quickly, within seconds. but in some error condition,
        # it fails the create entity type and we want to manually create it on AS side so we can continue.
        # however, WIOTP events won't go into data lake until ICS recovers
        logger.warning('entity_type=%s not automatically created from WIOTP device_type=%s, at least not yet, create it manually for now ...' % (entity_type_name, entity_type_name))

        table_already_exist = db.if_exists(table_name, db_schema)

        if not table_already_exist:
            # create table first by ourself
            iot_table = IotTypeTable(table_name, db, *columns, _timestamp=timestamp_column, extend_existing=True, schema=db_schema)

            # create timestamp index
            index_name = ('%s-timestamp' % table_name).upper()
            try:
                Index(index_name, iot_table.table.c[timestamp_column]).create(bind=db.engine)
            except Exception as e:
                logger.warning('failed creating index=%s for table=%s schema=%s: %s', index_name, table_name, db_schema, e)

        # create the entity type object using the already created table
        logger.info('In the if branch, before calling entity_type = EntityType timestamp_column=%s', timestamp_column)
        entity_type = EntityType(
            table_name,
            db,
            **{
                'auto_create_table': False,
                'logical_name': entity_type_name,
                '_timestamp_col': timestamp_column,
                '_db_schema': db_schema,
                '_dimension_table_name': dimension_table_name,
            })

        if dimensions is not None and isinstance(dimensions, list) and len(dimensions) > 0:
            entity_type.make_dimension(entity_type._dimension_table_name.lower(), *dimensions)

        if not is_local_mode():
            # what register() does is to use the table schema loaded to register data items to AS
            #resp = entity_type.register()
            logger.debug("v2 device type, no need to call entity_type.register()")
    else:
        logger.info('In the else branch , before calling entity_type = EntityType timestamp_column=%s', timestamp_column)
        entity_type = EntityType(
            table_name,
            db,
            **{
                'logical_name': entity_type_name,
                '_timestamp_col': timestamp_column,
                '_db_schema': db_schema,
                '_dimension_table_name': dimension_table_name,
            })

        #if wiotp_just_created and timestamp_column != default_timestamp_column:
            # call AS entity type creation API to set the correct time-base data item, if not using the default one
            # by default, when AS entity type is created and custom time-base data item is used, AS randomly chooses
            # one as the time base. if we know what we want to use, we have to call it again to set it correctly
            #resp = entity_type.register()
            #logger.debug('entity_type=%s register_response=%s', entity_type.logical_name, resp)
            #logger.debug("No need to call entity_type.register()")

    #MAS8.7 support for Monitor2.0 new table IOT_DEVICE_TYPE_{{entity_type_id}}
    
    json = { 'search': entity_type_name}
    logger.debug('before call api._get_entity_type_table_name_from_monitor json=%s', json)
    new_table_name = api._get_entity_type_table_name_from_monitor(json)
    logger.debug('new_table_name=%s', new_table_name)

    # use_wiotp will be false if import_only is true in setup_iot_type_v2
    if use_wiotp is False: # 
        device_type_uuid = api._get_entity_type_uuid_from_monitor(entity_type_name)

    # get_device_uid_map
    # For MAS8.8 Monitor only accepts 50 devices per request
    logger.info('before batch_size=%s', str(batch_size))
    #print('batch_size',batch_size)
    if batch_size > 50:
        batch_size = 50
    logger.info('after batch_size=%s', str(batch_size))  
    chunk=batch_size
    #print('chunk',chunk)
    total_device= len(devices)


    device_uid_map ={}
    import math
    for x in range(math.ceil(total_device/chunk)):
        #print(x)
        st = chunk * x
        end = chunk * (x+1)
        if end > total_device :
            end = total_device
        #print("chunk #" + str(x))
        logger.info('chunk #=%s', str(x))
        
        device_list_per_request = devices[st:end]
        #print(device_list_per_request)
        id_map=get_device_uid_map(device_type_uuid, device_list_per_request )
    
        #print('id_map',id_map)
        logger.info('id_map=%s', str(id_map))
        device_uid_map=mergeTwoDicts(device_uid_map,id_map)
    
    #print(device_uid_map)
    logger.info('device_uid_map=%s', str(device_uid_map))

    #device_uid_map = get_device_uid_map(device_type_uuid, devices )

    if write and df is not None:
        # it appears that sqlalchemy handles mixed-case columns in a bad way, so to make it simpler,
        # we always use lower case column names
        df = df.copy()

        #device_id_set=set(df['deviceid'])
        #first_device_id_name=device_id_set.pop()
        #print(first_device_id_name)

        df.columns = map(str.lower, df.columns)

        #######
        #WARNING: need to be replaced with Monitor API
        #######
        #df['device_uid'] = get_device_uid_from_monitor(device_type_uuid,first_device_id_name ) 

        df['device_uid'] = df['deviceid'].map(device_uid_map)

        df['devicetype'] = entity_type_name

        # For Monitor 2.0, the below columns are removed
        #df['logicalinterface_id'] = ''
        #df['eventtype'] = ''
        #df['format'] = ''
        df['updated_utc'] = dt.datetime.utcnow()

        # Need to create partition first
        create_partition_for_dataframe(db,df,timestamp_column,new_table_name)

        db.start_session()
        try:
            _write_dataframe(df=df, table_name=new_table_name, db=db, db_schema=db_schema,device_type=entity_type_name)
            logger.info('Succesfully finished execution of creating entity types')
        except:
            db.session.rollback()
            raise
        finally:
            db.commit()

    return entity_type


SERVER_ENTITY_TYPES = dict()

def get_device_uid_from_monitor(device_type_uuid,device_name):

    device_uid = 0

    input_json={"name": device_name}

    monitor_url_path='deviceTypes/'+device_type_uuid + "/devices"

    resp = _call_as_v2(url_path=monitor_url_path, method='post', json=input_json)
    
    '''
    [
    {
        "ddId": "mm_D_UHVtcF9BRk1fVEVTVDM5_c291cmRvdWdoXzAxMA",
        "name": "sourdough_010",
        "origin": null,
        "deviceUID": 58,
        "dimensions": {
            "deviceid": "sourdough_010"
        }
    },
    {
        "ddId": "mm_D_UHVtcF9BRk1fVEVTVDM5_c291cmRvdWdoXzAz",
        "name": "sourdough_03",
        "origin": null,
        "deviceUID": 59,
        "dimensions": {
            "deviceid": "sourdough_03"
        }
    }
]
    '''
    if resp is not None and resp.status_code == 200:
        json_array = json.loads( resp.text)
        for each_item in json_array:
            device_uid = each_item['deviceUID']

    return device_uid






def get_entity_type(entity_type_name, db, *args, **kwargs):
    """Get an EntityType instance by name. Name may be the logical name shown in the UI or the table name.'

    Parameters
    ----------
    entity_type_name : `str`
        The name of the entity type.
    """
    logger.debug('Retrieving EntityType instance by this name: %s', entity_type_name)
    if db is None:
        db = _get_db()

    if not is_local_mode():
        if entity_type_name not in SERVER_ENTITY_TYPES:
            db_schema = get_as_schema(None)
            # TODO: Health Check API Here
            is_91_instance = monitor_health_check()


            if (is_91_instance is True):
                entity_type_id = get_entity_type_id_by_entity_type_name(entity_type_name)
                logger.debug("Recevied  entity_type_id=%s", entity_type_id)
                SERVER_ENTITY_TYPES[entity_type_name] = ServerEntityType(resource_id=str(entity_type_id), db=db, db_schema=db_schema)
            else:

                uuid = get_uuid_by_entity_type_name(db,entity_type_name)
                logger.debug('Creating instance of ServerEntityType with uuid=%s', uuid)
                SERVER_ENTITY_TYPES[entity_type_name] = ServerEntityType(resource_id=str(uuid), db=db, db_schema=db_schema)
        return  SERVER_ENTITY_TYPES[entity_type_name] 
    else:
        metadata = db.entity_type_metadata[entity_type_name]
        logger.debug('Retrieved entity type metadata: %s', metadata)

        base_kwargs = {
            'auto_create_table': False,
            'logical_name': metadata['name'],
            '_timestamp': metadata['metricTimestampColumn'].lower(), # always use lower case column name (sqlalchemy)
            '_db_schema': metadata['schemaName'],
            '_dimension_table_name': metadata['dimensionTableName'],
            '_entity_type_id': metadata.get('entityTypeId', None),
        }
        base_kwargs.update(kwargs)
        return EntityType(
            metadata['metricTableName'],
            db,
            *args,
            **base_kwargs
        )


def _validate_resampling(entity_type_name, db, time_grain, agg_methods, agg_outputs):
    """Validate resampling parameters.

    This method also normalize the given time_grain, according to the support for pushing aggregation down to SQL.
    Whenever possible, the aggregation is pushed down to SQL to take advantge of that. Only when not possible, it
    is done by Pandas (with all data loaded back first).

    Since iotfunctions does push down some aggregation to SQLs, different database might have different support. For
    current supported databases, SQLite is the most limited. DB2 and PostgreSQL are same (in supported resampling
    features).

    There are also some limitation from defect of iotfunctions, mostly when 'first' or 'last' is used. Another
    limitation from iotfunctions currently is for minute and day, only 1 minute and 1 day can be pushed down to
    SQL.

    For DB2 and PostgreSQL, 'first' and 'last' only works when aggregation is pushed down to SQL, but not for
    Pandas aggregation (iotfunctions defect). So this means only {T, H, day, week, month, year} and their Pandas
    offset variants (note only n=1).

    For SQLite, 'first' and 'last' only works with 'day' and direct timestamp, nothing else. Any Pandas aggregation
    does not work for SQLite.
    """
    logger.debug('Validating resampling parameters for entity_type_name=%s', entity_type_name)

    # normalize parameters
    if agg_methods is None:
        agg_methods = dict()
    agg_methods = {k:v for k, v in agg_methods.items() if v is not None}
    for name, methods in agg_methods.items():
        if not isinstance(methods, list):
            agg_methods[name] = [methods]
    if agg_outputs is None:
        agg_outputs = dict()
    agg_outputs = {k:v for k, v in agg_outputs.items() if v is not None}
    for name, outputs in agg_outputs.items():
        if not isinstance(outputs, list):
            agg_outputs[name] = [outputs]

    db_query_used = False

    # find the timestamp column of the entity type
    resp= get_entity_from_metadata(db.entity_type_metadata,entity_type_name)
    if resp['name'] == entity_type_name:
        metadata= resp
    if metadata is None:
        raise RuntimeError('entity_type_name=%s not found' % entity_type_name)
    if isinstance(metadata, EntityType):
        timestamp_column = metadata._timestamp
    else:
        timestamp_column = metadata['metricTimestampColumn']
    
    #Somehow Monitor return 'metricTimestampColumn': None,
    if timestamp_column is None:
        timestamp_column ='evt_timestamp'
    # dataframe should always use lower case column name (sqlalchemy loaded)
    timestamp_column = timestamp_column.lower()

    # validate and normalize time_grain
    if time_grain is None:
        pass
    elif time_grain == timestamp_column:
        # resampling directly on timestamp, can be used to eliminate duplicate records at same time
        db_query_used = True
    elif time_grain.lower() == 'day':
        # 'day' works for all and can be pushed down to sql safely, make sure it's lower case
        time_grain = 'day'
        db_query_used = True
    elif time_grain.lower() == 'week':
        if db.db_type == 'sqlite':
            time_grain = 'W-SUN'
        else:
            time_grain = 'week'
            db_query_used = True
    elif time_grain.lower() == 'month':
        if db.db_type == 'sqlite':
            time_grain = 'MS'
        else:
            time_grain = 'month'
            db_query_used = True
    elif time_grain.lower() == 'year':
        if db.db_type == 'sqlite':
            time_grain = 'YS'
        else:
            time_grain = 'year'
            db_query_used = True
    else:
        # force using Pandas aggregation for the rest scenarios
        try:
            offset = to_offset(time_grain)
            if type(offset) == Minute:
                if db.db_type != 'sqlite' and offset.n == 1:
                    time_grain = '1min'
                    db_query_used = True
                else:
                    time_grain = '%dT' % offset.n
            elif type(offset) == Hour:
                if db.db_type != 'sqlite' and offset.n == 1:
                    time_grain = '1H'
                    db_query_used = True
                else:
                    time_grain = '%dh' % offset.n
            elif type(offset) == Day and offset.n == 1:
                time_grain = 'day'
                db_query_used = True
            elif type(offset) == MonthBegin and offset.n == 1 and db.db_type != 'sqlite':
                time_grain = 'month'
                db_query_used = True
            elif type(offset) == YearBegin and offset.n == 1 and db.db_type != 'sqlite':
                time_grain = 'year'
                db_query_used = True
        except:
            raise ValueError('invalid time_grain=%s' % time_grain)

    # verify whether given agg methods are supported
    first_used, last_used = None, None
    unsupported_methods = dict()
    supported_methods = {'count', 'first', 'last', 'max', 'mean', 'min', 'sum'}
    if db.db_type != 'sqlite':
        # alchemy sqlite does not support 'std'
        supported_methods.add('std')
    for name, methods in agg_methods.items():
        for method in methods:
            if method not in supported_methods:
                unsupported_methods.setdefault(name, set()).add(method)
            elif method == 'first':
                first_used = True
            elif method == 'last':
                last_used = True
    if len(unsupported_methods) > 0:
        raise ValueError('unsupported resampling methods: %s' % str(unsupported_methods))
    if first_used or last_used:
        # TODO iotfunctions has defect for first/last when using Pandas doing aggregation
        if not db_query_used:
            methods = 'first, last' if first_used and last_used else ('first' if first_used else 'last')
            raise ValueError('unsupported resampling methods %s when time_grain=%s and db_type=%s' % (methods, time_grain, db.db_type))

    # verify whether outputs are given correctly
    invalid_outputs = dict()
    for name, methods in agg_methods.items():
        # if for a name it is not given, we can automatically generate one, but if some
        # are given to a name but the number of outputs do not cover the required number
        # of methods, it is an error
        if name not in agg_outputs:
            for method in methods:
                agg_outputs.setdefault(name, list()).append('%s_%s' % (name, method))
        elif len(agg_outputs.setdefault(name, list())) > 0:
            if len(methods) > len(agg_outputs.setdefault(name, list())):
                invalid_outputs[name] = agg_outputs[name]
    if len(invalid_outputs) > 0:
        raise ValueError('invalid resampling outputs %s, not matching the number of given resampling methods: %s' % (str(invalid_outputs), str({name: methods for name, methods in agg_methods.items() if name in invalid_outputs})))
    invalid_outputs = dict()
    for name, outputs in agg_outputs.items():
        # outputs cannot be given to a name if it is not in agg_methods
        if name not in agg_methods:
            invalid_outputs[name] = outputs
    if len(invalid_outputs) > 0:
        raise ValueError('invalid resampling outputs %s, no matching keys in the given resampling methods: %s' % (str(invalid_outputs), str(agg_methods)))

    logger.debug('Finished validation for DB type %s and resampling parameters: time_grain=%s, agg_methods=%s, agg_outputs=%s', db.db_type, time_grain, agg_methods, agg_outputs)

    return (time_grain, agg_methods, agg_outputs)


# TODO this method is a wrapper to work-around a few issues in iotfunctions, change to use EntityType.get_data when all issues are fixed
def get_entity_type_data(entity_type_name, entity_type, start_ts=None, end_ts=None, entities=None, data_items=None, time_grain=None, agg_methods=None, agg_outputs=None):
    """This method gets entity type's data.

    The data retrieving support querying by time range and filtering by either entities. It also support column projects.
    Also, downsampling is supported. For example, reading all the data with incoming rate of 1 event per second can be
    downsampled to take 1 sample per 5 seconds. The aggregation methods of desampling can also be controlled, currently
    with options 'max', 'min', 'mean', 'count', 'sum', 'std', 'first', and 'last'. It is also possible to generate
    multiple desampled outputs (with different methods) from the same input event.

    The downsampling applies to all data_items retrieved, that is, if requesting downsampling to 1 sample per day, all data_items
    requested are downsampled to 1 sample per day.

    Here's an example. Given `time_grain` of `5T` (5 minutes) and `agg_methods` of `{'load': ['min', 'mean', 'max']}`,
    the returned dataframe has 'load_min', 'load_mean', and 'load_max' downsampled data (assuming 'load' is the only column
    in the entity type). But if the entity type as an extra column 'temperature' in this case, there would be an extra
    downsampled output 'temperature' representing the mean of every 5 minutes.

    Note that if the entity type has dimensions, they are also retrieved, joined properly and returned.

    Underlying, it relies on iotfunctions to retrieve the data. Currently, it only supports raw data, not derived data.

    Parameters
    ----------
    entity_type : `iotfunctions.metadata.EntityType`
        The entity type to get data.
    start_ts : str, optional
        The start of the range of the history to be retrieved, inclusive. Default is None, meaning all the way
        back the earliest record. It is given in the format like '2019-01-31 06:51:34.234561', with time portion
        optional.
    end_ts : str, optional
        The end of the range of the history to be retrieved, exclusive. Default is None, meaning all the way
        to the latest record. It is given in the format like '2019-01-31 06:51:34.234561', with time portion
        optional.
    entities : `list` of `str`, optional
        A list of assets to filter. Only data of these assets are retrieved. Default is no filtering.
    data_items : `list` of `str`, optional
        A list of data items to retrieved. Only the given data items are returned, but not other data items in the entity
        type. Default is None, tht is, all data items (and dimensions if available) of the entity type are returned.
    time_grain : `str`, optional
        The desampling frequency, given as [Pandas offset alias](http://pandas.pydata.org/pandas-docs/stable/user_guide/timeseries.html#offset-aliases).
        For example, '1D' to return 1 sample per day.
    agg_methods : `dict`, optional
        The downsampling aggregation methods used. The key is the column name with value being the list of
        aggregation methods for the column. Current aggregation methods available are {'max', 'min', 'mean',
        'count', 'sum', 'std', 'first', and 'last'}. Different column can have different set of methods. Default
        is empty `dict`. When there's no entry in this `dict` for a column, by default 'mean' is used for
        numeric types (integer, float) and 'max' is used for all other types.
    agg_outputs : `dict`, optional
        The donwsampled output names. The key is the column name with value being the list of output names,
        matching the corresponding list in `agg_methods` for the same key (list ordering matters). Any key of
        this `dict` not in `agg_methods` results in ValueError raised. If some key in `agg_methods` is not
        given in this `dict`, the output names are automatically generated in the format `<name>_<method>`.
        If a key exists in both `agg_methods` and this `dict`, the value (a list) must be of the same length
        otherwise a ValueError is raised. Default is empty `dict`.

    Returns
    -------
    `pandas.DataFrame`
        Note that the return dataframe does not guaranteed to be sorted by index.
    """

    if entity_type is None:
        raise ValueError('parameter entity_type cannot be None')
    if not isinstance(entity_type, EntityType):
        raise TypeError('parameter entity_type must be given an iotfunctions.metadata.EntityType object')

    if isinstance(data_items, list) and len(data_items) == 0:
        data_items = None

    logger.debug('Retrieving entity type data for entity type %s', entity_type_name)
    logger.debug('Data items to be retrieved: data_items=%s', data_items)
    logger.debug('Entity type data items: %s', entity_type._data_items)
    # TODO keep original data_items order after loaded

    # separate raw from derived data items
    raw_data_items, raw_data_items_rename, derived_data_items, derived_data_item_names, derived_data_item_types = list(), dict(), dict(), list(), dict()
    derived_non_agg, derived_ts_per_entity, derived_ts_all_entity, derived_non_ts_per_entity, derived_non_ts_all_entity, derived_with_dimentions = [], [], [], [], [], []
    for item in entity_type._data_items:
        item_name, item_type, column_name = item['name'], item['type'], item['columnName']
        if item_type == 'METRIC':
            if data_items is None or item_name in data_items:
                raw_data_items.append(column_name)
                # rename loaded table column name to data item name, if different (case-sensitive)
                raw_data_items_rename[column_name] = item_name
            elif data_items is not None:
                for dt in data_items:
                    if dt.lower() == item_name.lower():
                        raw_data_items.append(column_name)
                        raw_data_items_rename[column_name] = dt
        elif item_type == 'DERIVED_METRIC':
            if data_items is None or item_name in data_items:
                # if not for getting all data_items, raise error if any unsupported grain is requested
                grain = item['kpiFunctionDto']['granularity']
                g = entity_type._granularities_dict[grain] if grain is not None else None
                if data_items is not None:
                    if g is None:
                        derived_non_agg.append((item_name, item_type, grain, g))
                    elif g.dimensions is not None and len(g.dimensions) > 0:
                        derived_with_dimentions.append((item_name, item_type, grain, g))
                    elif g.entity_id is None and g.timestamp is None:
                        derived_non_ts_all_entity.append((item_name, item_type, grain, g))
                    elif g.timestamp is None:
                        derived_non_ts_per_entity.append((item_name, item_type, grain, g))
                    elif g.entity_id is None:
                        derived_ts_all_entity.append((item_name, item_type, grain, g))
                    else:
                        derived_ts_per_entity.append((item_name, item_type, grain, g))

                derived_data_items.setdefault(grain, []).append(item_name)
                derived_data_item_names.append(item_name)
                derived_data_item_types[item_name] = item['columnType']

    logger.debug('raw_data_items=%s', raw_data_items)
    logger.debug('raw_data_items_rename=%s', raw_data_items_rename)
    logger.debug('derived_data_items=%s', derived_data_items)

    logger.debug('derived_non_agg=%s, derived_ts_per_entity=%s, derived_ts_all_entity=%s, derived_non_ts_per_entity=%s, derived_non_ts_all_entity=%s, derived_with_dimentions=%s', derived_non_agg, derived_ts_per_entity, derived_ts_all_entity, derived_non_ts_per_entity, derived_non_ts_all_entity, derived_with_dimentions)

    if any(len(l) > 0 for l in [derived_ts_per_entity, derived_ts_all_entity, derived_non_ts_per_entity, derived_non_ts_all_entity, derived_with_dimentions]):
        msg = ''
        for iname, itype, gname, g in derived_with_dimentions:
            msg += 'data_item_name=%s of data_item_grain=%s using grain_dimensions=%s is not supported\n' % (iname, gname, g.dimensions)
        for iname, itype, gname, g in (derived_non_ts_per_entity + derived_non_ts_all_entity):
            msg += 'data_item_name=%s of non-time-series data_item_grain=%s is not supported\n' % (iname, gname)
        for iname, itype, gname, g in derived_ts_all_entity:
            msg += 'data_item_name=%s of group-wise time-series data_item_grain=%s is not supported\n' % (iname, gname)
        if len(raw_data_items) > 0 or len(derived_non_agg) > 0:
            for iname, itype, gname, g in derived_ts_per_entity:
                msg += 'data_item_name=%s of per-entity time-series aggregated data_item_grain=%s is not supported to be used together with either raw or per-entity derived time-series non-aggregated data items\n' % (iname, gname)

        if len(msg) > 0:
            raise ValueError(msg)

    if data_items is not None:
        invalid_data_items = list(set(data_items) - {entity_type._entity_id, entity_type._timestamp} - set(raw_data_items_rename.values()) - set(derived_data_item_names))
        if len(invalid_data_items) > 0:
            raise ValueError('invalid data_items=%s specified' % invalid_data_items)

    if len(raw_data_items) == 0 and len(derived_data_items) == 0:
        raise ValueError('invalid data_items=%s requested from entity_type=%s which has entity_type._data_items=%s' % (data_items, entity_type.logical_name, entity_type._data_items))

    # TODO validate derived data limitation (like only time-series allowed)
    time_grain, agg_methods, agg_outputs = _validate_resampling(entity_type_name, entity_type.db, time_grain, agg_methods, agg_outputs)

    # setup entity type pre aggregate properly
    entity_type._pre_aggregate_time_grain = time_grain
    entity_type._pre_agg_rules = agg_methods.copy()
    entity_type._pre_agg_outputs = agg_outputs.copy()

    # get raw data

    df_raw = None
    if len(raw_data_items) > 0:
        for item in [entity_type._entity_id, entity_type._timestamp]:
            if item not in raw_data_items:
                raw_data_items.insert(0, item)
        
        logger.debug('before entity_type.get_data raw_data_items=%s', raw_data_items)
        
        # according to Peter ,entity_type._pre_aggregate_time_grain  has to be None ;
        # revert back the entity_type._pre_aggregate_time_grain = None because we need to downsample Daily if the raw sensor data is in minute
        # entity_type._pre_aggregate_time_grain = None

        # per Peter, Predict needs to pass in data_items
        logger.debug('New code: before entity_type.get_data data_items=%s', data_items)
        df_raw = entity_type.get_data(start_ts=start_ts, end_ts=end_ts, entities=entities, columns=data_items)
        #df_raw = entity_type.get_data(start_ts=start_ts, end_ts=end_ts, entities=entities, columns=raw_data_items)

        #Kewei try to fix the https://github.ibm.com/wiotp/Maximo-Asset-Monitor/issues/5085 on predict
        feature_columns=set(list(df_raw.columns)) - set(['ENTITY_ID','_timestamp'])
         
        #Kewei keep backward compatible, we need to convert the column name to ALL upper case
        logger.debug('After entity_type.get_data, Data items to be retrieved: data_items=%s', data_items)
        if all([item.isupper() for item in data_items]):
            lower_to_upper_map={}
            for x in feature_columns:
                lower_to_upper_map[x]  = x.upper()
                df_raw.rename(columns=lower_to_upper_map,inplace=True)      


        #raw_data_items_rename = {k:v for k, v in raw_data_items_rename.items() if k != v}
        #if len(raw_data_items_rename) > 0:
        #    df_raw = df_raw.rename(columns=raw_data_items_rename)

        logger.debug('Retrieved sensor data from IOT_device_type table from IOTPlatform. df=%s', log_df_info(df_raw, head=5, logger=logger, log_level=logging.DEBUG))

        if time_grain is not None:
            # only needed if there's resampling

            if isinstance(df_raw.columns[0], tuple):
                # using Pandas to aggregate results in multi-level columns
                renamed_cols = {}
                for src, methods in agg_methods.items():
                    for idx, method in enumerate(methods):
                        out = entity_type._pre_agg_outputs[src][idx]
                        renamed_cols['%s|%s' % (src, method)] = out if out is not None else src

                new_columns = []
                for col in df_raw.columns:
                    if len(col[-1]) == 0:
                        # multi-level last level empty string
                        new_columns.append('|'.join(col[:-1]))
                    else:
                        new_columns.append('|'.join(col))

                    if new_columns[-1] not in renamed_cols:
                        renamed_cols[new_columns[-1]] = col[0]
                df_raw.columns = new_columns

                if len(renamed_cols) > 0:
                    df_raw = df_raw.rename(columns=renamed_cols)
            else:
                # non-Pandas way, using SQLAlchemy, columns are renamed directly
                # revert back the iotfunctions default renaming back to original ones
                new_columns = []
                for col in df_raw.columns:
                    if col.startswith('mean_'):
                        if col[5:] not in agg_methods:
                            new_columns.append(col[5:])
                            continue
                    elif col.startswith('max_'):
                        if col[4:] not in agg_methods:
                            new_columns.append(col[4:])
                            continue
                    new_columns.append(col)
                df_raw.columns = new_columns

                # day uses alchemy directly returns string timestamp column, need to convert it
                if time_grain == 'day':
                    # https://github.ibm.com/maximo/Asset-Health-Insight/issues/14774
                    column_list=list(df_raw.columns)
                    # fix the issue like columns={'ENTITY_ID': 'O', 'VELOCITYX': 'float64', 'VELOCITYZ': 'float64', 'VELOCITYY': 'float64', '_timestamp': 'O', 'RCV_TIMESTAMP_UTC': 'O'}, 
                    # entity_type._timestamp=RCV_TIMESTAMP_UTC
                    # ValueError: cannot insert RCV_TIMESTAMP_UTC, already exists
                    if entity_type._timestamp in column_list:
                        try:
                            # if entity_type._timestamp=RCV_TIMESTAMP_UTC is in the column list, then drop it
                            logger.debug(' entity_type._timestamp =%s ' ,entity_type._timestamp)
                            
                            df_raw.drop(columns=entity_type._timestamp,inplace=True)
                        except:
                            # for model training in CP4D, RCV_TIMESTAMP_UTC is not in the df_raw column list
                            pass
                    df_raw = df_raw.reset_index(level=entity_type._timestamp)
                    df_raw = df_raw.astype({entity_type._timestamp: 'datetime64[ms]'})
                    df_raw = df_raw.set_index(keys=entity_type._timestamp, append=True)

            logger.debug('Renamed DF columns to proper names: df_raw=%s', log_df_info(df_raw, head=5, logger=logger, log_level=logging.DEBUG))

    # get derived data
    df_pool_per_entity_ts_base, df_pool_per_entity_ts, df_pool_all_entity_ts, df_pool_per_entity_non_ts, df_pool_all_entity_non_ts = [], [], [], [], []
    for grain, items in derived_data_items.items():
        logger.debug('grain=%s, items=%s', grain, items)

        pool = None
        if grain is None:
            table_name = 'dm_%s' % entity_type.logical_name.lower()
            index_columns = ['entity_id', 'timestamp']
            pool = df_pool_per_entity_ts_base
        else:
            g = entity_type._granularities_dict[grain]
            table_name = g.table_name
            index_columns = []
            if g.entity_id is not None:
                index_columns.append('entity_id')
            if g.timestamp is not None:
                index_columns.append('timestamp')
            if g.dimensions is not None and len(g.dimensions) > 0:
                index_columns.extend(g.dimensions)

            if g.entity_id is not None and g.timestamp is not None and (g.dimensions is None or len(g.dimensions) == 0):
                pool = df_pool_per_entity_ts
            # TODO loading aggregated derived data to concatenated with raw data is not supported yet
            # elif g.entity_id is not None:
            #     pool = df_pool_per_entity_non_ts
            # elif g.timestamp is not None:
            #     pool = df_pool_all_entity_ts
            # else:
            #     pool = df_pool_all_entity_non_ts

        if pool is not None:
            df = _get_derived_data(db=entity_type.db, table_name=table_name, schema=entity_type._db_schema, index_columns=index_columns, data_items=items)

            if time_grain is not None:
                time_grain_lookup = {
                    'day': 'D',
                    'week': 'W-SUN',
                    'month': 'MS',
                    'year': 'YS',
                }
                freq = time_grain_lookup[time_grain] if time_grain in time_grain_lookup else time_grain
                derived_agg_methods = agg_methods.copy()
                for item in df.columns:
                    if item not in derived_agg_methods and item not in ['entity_id', 'timestamp']:
                        derived_agg_methods[item] = 'mean' if derived_data_item_types[item] == 'NUMBER' else 'max'
                logger.debug('freq=%s, derived_agg_methods=%s', freq, derived_agg_methods)

                df = resample(df=df, time_frequency=freq, timestamp='timestamp', dimensions=['entity_id'], agg=derived_agg_methods)
                df = df.set_index(keys=['entity_id', 'timestamp'])
                logger.debug('df_derived_resampled=%s', log_df_info(df, head=5, logger=logger, log_level=logging.DEBUG))

            pool.append(df)

    # concat all
    if len(df_pool_per_entity_ts_base) > 0 or len(df_pool_per_entity_ts) > 0:
        # concat() cannot handle duplicate multi-index, hence when there's upstream and we need
        # to concat them, we must remove dupcliate rows (on index) first.
        all_dfs = []
        if df_raw is not None:
            df_raw = df_raw.loc[~df_raw.index.duplicated(keep='last')]
            all_dfs.append(df_raw)

        # TODO concatenating both raw level or non-aggregated derived data with time-series aggregated derived data is not supported yet
        if len(df_pool_per_entity_ts_base) > 0:
            all_dfs.extend(df_pool_per_entity_ts_base)
        elif df_raw is None:
            all_dfs.extend(df_pool_per_entity_ts)

        df = pd.concat(all_dfs, axis=1, sort=True, copy=False)
    else:
        df = df_raw

    logger.debug('Concatenated DataFrames together. Result: %s', log_df_info(df, head=5, logger=logger, log_level=logging.DEBUG))

    return df


def _get_derived_data(db, table_name, schema=None, index_columns=None, data_items=None, dimension_table=None, start_ts=None, end_ts=None, entities=None, parse_dates=None, entityid_column='entity_id', timestamp_column='timestamp', key_column='key'):
    """This method get data of an entity's derived data from a specific grain table.

    It basically loads a grain table then unstacks and consolidate the 4 different typed value columns.
    """

    if db is None:
        raise ValueError('parameter db cannot be None')
    if table_name is None:
        raise ValueError('parameter table_name cannot be None')

    if index_columns is None:
        index_columns = []
    if entityid_column is not None and entityid_column not in index_columns:
        entityid_column = None
    if timestamp_column is not None and timestamp_column not in index_columns:
        timestamp_column = None

    if len(index_columns) == 0:
        raise ValueError('empty index columns: index_columns=%s, entityid_column=%s, timestamp_column=%s' % (index_columns, entityid_column, timestamp_column))

    if entityid_column is None:
        entities = None
        dimension_table = None

    if timestamp_column is None:
        start_ts = None
        end_ts = None

    if parse_dates is None:
        parse_dates = []
    if timestamp_column is not None and timestamp_column not in parse_dates:
        parse_dates.append(timestamp_column)
    if 'value_t' not in list(map(str.lower, parse_dates)):
        parse_dates.append('value_t')

    # empty entities list would return nothing, prevent it
    if isinstance(entities, list) and len(entities) == 0:
        entities = None

    filters = {}
    if data_items is None:
        filters = {}
    elif isinstance(data_items, list) and len(data_items) > 0:
        filters[key_column] = data_items

    logger.debug('table_name=%s, index_columns=%s, entities=%s, filters=%s', table_name, index_columns, entities, filters)

    column_values = {'value_n':'float64', 'value_b':bool, 'value_s':str, 'value_t':'datetime64[ms]'}
    column_names = index_columns + [key_column] + list(column_values.keys())
    column_aliases = list(map(str.lower, column_names))
    logger.debug('column_names=%s, column_aliases=%s', column_names, column_aliases)

    query, table = db.query(table_name=table_name, schema=schema, column_names=column_names, column_aliases=column_aliases,
        timestamp_col=timestamp_column, start_ts=start_ts, end_ts=end_ts, entities=entities, dimension=dimension_table, filters=None, deviceid_col=entityid_column)
    # TODO work-around with iotfunctions defect on filters
    if filters is not None:
        dim = None
        if dimension_table is not None:
            dim = db.get_table(table_name=dimension_table, schema=schema)
        for d, members in filters.items():
            try:
                col_obj = db.get_column_object(table, d)
            except KeyError:
                try:
                    col_obj = db.get_column_object(dim, d)
                except KeyError:
                    raise ValueError('invalid filter_column=%s not found in table or dimension' % d)
            if isinstance(members, str):
                members = [members]
            if not isinstance(members, list):
                raise ValueError('invalid filter_column_values=%s for filter_columns' % (members, d))
            elif len(members) == 1:
                query = query.filter(col_obj == members[0])
            elif len(members) == 0:
                pass
            else:
               query = query.filter(col_obj.in_(members))
    logger.debug('sql=%s', query.statement)

    df = pd.read_sql_query(sql=query.statement, con=db.engine, parse_dates=parse_dates)
    logger.debug('loaded_%s_df=%s', table_name, log_df_info(df, head=5, logger=logger, log_level=logging.DEBUG))

    df = df.set_index(keys=index_columns + [key_column])
    df_values = []
    for value_col, value_dtype in column_values.items():
        df_value = df[pd.notna(df[value_col])][value_col].unstack(level=key_column, fill_value=np.nan)
        df_value = df_value.astype(value_dtype)
        if not df_value.empty:
            df_values.append(df_value)
            logger.debug('df_%s=%s', value_col, log_df_info(df_value, head=10, logger=logger, log_level=logging.DEBUG))

    df = pd.concat(df_values, axis=1)
    logger.debug('unstacked_%s_df=%s', table_name, log_df_info(df, head=5, logger=logger, log_level=logging.DEBUG))

    if len(filters) > 0:
        for item in filters[key_column]:
            if item not in df.columns:
                df[item] = None

    logger.debug('df_derived=%s', log_df_info(df, head=5, logger=logger, log_level=logging.DEBUG))

    return df


def _delete_entity_type(entity_type, entity_type_table_prefix='IOT', db=None, db_schema=None):
    if is_local_mode():
        if db is None:
            db = _get_db()

        table_name = '%s_%s' % (entity_type_table_prefix.lower(), entity_type.lower())

        # drop the entity type's table and dimension table
        db.drop_table(table_name=table_name, schema=db_schema)
        db.drop_table(table_name='%s_dimension' % table_name, schema=db_schema)
        db.metadata = MetaData(db.engine)

        resp = {}
    else:
        # archive
        resp = _call_as_meta(url_path='/entityType/%s/archive' % entity_type, method='put', api_type='meta')
        if resp is not None:
            logger.info('archived entity_type=%s: status_code=%s, response_text=%s', entity_type, resp.status_code, resp.text)

        # delete
        resp = _call_as_meta(url_path='/entityType/%s' % entity_type, method='delete', api_type='meta')
        if resp is not None:
            logger.info('deleted entity_type=%s: status_code=%s, response_text=%s', entity_type, resp.status_code, resp.text)

    return resp

def _delete_entity_type_v2(entity_type, entity_type_table_prefix='IOT', db=None, db_schema=None):
    if is_local_mode():
        if db is None:
            db = _get_db()

        table_name = '%s_%s' % (entity_type_table_prefix.lower(), entity_type.lower())

        # drop the entity type's table and dimension table
        db.drop_table(table_name=table_name, schema=db_schema)
        db.drop_table(table_name='%s_dimension' % table_name, schema=db_schema)
        db.metadata = MetaData(db.engine)

        resp = {}
    else:

        entity_type_uuid = _get_entity_type_uuid_from_monitor(entity_type)
        logger.info(' _delete_entity_type_v2 %s', entity_type_uuid)
        # archive
        resp = _call_as_v2(url_path='core/deviceTypes/%s/archive' % entity_type_uuid, method='put')
        if resp is not None:
            logger.info('archived entity_type=%s: status_code=%s, response_text=%s', entity_type, resp.status_code, resp.text)

        # delete
        resp = _call_as_v2(url_path='core/deviceTypes/%s' % entity_type_uuid, method='delete')
        if resp is not None:
            logger.info('deleted entity_type=%s: status_code=%s, response_text=%s', entity_type, resp.status_code, resp.text)

    return resp


"""
Tenant initialization related functions.
"""

def _get_apm_info():
    """Get APM tenant information currently set on Aanlytic Service, mainly APM ID, APM API base URL, and API key.

    This method returns a dict containing keys APM_ID, APM_API_BASEURL, and APM_API_KEY.

    Note that currently, this is only for per-tenant level.
    """
    resp = _call_as_kpi(url_path='', method='get', api_type='constants')
    if resp is not None:
        meta = resp.json()
        for p in meta:
            key = p['name']
            if key == 'apm_info':
                return p['value']

    return None


def _set_apm_info():
    """Set APM tenant information to Analytic Service, mainly APM ID, APM API base URL, and API key.

    This method is intended for Data Scientist users to change their API key, as the usual case is for
    the model pipeline registger method to automatically refresh and keep tenant info in sync with
    whateer DS users use to train model (which is guaranteed to be correct after training is completed).
    For case like DS just changes his API key and do not want to re-train but simply wants to change the
    key, this method would help (also since this method needs to call APM API to retrieve AS credentials,
    so the tenant info is also guaranteed to be correct).

    Note that currently, this is only for per-tenant level.
    """

    apm_info_constant = [
        {
            'name': 'apm_info',
            'entityType': None,
            'enabled': True,
            'value': {
                'APM_ID': APM_ID,
                'APM_API_BASEURL': APM_API_BASEURL,
                'APM_API_KEY': APM_API_KEY
            },
            'metadata': {
                'type': 'CONSTANT',
                'dataType': 'LITERAL',
                'description': 'APM tenant information.',
                'tags': None,
                'required': True,
                'values': None
            }
        }
    ]

    #resp = _call_as(url_path='', method='put', api_type='constants', json=apm_info_constant)
    # constants api uses REST_KPI_URL
    resp = _call_as_kpi(url_path='', method='put', api_type='constants', json=apm_info_constant)
    if resp is None:
        ## constants api uses REST_KPI_URL
        #resp = _call_as(url_path='', method='post', api_type='constants', json=apm_info_constant)
        resp = _call_as_kpi(url_path='', method='post', api_type='constants', json=apm_info_constant)

    return resp


def _get_tenant():
    if APM_ID is None:
        if all([os.environ.get(env, None) is not None for env in ['APM_ID', 'APM_API_BASEURL', 'APM_API_KEY']]) or all([os.environ.get(env, None) is not None for env in ['API_BASEURL', 'API_KEY', 'API_TOKEN', 'TENANT_ID']]):
            init_environ()
        else:
            raise RuntimeError('missing mandatory environment variable APM_ID')

    return _call_apm(url_path='/ds/tenant?instanceId=%s' % APM_ID, method='get')


# TODO auto conversion of old PMI production URL to new one, remove when all users are migrated
def _convert_apm_api_baseurl():
    global APM_API_BASEURL
    if APM_API_BASEURL is not None and APM_API_BASEURL == 'https://pmi-prod-cluster.us-south.containers.appdomain.cloud':
        APM_API_BASEURL = 'https://prod.pmi.apm.maximo.ibm.com'


INIT_DONE = False
def init_environ():
    """Initialize all the necessary runtime information.

    This function is usually the very first to call in order for this library to work properly. It initialized
    the runtime environment properly so this library can communicate with server side.

    Two basic assumptions about environments:

    * On AS runtime, it is assumed (API_BASEURL, API_KEY, API_TOKEN, TENANT_ID) are set.
    * In other environments, like Notebook or Python devopment environment, it is assumed (APM_ID, APM_API_BASEURL, APM_API_KEY) are set.

    The pre-requisites to run this function is either one of the two environment variables are set: [APM_ID,
    APM_API_BASEURL, APM_API_KEY] or [API_BASEURL, API_KEY, API_TOKEN]. You just need one of them. The first set
    is expected to be set when running from notebook or any Python development environments. The second set
    is most likely for running on Analytic Service.

    The initialization flow is:

    1. If any of (APM_ID, APM_API_BASEURL, APM_API_KEY) or any of (API_BASEURL, API_KEY, API_TOKEN) is not set, raise runtime error.
    2. If any of (APM_ID, APM_API_BASEURL, APM_API_KEY) is not set, call AS API to get APM credentials from AS constants.
    3. If any of (APM_ID, APM_API_BASEURL, APM_API_KEY) is not set, raise runtime error.
    4. Call APM get_tenant API to retrieved all tenant configuration (overriding any environment variale if set already).
        * If error response, raise runtime error.
        * Otherwise, initialize based on the response, set all keys in 'info' as environment variables, also set 'wml_cred' and 'mahi_url'.

    Returns
    -------
    tuple
        (APM_ID, APM_API_BASEURL, APM_API_KEY)
    """
    global INIT_DONE, APM_ID, APM_API_BASEURL, APM_API_KEY, MAXIMO_BASEURL, MAXIMO_API_CONTEXT, MAXIMO_LINKED, API_BASEURL, API_KEY, API_TOKEN, TENANT_ID, AS_SCHEMA

    if is_local_mode():
        return (APM_ID, APM_API_BASEURL, APM_API_KEY)

    if INIT_DONE:
        return (APM_ID, APM_API_BASEURL, APM_API_KEY)

    logger.info('Initializing environment...')
    if INIT_DONE is False:
        APM_ID = os.environ.get('APM_ID', None)
        APM_API_BASEURL = os.environ.get('APM_API_BASEURL', None)
        _convert_apm_api_baseurl()
        APM_API_KEY = os.environ.get('APM_API_KEY', None)
        MAXIMO_BASEURL = os.environ.get('MAXIMO_BASEURL', None)
        MAXIMO_API_CONTEXT = os.environ.get('MAXIMO_API_CONTEXT', None)
        MAXIMO_LINKED = True if MAXIMO_BASEURL is not None else None
        API_BASEURL = os.environ.get('API_BASEURL', None)
        API_KEY = os.environ.get('API_KEY', None)
        API_TOKEN = os.environ.get('API_TOKEN', None)
        TENANT_ID = os.environ.get('TENANT_ID', None)
        AS_SCHEMA = os.environ.get('AS_SCHEMA', None)

    if any([env is None for env in [APM_ID, APM_API_BASEURL, APM_API_KEY]]) and any([env is None for env in [API_BASEURL, API_KEY, API_TOKEN, TENANT_ID]]):
        raise RuntimeError('missing mandatory environment variables, make sure either all of [APM_ID, APM_API_BASEURL, APM_API_KEY] is set or all of [API_BASEURL, API_KEY, API_TOKEN] are set')

    # priority: environment -> AS constants

    if any([env is None for env in [APM_ID, APM_API_BASEURL, APM_API_KEY]]):
        # if any environment variable is not set, check if we can get from AS constants
        logger.debug('API_BASEURL=%s, API_KEY=%s, API_TOKEN=%s, TENANT_ID=%s', API_BASEURL, '********' if API_KEY is not None else '(None)', '********' if API_TOKEN is not None else '(None)', TENANT_ID)

        value = _get_apm_info()
        if value is None:
            raise RuntimeError('missing mandatory APM tenant info [APM_ID, APM_API_BASEURL, APM_API_KEY] in AS constants, make sure it is initialized properly')

        if APM_ID is None:
            APM_ID = value.get('APM_ID', None)
            os.environ['APM_ID'] = APM_ID
        if APM_API_BASEURL is None:
            APM_API_BASEURL = value.get('APM_API_BASEURL', None)
            _convert_apm_api_baseurl()
            os.environ['APM_API_BASEURL'] = APM_API_BASEURL
        if APM_API_KEY is None:
            APM_API_KEY = value.get('APM_API_KEY', None)
            os.environ['APM_API_KEY'] = APM_API_KEY

        # check again to make sure all 3 are present
        if any([env is None for env in [APM_ID, APM_API_BASEURL, APM_API_KEY]]):
            raise RuntimeError('missing mandatory APM tenant info [APM_ID, APM_API_BASEURL, APM_API_KEY] in AS constants, make sure it is initialized properly')

    logger.debug('APM_ID=%s, APM_API_BASEURL=%s, APM_API_KEY=%s', APM_ID, APM_API_BASEURL, '********' if APM_API_KEY is not None else '(None)')
    os.environ["SSL_VERIFY_APM"] = 'False'
    resp = _get_tenant()
    if resp is None:
        raise RuntimeError('error getting tenant information, check if tenant ID, API URL, and API key are correct, or contact system admin')

    resp = json.loads(resp.text)
    if not isinstance(resp, dict) or 'info' not in resp:
        raise RuntimeError('error getting malformed tenant information, check if tenant ID, API URL, and API key are correct, or contact system admin: %s' % str(resp))

    maked_resp = mask_credential_in_dict(resp)
    maked_resp['info'] = mask_credential_in_dict(resp['info'])
    logger.debug('resp=%s', json.dumps(maked_resp, indent=4, sort_keys=True))

    if 'mahi_url' in resp and resp['mahi_url'] is not None:
        k = 'MAXIMO_BASEURL'
        v = resp['mahi_url']
        logger.debug('setting environment variable %s=%s', k, mask_credential(k, v))
        os.environ[k] = v
        MAXIMO_BASEURL = v
        MAXIMO_LINKED = True
    else:
        MAXIMO_LINKED = False

    k = 'MAXIMO_API_CONTEXT'
    if k in os.environ:
        del os.environ[k]
    MAXIMO_API_CONTEXT = None
    if 'mahi_auth_url' in resp and resp['mahi_auth_url'] is not None:
        k = 'MAXIMO_API_CONTEXT'
        v = resp['mahi_auth_url']
        p = re.compile(r'(/[^/]+)/permission/', re.IGNORECASE)
        r = p.search(v)
        if r is not None:
            v = r.group(1)
            logger.debug('setting environment variable %s=%s', k, mask_credential(k, v))
            os.environ[k] = v
            MAXIMO_API_CONTEXT = v


    if 'mas_as_schema' in resp and resp['mas_as_schema'] is not None:
        k = 'AS_SCHEMA'
        v = resp['mas_as_schema']
        logger.debug('setting environment variable %s=%s', k, mask_credential(k, v))
        os.environ[k] = v
        for ssl_verify, default_value in {'SSL_VERIFY_APM': False, 'SSL_VERIFY_AS': False, 'SSL_VERIFY_WIOTP': False, 'SSL_VERIFY_MAXIMO': False}.items():
            k = ssl_verify
            v = resp[k] if k in resp else default_value
            if k in os.environ:
                v = os.environ[k]
            logger.debug('setting environment variable %s=%s', k, mask_credential(k, v))
            os.environ[k] = str(v)
    else:
        for ssl_verify, default_value in {'SSL_VERIFY_APM': True, 'SSL_VERIFY_AS': True, 'SSL_VERIFY_WIOTP': True, 'SSL_VERIFY_MAXIMO': False}.items():
            k = ssl_verify
            v = resp[k] if k in resp else default_value
            if k in os.environ:
                v = os.environ[k]
            logger.debug('setting environment variable %s=%s', k, mask_credential(k, v))
            os.environ[k] = str(v)

    if 'wml_cred' in resp and resp['wml_cred'] is not None:
        k = 'WML_VCAPS'
        v = resp['wml_cred']
        if '$$$USER_ACCESS_TOKEN$$$' in v:
            if 'USER_ACCESS_TOKEN' in os.environ:
                v = v.replace('$$$USER_ACCESS_TOKEN$$$', os.environ['USER_ACCESS_TOKEN'])
        credentials = json.loads(v)
        if 'USER_PROVIDED_WML_ACCESS_TOKEN' in os.environ:
            credentials['token'] = os.environ['USER_PROVIDED_WML_ACCESS_TOKEN']
        if 'USER_PROVIDED_WML_URL' in os.environ:
            credentials['url'] = os.environ['USER_PROVIDED_WML_URL']
        if 'USER_PROVIDED_WML_INSTANCE_ID' in os.environ:
            credentials['instance_id'] = os.environ['USER_PROVIDED_WML_INSTANCE_ID']
        if 'USER_PROVIDED_WML_VERSION' in os.environ:
            credentials['version'] = os.environ['USER_PROVIDED_WML_VERSION']
        v = json.dumps(credentials)
        logger.debug('setting environment variable %s=%s', k, mask_credential(k, v))
        os.environ[k] = v
    

    for k, v in resp['info'].items():
        if v is not None and k not in os.environ:
            if k == 'DB_CONNECTION_STRING':
                # iotfunctions only compares upper case ...
                index = v.upper().find('SECURITY=SSL;')
                if index != -1:
                    v = v[:index] + 'SECURITY=SSL;' + v[index+len('SECURITY=SSL;'):]
                if v.startswith('postgres://'):
                    v = v[len('postgres://'):]
            elif k == 'COS_ENDPOINT' and not v.startswith('https://') and not v.startswith('http://'):
                v = 'https://' + v
            elif k == 'API_BASEURL':
                API_BASEURL = v
            elif k == 'API_KEY':
                API_KEY = v
            elif k == 'API_TOKEN':
                API_TOKEN = v
            elif k == 'DB_CERTIFICATE_FILE':
                DB_CERTIFICATE_FILE = v
            logger.debug('setting environment variable %s=%s', k, mask_credential(k, v))
            os.environ[k] = v
    
    if 'USER_PROVIDED_DB_CONNECTION_STRING' in os.environ:
        logger.info('Found user defined variable USER_PROVIDED_DB_CONNECTION_STRING, resetting environment variable DB_CONNECTION_STRING')
        os.environ['DB_CONNECTION_STRING'] = os.environ['USER_PROVIDED_DB_CONNECTION_STRING']
    if 'USER_PROVIDED_HEALTH_URL' in os.environ:
        logger.info('Found user defined variable USER_PROVIDED_HEALTH_URL, resetting environment variable MAXIMO_BASEURL=%s', os.environ['USER_PROVIDED_HEALTH_URL'])
        os.environ['MAXIMO_BASEURL'] = os.environ['USER_PROVIDED_HEALTH_URL']
        MAXIMO_BASEURL = os.environ['USER_PROVIDED_HEALTH_URL']
        MAXIMO_LINKED = True

    INIT_DONE = True

    # whenever the APM info is used successfully, set it to AS constants
    _set_apm_info()

    logger.info('Finished initializing environment.')

    return (APM_ID, APM_API_BASEURL, APM_API_KEY)


"""
Asset cache related.
"""
def _get_asset_cache_entity_type(db=None, db_schema=None):
    """Get, and create if not already created, the asset cache entity type.

    TODO asset cache entity type is static with 3 time-series fields and two dimensions. It needs to be dynamic.
    """

    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    _get_asset_cache_table(db=db, db_schema=db_schema)

    entity_type = create_entity_type(asset_cache_entity_type, **{
        'db': db,
        'db_schema': db_schema,
        'timestamp_column': asset_cache_entity_type_timestamp,
        'entity_type_table_prefix': asset_cache_entity_type_table_prefix,
        'dimension_table_name': asset_cache_dimension_table_name,
        # make sure all column names are lower cases or sqlalchemy would have hard time to work with them
        'dimension_table_name': asset_cache_dimension_table_name,
        'dimensions': [
            Column('installdate', DateTime(), nullable=True),
            Column('statusdate', DateTime(), nullable=True),
            Column('status', String(32), nullable=True),
        ],
        'write': False,
        'delete_table_first': False,
        'drop_table_first': False,
        'use_wiotp': False,
    })
    # make sure same as the actual underlying table column name
    entity_type._entity_id = 'deviceid'

    return entity_type


def _refresh_asset_cache(asset_group_id, db=None, db_schema=None, start_ts=None, end_ts=None, data_items=None, include_asset_data=False):
    logger.debug('Refreshing asset cache. Args: asset_group_id=%s, data_items=%s', asset_group_id, data_items)
    if is_local_mode() or not api.is_maximo_linked():
        # if local mode or no Maximo linked, return immediately since either asset cache in local mode
        # is managed by users or no Maximo to load data from, cache refresher is essentially no-op
        return

    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    # first make sure all tables and asset entity type are created

    table_asset_group_members = _get_asset_group_table(db=db, db_schema=db_schema)
    table_asset_device_mappings = _get_asset_device_mapping_table(db=db, db_schema=db_schema)
    table_asset_device_attribute_mappings = _get_asset_device_attribute_mapping_table(db=db, db_schema=db_schema)

    entity_type = _get_asset_cache_entity_type(db=db, db_schema=db_schema)

    # read asset group members and their asset-device-attribute mappings (from Maximo)

    raw_asset_device_mappings = get_maximo_asset_device_mappings(asset_group_id)
    logger.debug('Retrieved asset device mappings for asset group id %s. Mappings=%s', asset_group_id, raw_asset_device_mappings)
    assets = []
    asset_group_members = []
    asset_device_mappings = []

    # get all device type
    device_type_set = set()
    logger.debug('Generating device list... iterating over data_items: %s', data_items)
    for idx, item in enumerate(data_items):
        name_type, name = item.split(':')
        logger.debug('Iterating over data items. Current name_type: %s', name_type)
        if name_type is not None and len(name_type) >0 :
            device_type_set.add(name_type)
    device_list=list(device_type_set)
    logger.debug('Generated device types list: %s', device_list)

    if raw_asset_device_mappings is not None:
        for mapping in raw_asset_device_mappings:
            assets.append({'siteId': mapping['siteId'], 'assetNum': mapping['assetNum']})
            asset_group_members.append([asset_group_id, mapping['siteId'], mapping['assetNum'], '%s-____-%s' % (mapping['assetNum'], mapping['siteId'])])

            if 'devices' in mapping and len(mapping['devices']) > 0:
                for device in mapping['devices']:
                    devicetype, deviceid = device.split(':')
                    asset_device_mappings.append([mapping['siteId'], mapping['assetNum'], devicetype, deviceid, ''])
        
        # If there is no data in asset_device_mappings
        if len(asset_device_mappings) == 0:
            # No asset device mapping in Maximo, call new Monitor's Data Dictionary API to get asset device mapping
            logger.debug('No asset device mappings in Maximo... calling new Monitor Data Dictionary API to get asset device mapping')
            
            for each_device_type in device_list:
                request_json = {
                        "deviceTypeName": each_device_type,
                        "siteAssetSearchCriteria": [
                            {
                                "assetName": each_asset['assetNum'],
                                "siteName":  each_asset['siteId'] 
                            } for each_asset in assets 
                        ]
                }
                logger.debug('Monitor Request json=%s', request_json)
                asset_device_mapping_from_monitor_per_asset = api._get_asset_device_mapping_from_monitor(input_json=request_json)
                logger.debug('Retrieved asset device mappings per asset from Monitor: %s', asset_device_mapping_from_monitor_per_asset)
                # asset_device_mapping_from_monitor_per_asset=[['BEDFORD', 'MAS_PUMP_2', 'MAS_NOV21_PUMP', '7001', ''], ['BEDFORD', 'MAS_PUMP_2', 'MAS_NOV21_PUMP', '7002', '']]
                if asset_device_mapping_from_monitor_per_asset is not None:
                    for n in asset_device_mapping_from_monitor_per_asset:
                        asset_device_mappings.append(n)
                        
        logger.debug('Updated asset device mappings: %s', asset_device_mappings)

        df_asset_group_members = pd.DataFrame(np.array(asset_group_members), columns=[default_asset_group_column, default_site_column, default_asset_column, default_assetid_column])
    else:
        # we only refresh group and mapping and both are from the same PMI API, so if we go here,
        # there is really no mapping to refresh further
        logger.info('cannot find asset_group_id=%s in Maximo, checking local one now ...', asset_group_id)

        # check apm_asset_group table also, if still nothing, raise exception here.
        with db.engine.connect() as conn: 
            #if conn.execute(select([func.count()]).select_from(table_asset_group_members).where(table_asset_group_members.c[default_asset_group_column] == asset_group_id)).first()[0] == 0:
            #    logger.warning('cannot find asset_group_id=%s, please check whether it is a valid group, either in Maximo or in the local table %s', asset_group_id, asset_group_table_name)
            #raise RuntimeError('cannot find asset_group_id=%s, please check whether it is a valid group, either in Maximo or in the local table %s' % (asset_group_id, asset_group_table_name))
            db_schema = api.get_as_schema(db_schema)
            table_asset_group_members_name = db_schema+".APM_ASSET_GROUPS"
            sql_str = " select count(*) from  "+table_asset_group_members_name + " where ASSETGROUP=" + asset_group_id
            sql = text(sql_str)
            output= conn.execute(sql)
            group_count =output.fetchall()
            if group_count == 0:
                logger.warning('cannot find asset_group_id=%s, please check whether it is a valid group, either in Maximo or in the local table %s', asset_group_id, asset_group_table_name)



        return

    logger.debug('Retrieved all assets of the given asset group: df_asset_group_members=%s', log_df_info(df_asset_group_members, head=5, logger=logger, log_level=logging.DEBUG))

    if len(asset_device_mappings) > 0:
        df_asset_device_mappings = pd.DataFrame(np.array(asset_device_mappings), columns=[default_site_column, default_asset_column, default_devicetype_column, default_deviceid_column, default_attribute_column])

    if include_asset_data:
        dimensions = ['installdate', 'estendoflife']
        df_asset_metadata = get_asset_attributes(assets=assets, data_items=dimensions, df_id_column=default_deviceid_column)
        df_asset_failure_history = get_asset_failure_history(assets=assets, data_items=None, df_id_column=default_deviceid_column, df_timestamp_name=entity_type._timestamp, start_ts=start_ts, end_ts=end_ts, separate_asset_site_columns=True)

    db.start_session()
    try:


        #with db.engine.connect() as conn:    
        #    conn.execute(table_asset_group_members.delete().where(table_asset_group_members.c[default_asset_group_column] == asset_group_id))

        if df_asset_group_members.empty:
            raise RuntimeError('cannot find members of asset_group_id=%s, either it\'s an empty or invalid group' % asset_group_id)
        

        with db.engine.connect() as conn:    
            conn.execute(table_asset_group_members.delete().where(table_asset_group_members.c[default_asset_group_column] == asset_group_id))
            # print("--Will attempt to write dataframe--")
            df_asset_group_members.to_sql(
                name=asset_group_table_name,
                con=conn,
                schema=db_schema,
                if_exists='append',
                index=False,
                chunksize=db.write_chunk_size,
                dtype={col: String(255) for col in df_asset_group_members.columns}
            )
            # print("--Will finish attempt to write dataframe--"+asset_group_table_name)
            # print(df_asset_group_members.head())
            conn.commit()

        sql = 'DELETE FROM %s a WHERE EXISTS (SELECT 1 FROM %s b WHERE a.%s = b.%s AND a.%s = b.%s AND b.%s = \'%s\')' % (
                get_table_name(db_schema, asset_device_attribute_mappings_table_name),
                get_table_name(db_schema, asset_group_table_name),
                default_site_column,
                default_site_column,
                default_asset_column,
                default_asset_column,
                default_asset_group_column,
                asset_group_id)
        
        with db.engine.connect() as conn:    
            conn.execute(text(sql))
            conn.commit()

        if len(asset_device_mappings) > 0:
            df_asset_device_mappings.to_sql(name=asset_device_attribute_mappings_table_name, con=db.engine, schema=db_schema, if_exists='append', index=False, chunksize=db.write_chunk_size, dtype={c: String(255) for c in list(df_asset_device_mappings.columns)})

        if include_asset_data:
            sql = 'DELETE FROM %s a WHERE EXISTS (SELECT 1 FROM %s b WHERE a.%s = b.%s AND b.%s = \'%s\')' % (
                    get_table_name(db_schema, entity_type._dimension_table_name),
                    get_table_name(db_schema, asset_group_table_name),
                    default_deviceid_column,
                    default_assetid_column,
                    default_asset_group_column,
                    asset_group_id)
            logger.debug(sql)

            with db.engine.connect() as conn:    
                conn.execute(text(sql))
                conn.commit()

            _write_dataframe(df=df_asset_metadata, table_name=entity_type._dimension_table_name, db=db, db_schema=db_schema)

            sql = 'DELETE FROM %s a WHERE EXISTS (SELECT 1 FROM %s b WHERE a.%s = b.%s AND a.%s = b.%s AND b.%s = \'%s\')' % (
                    entity_type.name,
                    get_table_name(db_schema, asset_group_table_name),
                    default_site_column,
                    default_site_column,
                    default_asset_column,
                    default_asset_column,
                    default_asset_group_column,
                    asset_group_id)
            logger.debug(sql)
            with db.engine.connect() as conn:    
                conn.execute(text(sql))
                conn.commit()

            _write_dataframe(df=df_asset_failure_history, table_name=entity_type.name, db=db, db_schema=db_schema)
    except:
        db.session.rollback()
        raise
    finally:
        db.commit()


def _get_asset_group_table(db=None, db_schema=None):

    logger.debug('Getting asset group table...')
    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    try:
        table = db.get_table(asset_group_table_name, db_schema)
    except KeyError:
        table = Table(asset_group_table_name,
                      db.metadata,
                      Column(default_asset_group_column, String(64), nullable=False),
                      Column(default_site_column, String(64), nullable=False),
                      Column(default_asset_column, String(64), nullable=False),
                      Column(default_assetid_column, String(256), nullable=False),
                      schema=db_schema)
        
        engine=db.engine

        table.create(bind=engine, checkfirst=True)

        index_name = ('%s-unique' % asset_group_table_name).upper()
        try:
            Index(index_name,
                  table.c[default_asset_group_column],
                  table.c[default_site_column],
                  table.c[default_asset_column],
                  unique=True).create(bind=db.engine)
        except Exception as e:
            logger.warning('failed creating index=%s for table=%s schema=%s: %s', index_name, asset_group_table_name, db_schema, e)

        index_name = ('%s-composite-unique' % asset_group_table_name).upper()
        try:
            Index(index_name,
                  table.c[default_asset_group_column],
                  table.c[default_assetid_column],
                  unique=True).create(bind=db.engine)
        except Exception as e:
            logger.warning('failed creating index=%s for table=%s schema=%s: %s', index_name, asset_group_table_name, db_schema, e)

    return table


def _get_asset_device_mapping_table(db=None, db_schema=None):
    logger.info('Retrieving asset device mapping table...')
    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    try:
        logger.debug('DB Schema to use: %s', db_schema)
        table = db.get_table(asset_device_mappings_table_name, db_schema)
        logger.info('Successfully retrieved asset device mapping table.')
    except KeyError:
        logger.info('Asset device mapping table does not exist... Creating it now.')
        table = Table(asset_device_mappings_table_name,
                      db.metadata,
                      Column(default_site_column, String(64), nullable=False),
                      Column(default_asset_column, String(64), nullable=False),
                      Column(default_devicetype_column, String(64), nullable=False),
                      Column(default_deviceid_column, String(256), nullable=False),
                      schema=db_schema)

        #table.create()
        engine=db.engine

        table.create(bind=engine, checkfirst=True)

        index_name = ('%s-unique' % asset_device_mappings_table_name).upper()
        try:
            Index(index_name,
                  table.c[default_site_column],
                  table.c[default_asset_column],
                  table.c[default_devicetype_column],
                  table.c[default_deviceid_column],
                  unique=True).create(bind=db.engine)
        except Exception as e:
            logger.warning('failed creating index=%s for table=%s schema=%s: %s', index_name, asset_device_mappings_table_name, db_schema)

    return table


def _get_asset_device_attribute_mapping_table(db=None, db_schema=None):
    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    try:
        table = db.get_table(asset_device_attribute_mappings_table_name, db_schema)
    except KeyError:
        table = Table(asset_device_attribute_mappings_table_name,
                      db.metadata,
                      Column(default_site_column, String(64), nullable=False),
                      Column(default_asset_column, String(64), nullable=False),
                      Column(default_devicetype_column, String(64), nullable=False),
                      Column(default_deviceid_column, String(256), nullable=False),
                      Column(default_attribute_column, String(256), nullable=False),
                      schema=db_schema)

        #table.create()
        engine=db.engine

        table.create(bind=engine, checkfirst=True)

        index_name = ('%s-unique' % asset_device_attribute_mappings_table_name).upper()
        try:
            Index(index_name,
                  table.c[default_site_column],
                  table.c[default_asset_column],
                  table.c[default_devicetype_column],
                  table.c[default_deviceid_column],
                  table.c[default_attribute_column],
                  unique=True).create(bind=db.engine)
        except Exception as e:
            logger.warning('failed creating index=%s for table=%s schema=%s: %s', index_name, asset_device_attribute_mappings_table_name, db_schema, e)

    return table


def _get_asset_cache_table(db=None, db_schema=None):
    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    try:
        table = db.get_table(asset_cache_table_name, db_schema)
        if default_failurecode_column not in table.c or default_problemcode_column not in table.c:
            logger.warn('replacing old schema table=%s with new one', asset_cache_table_name)
            table.drop()
            db.metadata.remove(table)
            raise KeyError()
    except KeyError:
        # make sure all column names are lower cases or sqlalchemy would have hard time to work with them
        table = Table(asset_cache_table_name,
                      db.metadata,
                      Column(default_deviceid_column, String(256), nullable=False),
                      Column(asset_cache_entity_type_timestamp, DateTime(), nullable=False),
                      Column(default_site_column, String(64), nullable=False),
                      Column(default_asset_column, String(64), nullable=False),
                      Column(default_faildate_column, DateTime(), nullable=True),
                      Column(default_failurecode_column, String(64), nullable=True),
                      Column(default_problemcode_column, String(64), nullable=True),
                      schema=db_schema)

        #table.create()
        engine=db.engine

        table.create(bind=engine, checkfirst=True)

        index_name = ('%s-unique' % asset_cache_table_name).upper()
        try:
            Index(index_name,
                  table.c[default_asset_group_column],
                  unique=True).create(bind=db.engine)
        except Exception as e:
            logger.warning('failed creating index=%s for table=%s schema=%s: %s', index_name, asset_cache_table_name, db_schema, e)

    return table


def set_asset_cache(df, siteid_column=default_site_column, assetid_column=default_asset_column, faildate_column=default_faildate_column, failurecode_column=None, problemcode_column=None, db=None, db_schema=None, delete_df_asset_first=True):
    """Set asset cache from the given dataframe.

    This function only works for local asset groups, that is, IDs not existing in Maximo.
    This is meant for quick trial or PoC style work so you don't need to manage Maximo side
    first. The transition from local asset group to actual remote one is transparent, as
    long as you create the asset group in Maximo (with the same ID), the local group
    definition no longer works and is overriden by the remote group's members.

    Parameters
    ----------
    df : `pandas.DataFrame`
        The dataframe to get asset device mapping, assumed to have 4 mandatory columns.
    siteid_column : `str`, optional
        The column name in `df` representing the asset's site ID. Default is 'site'.
    assetid_column : `str`, optional
        The column name in `df` representing the asset's ID. Default is 'asset'.
    faildate_column : `str`, optional
        The column name in `df` representing the asset failure event's failure date. Default is 'faildate'.
        Note that failure date column must be present in the given `df` because it is used as the
        time-series base for the asset failure history.
    failurecode_column : `str`, optional
        The column name in `df` representing the asset failure event's failure code. Default is None.
    problemcode_column : `str`, optional
        The column name in `df` representing the asset failure event's problem code. Default is None.
    db_schema : `str`, optional
        The schema to be used. Default is None.
    delete_df_asset_first : `bool`, optional
        Whether to delete all data of the assets appearing in the dataframe df first. Default is True.
    """

    db_schema = get_as_schema(db_schema)
    if df is None:
        raise ValueError('parameter df cannot be None')
    if df.empty:
        return

    for param in [siteid_column, assetid_column, faildate_column]:
        if param is None:
            raise ValueError('parameter %s cannot be None' % param)
    for param in [siteid_column, assetid_column, faildate_column, failurecode_column, problemcode_column]:
        if param is not None and param not in df.columns:
            raise ValueError('invalid parameter %s not found in the given df' % param)

    if db is None:
        db = _get_db()

    entity_type_asset_cache = _get_asset_cache_entity_type(db=db, db_schema=db_schema)

    table = entity_type_asset_cache.table

    df = df.copy()

    # make both siteid and assetnum all upper case since Maximo does that
    if not df.empty:
        df[siteid_column] = df[siteid_column].str.upper()
        df[assetid_column] = df[assetid_column].str.upper()

    df[asset_cache_entity_type_timestamp] = df[faildate_column]
    df[default_deviceid_column] = df[assetid_column] + '-____-' + df[siteid_column]

    column_renames = {}
    if siteid_column != default_site_column:
        column_renames[siteid_column] = default_site_column
    if assetid_column != default_asset_column:
        column_renames[assetid_column] = default_asset_column
    if faildate_column != default_faildate_column:
        column_renames[faildate_column] = default_faildate_column
    if failurecode_column is not None:
        column_renames[failurecode_column] = default_failurecode_column
    if problemcode_column is not None:
        column_renames[problemcode_column] = default_problemcode_column
    df = df.rename(columns=column_renames)

    db.start_session()
    try:
        if delete_df_asset_first:
            asset_key_columns = [default_site_column, default_asset_column]
            df_assets = df.groupby(asset_key_columns).size().reset_index()[asset_key_columns]
            for row in df_assets.itertuples():
                row = row._asdict()
                with db.engine.connect() as conn:    
                    conn.execute(table.delete().where(and_(table.c[default_site_column] == row[default_site_column], table.c[default_asset_column] == row[default_asset_column])))

        _write_dataframe(df=df, table_name=asset_cache_table_name, db=db, db_schema=db_schema)
    except:
        db.session.rollback()
        raise
    finally:
        db.commit()


def delete_asset_cache(df, siteid_column=default_site_column, assetid_column=default_asset_column, db=None, db_schema=None):
    """Delete asset device cache from the given dataframe.

    This function deletes all time-series data of assets in the given dataframe.

    This function only works for local asset groups, that is, IDs not existing in Maximo. This is meant for quick trial or PoC style work so you don't need to manage Maximo side of work first. The transition from local asset group to actual remote one is transparent, as long as you create the asset group in Maximo (with the same ID), the local group definition no longer works and is overrides by the remote group's members.

    Parameters
    ----------
    df : DataFrame
        The dataframe to get asset device mapping, assumed to have 4 mandatory columns.
    siteid_column : str, optional
        The column name in dataframe df representing the asset's site ID. Default is 'site'.
    assetid_column : str, optional
        The column name in dataframe df representing the asset's ID. Default is 'asset'.
    db_schema : str, optional
        The schema to be used. Default is None.
    """

    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    _get_asset_cache_entity_type(db=db, db_schema=db_schema)

    table = _get_asset_cache_table(db=db, db_schema=db_schema)

    # make both siteid and assetnum all upper case since Maximo does that
    if not df.empty:
        df = df.copy()
        df[siteid_column] = df[siteid_column].str.upper()
        df[assetid_column] = df[assetid_column].str.upper()

    asset_key_columns = [siteid_column, assetid_column]
    df_assets = df.groupby(asset_key_columns).size().reset_index()[asset_key_columns]

    db.start_session()
    try:
        for row in df_assets.itertuples():
            row = row._asdict()
            with db.engine.connect() as conn:    
                conn.execute(table.delete().where(and_(table.c[default_site_column] == row[siteid_column], table.c[default_asset_column] == row[assetid_column])))
    except:
        db.session.rollback()
        raise
    finally:
        db.commit()


def set_asset_cache_dimension(df, siteid_column=default_site_column, assetid_column=default_asset_column, db=None, db_schema=None, delete_df_asset_first=True):
    """Set asset cache dimension from the given dataframe.

    This function only works for local asset groups, that is, IDs not existing in Maximo. This is meant for quick trial or PoC style work so you don't need to manage Maximo side of work first. The transition from local asset group to actual remote one is transparent, as long as you create the asset group in Maximo (with the same ID), the local group definition no longer works and is overrides by the remote group's members.

    Parameters
    ----------
    df : DataFrame
        The dataframe to get asset device mapping, assumed to have 4 mandatory columns.
    siteid_column : str, optional
        The column name in dataframe df representing the asset's site ID. Default is 'site'.
    assetid_column : str, optional
        The column name in dataframe df representing the asset's ID. Default is 'asset'.
    db_schema : str, optional
        The schema to be used. Default is None.
    delete_df_asset_first : bool, optional
        Whether to delete all data of the assets appearing in the dataframe df first. Default is True.
    """

    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    _get_asset_cache_entity_type(db=db, db_schema=db_schema)

    df = df.copy()

    # make both siteid and assetnum all upper case since Maximo does that
    if not df.empty:
        df[siteid_column] = df[siteid_column].str.upper()
        df[assetid_column] = df[assetid_column].str.upper()

    df[default_deviceid_column] = df[assetid_column] + '-____-' + df[siteid_column]
    key_columns = [default_deviceid_column, 'installdate', 'statusdate', 'status']
    df = df.reset_index()[key_columns]

    table = db.get_table(asset_cache_dimension_table_name, db_schema)

    db.start_session()
    try:
        if delete_df_asset_first:
            asset_key_columns = [default_deviceid_column]
            df_assets = df.groupby(asset_key_columns).size().reset_index()[asset_key_columns]
            for row in df_assets.itertuples():
                row = row._asdict()
                with db.engine.connect() as conn:    
                    conn.execute(table.delete().where(and_(table.c[default_deviceid_column] == row[default_deviceid_column])))

        _write_dataframe(df=df, table_name=asset_cache_dimension_table_name, db=db, db_schema=db_schema)
    except:
        db.session.rollback()
        raise
    finally:
        db.commit()


def delete_asset_cache_dimension(df, siteid_column=default_site_column, assetid_column=default_asset_column, db=None, db_schema=None):
    """Delete asset cache dimension from the given dataframe.

    This function deletes all dimensions of assets in the given dataframe.

    This function only works for local asset groups, that is, IDs not existing in Maximo. This is meant for quick trial or PoC style work so you don't need to manage Maximo side of work first. The transition from local asset group to actual remote one is transparent, as long as you create the asset group in Maximo (with the same ID), the local group definition no longer works and is overrides by the remote group's members.

    Parameters
    ----------
    df : DataFrame
        The dataframe to get asset device mapping, assumed to have 4 mandatory columns.
    siteid_column : str, optional
        The column name in dataframe df representing the asset's site ID. Default is 'site'.
    assetid_column : str, optional
        The column name in dataframe df representing the asset's ID. Default is 'asset'.
    db_schema : str, optional
        The schema to be used. Default is None.
    """

    db_schema = get_as_schema(db_schema)
    if db is None:
        db = _get_db()

    _get_asset_cache_entity_type(db=db, db_schema=db_schema)

    df = df.copy()

    # make both siteid and assetnum all upper case since Maximo does that
    if not df.empty:
        df[siteid_column] = df[siteid_column].str.upper()
        df[assetid_column] = df[assetid_column].str.upper()

    df[default_deviceid_column] = df[assetid_column] + '-____-' + df[siteid_column]

    table = db.get_table(asset_cache_dimension_table_name, db_schema)

    db.start_session()
    try:
        for row in df.itertuples():
            row = row._asdict()
            with db.engine.connect() as conn:    
                conn.execute(table.delete().where(and_(table.c[default_deviceid_column] == row[default_deviceid_column])))
    except:
        db.session.rollback()
        raise
    finally:
        db.commit()


def _write_dataframe(
        df,
        table_name,
        db,
        db_schema=None,
        if_exists = 'append',
        chunksize = None,
        device_type=None
        ):
    """
    Write a dataframe to a table

    This function does not do much validation but leave that to the underlying database. There
    is no need to ensure the given dataframe includes all columns the target table has, the
    minimum requirement is all the not null columns are provided. If any not null column is
    not provided, the underlying database driver raise exception and this funciton just
    propagates out.

    Parameters
    ----------
    df : `DataFrame`
        The DataFrame object to be written.
    table_name: `str`
        The name of the table to write the given dataframe to.
    db_schema : `str`, optional
        The table schema. Default is None, using default schema.
    if_exists : {‘fail’, ‘replace’, ‘append’}, optional
        How to behave if the table already exists. 'fail' raises a ValueError, 'replace' drops the table before inserting new values, 'append' inserts new values to the existing table. Default is ‘fail’.
    chunksize : `int`, optional
        The batch size for write. Default `db.write_chunk_size`.
    """
    logger.info('Writing dataframe to table %s...', table_name)
    db_schema = get_as_schema(db_schema)
    if df is None or not isinstance(df, pd.DataFrame):
        raise ValueError('parameter df must be a DataFrame object')
    if df.empty:
        return

    if table_name is None:
        raise ValueError('parameter table_name cannot be None')

    if db is None:
        raise ValueError('parameter db cannot be None')

    if chunksize is None:
        chunksize = db.write_chunk_size

    df = df.reset_index()

    # remove columns using reserved words
    df = df.drop(columns=['id', 'index'], errors='ignore')

    if if_exists == 'append':
        try:
            table = db.get_table(table_name, db_schema)
        except KeyError:
            pass
        else:
            table_cols = {column.key.lower():column.key for column in table.columns}

            hm=api.get_df_column_db_column_map(device_type)
            #logger.debug('after api.get_df_column_db_column_map hm=%s',str(hm))
            #print(hm)
            # hm is the dict that maps the df column name to db2 table's column name: {'speed': 'speed_8', 'velocityx': 'velocityx_9'}
            df = df.rename(columns=hm)

            df_cols = {col.lower():col for col in df.columns}
            matched_cols = set(df_cols.keys()) & set(table_cols.keys())

            df = df[[df_cols[col] for col in matched_cols]]
            df = df.rename(columns={df_cols[col]:table_cols[col] for col in matched_cols})

            

    # use a default type mapping for strings and booleans
    dtypes = {}
    for c in list(df.columns):
        if is_string_dtype(df[c]):
            dtypes[c] = String(255)
        elif is_bool_dtype(df[c]):
            dtypes[c] = SmallInteger()
    if 'default_timestamp_utc' not in df.columns:
        if 'rcv_timestamp_utc' in df.columns and monitor_health_check() is True:
            df['default_timestamp_utc'] = df['rcv_timestamp_utc']
        else:
            pass
    logger.debug('Converting Dataframe to SQL. columns=%s, shape=%s, index=%s', df.columns, df.shape, df.index)



    total_record = int(df.shape[0])

    #support up to 8 million rows
    if total_record > 200000:
        chunk_size = int(df.shape[0] / 40)

        #print(chunk_size)

        for start in range(0, df.shape[0], chunk_size):
            df_subset = df.iloc[start:start + chunk_size]
            with db.engine.connect() as conn:
                df_subset.to_sql(name=table_name, con=conn, schema=db_schema, if_exists=if_exists, index=False, chunksize=chunksize, dtype=dtypes)
                conn.commit()
    else:
        chunksize = df.shape[0]
        with db.engine.connect() as conn:
            df.to_sql(name=table_name, con=conn, schema=db_schema, if_exists=if_exists, index=False, chunksize=chunksize, dtype=dtypes)
            conn.commit()


    
    

    logger.info('Finished writing dataframe to table.')


def _update_kpi(entity_type, kpi_function_name, json_body):
    logger.debug('kpi_function_name: %s', kpi_function_name)
    logger.debug('json_body: %s', json_body)
    resp = _call_as(url_path='/entityType/%s/kpiFunction/%s' % (entity_type,kpi_function_name), method='put', api_type='kpi',json=json_body)

    

def replace_wml_deployment_id(entity_type,custom_pipeline_name, input_wml_deployment_uid):
    resp = _call_as(url_path='/entityType/%s/kpiFunction' % entity_type, method='get', api_type='kpi')

    if resp is not None:
        logger.debug('response: %s', resp)
        data = resp.json()
    else:
        logger.debug('No kpi function for %s', entity_type)
        return

    for each_kpi in data:
        if custom_pipeline_name in str(each_kpi):
            logger.debug('each_kpi: %s', each_kpi)
            kpi_input = each_kpi['input']
            model_pipeline = kpi_input['model_pipeline']
            wml_deployment_uid=model_pipeline['wml_deployment_uid']
            logger.debug('wml_deployment_uid: %s', wml_deployment_uid)
            model_pipeline['wml_deployment_uid'] =input_wml_deployment_uid
            kpi_function_name = each_kpi['name']
            logger.debug('kpi_function_name: %s', kpi_function_name)
            json_body=each_kpi
            _update_kpi(entity_type, kpi_function_name, json_body)


def get_entity_from_metadata(entity_type_metadata, entity_type_name):
    logger.debug('Retrieving entity type name "%s" from database entity type metadata', entity_type_name)
    queried_entity = None
    #Need to query the database again because CP4D has the cacche of the previous entity_type_metadata which does not have new created entity_type for asset_group_id
    db=_get_db()
    entity_type_metadata=db.entity_type_metadata
    entity_type_metadata_keys = entity_type_metadata.keys()
    logger.debug("List of keys retrieved from the entity type metadata: %s", entity_type_metadata_keys)
    for key_item in entity_type_metadata_keys:
        item = entity_type_metadata[key_item]
        if  item['name'] == entity_type_name:
            queried_entity = item
            logger.debug("Found matching entity type in the key %s", key_item)
            break
    return queried_entity

def get_all_entities_from_metadata(entity_type_metadata):
    entity_list = []
    entity_type_metadata_keys = entity_type_metadata.keys()
    #logger.debug("List of keys retried from the entity type metadata: "+ str(entity_type_metadata_keys))
    for key_item in entity_type_metadata_keys:
        item = entity_type_metadata[key_item]
        logger.debug("Found entity type "+ item['name']+ " with ID "+ str(key_item))
        entity_list.append(item['name'])
        
    return entity_list

def get_sensor_data(asset_group_id:str, device_type:str, desired_features:list, start_ts=None,\
                    end_ts=None, asset_group_label:str = None) -> pd.DataFrame:

    """
    get sensor data from the device_type table, for example, it device_type=MOTOR, the table name will be IOT_MOTOR
    This function gets sensor data from the device_type tabl

    Parameters
    ----------


    asset_group_id : `str`, required, example '1006'. If this is not provided, then 
                    the parameter `asset_group_label` must be provided
    device_type : `str`, required, example: 'Pump'
    desired_features: `list of str`, required. example: ['velocityx', 'velocityy', 'velocityz']
    start_ts: `str`, starting timestamp as a string. Optional. If provided, only data gathered since this timestamp
              will be retrieved
    end_ts: `str`, ending timestamp as a string. Optional. If provided only the data gathered till this timestamp
            will be retrieved
    asset_group_label: `str`, Not required if asset_group_id is provided, otherwise required.
    return dataframe with sensor readings / records
    """



    logger.debug('device_type=%s', device_type)
    device_type_qualified_features = [device_type+':'+col for col in desired_features]
    if asset_group_id == None:
        if asset_group_label == None:
            raise RuntimeError('Either the asset_group_id or asset_group_label must be provided. Both cannot be None')
        else:
            asset_group_id = get_maximo_asset_group_id(asset_group_label)
            logger.debug('get_sensor_data() - Found asset_group_id = %s for the asset group label = %s', asset_group_id, asset_group_label)
    db = _get_db(asset_group_id=asset_group_id)
    entity_type_metadata = db.entity_type_metadata#.copy()
    _entity_type_id= get_entity_from_metadata(entity_type_metadata, device_type)['entityTypeId']
   
    db_schema = os.environ.get('AS_SCHEMA', None)
    _entity_type = _AssetGroupEntityType(
            asset_group_id,
            db,
            **{
                '_timestamp_col': 'evt_timestamp',
                
                '_db_schema': db_schema,
                'logical_name': asset_group_id,
                '_entity_type_id': _entity_type_id
            })
    resamples=dict()

    asset_refresher = AssetCacheRefresher(asset_group_id, data_items=device_type_qualified_features, db=db, db_schema=db_schema)
    asset_refresher.execute(start_ts=start_ts, end_ts=end_ts, entities=None)

    loader = AssetLoader(
            asset_group=asset_group_id,
            _entity_type=_entity_type,
            data_items=device_type_qualified_features,
            names=desired_features,
            resamples=resamples,
            entity_type_metadata=entity_type_metadata,
            asset_device_mappings=None,
            fillna=None,

            fillna_exclude=None,
            dropna=None,
            dropna_exclude=None)

    df_sensor_data = loader.execute(start_ts=start_ts, end_ts=end_ts)
    return df_sensor_data


def get_new_monitor_table_name_for_prediction_result(db,entity_type_name, col_list):
    # entity_type_name is the asset group id
    logger.debug('get_new_monitor_table_name_for_prediction_result entity_type_name= %s   col_list= %s', entity_type_name, col_list)
    
    # get new db because there is DB2 cache that causes the different the table name
    # at group.register() table name=DM_DEVICE_TYPE_1041_1 for daily_predicted_failure_date
    # after group.register() table name=DM_DEVICE_TYPE_16_1 for daily_predicted_failure_date
    db = _get_db()
    resp= api.get_entity_from_metadata(db.entity_type_metadata,entity_type_name)
    logger.debug('get_new_monitor_table_name_for_prediction_result resp=%s',str(resp))

    if resp is None:
        raise Exception(f'entity_type_name {entity_type_name} does not exist in the Monitor database')
    if len(col_list) ==0:
        raise Exception('length of column name of the dataframe is 0')
    first_data_item = col_list[0]
    logger.debug('First data item in column list: %s', first_data_item)
    
    logger.debug('Iterating over data items...')
    for each_item in resp['dataItemDto']:
        logger.debug('Current iteration: %s', each_item)
        if each_item['name'] == first_data_item:
            logger.debug('Found matching column name. Returning this: %s', each_item['sourceTableName'])
            return each_item['sourceTableName']
    return None

def get_uuid_by_entity_type_name(db,entity_type_name):
    # entity_type_name is the asset group id
    #logger.debug('entity_type_name= %s   col_list= %s', entity_type_name, col_list)
    
    resp= api.get_entity_from_metadata(db.entity_type_metadata,entity_type_name)
    if resp is None:
        error_msg='entity_type_name ' +str(entity_type_name) +' does not exist in the Monitor database'
        raise Exception(error_msg)
    
    return resp.get('uuid',None)

def get_entity_type_id_by_entity_type_name(entity_type_name):
    # entity_type_name is the asset group id
    logger.debug('Retrieving entity type ID by entity type name=%s', entity_type_name)
    db = api._get_db()
    resp= api.get_entity_from_metadata(db.entity_type_metadata,entity_type_name)
    if resp is None:
        error_msg='entity_type_name ' +str(entity_type_name) +' does not exist in the Monitor database'
        raise Exception(error_msg)
    entity_type_id = resp.get('entityTypeId',None)
    logger.debug('Found entity type ID of %s for entity type name of %s', entity_type_id, entity_type_name)
    return entity_type_id

# get get_start_date_and_end_date from a dataframe
def get_start_date_and_end_date(df,timestamp_col_name):
    sensor_data_min =df[timestamp_col_name].min()
    logger.debug(sensor_data_min)
    #logger.debug(type(sensor_data_min))
    start_date=sensor_data_min.to_pydatetime().date()
    #logger.debug(type(sensor_data_min_pydatetime))
    
    sensor_data_max =df[timestamp_col_name].max()
    logger.debug(sensor_data_max)

    end_date=sensor_data_max.to_pydatetime().date()
    #logger.debug(type(end_date))
    
    return start_date, end_date


# get get_start_date_and_end_date_utc
def get_start_date_and_end_date_utc_and_partition_name( d ):
    # convert Timestamp to datetime
    #d_pydatetime=d.to_pydatetime()
    d_str= d.isoformat()
    d_utc_begin= d_str+'-00.00.00.000000'
    d_utc_end = d_str+ '-23.59.59.999999'
    partition_name= d_str.replace('-','')
    return d_utc_begin , d_utc_end, partition_name


# ALTER TABLE MASDEV_MAM.IOT_PUMP_AFM_INTHERANGE_79 ADD PARTITION 20220401 STARTING ('2022-04-01-00.00.00.000000') ENDING('2022-04-01-23.59.59.999999');

def create_partition_for_one_day(db,table_name,partition_name,d_begin,d_end ):
    if db is None:
        db = _get_db()
        
    db_schema = get_as_schema(None)
    table_name_with_schema=db_schema+"."+table_name
    
    db.start_session()
    
    
    sql =' ALTER TABLE %s ADD PARTITION %s STARTING (\'%s\') ENDING(\'%s\')' % (   
                    table_name_with_schema,
                    partition_name,
                    d_begin,d_end)
    logger.debug(sql)
    try:
        with db.engine.connect() as conn:    
            conn.execute(text(sql))
            conn.commit()
    except Exception as e:
        #ignore
        #("in except block")
        if hasattr(e, 'message'):
            logger.debug(e.message)
        else:
            logger.debug(e)
    finally:
        pass
        #print("in finally block")
    
def create_partition_for_dataframe(db,df,timestamp_col_name,table_name):
    start_date, end_date = get_start_date_and_end_date(df,timestamp_col_name)
    
    day_count = (end_date - start_date).days + 1
    for single_date in (start_date + timedelta(n) for n in range(day_count)):
        day_begin, day_end, partition_name= get_start_date_and_end_date_utc_and_partition_name(single_date)
        logger.debug('single_date = %s, day_begin = %s, day_end = %s, partition_name = %s', single_date, day_begin, day_end, partition_name)
        create_partition_for_one_day(db,table_name,partition_name,day_begin,day_end)
    
def get_devices_dd_id(device_type_uuid):
    
    devices_dd_id_map={}
    
    monitor_url_path='core/deviceTypes/'+device_type_uuid + "/devices"
    resp = _call_as_v2(url_path=monitor_url_path, method='get', json=None)
    if resp is not None and resp.status_code == 200:
        # resp.text is json array
        data = resp.json()
        #print(str(data))
        json_array= data['results']
        for each_item in json_array:
            device_ddId = each_item['ddId']
            device_name = each_item['name']
            devices_dd_id_map[device_name] = device_ddId
    #print(devices_dd_id_map)
    return devices_dd_id_map

def set_data_dictionary_in_monitor(df):
    db= _get_db()
    for index, row in df.iterrows():
        #print('dataframe row', row['asset_id'], row['site_id'],row['deviceid'] , row['devicetype'])
        device_type_uuid= api.get_uuid_by_entity_type_name(db,row['devicetype'])
        devices_dd_id_map = get_devices_dd_id(device_type_uuid)
    
        monitor_url_path= 'core/sites/'+ row['site_id'] +'/assets/' + row['asset_id'] + '/relations/devices/assign'
        #print(monitor_url_path)
        input_json= [ devices_dd_id_map[row['deviceid']] ]
        logger.debug(str(input_json))
        resp = _call_as_v2(url_path=monitor_url_path, method='put', json=input_json)




def get_table_name_for_aggregate_data(entity_type, columns_to_load):
    
    resp = api._call_as_meta(url_path='/entityType/%s' % entity_type, method='get', api_type='meta')
    if resp is not None:
        data = resp.json()
        #print(data)
    else:
        return None
    dataItemDto=data['dataItemDto']
    first_column_name=columns_to_load[0]
    for data_item in dataItemDto:
        #print(str(data_item['name']) + ' '+ str(data_item['sourceTableName']))
        if first_column_name == str(data_item['name']):
            return str(data_item['sourceTableName'])
    
    return None

def get_database_column_name_list(device_type_name):
    db=api._get_db()
    column_name_list=[]
    for k,v in db.entity_type_metadata.items():
    #print(k,v)
        if v['name'] == device_type_name:
            #print(v)
            #print(v['dataItemDto'])
            for each_item in v['dataItemDto']:
                if each_item['name'] not in ['ENTITY_ID','RCV_TIMESTAMP_UTC']:
                
                    column_name_list.append(each_item['columnName'])
    #print(column_name_list)
    return column_name_list

def get_feature_list(device_type_name):
    db=api._get_db()
    feature_name_list=[]
    for k,v in db.entity_type_metadata.items():
    #print(k,v)
        if v['name'] == device_type_name:
            #print(v)
            #print(v['dataItemDto'])
            for each_item in v['dataItemDto']:
                if each_item['name'] not in ['ENTITY_ID','RCV_TIMESTAMP_UTC']:
                
                    feature_name_list.append(each_item['name'])
    #print(column_name_list)
    return feature_name_list

def get_df_column_db_column_map(device_type):
    hm={}
    try:
        db=api._get_db()
        

        for k,v in db.entity_type_metadata.items():
        #print(k,v)
            if v['name'] == device_type:
                print(k,v)
                dataItemDto =v['dataItemDto']
                for i in dataItemDto:
                    print(i)
                    if ( i['columnType'] =='NUMBER'):
                        hm[i['name'].lower()] = i['columnName']
    except:
        pass
    
    #return empty mapping if there is DB2 error
    return hm

def monitor_health_check():
    try:
        logger.debug('health  check try none json')
        if any([env is None for env in [API_BASEURL, API_KEY, API_TOKEN]]):
            if all([env is not None for env in [APM_ID, APM_API_BASEURL, APM_API_KEY]]):
                init_environ()
            else:
                raise RuntimeError('missing mandatory environment variable API_BASEURL, API_KEY, API_TOKEN')

        api_baseurl = API_BASEURL
        if api_baseurl[-1] == '/':
            api_baseurl = api_baseurl[:-1]

        url = '%s/healthcheck/details' % (api_baseurl)
        final_headers = {
            'Content-Type': 'application/json',
            'X-api-key' : API_KEY,
            'X-api-token' : API_TOKEN,
            'Cache-Control': 'no-cache',
        }

        ssl_verify = os.environ.get('SSL_VERIFY_AS', 'true')
        if isinstance(ssl_verify, str) and ssl_verify.lower() in ('true', 'yes', '1'):
            ssl_verify = True
        else:
            ssl_verify = False
        # kwargs['ssl_verify'] = ssl_verify
        
        method = "get"

        resp = _call(url=url, method=method,headers=final_headers, ssl_verify=ssl_verify,json=None)
        data = resp.json()
        logger.debug('Health check api response = %s', str(data))
        if(data["monitorRelease"]=="9.1"):
            return True
        else:
            return False
    except Exception as e:
        logger.debug('health  check except')
        logger.warning('Exception: %s', e)
        return False