# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA, 5900-AMG
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

"""This module includes a wrapper for building Anomaly Detection model using either Unsupervised or Semi-supervised learning techniques"""

import copy
import logging
import logging.config
import os
from os import path
import re
from typing import Any, Union
import numpy as np
import pandas as pd
from datetime import datetime
import json
import srom
from srom.utils.anomaly_dag import anomaly_dag
from srom.utils.pipeline_utils import tabulate as tabulate
from srom.utils.no_op import NoOp
from srom.pipeline.utils.anomaly_utils import get_threshold_statistics
from srom.pipeline.utils.anomaly_utils import estimator_comparison
from srom.pipeline.utils.anomaly_utils import retrive_base_learner
from srom.pipeline.anomaly_pipeline import AnomalyPipeline
from srom.pipeline.srom_param_grid import SROMParamGrid
from srom.pipeline.hyper_params import anomaly_detection_fine_grid, anomaly_detection_sample_grid_for_rbfopt
from srom.anomaly_detection.generalized_anomaly_model import GeneralizedAnomalyModel

from sklearn.preprocessing import StandardScaler, MinMaxScaler, PowerTransformer, Normalizer, MaxAbsScaler, RobustScaler
from sklearn.ensemble import IsolationForest
from sklearn.mixture import GaussianMixture, BayesianGaussianMixture
from sklearn.svm import OneClassSVM
from sklearn.covariance import (EmpiricalCovariance, EllipticEnvelope, LedoitWolf, MinCovDet, OAS, ShrunkCovariance)
from srom.anomaly_detection.algorithms import NearestNeighborAnomalyModel, LOFNearestNeighborAnomalyModel
from srom.anomaly_detection.algorithms.pca_t2 import AnomalyPCA_T2
from srom.anomaly_detection.algorithms.pca_q import AnomalyPCA_Q
from srom.anomaly_detection.algorithms.bayesian_gmm_outlier import BayesianGMMOutlier
from srom.anomaly_detection.algorithms.anomaly_ensembler import AnomalyEnsembler
from srom.anomaly_detection.algorithms.gmm_outlier import GMMOutlier
from sklearn.ensemble import RandomForestClassifier
from srom.anomaly_detection.algorithms.sample_svdd import SampleSVDD
from srom.anomaly_detection.algorithms.random_partition_forest import RandomPartitionForest
from srom.anomaly_detection.algorithms.extended_isolation_forest import ExtendedIsolationForest
from srom.anomaly_detection.algorithms.negative_sample_anomaly import NSA
from srom.anomaly_detection.algorithms.anomaly_robust_pca import AnomalyRobustPCA
from srom.anomaly_detection.algorithms.neural_network_nsa import NeuralNetworkNSA
from srom.anomaly_detection.algorithms.ggm_quic import GraphQUIC
from srom.anomaly_detection.algorithms.ggm_pgscps import GraphPgscps 
from srom.anomaly_detection.algorithms.hotteling_t2 import HotellingT2
from srom.anomaly_detection.algorithms.cusum import CUSUM
from srom.anomaly_detection.algorithms.spad import SPAD
from srom.anomaly_detection.algorithms.extended_spad import ExtendedSPAD
from srom.anomaly_detection.algorithms.oob import OOB
from sklearn.neighbors import KernelDensity
from srom.deep_learning.anomaly_detector import DNNAutoEncoder
from srom.anomaly_detection.gaussian_graphical_anomaly_model import GaussianGraphicalModel

import uuid
from enum import Enum

DEFAULT_LOG_LEVEL = logging.ERROR
RUNTIME_INCLUDES_IPYTHON = False
logger = logging.getLogger(__name__)

try:
    from IPython.display import HTML, display, Image
    RUNTIME_INCLUDES_IPYTHON = True
except Exception as ex:
    print("Exception raised while attempting to import IPython", '\n', \
    ex, '\n', \
    "Environment does not support IPython. This function will be disabled in environments other\
         than IPython / Jupyter notebooks. This is a harmless exception and can be safely ignored in \
            non-IPython environment")
    
class AnomalyScoringMethod(Enum):
    EM_SCORE = 'em_score'
    AL_SCORE = 'al_score'
    MV_SCORE = 'mv_score'

class AnomalyThresholdCriteria(Enum):
    STDDEV = 'std'
    CONTAMINATION = 'contamination'
    ADAPTIVE_CONTAMINATION = 'adaptive-contamination'
    OTSU = 'otsu'
    MEDIANABSDEV = 'medianabsolutedev'
    QFUNCTION = 'qfunction'

class AnomalyDetectionWrapper:
    """
    Contains the Anomaly Detection DAG and the associated functionality to access, modify, execute, and
    read the DAG configuration.
    """

    def __init__(self, learning_mode = None, training_data_df:pd.DataFrame = None, \
                 validation_data_df:pd.DataFrame = None, standard_pipelines = True, include_deep_learning = False, include_covariance_based_techniques = False, full_dag = False, **kwargs):
        self.learning_mode = learning_mode
        if (validation_data_df is not None) and (not validation_data_df.empty):
            if learning_mode == None:
                learning_mode = 'semi_supervised'
        else:
            if learning_mode == 'semi_supervised':
                print('WARNING: Semi Supervised Learning is chosen without validation data. Switching to Unsupervised learning mode')
                learning_mode = 'unsupervised'
        if learning_mode == None:
            raise(ValueError('Learning mode is either not provided, or incompatible with the dataset, or could not be inferred from the dataset'))
        logger.setLevel(kwargs.get("log_level", logging.ERROR))
        print("Printing Logger = ", str(logger), " for the class = ", __name__)
        self.stages = None
        self.scoring_method = AnomalyScoringMethod.EM_SCORE.value
        self.threshold_grid = self.get_default_threshold_grid()
        self.pipeline = AnomalyPipeline()
        self.param_grid=SROMParamGrid(gridtype='anomaly_detection_fine_grid')
        self.execution_type=kwargs.get('execution_params','spark_node_random_search')
        self.num_option_per_pipeline=kwargs.get('number_of_option_per_pipeline',1)
        self.max_eval_time_minute=kwargs.get('maximum_evaluation_time_per_pipeline',20)
        self.total_execution_time = kwargs.get('total_execution_time', 30)
        print('Max aval per pipeline = ', self.max_eval_time_minute, '; total execution time = ', self.total_execution_time)
        self.random_state=kwargs.get('random_state',73)
        self.training_data = training_data_df
        self.validation_data = validation_data_df
        self.x_train = None
        self.y_train = None
        self.x_validate = None
        self.y_validate = None
        standard_algorithms = ['isolationforest']
        extended_algorithms = ['gaussiangraphicalmodel','neuralnetworknsa', 'randompartitionforest']
        covariance_based_techniques = ['mincovdet','shrunkcovariance','empiricalcovariance','ellipticenvelope','ledoitwolf','oas']
        self.anomaly_dag = None

    def display_default_dag(self, display_in_IPython = True):
        if self.learning_mode == 'unsupervised':
            self._display_unsupervised_ad_dag()
        elif self.learning_mode == 'semi_supervised':
            self._display_semi_supervised_ad_dag()
        else:
            raise(ValueError("Unrecognized learning mode. Make sure it is either `unsupervised` or `semi_supervised`"))
    def get_anomaly_detection_dag(self, scalers:list=None, append_or_replace:str = 'replace', estimators_to_keep:list = [], \
                                                   estimators_to_exclude:list = [], sliding_window_size = 50, sliding_window_data_cutoff = 15, custom_estimators:list=[], use_only_custom_estimators:bool = False):
        if self.learning_mode == 'unsupervised':
            return self._get_anomaly_dag_for_unsupervised_learning(scalers = scalers, append_or_replace = append_or_replace, estimators_to_keep= estimators_to_keep, \
                                                   estimators_to_exclude = estimators_to_exclude, sliding_window_size = 50, sliding_window_data_cutoff = 15, custom_estimators=custom_estimators,\
                                                                                      use_only_custom_estimators = use_only_custom_estimators)
        elif self.learning_mode == 'semi_supervised':
            return self._get_anomaly_dag_for_semi_supervised_learning(scalers, append_or_replace, estimators_to_keep= estimators_to_keep, \
                                                   estimators_to_exclude = estimators_to_exclude, sliding_window_size = 50, sliding_window_data_cutoff = 15, custom_estimators=custom_estimators,\
                                                                                      use_only_custom_estimators = use_only_custom_estimators)
    def override_stages(self, pipeline_stages:list):
        self.stages = pipeline_stages
        
    def set_scoring_method(self, scoring_method:AnomalyScoringMethod=AnomalyScoringMethod.EM_SCORE):
        self.scoring_method = scoring_method.value
        
    def get_anomaly_pipeline_for_best_estimator(self, best_estimator, estimator_name, threshold_method, threshold_cutoff):
        anomaly_pipeline_final = AnomalyPipeline(anomaly_threshold_method = threshold_method, std_threshold = threshold_cutoff)
        if isinstance(best_estimator,GaussianGraphicalModel):
            anomaly_pipeline_final.set_best_estimator(best_estimator)
        else:
            predict_fn, score_sign = [(algorithm[2], algorithm[1]) for algorithm in self.anomaly_dag if algorithm[3] == estimator_name][0]
            anomaly_pipeline_final.set_best_estimator(GeneralizedAnomalyModel(best_estimator, predict_function = predict_fn, score_sign  = score_sign))
        anomaly_pipeline_final._generate_anomaly_threshold_for_unsupervised_modeling(self.x_train)
        return anomaly_pipeline_final
    
    def execute_stages(self, stages:list, x_train, y_train=None, x_validate = None, y_validate = None, execution_params:dict = {}):
        self.x_train = x_train
        self.y_train = y_train
        self.x_vaidate = x_validate
        self.y_validate = y_validate
        if self.learning_mode == 'unsupervised':
            return self.execute_unsupervised_learning_dag(x_train, stages, execution_params)
        elif self.learning_mode == 'semi_supervised':
            return self.execute_semi_supervised_learning_dag(x_train, x_validate, y_validate, stages, execution_params)
        else:
            raise(ValueError('Unsupported learning mode provided ('+self.learning_mode+')'))
   
    def set_threshold_grid(self, threshold_grid:dict={}, append_or_replace = 'replace'):
        if append_or_replace == 'replace':
            self.threshold_grid = {}
            for key, vals in threshold_grid.items():
                if key == 'contamination':
                    self.threshold_grid[key] = [{key:v} for v in vals ]
                elif key.lower() == 'otsu':
                    self.threshold_grid[key] = [{'std_threshold':v} for v in vals ]
                else:
                    self.threshold_grid[key] = [{key+'_threshold':v} for v in vals ]
        else:
            self.threshold_grid = self.get_default_threshold_grid()
            for key, vals in threshold_grid.items():
                curr_entries = self.threshold_grid.get(key, None)
                new_vals = None
                if curr_entries == None:
                    new_vals = vals
                else:
                    curr_vals  = [list(d.values())[0] for d in curr_entries]
                    new_vals = list(set(curr_vals + vals))
                if key == 'contamination':
                    self.threshold_grid[key] = [{key:v} for v in new_vals]
                elif key.lower() == 'otsu':
                    self.threshold_grid[key] = [{'std_threshold':v} for v in vals ]
                else:
                    self.threshold_grid[key] = [{key+'_threshold':v} for v in new_vals]
                        
        return self.threshold_grid
                    
            
    def display_execution_results(self):
        if RUNTIME_INCLUDES_IPYTHON:
            execution_res = [[self.pipeline.best_estimator, self.pipeline.best_score]]
            display(HTML(tabulate(execution_res, headers = ['Best Pipeline', 'Score'], tablefmt='html')))
        else:
            raise(EnvironmentError('Not an IPython supported environment.\
                                   Cannot display formatted values. Use the get_execution_results method\
                                   to get the results as a data frame'))
            
    def get_execution_results(self)->pd.DataFrame:
        execution_res_df = pd.DataFrame([[self.pipeline.best_estimator, self.pipeline.best_score]],\
                                        columns = ['best_estimator','best_score'])
        return execution_res_df
    
    def execute_unsupervised_learning_dag(self, x_train, stages, execution_params):
        self.pipeline = AnomalyPipeline()
        scoring_method = execution_params.get('scoring_method', None)
        if scoring_method != None:
            self.scoring_method = scoring_method
            self.pipeline.set_scoring(self.scoring_method) # em_score, or mv_score or al_score
        self.pipeline.set_stages(stages)
        #print(pipeline)
        starttime = datetime.now()
        print('Starting pipelines exploration @ ', starttime)
        pipeline_output = self.pipeline.execute(trainX=x_train,
                        validX=None, validy=None, verbosity='low', 
                        param_grid=execution_params.get('param_grid',self.param_grid), 
                        exectype=execution_params.get('execution_type','spark_node_random_search'), 
                        num_option_per_pipeline=execution_params.get('number_of_option_per_pipeline',None), 
                        max_eval_time_minute=execution_params.get('maximum_evaluation_time_per_pipeline',self.max_eval_time_minute),
                        total_execution_time = execution_params.get('total_execution_time', self.total_execution_time),
                        random_state=execution_params.get('random_state',73))
        print("Finished fitting the estimators to the data set. Fitting time -", str(datetime.now() - starttime))
        return pipeline_output
    
    def execute_semi_supervised_learning_dag(self, x_train, x_validate, y_validate, stages, execution_params):
        print('maximum_evaluation_time_per_pipeline = ', execution_params.get('maximum_evaluation_time_per_pipeline',self.max_eval_time_minute))
        print('total_execution_time = ', execution_params.get('total_execution_time',self.total_execution_time))
        self.pipeline = AnomalyPipeline()
        self.pipeline.set_stages(stages)
        #print(pipeline)
        starttime = datetime.now()
        print('Starting pipelines exploration @ ', starttime)
        pipeline_output = self.pipeline.execute(trainX=x_train, validX=x_validate, validy=y_validate,\
                                         verbosity='low', param_grid=execution_params.get('param_grid',self.param_grid),\
                                         exectype=execution_params.get('execution_type','spark_node_random_search'),\
                                         num_option_per_pipeline=execution_params.get('number_of_option_per_pipeline',1),\
                                         max_eval_time_minute=execution_params.get('maximum_evaluation_time_per_pipeline',self.max_eval_time_minute),\
                                         total_execution_time = execution_params.get('total_execution_time', self.total_execution_time),\
                                         random_state=execution_params.get('random_state',73))
        print("Finished fitting the estimators to the data set. Fitting time -", str(datetime.now() - starttime))
        return pipeline_output
        
    def set_hyperparameter_grid(self, hyperparameter_dict:dict={}, replace_or_append='append'):
        hpg_ex = None
        if replace_or_append == 'append':
            hpg_ex = anomaly_detection_fine_grid.PARAM_GRID.copy()
            for key, value in hyperparameter_dict.items():
                existing_key_value = hpg_ex.get(key, None)
                if existing_key_value != None:
                    existing_key_value.extend(value)
                    hpg_ex[key] = existing_key_value
                else:
                    hpg_ex[key] = value
        else:
            hpg_ex = hyperparameter_dict
        
        self.param_grid = SROMParamGrid(gridtype = 'empty')
        self.param_grid.set_param_grid(hpg_ex)
            
    def profile_threshold_criteria(self, pipeline_output, number_of_pipelines_to_profile=1, max_eval_time_minute=None, total_execution_time=None, pipeline_index_to_profile=0):
        if self.learning_mode == 'semi_supervised':
            print('This operation is supported only for Unsupervised learning mode. For semi_supervised the thershold is computed differently using the ground truth')
            return None, None
        if max_eval_time_minute == None:
            max_eval_time_minute = self.max_eval_time_minute
        if total_execution_time == None:
            total_execution_time = self.total_execution_time
        return get_threshold_statistics(pipeline_output, self.x_train, number=number_of_pipelines_to_profile, threshold_parameters=self.threshold_grid,\
                                            max_eval_time_minute=max_eval_time_minute, total_execution_time=total_execution_time, pipeline_index=pipeline_index_to_profile)
        
    def compare_estimators(self, pipeline_output, threshold_method, threshold_cutoff, number_estimators_to_compare = 10, max_eval_time_minute = None, total_execution_time = None):
        if self.learning_mode == 'semi_supervised':
            print('This operation is supported only for Unsupervised learning mode. For semi_supervised the estimator is selected using the validation against the ground truth')
            return None, None
        else:
            return estimator_comparison(pipeline_output, self.x_train.values, threshold_method, threshold_cutoff, number=number_estimators_to_compare, \
                                    max_eval_time_minute = max_eval_time_minute, total_execution_time = total_execution_time)
        
    
    def get_default_threshold_grid(self):
        return srom.pipeline.utils.anomaly_utils.default_threshold_parameters
    
    def _get_anomaly_dag_for_unsupervised_learning(self, scalers:list=None, append_or_replace:str = 'replace', estimators_to_keep:list = [], \
                                                   estimators_to_exclude:list = [], sliding_window_size = 50, sliding_window_data_cutoff = 15, \
                                                   custom_estimators:list=[], use_only_custom_estimators:bool = False):
        
        anomaly_dag_ex = anomaly_dag.copy()

        anomaly_dag_ex.append((GaussianGraphicalModel(distance_metric='Stochastic_Nearest_Neighbors',sliding_window_size=sliding_window_size, \
                                                     sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True), 1, "predict", "ggm_snn"))
        anomaly_dag_ex.append((GaussianGraphicalModel(distance_metric='KL_Divergence_Dist',sliding_window_size=sliding_window_size,\
                                                     sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True), 1, "predict", "ggm_kl_div_dist"))
        anomaly_dag_ex.append((GaussianGraphicalModel(distance_metric='KL_Divergence',sliding_window_size=sliding_window_size,\
                                                     sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True), 1, "predict", "ggm_kl_divergence"))
        anomaly_dag_ex.append((GaussianGraphicalModel(distance_metric='Frobenius_Norm',sliding_window_size=sliding_window_size,\
                                                     sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True), 1, "predict", "ggm_frobenius_norm"))
        anomaly_dag_ex.append((GaussianGraphicalModel(distance_metric='Likelihood',sliding_window_size=sliding_window_size, \
                                                     sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True), 1, "predict", "ggm_likelihood"))
        anomaly_dag_ex.append((GaussianGraphicalModel(distance_metric='Spectral',sliding_window_size=sliding_window_size,\
                                                     sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True), 1, "predict", "ggm_spectral"))
        anomaly_dag_ex.append((GaussianGraphicalModel(distance_metric='Mahalanobis_Distance',sliding_window_size=sliding_window_size,\
                                                     sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True), 1, "predict", "ggm_mahalanobis"))
        anomaly_dag_ex.append((GaussianGraphicalModel(distance_metric='Sparsest_k_Subgraph',sliding_window_size=sliding_window_size,\
                                                     sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True), 1, "predict", "ggm_sparse_subgraph"))
        #graph_quic = GeneralizedAnomalyModel(GaussianGraphicalModel(base_learner=GraphQUIC(),sliding_window_size=0, scale=False),fit_function="fit",predict_function="predict",score_sign=1)
        anomaly_dag_ex.append((GraphQUIC(), 1, "mahalanobis", "graphquic"),)
        anomaly_dag_ex.append((RandomPartitionForest(), -1, "anomaly_score", "randompartitionforest"))

        
        stages = []
        scaler_list = []
        estimator_list = []
        if use_only_custom_estimators:
            if len(custom_estimators) == 0:
                raise(ValueError('No custom estimators are provided, but use_only_custom_estimators is set to True'))
            else:
                for ce in custom_estimators:
                    estimator_list.append(ce[3], GeneralizedAnomalyModel(base_learner = ce[0], fit_function = 'fit', predict_function = ce[2], score_sign = ce[1]))
        else:
            if len(estimators_to_keep) != 0:
                reg = re.compile(r"[*]$" )
                fuzzy_names = list(filter(reg.search, estimators_to_keep))
                full_names = list(filter(lambda x: not reg.search(x), estimators_to_keep))
                selected_estimators_1 = [(algorithm[3],GeneralizedAnomalyModel(base_learner=algorithm[0],fit_function="fit",
                                                                       predict_function=algorithm[2],score_sign=algorithm[1])) for algorithm in anomaly_dag_ex if algorithm[3] in full_names]
                selected_estimators_2 = {}
                for etk in fuzzy_names:
                    for algorithm in anomaly_dag_ex:
                        if etk.replace('*','') in algorithm[3]:
                            if not algorithm[3] in selected_estimators_2.keys():
                                selected_estimators_2[algorithm[3]] = (algorithm[3],GeneralizedAnomalyModel(base_learner=algorithm[0],fit_function="fit",
                                                                       predict_function=algorithm[2],score_sign=algorithm[1]))
                estimator_list = selected_estimators_1 + list(selected_estimators_2.values())
            elif len(estimators_to_exclude) != 0:
                reg = re.compile(r"[*]$" )
                fuzzy_names = list(filter(reg.search, estimators_to_exclude))
                full_names = list(filter(lambda x: not reg.search(x), estimators_to_exclude))
                selected_estimators_1 = [(algorithm[3],GeneralizedAnomalyModel(base_learner=algorithm[0],fit_function="fit",
                                                                       predict_function=algorithm[2],score_sign=algorithm[1])) for algorithm in anomaly_dag_ex if not algorithm[3] in full_names]
                selected_estimators_2 = selected_estimators_1.copy()
                for ete in fuzzy_names:
                    for n, algorithm in enumerate(selected_estimators_2):
                        if ete.replace('*','') in algorithm[3]:
                            selected_estimators_2 = selected_estimators_2.pop(n)
                estimator_list = selected_estimators_2.values()
            else:
                    #estimator_list.append( (algorithm[3],GeneralizedAnomalyModel(base_learner=algorithm[0],fit_function="fit",
                                                                       #predict_function=algorithm[2],score_sign=algorithm[1])))
                    estimator_list = [(algorithm[3],GeneralizedAnomalyModel(base_learner=algorithm[0],fit_function="fit",
                                                                       predict_function=algorithm[2],score_sign=algorithm[1])) for algorithm in anomaly_dag_ex]
            
#             for algorithm in anomaly_dag_ex:
#                 if algorithm[3] in estimators_to_keep:
#                         estimator_list.append( (algorithm[3],GeneralizedAnomalyModel(base_learner=algorithm[0],fit_function="fit",
#                                                                        predict_function=algorithm[2],score_sign=algorithm[1])))
                    
                    
#                 elif len(estimators_to_exclude) != 0:
#                     if not algorithm[3] in estimators_to_exclude:
#                         estimator_list.append( (algorithm[3],GeneralizedAnomalyModel(base_learner=algorithm[0],fit_function="fit",
#                                                                        predict_function=algorithm[2],score_sign=algorithm[1])))

            if len(custom_estimators) != 0:
                 for ce in custom_estimators:
                    estimator_list.append(ce[3], GeneralizedAnomalyModel(base_learner = ce[0], fit_function = 'fit', predict_function = ce[2], score_sign = ce[1]))
                    
        if scalers != None:
            if append_or_replace == 'replace':
                scaler_list = scalers
            elif append_or_replace == 'append':
                scaler_list = [MinMaxScaler(),StandardScaler()]
                existing_scalers = [item.__class__.__name__ for item in scaler_list]
                for input_item in scalers:
                    if not input_item.__class__.__name__ in existing_scalers:
                        scaler_list.append(input_item)
            else:
                raise(ValueError('Unsupported list operation for scalers - ', append_or_replace))
        else: # default scaler is MinMaxScaler & StandardScaler.
            scaler_list = [MinMaxScaler(),StandardScaler()]
        stages = [
            scaler_list,
            estimator_list
        ]
        self.anomaly_dag = anomaly_dag_ex
        return stages, anomaly_dag_ex
    
    def _get_anomaly_dag_for_semi_supervised_learning(self, scalers:list=None, append_or_replace:str = 'replace', estimators_to_keep:list = [], \
                                                   estimators_to_exclude:list = [], sliding_window_size = 50, sliding_window_data_cutoff = 15, custom_estimators:list=[], use_only_custom_estimators:bool = False):        
        stages = []
        scaler_list = []
        estimator_list = []
        if use_only_custom_estimators:
            if len(custom_estimators) == 0:
                raise(ValueError('No custom estimators are provided, but use_only_custom_estimators is set to True'))
            else:
                for ce in custom_estimators:
                    estimator_list.append(ce[3], GeneralizedAnomalyModel(base_learner = ce[0], fit_function = 'fit', predict_function = ce[2], score_sign = ce[1]))
        else:
            # Initialize Different Anomaly Learners

            # Rule/Density based Anomaly Models
            gam_if = GeneralizedAnomalyModel(base_learner=IsolationForest(), predict_function='decision_function', score_sign=-1)
            gam_gm = GeneralizedAnomalyModel(base_learner=GaussianMixture(), predict_function='score_samples', score_sign=1)
            gam_bgm = GeneralizedAnomalyModel(base_learner=BayesianGaussianMixture(), predict_function='score_samples', score_sign=1)
            gam_ocsvm= GeneralizedAnomalyModel(base_learner=OneClassSVM(), predict_function='decision_function', score_sign=1)
            gam_nnam = GeneralizedAnomalyModel(base_learner=NearestNeighborAnomalyModel(), predict_function='predict', score_sign=1)
            gam_lof_nnam = GeneralizedAnomalyModel(base_learner=LOFNearestNeighborAnomalyModel(), predict_function='predict', score_sign=1)
            gam_pcaT2 = GeneralizedAnomalyModel(base_learner=AnomalyPCA_T2(), predict_function='anomaly_score',score_sign=1)
            gam_pcaQ = GeneralizedAnomalyModel(base_learner=AnomalyPCA_Q(), predict_function='anomaly_score',score_sign=1)

            # Covariance Structure based Anomaly Models
            gam_empirical = GeneralizedAnomalyModel(base_learner=EmpiricalCovariance(), fit_function='fit', predict_function='mahalanobis',score_sign=1)
            gam_elliptic = GeneralizedAnomalyModel(base_learner=EllipticEnvelope(), fit_function='fit', predict_function='mahalanobis',score_sign=1)
            gam_ledoitwolf = GeneralizedAnomalyModel(base_learner=LedoitWolf(), fit_function='fit', predict_function='mahalanobis',score_sign=1)
            gam_mincovdet = GeneralizedAnomalyModel(base_learner=MinCovDet(), fit_function='fit', predict_function='mahalanobis',score_sign=1)
            gam_OAS = GeneralizedAnomalyModel(base_learner=OAS(), fit_function='fit', predict_function='mahalanobis',score_sign=1)
            gam_ShrunkCovariance = GeneralizedAnomalyModel(base_learner=ShrunkCovariance(), fit_function='fit', predict_function='mahalanobis',score_sign=1)

            # GaussianGraphicalModel
            ggm_Default = GaussianGraphicalModel(sliding_window_size=sliding_window_size, sliding_window_data_cutoff=15, scale=True)
            ggm_stochastic = GaussianGraphicalModel(distance_metric='Stochastic_Nearest_Neighbors',sliding_window_size=sliding_window_size,\
                                                    sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True)
            ggm_kl_diverse = GaussianGraphicalModel(distance_metric='KL_Divergence_Dist',sliding_window_size=sliding_window_size,\
                                                   sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True)
            ggm_kl_divergence = GaussianGraphicalModel(distance_metric='KL_Divergence',sliding_window_size=sliding_window_size,\
                                                       sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True) 
            ggm_frobenius = GaussianGraphicalModel(distance_metric='Frobenius_Norm',sliding_window_size=sliding_window_size,\
                                                   sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True)
            ggm_likelihood = GaussianGraphicalModel(distance_metric='Likelihood',sliding_window_size=sliding_window_size,\
                                                    sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True)
            ggm_spectral = GaussianGraphicalModel(distance_metric='Spectral',sliding_window_size=sliding_window_size,\
                                                  sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True)
            ggm_mahalanobis_distance = GaussianGraphicalModel(distance_metric='Mahalanobis_Distance',sliding_window_size=sliding_window_size,\
                                                              sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True)
            ggm_sparsest_k_subgraph = GaussianGraphicalModel(distance_metric='Sparsest_k_Subgraph',sliding_window_size=sliding_window_size,\
                                                             sliding_window_data_cutoff=sliding_window_data_cutoff, scale=True)

            #new
            bayesian_gmm_outlier = GeneralizedAnomalyModel(base_learner=BayesianGMMOutlier(n_components=2, random_state=2),fit_function="fit",predict_function="decision_function",score_sign=-1,)
            gmm_outlier = GeneralizedAnomalyModel(base_learner=GMMOutlier(random_state=2),fit_function="fit",predict_function="decision_function",score_sign=-1)
            nsa_rf = GeneralizedAnomalyModel(base_learner=NSA(scale=True,sample_ratio=0.95,sample_delta=0.5,base_model=RandomForestClassifier(random_state=2),anomaly_threshold=0.8,),fit_function="fit",predict_function="decision_function",score_sign=-1)
            sample_svdd = GeneralizedAnomalyModel(base_learner=SampleSVDD(outlier_fraction=0.001, kernel_s=5),fit_function="fit",score_sign=1,predict_function="decision_function")
            random_partition_forest = GeneralizedAnomalyModel(base_learner=RandomPartitionForest(anomaly_type="point_wise"),fit_function="fit",predict_function="decision_function",score_sign=-1,)
            extended_isolation_forest= GeneralizedAnomalyModel(base_learner=ExtendedIsolationForest(),fit_function="fit",predict_function="decision_function",score_sign=-1)
            anomaly_ensembler_predict_only = GeneralizedAnomalyModel(base_learner=AnomalyEnsembler(predict_only=True, random_state = 72),fit_function="fit",predict_function="anomaly_score",score_sign=1)
            anomaly_ensembler = GeneralizedAnomalyModel(base_learner=AnomalyEnsembler(random_state = 72),fit_function="fit",predict_function="anomaly_score",score_sign=1)
            anomaly_robust_pca = GeneralizedAnomalyModel(base_learner=AnomalyRobustPCA(),fit_function="fit",predict_function="anomaly_score",score_sign=1)
            #graph_quic = GeneralizedAnomalyModel(GaussianGraphicalModel(base_learner=GraphQUIC(),sliding_window_size=0, scale=False),fit_function="fit",predict_function="predict",score_sign=1)
            #graph_pgscps = GeneralizedAnomalyModel(GaussianGraphicalModel(base_learner=GraphPgscps(),sliding_window_size=0, scale=True),fit_function="fit",predict_function="predict",score_sign=1)  
            graph_quic = GeneralizedAnomalyModel(base_learner=GraphQUIC(),fit_function="fit",predict_function="mahalanobis",score_sign=1)
            graph_pgscps = GeneralizedAnomalyModel(base_learner=GraphPgscps(),fit_function="fit",predict_function="mahalanobis",score_sign=1)  
            #(GraphPgscps(), 1, "mahalanobis", "graphpgscps"),
            neural_network_nsa = GeneralizedAnomalyModel(base_learner=NeuralNetworkNSA(scale=True, sample_ratio=25.0, sample_delta=0.05, batch_size=10, epochs=5, dropout=0.85, layer_width=150, n_hidden_layers=2,),fit_function="fit",predict_function="anomaly_score",score_sign=-1)
            cusum = GeneralizedAnomalyModel(base_learner = CUSUM(), fit_function = 'fit', predict_function = "predict", score_sign = 1)
            hotellingt2 = GeneralizedAnomalyModel(base_learner = HotellingT2(), fit_function = 'fit', predict_function = "score_samples", score_sign = 1)
            #kernel_density = GeneralizedAnomalyModel(base_learner = KernelDensity(), fit_function = 'fit', predict_function = "decision_function", score_sign = 1)
            kernel_density = GeneralizedAnomalyModel(base_learner = KernelDensity(), fit_function = 'fit', predict_function = "score_samples", score_sign = -1)
            spad = GeneralizedAnomalyModel(base_learner = SPAD(), fit_function = 'fit', predict_function = "decision_function", score_sign = -1)
            spad_extended = GeneralizedAnomalyModel(base_learner = ExtendedSPAD(), fit_function = 'fit', predict_function = "decision_function", score_sign = -1)
            oob = GeneralizedAnomalyModel(base_learner = OOB(), fit_function = 'fit', predict_function = "decision_function", score_sign = 1)
            dnn_autoencoder = GeneralizedAnomalyModel(base_learner=DNNAutoEncoder(), fit_function="fit",predict_function="predict",score_sign=1)

            estimator_list = [
                       ('isolationforest', gam_if), ('gaussianmixture', gam_gm), ('bayesiangaussianmixture', gam_bgm), ('oneclasssvm', gam_ocsvm), ('nearestneighboranomalymodel', gam_nnam), 
                       ('lofnearestneighboranomalymodel', gam_lof_nnam), ('anomalypca_t2', gam_pcaT2), ('anomalypca_q', gam_pcaQ), ('empiricalcovariance', gam_empirical), 
                       ('ellipticenvelope', gam_elliptic), ('ledoitwolf', gam_ledoitwolf), ('mincovdet', gam_mincovdet), ('oas', gam_OAS), ('shrunkcovariance', gam_ShrunkCovariance),
                       ('gaussiangraphicalmodel', ggm_Default), ('gaussiangraphicalmodel_snn', ggm_stochastic), ('gaussiangraphicalmodel_kl', ggm_kl_diverse), 
                       ('gaussiangraphicalmodel_kl_divergence', ggm_kl_divergence), ('gaussiangraphicalmodel_frobenius', ggm_frobenius), ('gaussiangraphicalmodel_likelihood', ggm_likelihood),
                       ('gaussiangraphicalmodel_mahalanobis', ggm_mahalanobis_distance), ('gaussiangraphicalmodel_sparse', ggm_sparsest_k_subgraph), 
                       ('gaussiangraphicalmodel_spectral', ggm_spectral), ('bayesiangmmoutlier',bayesian_gmm_outlier), ('gmmoutlier',gmm_outlier), ('nsa',nsa_rf), ('samplesvdd',sample_svdd),
                       ('randompartitionforest',random_partition_forest), ('extendedisolationforest',extended_isolation_forest), ('predict_only_anomalyensembler',anomaly_ensembler_predict_only),
                       ('anomalyensembler',anomaly_ensembler), ('anomalyrobustpca', anomaly_robust_pca), ('graphquic', graph_quic), ('graphpgscps', graph_pgscps), ('cusum', cusum), 
                       ('hotellingt2',hotellingt2), ('oob',oob),
                       #Neural network algorithm needs GPUs and takes a very long time to complete
                       ('neuralnetworknsa', neural_network_nsa), ('kerneldensity',kernel_density), ('spad', spad), ('spad_extended',spad_extended), ('DNNAutoEncoder', dnn_autoencoder)
                      ]
            if len(estimators_to_keep) != 0:
                reg = re.compile(r"[*]$" )
                fuzzy_names = list(filter(reg.search, estimators_to_keep))
                full_names = list(filter(lambda x: not reg.search(x), estimators_to_keep))
                selected_estimators_1 = [es for es in estimator_list if es[0] in full_names]
                selected_estimators_2 = []
                for etk in fuzzy_names:
                    for es in estimator_list:
                        if etk.replace('*','') in es[0]:
                            if not es in selected_estimators_2:
                                selected_estimators_2.append(es)
                estimator_list = selected_estimators_1 + selected_estimators_2
            elif len(estimators_to_exclude) != 0:
                reg = re.compile(r"[*]$" )
                fuzzy_names = list(filter(reg.search, estimators_to_exclude))
                full_names = list(filter(lambda x: not reg.search(x), estimators_to_exclude))
                selected_estimators_1 = [es for es in estimator_list if not es[0] in full_names]
                selected_estimators_2 = selected_estimators_1.copy()
                for ete in fuzzy_names:
                    ete_pattern = ete.replace('*','')
                    #print(ete_pattern)
                    for es in selected_estimators_1:
                        #print(ete_pattern, ' vs ', es[0])
                        if ete_pattern in es[0]:
                            #print('Deleting the entry ', es[0])
                            selected_estimators_2.remove(es)
                estimator_list = selected_estimators_2 #selected_estimators_1 + selected_estimators_2
            if len(custom_estimators) != 0:
                 for ce in custom_estimators:
                    estimator_list.append((ce[3], GeneralizedAnomalyModel(base_learner = ce[0], fit_function = 'fit', predict_function = ce[2], score_sign = ce[1])))
                    
        if scalers != None:
            if append_or_replace == 'replace':
                scaler_list = scalers
            elif append_or_replace == 'append':
                scaler_list = [MinMaxScaler(),StandardScaler(),RobustScaler(),MaxAbsScaler()]
                existing_scalers = [item.__class__.__name__ for item in scaler_list]
                for input_item in scalers:
                    if not input_item.__class__.__name__ in existing_scalers:
                        scaler_list.append(input_item)
            else:
                raise(ValueError('Unsupported list operation for scalers - ', append_or_replace))
        else:
            scaler_list = [MinMaxScaler(),StandardScaler(),RobustScaler(),MaxAbsScaler()]
        
        stages = [
            scaler_list,
            estimator_list
        ]
        return stages, None
    
    def _display_unsupervised_ad_dag(self):
        stages, anomaly_dag_ex = self._get_anomaly_dag_for_unsupervised_learning()
        print(anomaly_dag_ex)
        headers = ['Algorithm', 'Polarity', 'Function', 'Name']
        if RUNTIME_INCLUDES_IPYTHON:
            display(HTML(tabulate(anomaly_dag_ex, headers, tablefmt='html')))
        return anomaly_dag_ex
    
    def _display_semi_supervised_ad_dag(self):
        anomaly_pipeline = AnomalyPipeline()
        stages, anomaly_dag_ex = self._get_anomaly_dag_for_semi_supervised_learning()
        print('stages = \,', stages)
        anomaly_pipeline.set_stages(stages)
        #headers = ['Algorithm', 'Polarity', 'Function', 'Name']
        headers = ['Algorithm', 'Definition']
        if RUNTIME_INCLUDES_IPYTHON:
            display(HTML(tabulate(stages[1], headers, tablefmt='html')))
            #Image(anomaly_pipeline.sromgraph.asimage(num_nodes = 50))
