# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA, 5900-AMG
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

import numpy as np
import pandas as pd
import pytest
from sqlalchemy import func, select

from pandas.tseries.frequencies import to_offset
from pandas.tseries.offsets import DateOffset

from .. import api
from ..time_to_failure import TimeToFailureAssetGroupPipeline
from ..util import current_directory, log_df_info


def test_prepare_execution_adding_training_features(asset_group_id='abcd', iot_type=None):
    if iot_type is None:
        import math
        import random
        iot_type = 'abcdsensor_%05d' % math.floor(random.random() * 10**5)

    group = TimeToFailureAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:%s' % (iot_type, col) for col in ['sa_humidity_episode','sa_temp_episode','ra_humidity_episode','ra_temp_episode','return_co2_episode','chw_valve_feedback_episode','lthw_valve_feedback_episode','off_coil_temp_episode']],
                    'features_for_training': [':installdate', ':faildate'],
                    'predictions': ['predicted_time_to_failure'],
                },
            )

    original_feature_for_training = group.pipeline_config['features_for_training'].copy()

    # test there should still be 2 training features changed by prepare_execute()
    group.prepare_execute(group._entity_type.get_calc_pipeline(), group.pipeline_config)
    assert 2 == len(group.pipeline_config['features_for_training'])

    # test multiple calls to prepare_execute() should not add multiple times of extra training features
    group.prepare_execute(group._entity_type.get_calc_pipeline(), group.pipeline_config)
    assert 2 == len(group.pipeline_config['features_for_training'])


def test_time_to_failure_prediction_backtrack(asset_group_id='abcd', iot_type=None):
    if iot_type is None:
        import math
        import random
        iot_type = 'abcdsensor_%05d' % math.floor(random.random() * 10**5)

    # test default summary
    group = TimeToFailureAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:%s' % (iot_type, col) for col in ['sa_humidity_episode','sa_temp_episode','ra_humidity_episode','ra_temp_episode','return_co2_episode','chw_valve_feedback_episode','lthw_valve_feedback_episode','off_coil_temp_episode']],
                    'features_for_training': [':installdate', ':faildate'],
                    'predictions': ['predicted_time_to_failure'],
                },
            )
    assert [[DateOffset(**{"hour": 0, "minute": 0, "second": 0, "microsecond": 0}), DateOffset(**{"days": 1})], []] == group.get_prediction_backtrack(group.pipeline_config)

    # test custom hourly summary
    group = TimeToFailureAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:%s' % (iot_type, col) for col in ['sa_humidity_episode','sa_temp_episode','ra_humidity_episode','ra_temp_episode','return_co2_episode','chw_valve_feedback_episode','lthw_valve_feedback_episode','off_coil_temp_episode']],
                    'features_for_training': [':installdate', ':faildate'],
                    'predictions': ['predicted_time_to_failure'],
                },
                summary={
                    '${predictions[0]}': {
                        'hourly': {
                            'max': None,
                        },
                    },
                },
            )
    assert [[DateOffset(**{"minute": 0, "second": 0, "microsecond": 0}), DateOffset(**{"hours": 1})], []] == group.get_prediction_backtrack(group.pipeline_config)

    # test custom weekly summary
    group = TimeToFailureAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:%s' % (iot_type, col) for col in ['sa_humidity_episode','sa_temp_episode','ra_humidity_episode','ra_temp_episode','return_co2_episode','chw_valve_feedback_episode','lthw_valve_feedback_episode','off_coil_temp_episode']],
                    'features_for_training': [':installdate', ':faildate'],
                    'predictions': ['predicted_time_to_failure'],
                },
                summary={
                    '${predictions[0]}': {
                        'weekly': {
                            'max': None,
                        },
                    },
                },
            )
    assert [[DateOffset(**{"weekday": 6, "hour": 0, "minute": 0, "second": 0, "microsecond": 0}), DateOffset(n=1 if pd.Timestamp('today').weekday() == 6 else 2, **{"weeks": 1})], []] == group.get_prediction_backtrack(group.pipeline_config)

    # test custom monthly summary
    group = TimeToFailureAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:%s' % (iot_type, col) for col in ['sa_humidity_episode','sa_temp_episode','ra_humidity_episode','ra_temp_episode','return_co2_episode','chw_valve_feedback_episode','lthw_valve_feedback_episode','off_coil_temp_episode']],
                    'features_for_training': [':installdate', ':faildate'],
                    'predictions': ['predicted_time_to_failure'],
                },
                summary={
                    '${predictions[0]}': {
                        'monthly': {
                            'max': None,
                        },
                    },
                },
            )
    assert [[DateOffset(**{"day": 1, "hour": 0, "minute": 0, "second": 0, "microsecond": 0}), DateOffset(**{"months": 1})], []] == group.get_prediction_backtrack(group.pipeline_config)


def test_time_to_failure_summary_config(asset_group_id='abcd', iot_type=None):
    if iot_type is None:
        import math
        import random
        iot_type = 'abcdsensor_%05d' % math.floor(random.random() * 10**5)

    # test default summary
    group = TimeToFailureAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:%s' % (iot_type, col) for col in ['sa_humidity_episode','sa_temp_episode','ra_humidity_episode','ra_temp_episode','return_co2_episode','chw_valve_feedback_episode','lthw_valve_feedback_episode','off_coil_temp_episode']],
                    'features_for_training': [':installdate', ':faildate'],
                    'predictions': ['predicted_time_to_failure'],
                },
            )
    assert group.post_processing == [
        {
            "functionName": "Maximum",
            "enabled": True,
            "granularity": "Daily",
            "output": {
                "name": "daily_predicted_time_to_failure"
            },
            "input": {
                "source": "predicted_time_to_failure"
            },
        }
    ]


def test_time_to_failure_data_substitution(asset_group_id='abcd', iot_type=None):
    if iot_type is None:
        import math
        import random
        iot_type = 'abcdsensor_%05d' % math.floor(random.random() * 10**5)

    df_data = pd.read_csv('%s/ahu_time_to_failure.csv' % current_directory(file=__file__), parse_dates=['faildate'])
    df_data['deviceid'] = df_data['id'].str.replace('-____-BEDFORD', '')
    df_data = df_data.drop(columns='id')
    print('df_data=%s' % log_df_info(df_data, head=5))

    # to make it simple, we just use same id for asset/sensor pair, split the dataframe 
    # into 3 for data substitution
    df_asset_ts = df_data[['deviceid', 'faildate']].dropna(subset=['faildate'])
    df_asset_ts['event_timestamp'] = df_data['faildate']
    print('df_asset_ts=%s' % log_df_info(df_asset_ts, head=5))
    # df_asset_dim = df_data[pd.notna(df_data['installdate'])][]
    df_asset_dim = df_data[['deviceid', 'installdate']].groupby('deviceid').first().reset_index()
    print('df_asset_dim=%s' % log_df_info(df_asset_dim, head=5))
    df_sensor = df_data.drop(columns=['installdate', 'faildate'])
    print('df_sensor=%s' % log_df_info(df_sensor, head=5))

    group = TimeToFailureAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:%s' % (iot_type, col) for col in ['sa_humidity_episode','sa_temp_episode','ra_humidity_episode','ra_temp_episode','return_co2_episode','chw_valve_feedback_episode','lthw_valve_feedback_episode','off_coil_temp_episode']],
                    'features_for_training': [':installdate', ':faildate'],
                    'predictions': ['predicted_time_to_failure'],
                },
                asset_device_mappings={deviceid: ['%s:%s' % (iot_type, deviceid)] for deviceid in df_asset_dim['deviceid']},
                data_substitution={
                    '': [
                        {
                            'df': df_asset_ts,
                            'keys': ['deviceid'],
                            'columns': ['faildate'],
                            'timestamp': 'event_timestamp'
                        },
                        {
                            'df': df_asset_dim,
                            'keys': ['deviceid'],
                            'columns': ['installdate'],
                        },
                    ],
                    iot_type: [
                        {
                            'df': df_sensor,
                            'keys': ['deviceid'],
                            'columns': [
                                'sa_humidity_episode',
                                'sa_temp_episode',
                                'ra_humidity_episode',
                                'ra_temp_episode',
                                'return_co2_episode',
                                'chw_valve_feedback_episode',
                                'lthw_valve_feedback_episode',
                                'off_coil_temp_episode',
                            ],
                            'timestamp': 'event_timestamp',
                        },
                    ],
                }
            )
    df = group.execute()

    print(log_df_info(df, head=0))

    assert group.new_training
    assert False == group.training
    assert 'TimeToFailureEstimatorSrom' in group.model_timestamp
    assert (4113, 1) == df.shape
    assert 'predicted_time_to_failure' in df.columns


if __name__ == '__main__':
    test_prepare_execution_adding_training_features()
    test_time_to_failure_prediction_backtrack()
    test_time_to_failure_summary_config()
    test_time_to_failure_data_substitution()

