# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA, 5900-AMG
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

import math
import random

import numpy as np
import pandas as pd
import pytest

from .. import api
from ..anomaly_detection import AnomalyDetectionAssetGroupPipeline
from ..failure_prediction import FailurePredictionAssetGroupPipeline
from ..pipeline import _ModelPipelineConfig, AssetGroupPipeline
from ..time_to_failure import TimeToFailureAssetGroupPipeline
from ..util import current_directory, log_df_info


@pytest.fixture(scope='module')
def asset_group_id():
    return 'abcd'


@pytest.fixture
def iot_type():
    return 'abcdsensor_%05d' % math.floor(random.random() * 10**5)


def test_immutable_modelconfigpipeline_keys():
    db = api._get_db()

    model_pipeline = {
        'features': ['abcd:axlevibration', 'abcd:axlemomentum'],
        'features_for_training': [':faildate'],
        'predictions': ['anomaly_score', 'anomaly_threshold'],
    }
    pipeline_config = _ModelPipelineConfig(db=db, entity_type_metadata=db.entity_type_metadata, training=True, **model_pipeline)
    assert pipeline_config['features'] == ['axlevibration', 'axlemomentum']
    assert pipeline_config['features_for_training'] == ['faildate']
    assert pipeline_config['predictions'] == ['anomaly_score', 'anomaly_threshold']
    assert pipeline_config['inputs'] == ('abcd:axlevibration', 'abcd:axlemomentum', ':faildate')
    assert pipeline_config['renamed_inputs'] == ('axlevibration', 'axlemomentum', 'faildate')

    print(pipeline_config)

    pipeline_config['features'].append('new_feature')
    assert pipeline_config['features'] == ['axlevibration', 'axlemomentum', 'new_feature']
    pipeline_config['features_for_training'].append('new_feature_2')
    assert pipeline_config['features_for_training'] == ['faildate', 'new_feature_2']
    pipeline_config['predictions'].append('new_prediction')
    assert pipeline_config['predictions'] == ['anomaly_score', 'anomaly_threshold', 'new_prediction']

    pipeline_config['new_config'] = 1

    with pytest.raises(ValueError):
        pipeline_config['inputs'] = []
    with pytest.raises(AttributeError):
        pipeline_config['inputs'].append(['new_feature'])
    with pytest.raises(ValueError):
        pipeline_config['renamed_inputs'] = []
    with pytest.raises(AttributeError):
        pipeline_config['renamed_inputs'].append(['new_feature'])


def test_duplicated_feature_name(asset_group_id, iot_type):
    # normal case without feature renaming, different name
    group = AssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'predictions': ['out'],
                },
            )
    assert group.pipeline_config['features'] == ['name1', 'name2']

    # normal case without feature renaming, same name same type, two occurance
    with pytest.raises(ValueError):
        group = AssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name' % iot_type, '%s:name' % iot_type],
                        'predictions': ['out'],
                    },
                )

    # normal case without feature renaming, same name same type, two occurance separatated by another feature
    with pytest.raises(ValueError):
        group = AssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name' % iot_type, '%s:other_name' % iot_type, '%s:name' % iot_type],
                        'predictions': ['out'],
                    },
                )

    # normal case without feature renaming, same name same type, three occurance
    with pytest.raises(ValueError):
        group = AssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name' % iot_type, '%s:name' % iot_type, '%s:name' % iot_type],
                        'predictions': ['out'],
                    },
                )

    # normal case without feature renaming, same name same type, two sets of duplicates
    with pytest.raises(ValueError):
        group = AssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name' % iot_type, '%s:name2' % iot_type, '%s:name' % iot_type, '%s:name2' % iot_type, '%s:name' % iot_type],
                        'predictions': ['out'],
                    },
                )

    # without feature renaming, same name from different types still not allowed
    with pytest.raises(ValueError):
        group = AssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name' % iot_type, '%s_2:name' % iot_type],
                        'predictions': ['out'],
                    },
                )

    # with feature renaming
    group = AssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name' % iot_type, '%s_2:name:name2' % iot_type],
                    'predictions': ['out'],
                },
            )
    assert group.pipeline_config['features'] == ['name', 'name2']

    # with feature renaming, different names from different types are renamed to be the same
    with pytest.raises(ValueError):
        group = AssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name1:name' % iot_type, '%s_2:name_2:name' % iot_type],
                        'predictions': ['out'],
                    },
                )

    # features and features_for_training together

    # normal case without feature renaming, different name
    group = AssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'features_for_training': [':installdate'],
                    'predictions': ['out'],
                },
            )
    assert group.pipeline_config['features'] == ['name1', 'name2']
    assert group.pipeline_config['features_for_training'] == ['installdate']

    # normal case without feature renaming, same name same type across features and feature_for_training
    with pytest.raises(ValueError):
        group = AssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name' % iot_type, '%s:name2' % iot_type],
                        'features_for_training': ['%s:name' % iot_type],
                        'predictions': ['out'],
                    },
                )

    # normal case without feature renaming, same name same type across features and feature_for_training
    with pytest.raises(ValueError):
        group = AssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name:name' % iot_type, '%s:name2:name' % iot_type],
                        'features_for_training': [':installdate', '%s:name3:name' % iot_type],
                        'predictions': ['out'],
                    },
                )


def test_published_outputs(asset_group_id, iot_type):
    group = AssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'predictions': ['out'],
                },
            )
    assert group.published_outputs == {}

    # override by paeameter
    group = AssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'predictions': ['out'],
                },
                published_outputs={'out': 'dummy'},
            )
    assert group.published_outputs == {'out': 'dummy'}

    # invalid output to be published
    with pytest.raises(ValueError):
        group = AssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                        'predictions': ['out'],
                    },
                    published_outputs={'nonexistent': 'dummy'},
                )

    # anomaly
    group = AnomalyDetectionAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'predictions': ['anomaly_score', 'anomaly_threshold'],
                },
            )
    assert group.published_outputs == {'daily_anomaly_score': 'anomaly_score'}

    # override by parameter, publishing raw level outputs instead
    group = AnomalyDetectionAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'predictions': ['anomaly_score', 'anomaly_threshold'],
                },
                published_outputs={'anomaly_score': 'anomaly_score_override'},
            )
    assert group.published_outputs == {'anomaly_score': 'anomaly_score_override'}

    # override by parameter, publishing both summary and raw level outputs instead
    group = AnomalyDetectionAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'predictions': ['anomaly_score', 'anomaly_threshold'],
                },
                published_outputs={'daily_anomaly_score': 'anomaly_score', 'anomaly_threshold': 'anomaly_threshold'},
            )
    assert group.published_outputs == {'daily_anomaly_score': 'anomaly_score', 'anomaly_threshold': 'anomaly_threshold'}

    # partial published outputs are invalid
    with pytest.raises(ValueError):
        group = AnomalyDetectionAssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                        'predictions': ['anomaly_score', 'anomaly_threshold'],
                    },
                    published_outputs={'daily_anomaly_score': 'anomaly_score', 'daily_anomaly_threshold': 'dummy'},
                )

    # duplicated Health side names of published outputs
    with pytest.raises(ValueError):
        group = AnomalyDetectionAssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                        'predictions': ['anomaly_score', 'anomaly_threshold'],
                    },
                    published_outputs={'daily_anomaly_score': 'anomaly', 'anomaly_threshold': 'anomaly'},
                )

    # failure probability
    group = FailurePredictionAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'features_for_training': [':faildate'],
                    'predictions': ['failure_probability', 'rca_path'],
                    'prediction_window_size': '5d',
                    'aggregation_methods': ['mean'],
                },
            )
    assert group.published_outputs == {'daily_failure_probability_5d': 'failure_probability_5d'}

    group = FailurePredictionAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'features_for_training': [':faildate'],
                    'predictions': ['failure_probability', 'rca_path'],
                    'prediction_window_size': '10d',
                    'aggregation_methods': ['mean'],
                },
            )
    assert group.published_outputs == {'daily_failure_probability_10d': 'failure_probability_10d'}

    # multiclass failure probability
    group = FailurePredictionAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'predictions': ['failure_probability', 'rca_path'],
                    'prediction_window_size': '10d',
                    'aggregation_methods': ['mean'],
                    'multiclass': True,
                    'failure_modes': ['STOPPED','BROKEN'],
                },
            )
    assert group.published_outputs == {
        'daily_failure_probability_stopped_10d': 'failure_probability_stopped_10d',
        'daily_failure_probability_broken_10d': 'failure_probability_broken_10d',
    }

    # features_for_training cannot be given with faildate when failure_modes is given
    with pytest.raises(ValueError):
        group = FailurePredictionAssetGroupPipeline(
                    asset_group_id=asset_group_id,
                    model_pipeline={
                        'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                        'features_for_training': [':faildate'],
                        'predictions': ['failure_probability', 'rca_path'],
                        'prediction_window_size': '10d',
                        'aggregation_methods': ['mean'],
                        'multiclass': True,
                        'failure_modes': ['STOPPED','BROKEN'],
                    },
                )

    # failure date
    group = TimeToFailureAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'features_for_training': [':installdate', ':faildate'],
                    'predictions': ['predicted_time_to_failure'],
                },
            )
    assert group.published_outputs == {'daily_predicted_time_to_failure': 'predicted_time_to_failure'}

    # failure_mode is ignore when not using smart_regression
    group = TimeToFailureAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'features_for_training': [':installdate', ':faildate'],
                    'predictions': ['predicted_time_to_failure'],
                    'failure_mode': 'PUMPS/LEAKING/SEALBROKEN/REPLACE',
                },
            )
    assert group.published_outputs == {'daily_predicted_time_to_failure': 'predicted_time_to_failure'}

    # smart regression failure date
    group = TimeToFailureAssetGroupPipeline(
                asset_group_id=asset_group_id,
                model_pipeline={
                    'features': ['%s:name1' % iot_type, '%s:name2' % iot_type],
                    'features_for_training': [':installdate', ':faildate'],
                    'predictions': ['predicted_time_to_failure'],
                    'failure_mode': 'PUMPS/LEAKING/SEALBROKEN/REPLACE',
                    'smart_regression': True,
                },
            )
    assert group.published_outputs == {
        'daily_predicted_time_to_failure_pumps_leaking_sealbroken_replace': 'predicted_time_to_failure_pumps_leaking_sealbroken_replace',
        'daily_prediction_interval_pumps_leaking_sealbroken_replace_high': 'prediction_interval_pumps_leaking_sealbroken_replace_high',
        'daily_prediction_interval_pumps_leaking_sealbroken_replace_low': 'prediction_interval_pumps_leaking_sealbroken_replace_low',
    }


if __name__ == '__main__':
    test_immutable_modelconfigpipeline_keys()
    test_duplicated_feature_name()

