# -*- coding: utf-8 -*-
# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA, 5900-AMG
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

import pandas as pd
import numpy as np
from datetime import timedelta
import logging
import timeit
#from IPython.display import display, HTML, Image

import json
import os
#from pmlib.time_to_event_feature_eng import TSAggOps
from srom.feature_engineering.timeseries.rolling_window_feature_extraction import simple_summary_statistics, \
                                advance_summary_statistics, rate_of_change

from enum import IntEnum, unique


class BaseAnomalyContext():
    def __init__(self, df:pd.DataFrame, context_name:str, expression:str, custom_function = None,
                 custom_function_arg_list:list = None, custom_function_arg_dict:dict = None, **kwargs):
        """

        Parameters
        ----------
        timeunits : TYPE, optional
            DESCRIPTION. Values permitted by Pandas Timedelta - [weeks, days, hours, minutes, seconds, milliseconds, microseconds, nanoseconds]. The default is 'days'.

        Returns
        -------
        None.

        """
        self.context_name = context_name
        self.source_df = df
        self.expression = expression
        self.custom_function = custom_function
        self.custom_function_arg_list = custom_function_arg_list
        self.custom_function_arg_dict = custom_function_arg_dict
        self.context_kwargs = kwargs
        self.modified_df = None
        self.__logger = logging.getLogger()
        self.__logger.setLevel(kwargs.get('log_level',logging.DEBUG))


    def apply_context(self):
        if self.custom_function == None:
            self.modified_df = self.source_df.query(self.expression)
        else:
            self.modified_df = self.execute_context_rules(df,self.custom_function_arg_list,
                                                     self.custom_function_arg_dict, **self.context_kwargs)
        return self.modified_df

    def execute_context_rules(self, df:pd.DataFrame, anyList:list, anyDict:dict, **kwargs):
        # This function should be overridden using a custom logic / implementation to produce
        # the modified_df. Once that's done in the derived class, call this function using super()
        # and that will return the modified_df
        return self.modified_df

    def compute_temporal_features(self, device_id_column_name, timestamp_column_name,
                                  timestamp_column_format, rolling_window_size, agg_fns,
                                  variable_clmns = None, aggregation_type = 'time', min_periods = None):
        self.__logger.debug('Beginning of compute_temporal_features \n')
        source_temporal_df = simple_summary_statistics(self.source_df, rolling_window_size, variable_clmns,\
                     device_id_column_name, timestamp_column_name, timestamp_column_format, min_periods, \
                     aggregation_type = aggregation_type, aggregation_methods = agg_fns)
        source_temporal_df.rename({'asset_id':device_id_column_name, 'datetime':timestamp_column_name}, inplace = True, axis = 1)
        return source_temporal_df
    def compute_rate_change(self, device_id_column_name, timestamp_column_name,
                                  timestamp_column_format, rolling_window_size, 
                                  variable_clmns = None, aggregation_type = 'time', min_periods = None):
        self.__logger.debug('Beginning of compute_rate_change_features \n')
        source_rate_change_df = advance_summary_statistics(self.source_df, rolling_window_size, variable_clmns,\
                     device_id_column_name, timestamp_column_name, timestamp_column_format, min_periods, \
                     aggregation_type = aggregation_type, aggregation_methods = ['rate_of_change'])
        source_rate_change_df.rename({'asset_id':device_id_column_name, 'datetime':timestamp_column_name}, inplace = True, axis = 1)
        return source_rate_change_df
    def compute_temporal_rate_change_features(self, device_id_column_name, timestamp_column_name,
                                  timestamp_column_format, rolling_window_size, agg_fns,
                                  variable_clmns = None, aggregation_type = 'time', min_periods = None):
        feature_df = simple_summary_statistics(self.source_df, rolling_window_size, variable_clmns,\
                     device_id_column_name, timestamp_column_name, timestamp_column_format, min_periods, \
                     aggregation_type = aggregation_type, aggregation_methods = agg_fns)
        print('Shape of the data frame after feature creation = ', str(feature_df.shape))
        feature_df.rename({'asset_id':device_id_column_name, 'datetime':timestamp_column_name}, inplace = True, axis = 1)
        feature_df[timestamp_column_name] = pd.to_datetime(feature_df[timestamp_column_name], format = timestamp_column_format)
        print(feature_df.columns.values)
        #display(feature_df.head())
        rate_change_df = advance_summary_statistics(feature_df, rolling_window_size, variable_clmns,\
                     device_id_column_name, timestamp_column_name, timestamp_column_format, min_periods, \
                     aggregation_type = aggregation_type, aggregation_methods = ['rate_of_change'])
        print('Shape of data frame after rate change = ', str(rate_change_df.shape))
        rate_change_df.rename({'asset_id':device_id_column_name, 'datetime':timestamp_column_name}, inplace = True, axis = 1)
        print(rate_change_df.columns.values)
        
        rate_change_features = rate_change_df.columns.values.tolist()
        rate_change_features = list( set(rate_change_features) - set([timestamp_column_name, device_id_column_name] + variable_clmns))
        for rcf in rate_change_features:
            feature_df[rcf] = rate_change_df[rcf]
        
        print('Shape of data frame after combining the results = ', str(feature_df.shape))
        print(feature_df.columns.values)
        
        return feature_df