# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

import json

import numpy as np
import pandas as pd
import pytest
from requests import Session

from watson_machine_learning_client import WatsonMachineLearningAPIClient

from .. import api
from ..estimator import WmlDeploymentEstimator
from ..pipeline import WmlScoringAssetGroupPipeline
from ..util import current_directory, log_df_info


def test_wml_client_credentials(mocker):
    asset_group_id = 'abcd'

    import math
    import random
    iot_type = 'abcdsensor_%05d' % math.floor(random.random() * 10**5)

    siteid = 'BEDFORD'

    df_data_asset = pd.read_csv('%s/trainbrake_asset_faildates.csv' % current_directory(file=__file__), parse_dates=['faildate'])
    df_data_asset['asset'] = np.where(df_data_asset['asset'] == 'TRAINBRAKE1', 'abcd-1', 'abcd-2')
    df_asset_group = df_data_asset.groupby(['site', 'asset']).size().reset_index()[['site', 'asset']]

    df_data_sensor = pd.read_csv('%s/trainbrake_device_data.csv' % current_directory(file=__file__), parse_dates=['RCV_TIMESTAMP_UTC'])[:10000]
    df_data_sensor['DEVICEID'] = np.where(df_data_sensor['DEVICEID'] == 'TrainBrake_1', 'abcd-1', 'abcd-2')

    # generate 2 columns one datetime the other boolean to test if WML pipeline can handle them
    df_data_sensor['datetime'] = pd.to_datetime(df_data_sensor['RCV_TIMESTAMP_UTC'])
    df_data_sensor['boolean'] = True

    df_mappings = pd.DataFrame(
        columns=['site', 'asset', 'devicetype', 'deviceid'],
        data=[
            ['BEDFORD', 'abcd-1', iot_type, 'abcd-1'],
        ],
    )

    model = mocker.Mock()
    model.score.return_value = {'values': [[i] for i in range(5375)]}
    wml_client = mocker.Mock()
    wml_client.deployments.get_details.return_value = {'entity': {'scoring_url': 'abc'}}
    mocker.patch.object(WmlDeploymentEstimator, 'wml_client', new=wml_client)
    mocker.patch.object(WmlDeploymentEstimator, 'train_model', return_value=model)
    mocker.patch.object(WmlDeploymentEstimator, 'get_models_for_predict', return_value=[model])

    try:
        db = api._get_db()
        db_schema = None

        api.set_asset_group_members(asset_group_id=asset_group_id, df=df_asset_group, db=db, db_schema=db_schema)

        api.set_asset_device_mappings(df=df_mappings, db=db, db_schema=db_schema)

        api.set_asset_cache(df=df_data_asset, siteid_column='site', assetid_column='asset', faildate_column='faildate', db=db, db_schema=db_schema)

        api.setup_iot_type(iot_type, df_data_sensor, columns=['TRAINBRAKESIMULATION_AXLEVIBRATION', 'TRAINBRAKESIMULATION_AXLEMOMENTUM', 'datetime', 'boolean'], deviceid_column='DEVICEID', timestamp_column='RCV_TIMESTAMP_UTC', timestamp_in_payload=False, parse_dates=None, rename_columns={'TRAINBRAKESIMULATION_AXLEVIBRATION': 'axlevibration', 'TRAINBRAKESIMULATION_AXLEMOMENTUM': 'axlemomentum'}, write='deletefirst', use_wiotp=False, import_only=False, db=db, db_schema=db_schema)

        your_api_key = "dummy"
        # new IAM apikey style
        group = WmlScoringAssetGroupPipeline(
                asset_group_id=asset_group_id, 
                model_pipeline={
                    'features': ['%s:%s' % (iot_type, col) for col in ['axlevibration', 'axlemomentum', 'datetime', 'boolean']],
                    'predictions': ['prediction'],
                    'wml_credentials': {
                        'apikey': your_api_key,
                        'iam_apikey_description': 'Auto-generated for key 345311bf-0b7f-42fa-b951-e0eb5f91988d',
                        'iam_apikey_name': 'Service credentials-1',
                        'iam_role_crn': 'crn:v1:bluemix:public:iam::::serviceRole:Writer',
                        'iam_serviceid_crn': 'crn:v1:bluemix:public:iam-identity::a/3d2f4d5b026077a5be328bbfde7abb75::serviceid:ServiceId-cd98659f-f720-4c7a-bad3-c71d3d3ad4cd',
                        'instance_id': 'c59e332a-c55d-4d00-bcb7-93e1a6ade7ab',
                        'url': 'https://us-south.ml.cloud.ibm.com',
                    },
                    'wml_deployment_uid': 'f056bce2-e762-414a-b4a3-ef9dbf558f88',
                    'result_value_index': [0],
                },
                summary={
                    '${predictions[0]}': {
                        'daily': {
                            'max': None,
                        },
                    },
                },
            )
        df = group.execute()

        print(log_df_info(df, head=0))

        assert len(df) == 5375
        assert df.iloc[0, 0] == 0
        assert df.iloc[1 , 0] == 1
        assert df.iloc[5374, 0] == 5374
        assert group.post_processing == [
            {
                "functionName": "Maximum",
                "enabled": True,
                "granularity": "Daily",
                "output": {
                    "name": "daily_prediction_max"
                },
                "input": {
                    "source": "prediction"
                },
            },
        ]
    except:
        raise
    finally:
        api.delete_iot_type(iot_type, use_wiotp=False, db=db, db_schema=db_schema)

        api.delete_asset_cache(df=df_data_asset, db=db, db_schema=db_schema)

        api.delete_asset_device_mappings(df=df_mappings, db=db, db_schema=db_schema)

        api.delete_asset_group_members(asset_group_id=asset_group_id, db=db, db_schema=db_schema)

