# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

import numpy as np
import pandas as pd
import pytest

from .. import api


def test_multi_timezone_df_column(mocker):
    asset_metadata = {
        'MC20-____-BEDFORD': {'installdate': '2010-07-01T00:00:00-05:00'}, 
        'MC204-____-BEDFORD': {'installdate': '2017-01-19T00:00:00-06:00'},
        'MC205-____-BEDFORD': {'installdate': '2017-01-19T00:00:00+08:00'},
    }

    # mock the underlying API call for api.get_asset_attributes, using pytest_mock
    mocker.patch.object(api, 'get_maximo_asset_metadata', autospec=True, return_value=asset_metadata)

    api.get_maximo_asset_metadata.assert_not_called
    assert asset_metadata == api.get_maximo_asset_metadata(assets=None, data_items=None)
    api.get_maximo_asset_metadata.assert_called
    api.get_maximo_asset_metadata.assert_called_with(assets=None, data_items=None)

    assets = []
    for asset_id in asset_metadata.keys():
        asset_num, site_id = asset_id.split('-____-')
        assets.append({"assetNum":asset_num, "siteId": site_id})
    df = api.get_asset_attributes(assets=assets, data_items=['installdate'], df_id_column='asset_id')

    df = df.set_index('asset_id')
    assert df.loc['MC20-____-BEDFORD', 'installdate'] == pd.to_datetime('2010-07-01').tz_localize(None)
    assert df.loc['MC204-____-BEDFORD', 'installdate'] == pd.to_datetime('2017-01-19').tz_localize(None)
    assert df.loc['MC205-____-BEDFORD', 'installdate'] == pd.to_datetime('2017-01-19').tz_localize(None)

    # white-box test to convert multi-timezone datetime column-wise which throws error

    asset_metadata_list = []
    for asset_id, asset_meta in asset_metadata.copy().items():
        asset_meta = asset_meta.copy()
        asset_meta.update({
            'asset_id': asset_id,
        })
        asset_metadata_list.append(asset_meta)

    df = pd.DataFrame(asset_metadata_list)
    df = df.set_index('asset_id')
    assert df.loc['MC20-____-BEDFORD', 'installdate'] == '2010-07-01T00:00:00-05:00'
    assert df.loc['MC204-____-BEDFORD', 'installdate'] == '2017-01-19T00:00:00-06:00'
    assert df.loc['MC205-____-BEDFORD', 'installdate'] == '2017-01-19T00:00:00+08:00'

    with pytest.raises(ValueError):
        df['installdate'] = pd.to_datetime(df['installdate']).dt.tz_localize(None)

