# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

from functools import partial
import json
import os

import numpy as np
import pandas as pd
import pytest
from requests import Session

from .. import api
from ..util import log_df_info


def test_get_asset_attributes(mocker):
    # white box testing, to pass some function level validation
    mocker.patch.object(api, 'APM_ID', 'apm_id')
    mocker.patch.object(api, 'APM_API_BASEURL', 'apm_api_baseurl')
    mocker.patch.object(api, 'APM_API_KEY', 'apm_api_key')

    # mock requests response
    resp = mocker.MagicMock()
    resp.status_code = 200
    resp.text = json.dumps({
        "meta": [
            {
                "installdate": "2019-10-14T00:00:00-05:00",
                "assetNum": "INTGTST0000",
                "siteId": "BEDFORD",
                "orgid": "EAGLENA",
                "status": "NOT READY",
                "statusdate": "2019-11-27T03:58:36-05:00",
            },
            {
                "installdate": "2019-10-16T00:00:00-05:00",
                "assetNum": "INTGTST0001",
                "siteId": "BEDFORD",
                "estendoflife": "2019-10-16T00:00:00-05:00",
                "orgid": "EAGLENA",
                "status": "NOT READY",
                "statusdate": "2019-11-27T03:58:36-05:00",
            },
        ]
    })
    resp.json = partial(json.loads, resp.text)
    mocker.patch.object(Session, 'request', return_value=resp)

    assets = [{'assetNum': 'INTGTST%04d' % i, 'siteId': 'BEDFORD'} for i in range(2)]

    # just installdate (timezone neutral date)
    df = api.get_asset_attributes(assets=assets, data_items=['installdate'], df_id_column='id')
    print(log_df_info(df, head=5))
    assert len(df) == 2
    assert set(df.columns) == set(['id', 'installdate'])
    assert df.iloc[0]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[0]['installdate'] == pd.Timestamp('2019-10-14T00:00:00-05:00').tz_localize(None)
    assert df.iloc[1]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[1]['installdate'] == pd.Timestamp('2019-10-16T00:00:00-05:00').tz_localize(None)

    # 2 data items
    df = api.get_asset_attributes(assets=assets, data_items=['installdate', 'orgid'], df_id_column='id')
    print(log_df_info(df, head=5))
    assert len(df) == 2
    assert set(df.columns) == set(['id', 'installdate', 'orgid'])
    assert df.iloc[0]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[0]['installdate'] == pd.Timestamp('2019-10-14T00:00:00-05:00').tz_localize(None)
    assert df.iloc[0]['orgid'] == 'EAGLENA'
    assert df.iloc[1]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[1]['installdate'] == pd.Timestamp('2019-10-16T00:00:00-05:00').tz_localize(None)
    assert df.iloc[1]['orgid'] == 'EAGLENA'

    # with timezone-aware datetime field
    df = api.get_asset_attributes(assets=assets, data_items=['estendoflife', 'status', 'statusdate'], df_id_column='id')
    print(log_df_info(df, head=5))
    assert len(df) == 2
    assert set(df.columns) == set(['id', 'estendoflife', 'status', 'statusdate'])
    assert df.iloc[0]['id'] == 'INTGTST0000-____-BEDFORD'
    assert pd.isna(df.iloc[0]['estendoflife'])
    assert df.iloc[0]['status'] == 'NOT READY'
    assert df.iloc[0]['statusdate'] == pd.Timestamp('2019-11-27T03:58:36-05:00')
    assert df.iloc[1]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[1]['estendoflife'] == pd.Timestamp('2019-10-16T00:00:00-05:00').tz_localize(None)
    assert df.iloc[1]['status'] == 'NOT READY'
    assert df.iloc[1]['statusdate'] == pd.Timestamp('2019-11-27T03:58:36-05:00')

    # test parameter validation

    with pytest.raises(ValueError):
        api.get_asset_attributes(assets=None, data_items=['installdate'], df_id_column='id')
    with pytest.raises(ValueError):
        api.get_asset_attributes(assets=[], data_items=['installdate'], df_id_column='id')
    with pytest.raises(ValueError):
        api.get_asset_attributes(assets=[{}, {}], data_items=['installdate'], df_id_column='id')
    with pytest.raises(ValueError):
        api.get_asset_attributes(assets=[{'assetNum': 'INTGTST0000'}], data_items=None, df_id_column='id')
    with pytest.raises(ValueError):
        api.get_asset_attributes(assets=[{'assetnum': 'INTGTST0000', 'siteId': 'BEDFORD'}], data_items=None, df_id_column='id')
    with pytest.raises(ValueError):
        api.get_asset_attributes(assets=[{'assetNum': 'INTGTST0000', 'siteid': 'BEDFORD'}], data_items=None, df_id_column='id')
    with pytest.raises(ValueError):
        api.get_asset_attributes(assets=assets, data_items=None, df_id_column='id')
    with pytest.raises(ValueError):
        api.get_asset_attributes(assets=assets, data_items=[], df_id_column='id')
    with pytest.raises(ValueError):
        api.get_asset_attributes(assets=assets, data_items=['installdate'], df_id_column=None)
    with pytest.raises(ValueError):
        api.get_asset_attributes(assets=assets, data_items=['installdate'], df_id_column='')


def test_get_failure_history(mocker):
    # white box testing, to pass some function level validation
    mocker.patch.object(api, 'APM_ID', 'apm_id')
    mocker.patch.object(api, 'APM_API_BASEURL', 'apm_api_baseurl')
    mocker.patch.object(api, 'APM_API_KEY', 'apm_api_key')

    # mock requests response
    resp = mocker.MagicMock()
    resp.status_code = 200
    resp.text = json.dumps({
        "pastFailure": [
            {
                "date": "2019-10-14T03:03:05+00:00",
                'classcode': 'PUMPS',
                "problemcode": "STOPPED",
                'causecode': 'PUMPJAM',
                'remedycode': 'RESET',
                "wonum": "W2_0000_00"
            },
            {
                "date": "2019-10-16T01:02:08+00:00",
                'classcode': 'PUMPS',
                "problemcode": "STOPPED",
                'causecode': 'PUMPJAM',
                'remedycode': 'RESET',
                "wonum": "W2_0000_01"
            },
        ]
    })
    resp.json = partial(json.loads, resp.text)
    mocker.patch.object(Session, 'request', return_value=resp)

    assets = [{'assetNum': 'INTGTST%04d' % i, 'siteId': 'BEDFORD'} for i in range(10)]

    # synchronous

    df = api.get_asset_failure_history(assets=assets, data_items=None, df_id_column='id', df_timestamp_name='timestamp', async_mode=False)
    print(log_df_info(df, head=5))
    assert len(df) == 20 # 10 assets each with 2 failure records
    assert set(df.columns) == set(['id', 'timestamp', 'faildate', 'problemcode','causecode','remedycode'])
    assert df.iloc[0]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[0]['timestamp'] == pd.Timestamp('2019-10-14T03:03:05')
    assert df.iloc[1]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[1]['timestamp'] == pd.Timestamp('2019-10-16T01:02:08')
    assert df.iloc[2]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[2]['timestamp'] == pd.Timestamp('2019-10-14T03:03:05')
    assert df.iloc[3]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[3]['timestamp'] == pd.Timestamp('2019-10-16T01:02:08')

    # asynchronous

    df = api.get_asset_failure_history(assets=assets, data_items=None, df_id_column='id', df_timestamp_name='timestamp', async_mode=True)
    print(log_df_info(df, head=5))
    assert len(df) == 20 # 10 assets each with 2 failure records
    assert set(df.columns) == set(['id', 'timestamp', 'faildate', 'problemcode','causecode','remedycode'])
    assert df.iloc[0]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[0]['timestamp'] == pd.Timestamp('2019-10-14T03:03:05')
    assert df.iloc[1]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[1]['timestamp'] == pd.Timestamp('2019-10-16T01:02:08')
    assert df.iloc[2]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[2]['timestamp'] == pd.Timestamp('2019-10-14T03:03:05')
    assert df.iloc[3]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[3]['timestamp'] == pd.Timestamp('2019-10-16T01:02:08')

    # test API response containing also 'failurecode'

    resp = mocker.MagicMock()
    resp.status_code = 200
    resp.text = json.dumps({
        "pastFailure": [
            {
                "date": "2019-10-14T03:03:05+00:00",
		        'classcode': 'PUMPS',
                "failurecode": "PUMPS",
                "problemcode": "STOPPED",
                'causecode': 'PUMPJAM',
                'remedycode': 'RESET',
                "wonum": "W2_0000_00"
            },
            {
                "date": "2019-10-16T01:02:08+00:00",
		        'classcode': 'PUMPS',
                "failurecode": "PUMPS",
                "problemcode": "STOPPED",
                'causecode': 'PUMPJAM',
                'remedycode': 'RESET',
                "wonum": "W2_0000_01"
            },
        ]
    })
    resp.json = partial(json.loads, resp.text)
    mocker.patch.object(Session, 'request', return_value=resp)

    df = api.get_asset_failure_history(assets=assets, data_items=None, df_id_column='id', df_timestamp_name='timestamp', async_mode=False)
    print(log_df_info(df, head=5))
    assert len(df) == 20 # 10 assets each with 2 failure records
    assert set(df.columns) == set(['id', 'timestamp', 'faildate', 'failurecode', 'problemcode','causecode','remedycode'])
    assert df.iloc[0]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[0]['timestamp'] == pd.Timestamp('2019-10-14T03:03:05')
    assert df.iloc[0]['failurecode'] == 'PUMPS'
    assert df.iloc[0]['problemcode'] == 'PUMPS/STOPPED'
    assert df.iloc[1]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[1]['timestamp'] == pd.Timestamp('2019-10-16T01:02:08')
    #assert pd.isna(df.iloc[1]['failurecode'])
    assert df.iloc[1]['problemcode'] == 'PUMPS/STOPPED'
    assert df.iloc[2]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[2]['timestamp'] == pd.Timestamp('2019-10-14T03:03:05')
    assert df.iloc[2]['failurecode'] == 'PUMPS'
    assert df.iloc[2]['problemcode'] == 'PUMPS/STOPPED'
    assert df.iloc[3]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[3]['timestamp'] == pd.Timestamp('2019-10-16T01:02:08')
    #assert pd.isna(df.iloc[3]['failurecode'])
    assert df.iloc[3]['problemcode'] == 'PUMPS/STOPPED'

    # test data_items filtering

    df = api.get_asset_failure_history(assets=assets, data_items=['faildate'], df_id_column='id', df_timestamp_name='timestamp', async_mode=False)
    print(log_df_info(df, head=5))
    assert len(df) == 20 # 10 assets each with 2 failure records
    assert set(df.columns) == set(['id', 'timestamp', 'faildate'])
    assert df.iloc[0]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[0]['timestamp'] == pd.Timestamp('2019-10-14T03:03:05')
    assert df.iloc[1]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[1]['timestamp'] == pd.Timestamp('2019-10-16T01:02:08')
    assert df.iloc[2]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[2]['timestamp'] == pd.Timestamp('2019-10-14T03:03:05')
    assert df.iloc[3]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[3]['timestamp'] == pd.Timestamp('2019-10-16T01:02:08')

    df = api.get_asset_failure_history(assets=assets, data_items=['faildate', 'problemcode'], df_id_column='id', df_timestamp_name='timestamp', async_mode=False)
    print(log_df_info(df, head=5))
    assert len(df) == 20 # 10 assets each with 2 failure records
    assert set(df.columns) == set(['id', 'timestamp', 'faildate', 'problemcode'])
    assert df.iloc[0]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[0]['timestamp'] == pd.Timestamp('2019-10-14T03:03:05')
    assert df.iloc[0]['problemcode'] == 'PUMPS/STOPPED'
    assert df.iloc[1]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[1]['timestamp'] == pd.Timestamp('2019-10-16T01:02:08')
    assert df.iloc[1]['problemcode'] == 'PUMPS/STOPPED'
    assert df.iloc[2]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[2]['timestamp'] == pd.Timestamp('2019-10-14T03:03:05')
    assert df.iloc[2]['problemcode'] == 'PUMPS/STOPPED'
    assert df.iloc[3]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[3]['timestamp'] == pd.Timestamp('2019-10-16T01:02:08')
    assert df.iloc[3]['problemcode'] == 'PUMPS/STOPPED'

    df = api.get_asset_failure_history(assets=assets, data_items=['failurecode', 'problemcode'], df_id_column='id', df_timestamp_name='timestamp', async_mode=False)
    print(log_df_info(df, head=5))
    assert len(df) == 20 # 10 assets each with 2 failure records
    assert all([col in df.columns for col in ['id', 'timestamp', 'failurecode', 'problemcode']])
    assert df.iloc[0]['id'] == 'INTGTST0000-____-BEDFORD'
    assert df.iloc[0]['failurecode'] == 'PUMPS'
    assert df.iloc[0]['problemcode'] == 'PUMPS/STOPPED'
    assert df.iloc[1]['id'] == 'INTGTST0000-____-BEDFORD'
    #assert pd.isna(df.iloc[1]['failurecode'])
    assert df.iloc[1]['problemcode'] == 'PUMPS/STOPPED'
    assert df.iloc[2]['id'] == 'INTGTST0001-____-BEDFORD'
    assert df.iloc[2]['failurecode'] == 'PUMPS'
    assert df.iloc[2]['problemcode'] == 'PUMPS/STOPPED'
    assert df.iloc[3]['id'] == 'INTGTST0001-____-BEDFORD'
    #assert pd.isna(df.iloc[3]['failurecode'])
    assert df.iloc[3]['problemcode'] == 'PUMPS/STOPPED'

    # test parameter validation

    with pytest.raises(ValueError):
        api.get_asset_failure_history(assets=None, data_items=None, df_id_column='id', df_timestamp_name='timestamp')
    with pytest.raises(ValueError):
        api.get_asset_failure_history(assets=[], data_items=None, df_id_column='id', df_timestamp_name='timestamp')
    with pytest.raises(ValueError):
        api.get_asset_failure_history(assets=[{}, {}], data_items=None, df_id_column='id', df_timestamp_name='timestamp')
    with pytest.raises(ValueError):
        api.get_asset_failure_history(assets=[{'assetNum': 'INTGTST0000'}], data_items=None, df_id_column='id', df_timestamp_name='timestamp')
    with pytest.raises(ValueError):
        api.get_asset_failure_history(assets=[{'assetnum': 'INTGTST0000', 'siteId': 'BEDFORD'}], data_items=None, df_id_column='id', df_timestamp_name='timestamp')
    with pytest.raises(ValueError):
        api.get_asset_failure_history(assets=[{'assetNum': 'INTGTST0000', 'siteid': 'BEDFORD'}], data_items=None, df_id_column='id', df_timestamp_name='timestamp')
    with pytest.raises(ValueError):
        api.get_asset_failure_history(assets=assets, data_items=None, df_id_column=None, df_timestamp_name='timestamp')
    with pytest.raises(ValueError):
        api.get_asset_failure_history(assets=assets, data_items=None, df_id_column='', df_timestamp_name='timestamp')
    with pytest.raises(ValueError):
        api.get_asset_failure_history(assets=assets, data_items=None, df_id_column='id', df_timestamp_name=None)
    with pytest.raises(ValueError):
        api.get_asset_failure_history(assets=assets, data_items=None, df_id_column='id', df_timestamp_name='')


def test_maximo_api_context(mocker):
    # mock the underlying API call for api.get_asset_attributes, using pytest_mock
    mocker.patch.object(api, '_call', autospec=True, return_value={})

    maximo_baseurl = 'https://52.116.33.175:9443/maximo'
    url_path = '/path'
    method = 'get'
    apm_api_key = 'abcd'
    headers = {'apikey': apm_api_key}

    mocker.patch.object(api, 'is_local_mode', return_value=False)
    mocker.patch.object(api, '_set_apm_info', return_value=None)
    mocker.patch.dict(os.environ, {
        'APM_ID': 'apm_id',
        'APM_API_BASEURL': 'apm_api_baseurl',
        'APM_API_KEY': 'apm_api_key',
    })

    tenant_info_resp = mocker.MagicMock()
    tenant_info_resp.text = json.dumps({
        "mahi_auth_url": "https://52.116.33.175/maximo/api/permission/allowedappoptions?app=RELENGINEER&lean=1",
        "mahi_url": maximo_baseurl,
        "info": {
        }
    })
    mocker.patch.object(api, '_get_tenant', return_value=tenant_info_resp)
    mocker.patch.object(api, 'INIT_DONE', new=False)

    # test init_environ correctly handle maximo api context
    assert ('apm_id', 'apm_api_baseurl', 'apm_api_key') == api.init_environ()
    assert maximo_baseurl == api.MAXIMO_BASEURL
    assert '/api' == api.MAXIMO_API_CONTEXT

    tenant_info_resp = mocker.MagicMock()
    tenant_info_resp.text = json.dumps({
        "mahi_auth_url": "https://52.116.33.175:9443/maximo/api/permission/allowedappoptions?app=RELENGINEER&lean=1",
        "mahi_url": maximo_baseurl,
        "info": {
        }
    })
    mocker.patch.object(api, '_get_tenant', return_value=tenant_info_resp)
    mocker.patch.object(api, 'INIT_DONE', new=False)

    # test init_environ correctly handle maximo api context
    assert ('apm_id', 'apm_api_baseurl', 'apm_api_key') == api.init_environ()
    assert maximo_baseurl == api.MAXIMO_BASEURL
    assert '/api' == api.MAXIMO_API_CONTEXT

    tenant_info_resp = mocker.MagicMock()
    tenant_info_resp.text = json.dumps({
        "mahi_auth_url": "https://52.116.33.175:9443/maximo/oslc/permission/allowedappoptions?app=RELENGINEER&lean=1",
        "mahi_url": maximo_baseurl,
        # "mahi_api_context": new_maximo_api_context,
        "info": {
        }
    })
    mocker.patch.object(api, '_get_tenant', return_value=tenant_info_resp)
    mocker.patch.object(api, 'INIT_DONE', new=False)

    # test init_environ correctly handle maximo api context
    assert ('apm_id', 'apm_api_baseurl', 'apm_api_key') == api.init_environ()
    assert maximo_baseurl == api.MAXIMO_BASEURL
    assert '/oslc' == api.MAXIMO_API_CONTEXT

    default_maximo_api_context = '/api'
    new_maximo_api_context = '/oslc'

    # white box testing, to pass some function level validation
    mocker.patch.object(api, 'APM_API_KEY', new=apm_api_key)
    mocker.patch.object(api, 'MAXIMO_BASEURL', new=maximo_baseurl)
    mocker.patch.object(api, 'MAXIMO_API_CONTEXT', new=default_maximo_api_context)

    # test _call_maximo composed url is correct
    api._call.assert_not_called
    assert {} == api._call_maximo(url_path=url_path, method=method, headers=headers)
    api._call.assert_called_with(url=maximo_baseurl + default_maximo_api_context + url_path, method=method, json=None, headers=headers, timeout=300, ssl_verify=False)

    # test _call_maximo composed url is correct, when maximo api context is given
    mocker.patch.object(api, 'MAXIMO_API_CONTEXT', new=new_maximo_api_context)

    assert {} == api._call_maximo(url_path=url_path, method=method, headers=headers)
    api._call.assert_called_with(url=maximo_baseurl + new_maximo_api_context + url_path, method=method, json=None, headers=headers, timeout=300, ssl_verify=False)


def test_ssl_verify_switch(mocker):
    # mock the underlying API call for api.get_asset_attributes, using pytest_mock
    mocker.patch.object(api, '_call', autospec=True, return_value={})

    json = None

    maximo_baseurl = 'http://maximo'
    maximo_api_context = '/api'
    url_path = '/path'
    method = 'get'
    apm_api_key = 'abcd'
    headers = {'apikey': apm_api_key}

    # white box testing, to pass some function level validation
    mocker.patch.object(api, 'APM_API_KEY', new=apm_api_key)
    mocker.patch.object(api, 'MAXIMO_BASEURL', new=maximo_baseurl)
    mocker.patch.object(api, 'MAXIMO_API_CONTEXT', new=maximo_api_context)

    # test by default, ssl_verify should be False for calling Maximo APIs
    api._call.assert_not_called
    assert {} == api._call_maximo(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called
    api._call.assert_called_with(url=maximo_baseurl + maximo_api_context + url_path, method=method, json=json, headers=headers, timeout=300, ssl_verify=False)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_MAXIMO': ''})
    api._call_maximo(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=maximo_baseurl + maximo_api_context + url_path, method=method, json=json, headers=headers, timeout=300, ssl_verify=False)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_MAXIMO': '0'})
    api._call_maximo(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=maximo_baseurl + maximo_api_context + url_path, method=method, json=json, headers=headers, timeout=300, ssl_verify=False)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_MAXIMO': '1'})
    api._call_maximo(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=maximo_baseurl + maximo_api_context + url_path, method=method, json=json, headers=headers, timeout=300, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_MAXIMO': 'true'})
    api._call_maximo(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=maximo_baseurl + maximo_api_context + url_path, method=method, json=json, headers=headers, timeout=300, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_MAXIMO': 'True'})
    api._call_maximo(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=maximo_baseurl + maximo_api_context + url_path, method=method, json=json, headers=headers, timeout=300, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_MAXIMO': 'YES'})
    api._call_maximo(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=maximo_baseurl + maximo_api_context + url_path, method=method, json=json, headers=headers, timeout=300, ssl_verify=True)

    del os.environ['SSL_VERIFY_MAXIMO']
    api._call_maximo(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=maximo_baseurl + maximo_api_context + url_path, method=method, json=json, headers=headers, timeout=300, ssl_verify=False)

    apm_id = 'abcd'
    apm_baseurl = 'http://apm'
    apm_api_key = 'abcd'
    headers = {'apmapitoken': apm_api_key}

    mocker.patch.object(api, 'APM_ID', new=apm_id)
    mocker.patch.object(api, 'APM_API_BASEURL', new=apm_baseurl)
    mocker.patch.object(api, 'APM_API_KEY', new=apm_api_key)

    assert {} == api._call_apm(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=api._normalize_apm_api_baseurl() + url_path, method=method, json=json, headers=headers, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_APM': 'YES'})
    assert {} == api._call_apm(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=api._normalize_apm_api_baseurl() + url_path, method=method, json=json, headers=headers, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_APM': 'truE'})
    assert {} == api._call_apm(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=api._normalize_apm_api_baseurl() + url_path, method=method, json=json, headers=headers, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_APM': '1'})
    assert {} == api._call_apm(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=api._normalize_apm_api_baseurl() + url_path, method=method, json=json, headers=headers, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_APM': '0'})
    assert {} == api._call_apm(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=api._normalize_apm_api_baseurl() + url_path, method=method, json=json, headers=headers, ssl_verify=False)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_APM': 'False'})
    assert {} == api._call_apm(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=api._normalize_apm_api_baseurl() + url_path, method=method, json=json, headers=headers, ssl_verify=False)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_APM': 'no'})
    assert {} == api._call_apm(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=api._normalize_apm_api_baseurl() + url_path, method=method, json=json, headers=headers, ssl_verify=False)

    del os.environ['SSL_VERIFY_APM']
    assert {} == api._call_apm(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url=api._normalize_apm_api_baseurl() + url_path, method=method, json=json, headers=headers, ssl_verify=True)

    org_id = 'defg'
    api_key = '1234-%s-xyz' % org_id
    api_token = 'abcd'
    headers = {}

    mocker.patch.object(api, 'API_KEY', new=api_key)
    mocker.patch.object(api, 'API_TOKEN', new=api_token)

    assert {} == api._call_wiotp(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url='https://%s.internetofthings.ibmcloud.com/api/v0002%s' % (org_id, url_path), method=method, json=json, headers=headers, auth=(api_key, api_token), ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_WIOTP': 'YES'})
    assert {} == api._call_wiotp(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url='https://%s.internetofthings.ibmcloud.com/api/v0002%s' % (org_id, url_path), method=method, json=json, headers=headers, auth=(api_key, api_token), ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_WIOTP': 'True'})
    assert {} == api._call_wiotp(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url='https://%s.internetofthings.ibmcloud.com/api/v0002%s' % (org_id, url_path), method=method, json=json, headers=headers, auth=(api_key, api_token), ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_WIOTP': '1'})
    assert {} == api._call_wiotp(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url='https://%s.internetofthings.ibmcloud.com/api/v0002%s' % (org_id, url_path), method=method, json=json, headers=headers, auth=(api_key, api_token), ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_WIOTP': '0'})
    assert {} == api._call_wiotp(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url='https://%s.internetofthings.ibmcloud.com/api/v0002%s' % (org_id, url_path), method=method, json=json, headers=headers, auth=(api_key, api_token), ssl_verify=False)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_WIOTP': 'no'})
    assert {} == api._call_wiotp(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url='https://%s.internetofthings.ibmcloud.com/api/v0002%s' % (org_id, url_path), method=method, json=json, headers=headers, auth=(api_key, api_token), ssl_verify=False)

    del os.environ['SSL_VERIFY_WIOTP']
    assert {} == api._call_wiotp(url_path=url_path, method=method, json=json, headers=headers)
    api._call.assert_called_with(url='https://%s.internetofthings.ibmcloud.com/api/v0002%s' % (org_id, url_path), method=method, json=json, headers=headers, auth=(api_key, api_token), ssl_verify=True)

    api_baseurl = 'http://as'
    api_type = 'meta'
    tenant_id = 't1'
    headers = {
        'Content-Type': 'application/json',
        'X-api-key' : api_key,
        'X-api-token' : api_token,
        'Cache-Control': 'no-cache',
    }

    mocker.patch.object(api, 'API_BASEURL', new=api_baseurl)
    mocker.patch.dict(os.environ, {'TENANT_ID': tenant_id})

    assert {} == api._call_as(url_path=url_path, method=method, api_type=api_type, json=json, headers=headers)
    api._call.assert_called_with(url='%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path), method=method, json=json, headers=headers, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_AS': 'YES'})
    assert {} == api._call_as(url_path=url_path, method=method, api_type=api_type, json=json, headers=headers)
    api._call.assert_called_with(url='%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path), method=method, json=json, headers=headers, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_AS': '1'})
    assert {} == api._call_as(url_path=url_path, method=method, api_type=api_type, json=json, headers=headers)
    api._call.assert_called_with(url='%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path), method=method, json=json, headers=headers, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_AS': 'True'})
    assert {} == api._call_as(url_path=url_path, method=method, api_type=api_type, json=json, headers=headers)
    api._call.assert_called_with(url='%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path), method=method, json=json, headers=headers, ssl_verify=True)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_AS': 'False'})
    assert {} == api._call_as(url_path=url_path, method=method, api_type=api_type, json=json, headers=headers)
    api._call.assert_called_with(url='%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path), method=method, json=json, headers=headers, ssl_verify=False)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_AS': 'no'})
    assert {} == api._call_as(url_path=url_path, method=method, api_type=api_type, json=json, headers=headers)
    api._call.assert_called_with(url='%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path), method=method, json=json, headers=headers, ssl_verify=False)

    mocker.patch.dict(os.environ, {'SSL_VERIFY_AS': '0'})
    assert {} == api._call_as(url_path=url_path, method=method, api_type=api_type, json=json, headers=headers)
    api._call.assert_called_with(url='%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path), method=method, json=json, headers=headers, ssl_verify=False)

    del os.environ['SSL_VERIFY_AS']
    assert {} == api._call_as(url_path=url_path, method=method, api_type=api_type, json=json, headers=headers)
    api._call.assert_called_with(url='%s/api/%s/v1/%s%s' % (api_baseurl, api_type, tenant_id, url_path), method=method, json=json, headers=headers, ssl_verify=True)

