# Licensed Materials - Property of IBM
# 5737-M66, 5900-AAA
# (C) Copyright IBM Corp. 2019, 2025 All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication, or disclosure
# restricted by GSA ADP Schedule Contract with IBM Corp.

import logging
from typing import Dict, List

from . import api

logger = logging.getLogger(__name__)


def search_for_device_types(query: str) -> dict:
    """Search for devices types using the following string. It could be a device type name, UUID, etc.

    Args:
        query (str): query to use for search

    Raises:
        RuntimeError: if there is an error with the API request to Monitor

    Returns:
        dict: dictionary containing devices types found in Monitor
    """
    path = 'core/deviceTypes/search'
    body = {"search": query}
    response = api._call_as_v2(url_path=path, method='post', json=body)
    if response is None:
        msg = f'The API request to monitor failed when trying to search for device types using the string: {query}. Check the logs to see the potential problem.'
        logger.warning(msg)
        raise RuntimeError(msg)

    return response.json()['results']


def get_device_type_uuid_from_name(device_type_name: str) -> str:
    """Returns the device type UUID given it's name or None if the device type cannot be found.

    Args:
        device_type_name (str): device type name to get UUID for

    Returns:
        str: UUID of device type
    """
    results = search_for_device_types(device_type_name)
    for result in results:
        if result['name'] == device_type_name:
            return result['uuid']
    return None


def get_granularities_for_device_uuid(device_type_uuid: str) -> List[Dict]:
    """Returns the existing granularities for the device type given its UUID.

    Args:
        device_type_uuid (str): device type to get granularities for

    Raises:
        RuntimeError: if the API request to Monitor fails

    Returns:
        List[Dict]: list of dictionaries containing granularities
    """
    path = f'core/deviceTypes/{device_type_uuid}/granularity'
    response = api._call_as_v2(url_path=path, method='get')
    if response is None:
        msg = f'The API request to monitor failed when trying to get the granularities for device type with UUID: {device_type_uuid}. Check the logs to see the potential problem.'
        logger.warning(msg)
        raise RuntimeError(msg)
    elif response.status_code == 204:
        # HTTP 204 = No content, meaning no granularities
        return []

    return response.json()


def add_missing_granularities(device_type_name: str, local_granularities_list: List[Dict]):
    """Adds the missing granularities to Monitor for the device type given a list of local granularities.

    Args:
        device_type_name (str): name of device type to add granularities for
        local_granularities_list (List[Dict]): list of granularity dictionaries to add
    """
    device_type_uuid = get_device_type_uuid_from_name(device_type_name)

    existing_granularities = get_granularities_for_device_uuid(
        device_type_uuid)

    local_granularities = {
        granularity['name']: granularity for granularity in local_granularities_list}
    remote_granularities = {
        granularity['name']: granularity for granularity in existing_granularities}

    # update local grains with contents from remote grains
    for remote_name, remote_granularity in remote_granularities.items():
        if remote_name in local_granularities:
            local_granularities[remote_name].update(remote_granularity)
        else:
            local_granularities[remote_name] = remote_granularity

    for local_name, local_granularity in local_granularities.items():
        if local_name not in remote_granularities:
            path = f'core/deviceTypes/{device_type_uuid}/granularity'
            resp = api._call_as_v2(
                url_path=path, method='post', json=local_granularity)
            if resp is not None:
                local_granularities[local_name].update(resp.json())

    return local_granularities
