#!/usr/bin/perl
# IBM_PROLOG_BEGIN_TAG
# This is an automatically generated prolog.
#
# Contributors Listed Below - COPYRIGHT 2015
# [+] International Business Machines Corp.
#
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied. See the License for the specific language governing
# permissions and limitations under the License.
#
# IBM_PROLOG_END_TAG

use strict;
use Getopt::Std;

my $VERSION = "3.0";
my ($BMC, $ADMIN_PW, $SYSADMIN_PW, $HOST, $USER, $PASSWORD, $VERBOSE, $PNOR) = "";

my %options=();
print "Power LC Data Collection Script Version $VERSION\n\n";
getopts("a:b:s:h:u:p:vif", \%options);
if ( $options{i} ) {
	interactive();
} else {
	unless ( defined $options{b} ) {
		print "ERROR: no BMC host specified with -b flag\n";
		help();
		exit 1;
	} else {
		$SYSADMIN_PW = ( $options{s} ? $options{s} : "superuser" );
		$ADMIN_PW = ( $options{a} ? $options{a} : "admin" );
		$HOST = $options{h};
		$USER = $options{u};
		$PASSWORD = $options{p};
		$BMC = $options{b};
		$VERBOSE = $options{v};
		$PNOR = $options{f};
	}
}

my $OUTFILE = $BMC . "-" . `date "+%F.%H%M"`;
chop $OUTFILE;
my $ESELFILE 		= $OUTFILE . ".eSel";
my $BMC_OUTFILE 	= $OUTFILE . ".bmc.txt";
my $FILES_OUTFILE	= $OUTFILE . ".files.txt";
my $LED_OUTFILE 	= $OUTFILE . ".led.txt";
my $STATUS_OUTFILE	= $OUTFILE . ".status.txt";
my $HOST_OUTFILE	= $OUTFILE . ".host.txt";
my $IPMI_OUTFILE	= $OUTFILE . ".IPMI.txt";
my $PARTITION_OUTFILE	= $OUTFILE . ".partition.txt";
my $PNOR_OUTFILE	= $OUTFILE . ".pnor.img";

my $CHUNK = 254.0;

my %rawSize = ( 
	"gard",  "raw 0x3a 0x0c 0x47 0x55 0x41 0x52 0x44 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
	"hbel",  "raw 0x3a 0x0c 0x48 0x42 0x45 0x4c 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
	"fir",   "raw 0x3a 0x0c 0x46 0x49 0x52 0x44 0x41 0x54 0x41 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
	"attrp", "raw 0x3a 0x0c 0x41 0x54 0x54 0x52 0x5f 0x50 0x45 0x52 0x4d 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
	"attrt", "raw 0x3a 0x0c 0x41 0x54 0x54 0x52 0x5f 0x54 0x4d 0x50 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
	"ver",   "raw 0x3a 0x0c 0x56 0x45 0x52 0x53 0x49 0x4f 0x4e 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
);

my %rawData = ( 
	"gard",  "raw 0x3a 0x0b 0x47 0x55 0x41 0x52 0x44 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
	"hbel",  "raw 0x3a 0x0b 0x48 0x42 0x45 0x4c 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
	"fir",   "raw 0x3a 0x0b 0x46 0x49 0x52 0x44 0x41 0x54 0x41 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
	"attrp", "raw 0x3a 0x0b 0x41 0x54 0x54 0x52 0x5f 0x50 0x45 0x52 0x4d 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
	"attrt", "raw 0x3a 0x0b 0x41 0x54 0x54 0x52 0x5f 0x54 0x4d 0x50 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
	"ver",   "raw 0x3a 0x0b 0x56 0x45 0x52 0x53 0x49 0x4f 0x4e 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
);


prereqs();
pingTest($BMC);
my $POWER_STATE = `ipmitool -I lanplus -H $BMC -U ADMIN -P $ADMIN_PW chassis power status 2>/dev/null`;
get_bmclogs();
get_esels();
get_led();
get_ipmi();
if ( $POWER_STATE =~ /is off/ ) {
	get_partition();
}
if ( $HOST ) {
	get_host_data();
}
if ( $PNOR ) {
	get_fw_image();
}
package_itup();
print "\nComplete\n";

sub get_bmclogs {
	my @output;
	my $CMD;
	
	print "Output File: $OUTFILE.powerlc.tar.gz\n";
	logger("Getting BMC logs from $BMC\n");
	printf "\nGetting BMC data\n";
	my @commands = 	("cat /proc/ractrends/Helper/FwInfo",
			"date",
			"uptime",
			"cat /proc/cmdline",
			"cat /proc/mtd",
			"ps -ax",
			"ps -eT",
			"free",
			"cat /proc/meminfo",
			"lsmod",
			"ls -l /proc/*/fd/",
			"df -h",
			"ls -laR /conf",
			"ls -laR /bkupconf",
			"ls -laR /extlog",
			"cat /conf/ncml.conf",
			"ls -la /var/",
			"ls -la /var/log",
			"cat /var/log/alert.log",
			"cat /var/log/crit.log",
			"cat /var/log/debug.log",
			"cat /var/log/emerg.log",
			"cat /var/log/info.log",
			"cat /var/log/warning.log",
			"cat /extlog/sollog/SOLHostCapture.log",
			"cat /extlog/sollog/SOLHostCapture.log.1",
			);
	foreach (@commands) {
		system "echo '********************' Collecting $_ '********************' >> $BMC_OUTFILE";
		$CMD = "sshpass -p $SYSADMIN_PW ssh -k -o StrictHostKeyChecking=no sysadmin@". $BMC ." '$_ 2>/dev/null'>> $BMC_OUTFILE";
		`$CMD`;
		if ( $? == 111111 ) {
			print "\n ERROR: Failure accessing BMC: Return Code $?\n";
			die;
		}
		printf ".";
		if ( $VERBOSE ) { print "$_\n"; }
	}
	get_ipmi();
	$CMD = "sshpass -p $SYSADMIN_PW ssh -k -o StrictHostKeyChecking=no sysadmin@". $BMC ." 'find /extlog -type f'";
	@output = `$CMD`;
	foreach ( @output ) {
		chop;
		system "echo '********************' Collecting $_ '********************' >> $FILES_OUTFILE";
		$CMD = "sshpass -p $SYSADMIN_PW ssh -k -o StrictHostKeyChecking=no sysadmin@". $BMC ." 'cat $_' >> $FILES_OUTFILE";
		`$CMD`;
	}
}


sub get_esels {
	logger("Getting eSEL events\n");
	`./eSEL2.pl -P $ADMIN_PW -t $BMC -l $ESELFILE`;
}

sub get_led {
	logger("Getting LED status\n");
	`./led_status.sh $BMC $ADMIN_PW > $LED_OUTFILE`;
}

sub get_fw_image {
	my $CMD = "";

	logger("Getting PNOR Image\n");
	printf "\nGetting PNOR Firmware Image\n";
	$CMD = "sshpass -p $SYSADMIN_PW ssh -k -o StrictHostKeyChecking=no sysadmin@". $BMC ." 'dd if=/dev/mtd11 bs=64k ' > $PNOR_OUTFILE";
	system "$CMD 2>/dev/null";
}

sub get_host_data {
	my $CMD = "";
	my @commands = 	("cat /sys/firmware/opal/msglog",
			"cat /proc/cpuinfo",
			"dmesg",
			"cat /proc/cpuinfo",
			"cat /proc/meminfo",
			"ppc64_cpu --frequency"
			);			
			
	logger("Getting host data from $HOST\n");
	printf "\nGetting Host Data\n";
	my $rc = system("ping -c 1 $HOST >/dev/null");
	if ( $rc == 256 ) {
		logger("ERROR unable to ping host at $HOST\n");
		print " Unable to ping host at $HOST\n";
	} else {
		foreach my $cmd ( @commands ) {
			system "echo RUNNING: $cmd >> $HOST_OUTFILE";
			$CMD = "sshpass -p $PASSWORD ssh -k -o StrictHostKeyChecking=no $USER@". $HOST ." '$cmd'>> $HOST_OUTFILE";
			system "$CMD 2>/dev/null";
			if ( $? ) {	
				print "\n FAIL: $CMD Return Code: $?\n";
				die
			}
		}
	}
}

sub get_partition {
	my ($data, $size, $size2, $blocks, $blocks2, $offset, $offset2, $cmd) = (0);

	logger("Getting Partition Data");
	printf "\nGetting Patition Data\n";
	foreach $data (keys %rawSize) {
		printf ".";
		if ( $VERBOSE ) { print "Partition: $data\n"; }
		$cmd = "ipmitool -I lanplus -H $BMC -U ADMIN -P $ADMIN_PW " . $rawSize{$data};
		system "echo Command: $data $cmd >> $PARTITION_OUTFILE";
		$size = `$cmd`;
		chop $size;
		system "echo Size: $size >> $PARTITION_OUTFILE" ;
		$size2 = "0x" . join '', reverse split /(\w\w)/, $size;
		$size2 =~ s/ //g;
		$blocks = int(hex($size2) / $CHUNK );
		$blocks2 = $blocks - 1;
		system "echo Partition Size: $size Blocks: $blocks of $CHUNK bytes >> $PARTITION_OUTFILE";
		for ( my $count = 0; $count < $blocks; $count++) {
			$offset = sprintf "%08x", $count * 254;
			$offset =~ m/(..)(..)(..)(..)/;
			$offset2 = "0x" . $4 . " 0x" . $3 . " 0x" . $2 . " 0x" . $1;
			#printf "XXXX %s %s XX %s YY %s\n", $count, $blocks, $offset, $offset2;
			$cmd =  "ipmitool -I lanplus -H $BMC -U ADMIN -P $ADMIN_PW " . $rawData{$data} . " " . $offset2;
			system  "echo Getting $data Block $count of $blocks2 $cmd >> $PARTITION_OUTFILE";
			system "$cmd >> $PARTITION_OUTFILE";
		}
	}	
}

sub get_ipmi {
	my $CMD = "";
	my @commands = ( "raw 6 1",
			"sel elist",
			"sel list -v",
			"sdr -v",
			"sensor list",
			"lan print 1",
			"fru print",
			"chassis poh"
			);			
	logger("Getting IPMI Data\n");
	printf "\nGetting IPMI Data\n";
	foreach my $cmd ( @commands ) {
		printf ".";
		if ( $VERBOSE ) { print "$cmd\n"; }
		system "echo '********************' Collecting  $cmd '********************' >> $BMC_OUTFILE";
		$CMD = "ipmitool -I lanplus -H $BMC -U ADMIN -P $ADMIN_PW $cmd 2>/dev/null";
		system "$CMD >> $BMC_OUTFILE";
		if ( $? == 1111 ) {	
			print "\n ERROR: $CMD Return Code: $?\n";
			die "Script aborted: Possible incorrect ADMIN password for BMC\n";
		}
	}
}
		
	

sub package_itup {
# package everything up
	logger("\nPackaging data files to $OUTFILE.powerlc.tar\n");
	system "tar -czvf $OUTFILE.powerlc.tar.gz $OUTFILE*";
}

# check that the specified address is reachable
sub pingTest {
	(my $address) = @_;

	my $rc = system("ping -c 1 $address >/dev/null"); 
	if ( $rc == 256 ) {
		logger("ERROR unable to ping BMC at $BMC\n");
		print " Unable to ping $address\n";
		exit 2;
	}
	`ssh-keygen -R $BMC 2>/dev/null`; # remove ssh key as could be old
}

sub prereqs {
	my $date = `date`;
	logger("POWER LC data collection version $VERSION\n");
	logger("Date $date");
	my $QUIT = 0;
	unless ( `which sshpass` ) {
		logger("ERROR: shhpass not found\n");
		print " sshpass is required and not found on this system\n";
		$QUIT = 1;
	}
	unless ( `which ipmitool` ) {
		logger("ERROR: ipmitool not found\n");
		print " ipmitool is required and not found on this system\n";
		$QUIT = 1;
	}
	if ( $QUIT ) {
		print " Exiting\n";
		exit 1;
	}
	my $ipmiVersion = `ipmitool -V`;
	logger("$ipmiVersion");
}

sub logger {
	(my $msg) = @_;
	open (my $fh, '>>', $STATUS_OUTFILE);
	print $fh $msg;
	close $fh;
}

sub interactive {
	print "Interactive Mode\n";
	print "BMC address (IP or hostname)               : ";
	$BMC = <STDIN>; chop $BMC;
	print "BMC admin password (default admin)         : ";
	$ADMIN_PW = <STDIN>; chop $ADMIN_PW; unless ($ADMIN_PW) { $ADMIN_PW = "admin"; }
	print "BMC sysadmin password (default superuser)  : ";
	$SYSADMIN_PW = <STDIN>; chop $SYSADMIN_PW; unless ($SYSADMIN_PW) { $SYSADMIN_PW = "superuser"; }
	print "Host data (leave blank to skip) IP address : ";
	$HOST = <STDIN>; chop $HOST;
	if ( $HOST ) {
		print "Host username                              : ";	
		$USER = <STDIN>; chop $USER;
		print "Host password                              : ";	
		$PASSWORD = <STDIN>; chop $PASSWORD;
	}
	print "BMC $BMC PW $ADMIN_PW sysadmin $SYSADMIN_PW Host $HOST user $USER pw $PASSWORD\n";
}


sub help {
	print "POWER LC BMC Data Collection version $VERSION\n\n";
	print "usage: $0 { -b bmc_address | -i } [-a admin_pw] [-s sysadmin_pw] [-h host -u user -p password]\n";
	print "Flags:\n";
	print "   -b BMC hostname or IP address\n";
	print "   -a BMC ADMIN password if changed from default (admin)\n";
	print "   -s BMC sysadmin password if changed from default (superuser)\n";
	print "   -f Collect PNOR Firmware Image\n";
	print "   -h Linux host address\n";
	print "   -u Linux host user ID\n";
	print "   -p Linux host password\n";
	print "   -i Interactive mode\n";
}
	
