#!/usr/bin/perl
# IBM_PROLOG_BEGIN_TAG
# This is an automatically generated prolog.
#
# Contributors Listed Below - COPYRIGHT 2015
# [+] International Business Machines Corp.
#
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied. See the License for the specific language governing
# permissions and limitations under the License.
#
# IBM_PROLOG_END_TAG

use strict;
use Getopt::Std;
use Cwd qw(abs_path getcwd);
use File::Basename;

my $VERSION = "1.02";
my ($BMC, $ADMINID, $ADMIN_PW, $HOST, $USER, $PASSWORD, $VERBOSE, $OUTDIR, $vflag) = "";

my %options=();
getopts("n:a:b:h:u:p:d:vi", \%options);
if ( $options{i} ) {
    interactive();
    $VERBOSE = $options{v};
} else {
    unless ( defined $options{b} ) {
        print "ERROR: no BMC host specified with -b flag\n";
        help();
        exit 1;
    } else {
        $ADMINID = ( $options{n} ? $options{n} : "ADMIN" );
        $ADMIN_PW = ( $options{a} ? $options{a} : "ADMIN" );
        $HOST = $options{h};
        $USER = $options{u};
        $PASSWORD = $options{p};
        $BMC = $options{b};
        $VERBOSE = $options{v};
        $OUTDIR = $options{d};
        unless ($OUTDIR) { $OUTDIR = "."; }

    }
}
if( $VERBOSE ) { $vflag = "--v"; }
#logger( "Verbose is \"$VERBOSE\" and flag is \"$vflag\"\n");

my $OUTFILE = $BMC . "-" . `date "+%F.%H%M"`;
chop $OUTFILE;
my $ESELFILE            = $OUTFILE . ".eSel";
my $BMC_PLN_LOG_OUTFILE = $OUTFILE . ".bmc_log.tar.gz";     # un-encrypted file
my $BMC_ENC_LOG_OUTFILE = $OUTFILE . ".bmc_log.bin";        # encrypted file
my $BMC_LOG_OUTFILE     = "";                               # one of the above two
my $BMC_OUTFILE         = $OUTFILE . ".bmc.txt";
my $FILES_OUTFILE       = $OUTFILE . ".files.txt";
my $LED_OUTFILE         = $OUTFILE . ".led.txt";
my $STATUS_OUTFILE      = $OUTFILE . ".status.txt";
my $HOST_OUTFILE        = $OUTFILE . ".host.txt";
my $IPMI_OUTFILE        = $OUTFILE . ".IPMI.txt";
my $PARTITION_OUTFILE   = $OUTFILE . ".partition.txt";
my $ARCH                = -`arch`;
chomp($ARCH);
if ($ARCH =~ /i686/) { $ARCH = "-x86_64"; }
#unless (!($ARCH =~ /i686/)) { $ARCH = ""; }

my $CHUNK = 254.0;
my $processor = "unknown";
# 20171115 DJM Find where script home is located so we can find the rest of our functions
my $cwd = dirname(abs_path($0));
unless ($cwd =~ /\/$/) { $cwd = $cwd . "/"; }
unless ($OUTDIR =~ /\/$/) { $OUTDIR = $OUTDIR . "/"; }
$ENV{PERL5LIB} = $cwd;
logger( "Power LC Data Collection Script Version $VERSION\n\n");
logger( "Running from $cwd\n");
logger( "Using output directory $OUTDIR\n");

my %rawSize = (
    "gard",  "raw 0x3a 0x0c 0x47 0x55 0x41 0x52 0x44 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
    "hbel",  "raw 0x3a 0x0c 0x48 0x42 0x45 0x4c 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
    "fir",   "raw 0x3a 0x0c 0x46 0x49 0x52 0x44 0x41 0x54 0x41 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
    "attrp", "raw 0x3a 0x0c 0x41 0x54 0x54 0x52 0x5f 0x50 0x45 0x52 0x4d 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
    "attrt", "raw 0x3a 0x0c 0x41 0x54 0x54 0x52 0x5f 0x54 0x4d 0x50 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
    "ver",   "raw 0x3a 0x0c 0x56 0x45 0x52 0x53 0x49 0x4f 0x4e 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
);

my %rawData = (
    "gard",  "raw 0x3a 0x0b 0x47 0x55 0x41 0x52 0x44 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
    "hbel",  "raw 0x3a 0x0b 0x48 0x42 0x45 0x4c 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
    "fir",   "raw 0x3a 0x0b 0x46 0x49 0x52 0x44 0x41 0x54 0x41 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
    "attrp", "raw 0x3a 0x0b 0x41 0x54 0x54 0x52 0x5f 0x50 0x45 0x52 0x4d 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
    "attrt", "raw 0x3a 0x0b 0x41 0x54 0x54 0x52 0x5f 0x54 0x4d 0x50 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
    "ver",   "raw 0x3a 0x0b 0x56 0x45 0x52 0x53 0x49 0x4f 0x4e 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00 0x00",
);


prereqs();
pingTest($BMC);
get_bmclogs();
get_esels();
get_led();
get_ipmi();
get_partition();

if ( $HOST ) {
    get_host_data();
}
package_itup();
logger( "\nComplete\n");

# 20180510 DJM Adding routine that logs verbose output stripping PI data (GDPR)

sub dbglog {
  my $logstr = @_;
}

sub ipmi_get_sys_guid {
    my $cmd = "ipmitool -I lanplus -H $BMC -U $ADMINID -P $ADMIN_PW raw 0x06 0x37 2>&1";
    logger("Sleeping a second after ping\n");
    sleep(1);
    if ( $VERBOSE ) { logger("running $cmd\n"); }
    my @ipmiRet = qx/$cmd/;
    if ($? != 0) {
      logger("Failed get_sys_guid rc $? with: " . join(" ",@ipmiRet) . "\nPlease check your userid and password passed for the BMC\n");
      exit 1;
    }
    logger("Get Processor type status output: " . join(" ",@ipmiRet));
    if (@ipmiRet[0] =~ /^\s*([[:xdigit:]]+)\s+([[:xdigit:]]+)\s+([[:xdigit:]]+)\s+([[:xdigit:]]+)\s+([[:xdigit:]]+)\s+([[:xdigit:]]+)/) {
        if ( (lc($1) eq "37") && (lc($2) eq "31") && (lc($3) eq "30") &&
             (lc($4) eq "31") && (lc($5) eq "4d") && (lc($6) eq "53") ) {
            $processor = "P9";
        }
        elsif ( (lc($1) eq "35") && (lc($2) eq "31") && (lc($3) eq "30") &&
                (lc($4) eq "31") && (lc($5) eq "4d") && (lc($6) eq "53") ) {
            $processor = "P8";
        }
    }
    logger("Processor : $processor\n");
#    print  "Processor : $processor\n";
}

sub get_bmclogs {
    logger("Getting BMC logs from $BMC\n");
    ipmi_get_sys_guid();
    if ($processor eq "P9") {
#        $BMC_LOG_OUTFILE = $OUTDIR . $BMC_PLN_LOG_OUTFILE;    # un-encrypted
        $BMC_LOG_OUTFILE = $BMC_PLN_LOG_OUTFILE;    # un-encrypted
    }
    else {
#        $BMC_LOG_OUTFILE = $OUTDIR . $BMC_ENC_LOG_OUTFILE;    # encrypted
        $BMC_LOG_OUTFILE = $BMC_ENC_LOG_OUTFILE;    # encrypted
    }
    `$cwd/bmclog.pl -t $BMC -U $ADMINID -P $ADMIN_PW -o $OUTDIR -l $BMC_LOG_OUTFILE $vflag`;
}


sub get_esels {
    logger("Getting eSEL events\n");
    my $cmd = "$cwd/eSEL2.pl -U $ADMINID -P $ADMIN_PW -t $BMC -l $ESELFILE -e \"$cwd/$processor$ARCH\" -f \"$cwd/$processor$ARCH\" -i \"$cwd/$processor$ARCH\" -o \"$OUTDIR\" -s \"$STATUS_OUTFILE\" $vflag";
    if($VERBOSE) { logger("Running $cmd\n"); }
    my $op = qx/$cmd/;
    logger( $op);
}

sub get_led {
    logger("Getting LED status\n");
    `$cwd/led_status.sh $BMC $ADMINID $ADMIN_PW > $OUTDIR/$LED_OUTFILE $vflag`;
}

sub get_host_data {
    my $CMD = "";
    my @commands = ("cat /sys/firmware/opal/msglog",
                    "cat /proc/cpuinfo",
                    "dmesg",
                    "cat /proc/cpuinfo",
                    "cat /proc/meminfo",
                    "ppc64_cpu --frequency"
                   );

    logger("Getting host data from $HOST\n");
#    printf "Getting Host Data from $HOST\n";
    my $rc = system("ping -c 1 $HOST >/dev/null");
    if ( $rc == 256 ) {
        logger("ERROR unable to ping host at $HOST\n");
#        print " Unable to ping host at $HOST\n";
    } else {
        foreach my $cmd ( @commands ) {
            system "echo RUNNING: $cmd >> $HOST_OUTFILE";
            $CMD = "sshpass -p $PASSWORD ssh -k -o StrictHostKeyChecking=no $USER@". $HOST ." '$cmd'>> $HOST_OUTFILE";
            system "$CMD 2>/dev/null";
            if ( $? ) {
                logger("\n FAIL: $CMD Return Code: $?\n");
                die
            }
        }
    }
    logger("\n");
}

sub get_partition {
    my ($data, $size, $size2, $blocks, $blocks2, $offset, $offset2, $cmd) = (0);

    logger("Getting Partition Data");
#    printf "Getting Partition Data\n";
    foreach $data (keys %rawSize) {
        printf ".";
        if ( $VERBOSE ) { logger("Partition: $data\n"); }
        $cmd = "ipmitool -I lanplus -H $BMC -U $ADMINID -P $ADMIN_PW " . $rawSize{$data};
        if ( $VERBOSE ) { logger("running $cmd\n"); }
        system cln_gdpr("echo Command: $data '$cmd' >> $OUTDIR/$PARTITION_OUTFILE");
        $size = `$cmd 2>&1`;
        unless (!($size =~ /rsp=/)) {
            system cln_gdpr("echo 'Response: $size' >> $OUTDIR/$PARTITION_OUTFILE");
            next;
        }
        chop $size;
        system cln_gdpr("echo Size: $size >> $OUTDIR/$PARTITION_OUTFILE");
        $size2 = "0x" . join '', reverse split /(\w\w)/, $size;
        $size2 =~ s/ //g;
        $blocks = int(hex($size2) / $CHUNK );
        $blocks2 = $blocks - 1;
        system cln_gdpr("echo Partition Size: $size Blocks: $blocks of $CHUNK bytes >> $OUTDIR/$PARTITION_OUTFILE");
        for ( my $count = 0; $count < $blocks; $count++) {
            $offset = sprintf "%08x", $count * 254;
            $offset =~ m/(..)(..)(..)(..)/;
            $offset2 = "0x" . $4 . " 0x" . $3 . " 0x" . $2 . " 0x" . $1;
            #printf "XXXX %s %s XX %s YY %s\n", $count, $blocks, $offset, $offset2;
            $cmd =  "ipmitool -I lanplus -H $BMC -U $ADMINID -P $ADMIN_PW " . $rawData{$data} . " " . $offset2;
            if ( $VERBOSE ) { logger("running $cmd\n"); }
            system  cln_gdpr("echo Getting $data Block $count of $blocks2 '$cmd' >> $OUTDIR/$PARTITION_OUTFILE");
            system "$cmd >> $OUTDIR/$PARTITION_OUTFILE";
        }
    }
    logger("\n");
}

sub get_ipmi {
    my $CMD = "";
    my @commands = ("raw 6 1",
                    "sel elist",
                    "sel list -v",
                    "sdr -v",
                    "sensor list",
                    "lan print 1",
                    "fru print",
                    "mc info",
                    "chassis poh"
                   );
    logger("Getting IPMI Data\n");
#    printf "Getting IPMI Data\n";
    foreach my $cmd ( @commands ) {
        printf ".";
        if ( $VERBOSE ) { logger("$cmd\n"); }
        system "echo '********************' Collecting  $cmd '********************' >> $OUTDIR/$BMC_OUTFILE";
        $CMD = "ipmitool -I lanplus -H $BMC -U $ADMINID -P $ADMIN_PW $cmd 2>/dev/null";
        if ( $VERBOSE ) { logger("running $cmd\n"); }
        system "$CMD >> $OUTDIR/$BMC_OUTFILE";
        if ( $? == 1111 ) {
            logger("\n ERROR: $CMD Return Code: $?\n");
            die "Script aborted: Possible incorrect ADMIN password for BMC\n";
        }
    }
    logger("\n");
}


sub package_itup {
# package everything up
    logger("Packaging data files to $OUTFILE-gdp.powerlc.tar\n");
#    print  "Packaging data files to $OUTFILE.powerlc.tar\n";
    system "tar -czvf $OUTDIR/$OUTFILE-gdp.powerlc.tar.gz $OUTDIR/$OUTFILE* 2>&1 | tee $OUTDIR" . "$STATUS_OUTFILE";
}

# check that the specified address is reachable
sub pingTest {
    (my $address) = @_;

    my $rc = system("ping -c 1 $address >/dev/null");
# 1.02 ping sometimes returns other than 256 for failure.
#    if ( $rc == 256 ) {
    if ( $rc != 0 ) {
        logger("ERROR unable to ping BMC at $BMC\n");
#        print " Unable to ping $address\n";
        exit 2;
    }
    `ssh-keygen -R $BMC 2>/dev/null`; # remove ssh key as could be old
}

sub prereqs {
    my $date = `date`;
    logger("POWER LC data collection version $VERSION\n");
    logger("Date $date");
    my $QUIT = 0;
# 20171115 DJM Removed dependency for CORAL
#    unless ( `which sshpass` ) {
#        logger("ERROR: shhpass not found\n");
#        print " sshpass is required and not found on this system\n";
#        $QUIT = 1;
#    }
    unless ( `which ipmitool` ) {
        logger("ERROR: ipmitool not found\n");
#        print " ipmitool is required and not found on this system\n";
        $QUIT = 1;
    }
    if ( $QUIT ) {
        logger(" Exiting\n");
        exit 1;
    }
}

# 20180510 DJM
# plc.pl needs a UID/PWD puller for the logging functions above
#
sub cln_gdpr {
  (my $val) = @_;
  $val =~ s/-U\s{1,}\S{1,}\s/-U ****** /g;
  $val =~ s/-P\s{1,}\S{1,}\s/-P ****** /g;
  return $val;
}

# 20180510 DJM
# Use logger for verbose printing as well as output display messages so that
# GDPR PI data can be removed (user/password fields)
#
sub logger {
    (my $msg) = @_;
    open (my $fh, '>>', $OUTDIR . $STATUS_OUTFILE);
    $msg = cln_gdpr($msg);
    print $fh $msg;
    print $msg;
    close $fh;
}

sub interactive {
    print "Interactive Mode\n";
    print "BMC address (IP or hostname)               : ";
    $BMC = <STDIN>; chop $BMC;
    print "BMC admin ID (default ADMIN)               : ";
    $ADMINID = <STDIN>; chop $ADMINID; unless ($ADMINID) { $ADMINID = "ADMIN"; }
    print "BMC admin password (default ADMIN)         : ";
    $ADMIN_PW = <STDIN>; chop $ADMIN_PW; unless ($ADMIN_PW) { $ADMIN_PW = "ADMIN"; }
    print "Output directory (default .)               : ";
    $OUTDIR = <STDIN>; chop $OUTDIR ; unless ($OUTDIR) { $OUTDIR = getcwd(); }
# 20171115 DJM Removed dependency for CORAL
#    print "Host data (leave blank to skip) IP address : ";
#    $HOST = <STDIN>; chop $HOST;
#    if ( $HOST ) {
#        print "Host username                              : ";
#        $USER = <STDIN>; chop $USER;
#        print "Host password                              : ";
#        $PASSWORD = <STDIN>; chop $PASSWORD;
#    }
#    print "BMC $BMC PW $ADMIN_PW Host $HOST user $USER pw $PASSWORD\n";
}


sub help {
    print "POWER LC BMC Data Collection version $VERSION\n\n";
    print "usage: $0 { -b bmc_address | -i } [-a admin_pw] [-h host -u user -p password]\n";
    print "Flags:\n";
    print "   -b BMC hostname or IP address\n";
    print "   -n BMC ADMIN ID if changed from default (ADMIN)\n";
    print "   -a BMC ADMIN password if changed from default (ADMIN)\n";
    print "   -h Linux host address\n";
    print "   -u Linux host user ID\n";
    print "   -p Linux host password\n";
    print "   -i Interactive mode\n";
    print "   -d Output directory\n";
}

