#!/bin/bash
#-----------------------------------------------------------------------
# (C) Copyright IBM Corporation 2019.
#
# The source code for this program is not published or otherwise
# divested of its trade secrets, irrespective of what has
# been deposited with the U.S. Copyright Office.
# ----------------------------------------------------------------------
#

versionNum=0.4.0

# Default values
jobImageName="alpine"
scriptDir=$(cd $(dirname $0) && pwd)
release="noi"
baseDir="/root/ibm-netcool-prod"
nodes=""
yamlOnly=""
namespace="default"

pvs=( cassandra
      cassandra-bak
      kafka
      zookeeper
      couchdb
      db2
      impactgui
      ncobackup
      ncoprimary
      openldap
      nciserver
      elastic
      observer
      topology-cassandra
      topology-cassandra-bak
      topology-elastic
      minio )

hybrid_pvs=(cassandra
      cassandra-bak
      kafka
      zookeeper
      couchdb
      minio )

# Default sizes
cassandraStorageSize="50Gi"
kafkaStorageSize="50Gi"
zookeeperStorageSize="5Gi"
couchDbStorageSize="20Gi"
db2StorageSize="5Gi"
ncoStorageSize="5Gi"
nciStorageSize="5Gi"
nciUIStorageSize="5Gi"
ldapStorageSize="1Gi"
elasticStorageSize="75Gi"
observerStorageSize="5Gi"
storageSizeMinio="10Gi"

version()
{
  echo "$0 v${versionNum}"
  exit 0
}

usage()
{
  echo "This script will create local Storage Classes, Persistant Volumes and Directories required by Netcool Operations Insight"
  echo "Usage: $0"
  echo "  [-d, --directory <dir>        : base directory path]"
  echo "  [-n, --nodes <node list>      : installation nodes]"
  echo "  [-r, --release <release name> : release name]"
  echo "  [-a, --namespace <name>       : release namespace]"
  echo "  [-b, --hybrid                 : hybrid deployment]"
  echo "  [-y, --yaml                   : yaml output only]"
  echo "  [-s, --silent                 : silent mode, no confirmation]"
  echo "  [-h, --help]"
  echo "  [-v, --version]"
  exit -1
}


process_options()
{
  while [ "$1" != "" ]; do
  case $1 in
    -a|--namespace)
      shift
      namespace=$1
      shift
      ;;
    -r|--release)
      shift
      release=$1
      shift
      ;;
    -d|--directory)
      shift
      baseDir=$1
      shift
      ;;
    -n|--nodes)
      shift
      nodes=$1
      shift
      ;;
    -y|--yaml)
      shift
      yamlOnly="true"
      ;;
    -s|--silent)
      shift
      silent="true"
      ;;
    -b|--hybrid)
      shift
      hybrid="true"
      ;;
    -h|--help)
      shift
      usage
      ;;
    -v|--version)
      shift
      version
      ;;
    *)
      usage
  esac
  done

  if [ -z "$nodes" ]; then
    nodes=$(kubectl get nodes --show-labels | grep node-role.kubernetes.io/worker | grep -v NotReady | cut -d" " -f 1 | tr '\n' ' ')
  fi

  if [[ -z $release || -z $baseDir ]]; then
    echo "Error: Invalid arguments"
    usage
  fi

  
}


enterYToContinue() {
  read -p "Enter y to continue: " reply
  echo
  case $reply in
    [Yy]*) 
      ;;
    *)
      exit 1
      ;; 
  esac  
}


output_yaml()
{
  local yaml=$1
  echo "$yaml"
  echo "---"
}


create_storage_class()
{
  local name=$1
  local release=$2

  yaml=$(cat <<-EOF
	kind: StorageClass
	apiVersion: storage.k8s.io/v1
	metadata:
	  name: local-storage-${name}
	  labels:
	    release: ${release}
	provisioner: kubernetes.io/no-provisioner
	volumeBindingMode: WaitForFirstConsumer
	EOF
  )

  if [ -n "$yamlOnly" ]; then
    output_yaml "$yaml"
  else
    echo "$yaml" | kubectl create -f -
  fi
}


create_storage_classes()
{
  if [ -n "$hybrid" ]; then
    local_pvs=${hybrid_pvs[@]}
  else
    local_pvs=${pvs[@]}
  fi

  # Create Storage Classes
  for name in ${local_pvs[@]}
  do
    create_storage_class ${name} ${release}
  done
}


delete_storage_class()
{
  local name=$1
  local release=$2

  yaml=$(cat <<-EOF
	kind: StorageClass
	apiVersion: storage.k8s.io/v1
	metadata:
	  name: local-storage-${name}
	  labels:
	    release: ${release}
	provisioner: kubernetes.io/no-provisioner
	volumeBindingMode: WaitForFirstConsumer
	EOF
  )

  if [ -n "$yamlOnly" ]; then
    output_yaml "$yaml"
  else
    echo "$yaml" | kubectl delete -f -
  fi
}


delete_storage_classes()
{
  if [ -n "$hybrid" ]; then
    local_pvs=${hybrid_pvs[@]}
  else
    local_pvs=${pvs[@]}
  fi

  # Delete Storage Classes
  for name in ${local_pvs[@]}
  do
    delete_storage_class ${name} ${release}
  done
}

create_local_pv()
{
  name=$1
  node=$2
  root=$3
  size=$4
  release=$5
  reclaim=$6

  yaml=$(cat <<-EOF
	apiVersion: v1
	kind: PersistentVolume
	metadata:
	  name: ${release}-local-storage-${name}-${node}
	  labels:
	    release: ${release} 
	spec:
	  capacity:
	    storage: ${size}
	  storageClassName: local-storage-${name}
	  local:
	    path: ${root}/${release}/${name}
	  nodeAffinity: 
	    required:
	      nodeSelectorTerms:
	      - matchExpressions:
	        - key: kubernetes.io/hostname
	          operator: In
	          values: ["${node}"]
	  accessModes:
	  - ReadWriteOnce
	  persistentVolumeReclaimPolicy: ${reclaim}
	EOF
  )

  if [ -n "$yamlOnly" ]; then
    output_yaml "$yaml"
  else
    echo "$yaml" | kubectl create -f -
  fi
}


delete_local_pv()
{
  name=$1
  node=$2
  root=$3
  size=$4
  release=$5
  reclaim=$6

  yaml=$(cat <<-EOF
	apiVersion: v1
	kind: PersistentVolume
	metadata:
	  name: ${release}-local-storage-${name}-${node}
	  labels:
	    release: ${release} 
	spec:
	  capacity:
	    storage: ${size}
	  storageClassName: local-storage-${name}
	  local:
	    path: ${root}/${release}/${name}
	  nodeAffinity: 
	    required:
	      nodeSelectorTerms:
	      - matchExpressions:
	        - key: kubernetes.io/hostname
	          operator: In
	          values: ["${node}"]
	  accessModes:
	  - ReadWriteOnce
	  persistentVolumeReclaimPolicy: ${reclaim}
	EOF
  )

  if [ -n "$yamlOnly" ]; then
    output_yaml "$yaml"
  else
    echo "$yaml" | kubectl delete -f -
  fi
}


create_persistent_volumes()
{
  mkdirCmd="mkdir -p "
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/cassandra"
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/cassandra-bak"
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/kafka"
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/zookeeper"
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/couchdb"
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/elastic"
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/observer"
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/topology-cassandra"
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/topology-cassandra-bak"
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/topology-elastic"
  mkdirCmd="${mkdirCmd} ${baseDir}/${release}/minio"
  if [ -z "$hybrid" ]; then
    mkdirCmd="${mkdirCmd} ${baseDir}/${release}/db2"
    mkdirCmd="${mkdirCmd} ${baseDir}/${release}/impactgui"
    mkdirCmd="${mkdirCmd} ${baseDir}/${release}/ncobackup"
    mkdirCmd="${mkdirCmd} ${baseDir}/${release}/ncoprimary"
    mkdirCmd="${mkdirCmd} ${baseDir}/${release}/openldap"
    mkdirCmd="${mkdirCmd} ${baseDir}/${release}/nciserver"
  fi

  # Create Persistent Volumes
  for node in $nodes
  do
    create_local_pv cassandra ${node} ${baseDir} ${cassandraStorageSize} ${release} Retain
    create_local_pv cassandra-bak ${node} ${baseDir} ${cassandraStorageSize} ${release} Retain
    create_local_pv kafka ${node} ${baseDir} ${kafkaStorageSize} ${release} Retain
    create_local_pv zookeeper ${node} ${baseDir} ${zookeeperStorageSize} ${release} Retain
    create_local_pv couchdb ${node} ${baseDir} ${couchDbStorageSize} ${release} Retain
    create_local_pv elastic ${node} ${baseDir} ${elasticStorageSize} ${release} Retain
    create_local_pv observer ${node} ${baseDir} ${observerStorageSize} ${release} Retain
    create_local_pv topology-cassandra ${node} ${baseDir} ${cassandraStorageSize} ${release} Retain
    create_local_pv topology-cassandra-bak ${node} ${baseDir} ${cassandraStorageSize} ${release} Retain
    create_local_pv topology-elastic ${node} ${baseDir} ${elasticStorageSize} ${release} Retain
    create_local_pv minio ${node} ${baseDir} ${storageSizeMinio} ${release} Retain

    if [ -z "$hybrid" ]; then
      create_local_pv db2 ${node} ${baseDir} ${db2StorageSize} ${release} Retain
      create_local_pv impactgui ${node} ${baseDir} ${nciUIStorageSize} ${release} Retain
      create_local_pv ncobackup ${node} ${baseDir} ${ncoStorageSize} ${release} Retain
      create_local_pv ncoprimary ${node} ${baseDir} ${ncoStorageSize} ${release} Retain
      create_local_pv openldap ${node} ${baseDir} ${ldapStorageSize} ${release} Retain
      create_local_pv nciserver ${node} ${baseDir} ${nciStorageSize} ${release} Retain
    fi  
    create_pv_directories_on_node "${mkdirCmd}" "${baseDir}" $node
  done
}


create_pv_directories_on_node()
{
  pv_path_cmd=$1
  pv_root=$2
  worker_node=$3
  job_name=create-local-storage-${worker_node}

  yaml=$(cat <<-EOF
	apiVersion: batch/v1
	kind: Job
	metadata:
	  name: ${job_name}
	spec:
	  template:
	    spec:
	      serviceAccountName: default
	      nodeName: '${worker_node}'
	      restartPolicy: Never
	      volumes:
	      - name: localstorage
	        hostPath:
	          path: ${pv_root}
	      containers:
	      - name: local-storage-creator
	        command: ["sh", "-c", "${pv_path_cmd}"]
	        image: ${jobImageName}
	        imagePullPolicy: IfNotPresent
	        securityContext:
	          privileged: true
	        volumeMounts:
	        - name: localstorage
	          mountPath: ${pv_root}
	EOF
  )

  if [ -n "$yamlOnly" ]; then
    output_yaml "$yaml"
  else
    # If any job was left hanging around from before, delete it:
    if [ $(kubectl -n ${namespace} get jobs --ignore-not-found |grep -w $job_name|wc -l) -gt 0 ]; then
      kubectl delete -n ${namespace} job ${job_name}
    fi
    echo "$yaml" | kubectl create -n ${namespace} -f - 
    if [ $? -ne 0 ]; then
      echo "Failed to create directories on ${worker_node}"
      kubectl -n ${namespace} logs job/${job_name}
      kubectl -n ${namespace} delete job ${job_name}
      return;
    fi

    echo -n "waiting for job to become active"
    active=$(kubectl -n ${namespace} get jobs ${job_name} -o jsonpath='{.status.active}')
    until [ -z ${active} ]
    do
      echo -n "."
      sleep 1
      active=$(kubectl -n ${namespace} get jobs ${job_name} -o jsonpath='{.status.active}')
    done
    echo 

    pod=$(kubectl get pods --selector=job-name=${job_name} -n ${namespace} --output=jsonpath='{.items[*].metadata.name}')
    phase="$(kubectl get pod "${pod}" -n ${namespace} -o jsonpath={.status.phase})"
    echo -n "waiting for '${pod}' to complete"
    until [ "${phase}" != "Pending" ] && [ "${phase}" != "Running" ]
    do
      echo -n "."
      sleep 1
      phase="$(kubectl get pod "${pod}" -n ${namespace} -o jsonpath={.status.phase})"
    done
    echo

    kubectl -n ${namespace} logs job/${job_name}
    kubectl -n ${namespace} delete job ${job_name}
  fi
}


delete_persistent_volumes()
{

  rmrfCmd="rm -rf ${baseDir}/${release}"

  # Delete Persistent Volumes
  for node in $nodes
  do
    delete_local_pv cassandra ${node} ${baseDir} ${cassandraStorageSize} ${release} Retain
    delete_local_pv cassandra-bak ${node} ${baseDir} ${cassandraStorageSize} ${release} Retain
    delete_local_pv kafka ${node} ${baseDir} ${kafkaStorageSize} ${release} Retain
    delete_local_pv zookeeper ${node} ${baseDir} ${zookeeperStorageSize} ${release} Retain
    delete_local_pv couchdb ${node} ${baseDir} ${couchDbStorageSize} ${release} Retain
    delete_local_pv elastic ${node} ${baseDir} ${elasticStorageSize} ${release} Retain
    delete_local_pv observer ${node} ${baseDir} ${observerStorageSize} ${release} Retain
    delete_local_pv topology-cassandra ${node} ${baseDir} ${cassandraStorageSize} ${release} Retain
    delete_local_pv topology-cassandra-bak ${node} ${baseDir} ${cassandraStorageSize} ${release} Retain
    delete_local_pv topology-elastic ${node} ${baseDir} ${elasticStorageSize} ${release} Retain
    delete_local_pv minio ${node} ${baseDir} ${storageSizeMinio} ${release} Retain

    if [ -z "$hybrid" ]; then
      delete_local_pv db2 ${node} ${baseDir} ${db2StorageSize} ${release} Retain
      delete_local_pv impactgui ${node} ${baseDir} ${nciUIStorageSize} ${release} Retain
      delete_local_pv ncobackup ${node} ${baseDir} ${ncoStorageSize} ${release} Retain
      delete_local_pv ncoprimary ${node} ${baseDir} ${ncoStorageSize} ${release} Retain
      delete_local_pv openldap ${node} ${baseDir} ${ldapStorageSize} ${release} Retain
      delete_local_pv nciserver ${node} ${baseDir} ${nciStorageSize} ${release} Retain
    fi  
    delete_pv_directories_on_node "${rmrfCmd}" "${baseDir}" $node
  done

}


delete_pv_directories_on_node()
{
  pv_path_cmd=$1
  pv_root=$2
  worker_node=$3
  job_name=delete-local-storage-${worker_node}

  yaml=$(cat <<-EOF
	apiVersion: batch/v1
	kind: Job
	metadata:
	  name: ${job_name}
	spec:
	  template:
	    spec:
	      serviceAccountName: default
	      nodeName: '${worker_node}'
	      restartPolicy: Never
	      volumes:
	      - name: localstorage
	        hostPath:
	          path: ${pv_root}
	      containers:
	      - name: local-storage-deleter
	        command: ["sh", "-c", "${pv_path_cmd}"]
	        image: ${jobImageName}
	        imagePullPolicy: IfNotPresent
	        securityContext:
	          privileged: true
	        volumeMounts:
	        - name: localstorage
	          mountPath: ${pv_root}
	EOF
  )

  if [ -n "$yamlOnly" ]; then
    output_yaml "$yaml"
  else
    # If any job was left hanging around from before, delete it:
    if [ $(kubectl -n ${namespace} get jobs --ignore-not-found |grep -w $job_name|wc -l) -gt 0 ]; then
      kubectl delete -n ${namespace} job ${job_name}
    fi
    echo "$yaml" | kubectl create -n ${namespace} -f - 
    if [ $? -ne 0 ]; then
      echo "Failed to delete directories on ${worker_node}"
      kubectl -n ${namespace} logs job/${job_name}
      kubectl -n ${namespace} delete job ${job_name}
      return;
    fi

    echo -n "waiting for job to become active"
    active=$(kubectl -n ${namespace} get jobs ${job_name} -o jsonpath='{.status.active}')
    until [ -z "${active}" ]
    do
      echo -n "."
      sleep 1
      active=$(kubectl -n ${namespace} get jobs ${job_name} -o jsonpath='{.status.active}')
    done
    echo 

    pod=$(kubectl get pods --selector=job-name=${job_name} -n ${namespace} --output=jsonpath='{.items[*].metadata.name}')
    phase=$(kubectl get pod "${pod}" -n ${namespace} -o jsonpath={.status.phase})
    echo -n "waiting for '${pod}' to complete"
    until [ "${phase}" != "Pending" ] && [ "${phase}" != "Running" ]
    do
      echo -n "."
      sleep 1
      phase="$(kubectl get pod "${pod}" -n ${namespace} -o jsonpath={.status.phase})"
    done
    echo

    kubectl -n ${namespace} logs job/${job_name}
    kubectl -n ${namespace} delete job ${job_name}
  fi
}

delete_pvc()
{
  pvcs=( RELEASE-db2ese-pvc
    RELEASE-impactgui-pvc
    RELEASE-ncobackup-pvc
    RELEASE-ncoprimary-pvc
    RELEASE-openldap-pvc
    back-RELEASE-cassandra-0
    data-RELEASE-cassandra-0
    data-RELEASE-couchdb-0
    data-RELEASE-elasticsearch-0
    data-RELEASE-kafka-0
    data-RELEASE-zookeeper-0
    space-RELEASE-nciserver-0
    back-RELEASE-topology-cassandra-0
    data-RELEASE-topology-cassandra-0
    data-RELEASE-topology-elasticsearch-0
    data-RELEASE-topology-file-observer
    data-RELEASE-minio
    )

  hybrid_pvcs=( back-RELEASE-cassandra-0 
    data-RELEASE-cassandra-0
    data-RELEASE-couchdb-0
    data-RELEASE-kafka-0
    data-RELEASE-zookeeper-0
    )

  if [ -z "$hybrid" ]; then 
    local_pvcs=${hybrid_pvcs[@]}
  else
    local_pvcs=${pvcs[@]}
  fi

  for pvc in ${local_pvcs[@]}
  do
    pvc="$(echo "${pvc}" | sed s/RELEASE/"${release}"/)"
    kubectl delete pvc ${pvc} -n ${namespace}
    if [[ ${pvc} =~ [0-9]$ ]]; then
      for i in {1..10}
      do
        pvc=$(echo "$pvc" | sed "s/[0-9]$/$i/")
        if ! kubectl delete pvc "${pvc}" -n ${namespace} ; then
          break;
        fi
      done
    fi
  done  
}

create_delete_script()
{
  if [ -f $(dirname $0)/deleteStorageAllNodes.sh ]; then
    mv $(dirname $0)/deleteStorageAllNodes.sh $(dirname $0)/deleteStorageAllNodes.sh_$(date +%s)
  fi
  echo
  echo "Creating the delete script (deleteStorageAllNodes.sh) for the resources created by this script."
  echo "It will be found at  $scriptDir/deleteStorageAllNodes.sh"
  echo
	cat <<- EOF > $(dirname $0)/deleteStorageAllNodes.sh
	#!/bin/bash
	#-----------------------------------------------------------------------
	# (C) Copyright IBM Corporation 2019.
	#
	# The source code for this program is not published or otherwise
	# divested of its trade secrets, irrespective of what has
	# been deposited with the U.S. Copyright Office.
	# ----------------------------------------------------------------------
	source $scriptDir/$(basename $0) --source-only
	usage()
	{
	  echo "This script will delete PVCs, PVs, Storage Classes and directories created by ./createStorageAllNodes.sh"
	  echo "Usage:  \$0"
	  echo "  [-n, --namespace <name> : release namespace]"
	  echo "  [-y, --yaml             : yaml output only]"
	  echo "  [-s, --silent           : silent mode, no confirmation]"
	  echo "  [-b, --hybrid           : hybrid deployment]"
	  echo "  [-h, --help]"
	  echo "  [-v, --version]"

	  exit  -1
	}
	process_options()
	{
	  while [ "\$1" != "" ]; do
	  case \$1 in
	    -n|--namespace)
	      shift
	      namespace=\$1
	      shift
	      ;;
	    -y|--yaml)
	      shift
	      yamlOnly="true"
	      ;;
	    -s|--silent)
	      shift
	      silent="true"
	      ;;
	    -b|--hybrid)
	      shift
	      hybrid="true"
	      ;;
	    -h|--help)
	      shift
	      help
	      ;;
	    -v|--version)
	      shift
	      version
	      ;;
	    *)
	      usage
	  esac
	  done
	}

	process_options "\$@"

	if [ -z "\$silent" ] && [ -z "\$yamlOnly" ]; then
	  echo "Warning! This script will delete storage for release '${release}' of Netcool Operations Insight. Ensure that you have removed your release i.e. 'helm delete --purge ${release}'"
	  enterYToContinue
	fi

	release="$release"
	nodes="$nodes"
	baseDir="$baseDir"

	if [ -z "\$namespace" ]; then
	  namespace="default"
	fi

	delete_pvc
	delete_storage_classes
	delete_persistent_volumes

	EOF
  chmod 744 $(dirname $0)/deleteStorageAllNodes.sh
}

main()
{
  process_options "$@"
  if [ -z "$silent" ] && [ -z "$yamlOnly" ]; then
    echo "This script will create local Storage Classes, Persistant Volumes and directories required by Netcool Operations Insight"
    enterYToContinue
  fi

  create_storage_classes
  create_persistent_volumes
  
  if [ -z "$yamlOnly" ]; then
    create_delete_script
  fi
}

if [ "$1" = "--source-only" ]; then
  return
else
  main "$@"
fi


