#!/bin/sh
#######################################################################################
# FILENAME: postCloneApp.sh
# Copyright IBM Corp. 2014
# All Rights Reserved
#
# SYNOPSIS: virtual system pattern script for the Oracle EBS 12.2 App tier (RHEL 6.4/x86)
#
## TBD: update desc
#           The script accomplishes following tasks:
#           - connect to DB tier and checks if DB services are up
#           - if DB tier services are up:
#              o set the environment from supplied parameters or default values before 
#                calling post cloning produre
#              o execute the config script for DB tier as user $EBSVA_ORACLE_APP_USER
#
# USAGE: postCloneApp.sh -b APPS_BASE -h DB_HOSTNAME -d DB_DOMAIN -i DB_IP_ADDR 
#                       -w WEBLOGICADMIN_PASSWORD [-P DB_PORT] [-s ORACLE_SID]
#                       [-u ORACLE_DB_USER] [-a APPS_USER] [-p APPS_PASSWORD] [-t TIMEOUT_MINS]
# REQUIRED PARAMS: -b APPS_BASE,-h DB_HOSTNAME, -d DB_DOMAIN -i DB_IP_ADDR
# EXAMPLES: postCloneApp.sh -b /u01/app/oracle/VIS -h icc-150 -i 1.2.3.4 -d usca.ibm.com -w WLAPassw0rd
#           postCloneApp.sh -b /u01/app/oracle/VIS -h icc-150 -i 1.2.3.4 -d usca.ibm.com -w WLAPassw0rd
#                           -P 1521 -s VIS -u oracle -a applmgr -p apps -t 60
#
# VERSION:
# AUTHOR: Vishal Agarwal (v.agarwal@in.ibm.com)
# QUESTIONS/COMMENTS: ibmoracle@us.ibm.com
#
# DISCLAIMER:
#  THIS INFORMATION  CONTAINS SAMPLE CODE  PROVIDED  IN SOURCE CODE FORM. YOU MAY COPY,
#  MODIFY, AND DISTRIBUTE  THESE  SAMPLE  PROGRAMS IN  ANY  FORM WITHOUT PAYMENT TO IBM
#  FOR  THE  PURPOSES  OF  DEVELOPING,  USING,  MARKETING OR  DISTRIBUTING  APPLICATION
#  PROGRAMS  CONFORMING  TO  THE  APPLICATION  PROGRAMMING  INTERFACE FOR THE OPERATING
#  PLATFORM  FOR  WHICH  THE  SAMPLE  CODE  IS WRITTEN. NOTWITHSTANDING ANYTHING TO THE
#  CONTRARY,  IBM PROVIDES THE SAMPLE SOURCE CODE ON AN "AS IS" BASIS AND IBM DISCLAIMS
#  ALL  WARRANTIES,  EXPRESS  OR  IMPLIED,  INCLUDING, BUT  NOT LIMITED TO, ANY IMPLIED
#  WARRANTIES  OR  CONDITIONS  OF MERCHANTABILITY,  SATISFACTORY QUALITY, FITNESS FOR A
#  PARTICULAR  PURPOSE,  TITLE, AND ANY WARRANTY OR CONDITION OF NON-INFRINGEMENT.  IBM
#  SHALL  NOT  BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL OR CONSEQUENTIAL
#  DAMAGES  ARISING  OUT OF THE USE OR OPERATION OF THE SAMPLE SOURCE CODE.  IBM HAS NO
#  OBLIGATION TO  PROVIDE  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS
#  TO THE SAMPLE SOURCE CODE.
########################################################################################

usage="Usage: postCloneApp.sh -b APPS_BASE -h DB_HOSTNAME -d DB_DOMAIN -i DB_IP_ADDR -w WEBLOGIC_ADMIN_PASSWORD [-P DB_PORT]\n\t\t\
                             [-s ORACLE_SID] [-u ORACLE_DB_USER] [-a APPS_USER] [-p APPS_PASSWORD] [-t TIMEOUT_MINS]\n\
Required parameters: -b APPS_BASE,-h DB_HOSTNAME, -d DB_DOMAIN\n\
Example: postCloneApp.sh -b /u01/app/oracle/VIS -h icc-150 -i 1.2.3.4 -d usca.ibm.com -w weblogic_admin_password\n\t\
         postCloneApp.sh -b /u01/app/oracle/VIS -h icc-150 -i 1.2.3.4 -d usca.ibm.com -w weblogic_admin_password -P 1521 -s VIS -u oracle -a applmgr -p apps -t 60\n"

logfile="/tmp/postCloneApp.log"
echo -e "[INFO] starting App tier cloning" | tee -a $logfile

chmod 777 $logfile

# parse command line options
bFlag=0
hFlag=0
dFlag=0
iFlag=0
PFlag=0
sFlag=0
uFlag=0
aFlag=0
pFlag=0
tFlag=0
wFlag=0

while getopts b:h:d:i:w:P:s:u:a:p:t: arg 
do
    case $arg in
	b)  bFlag=1
	    appsBase="$OPTARG";;
	h)  hFlag=1
	    dbHost="$OPTARG";;
	d)  dFlag=1
	    dbDomain="$OPTARG";;
	i)  iFlag=1
	    ipAddr="$OPTARG";;
	w)  wFlag=1
	    wlaPassword="$OPTARG";;
	P)  PFlag=1
	    dbPort="$OPTARG";;
	s)  sFlag=1
	    oracleSID="$OPTARG";;
	u)  uFlag=1
	    oracleDBUser="$OPTARG";;
	a)  aFlag=1
	    appsUser="$OPTARG";;
	p)  pFlag=1
	    appsPassword="$OPTARG";;
	t)  tFlag=1
	    timeoutMins="$OPTARG";;
	?)  echo -e "[ERROR] Unknown argument - $arg" | tee -a $logfile
	    echo -e $usage | tee -a $logfile
	    exit 2;;
    esac
done

# arg1: environment variable to set
# arg2: flag
# arg3: flag value
# arg4: [opt|req]
# arg5: default value
# arg6: user supplied value
setArg()
{
    # echo -e "[DEBUG] var: '$1' def: '$2' flag: '$3' flagVal: '$4' opt/req: '$5' user: '$6'" | tee -a $logfile

    if [ $4 == 0 ]; then	# if flag not specified
	if [ $5 == "req" ]; then
	    echo -e "[ERROR] Required input \$$1 ($3) missing, exiting!" | tee -a $logfile
	    echo -e $usage | tee -a $logfile
	    exit 2
	else
	    echo -e "[INFO] \$$1 = '$2' (default)"  | tee -a $logfile# set default value
	    export $1=$2
	fi
    else			   # flag specified
	echo -e "[INFO] \$$1 = '$6' "  | tee -a $logfile # set user supplied value
	export $1=$6
    fi
}

# arg1 remote server
# arg2 remote port
# arg3 timeout in minutes
# return 0: success, 1: fail (connection on port refused) , 2: fail (connection timed out)
checkDBStatus()
{
    attempt=1
    sleepSecs=10	    # have divisible of 60 for correct results
    let timeoutSecs=$3*60

    export TERM=vt220 # must set TERM else expect fails to read response
    echo -e "[INFO] Testing connectivity to DB server $1 on port $2" | tee -a $logfile
    while [[ $timeoutSecs -gt 0 ]]
    do
	echo -e "[INFO] Telnet attempt #$attempt to $1:$2 on `date`" | tee -a $logfile
	expect <<EOF
            spawn telnet $1 $2
            expect {
                "Connected to*" {
                     send \003
                     exit 1
                } "*remote host refused*" {
                     puts "failed"
                     exit 2
                }
            }
EOF
	if [ $? == 1 ]; then
	    echo -e "[INFO] Successfully tested connection to port $2 on $1!" | tee -a $logfile
	    return 0
	fi
	# else if server/port not reachable, etc.
	sleep $sleepSecs
	echo -e "[INFO] Telnet attempt failed, will retry after $sleepSecs seconds.." | tee -a $logfile
	let timeoutSecs=timeoutSecs-sleepSecs
	let attempt=attempt+1
    done

    echo -e "[ERROR] Connection to port $2 on $1 timed out!" | tee -a $logfile
    return 2
}

# set environment variables based upon supplied or default values
setArg "EBSVA_APPS_BASE" "/u01/app/oracle/VIS" "-b" $bFlag "req" $appsBase
setArg "EBSVA_DB_HOSTNAME" "localhost" "-h" $hFlag "req" $dbHost
setArg "EBSVA_DB_DOMAIN" "localdomain" "-d" $dFlag "req" $dbDomain
setArg "EBSVA_DB_IP" "127.0.0.1" "-i" $iFlag "req" $ipAddr
setArg "EBSVA_WLA_PASS" "wla_password" "-w" $wFlag "req" $wlaPassword
setArg "EBSVA_DB_PORT" "1521" "-P" $PFlag "opt" $dbPort
setArg "ORACLE_SID" "VIS" "-s" $sFlag "opt" $oracleSID
setArg "EBSVA_ORACLE_DB_USER" "oracle" "-u" $uFlag "opt" $oracleDBUser
setArg "EBSVA_ORACLE_APPS_USER" "applmgr" "-a" $aFlag "opt" $appsUser
setArg "EBSVA_APPS_PASS" "apps" "-p" $pFlag "opt" $appsPassword
setArg "EBSVA_DB_CONN_TIMEOUT_MINS" "60" "-t" $tFlag "opt" $timeoutMins

# set apps hostname variable
# NOTE: ICCT activation engine may invoke this script before setting hostname correctly;
#       wait until maxSleep time to see if hostname is set up correctly.
sleepSecs=10
sleepCount=0
maxSleep=1800 # give up after half an hour
EBSVA_APPS_HOSTNAME=`hostname -s`

while [[ $EBSVA_APPS_HOSTNAME ==  "localhost" && $sleepCount -lt $maxSleep ]]
do
    sleep $sleepSecs
    let sleepCount=sleepCount+sleepSecs
    EBSVA_APPS_HOSTNAME = `hostname -s`
done

if [ $EBSVA_APPS_HOSTNAME == "localhost" ]; then
    echo -e "[ERROR] system hostname not yet set, can't continue" | tee -a $logfile
    exit 1
fi

export EBSVA_APPS_HOSTNAME
echo -e "[INFO] \$EBSVA_APPS_HOSTNAME=$EBSVA_APPS_HOSTNAME" | tee -a $logfile

export EBSVA_APPS_CLONE_RUN_SCRIPT=postCloneAppRun.exp
echo -e "[INFO] \$EBSVA_APPS_CLONE_RUN_SCRIPT=$EBSVA_APPS_CLONE_RUN_SCRIPT" | tee -a $logfile

export EBSVA_APPS_PRECLONE_SCRIPT=preCloneApp.exp
echo -e "[INFO] \$EBSVA_APPS_PRECLONE_SCRIPT=$EBSVA_APPS_PRECLONE_SCRIPT" | tee -a $logfile

export EBSVA_APPS_CLONE_PATCH_SCRIPT=postCloneAppPatch.exp
echo -e "[INFO] \$EBSVA_APPS_CLONE_PATCH_SCRIPT=$EBSVA_APPS_CLONE_PATCH_SCRIPT" | tee -a $logfile

# 
# find and set expect scripts path
#
postCloneAppsRunScriptPath=`find $PWD -name "$EBSVA_APPS_CLONE_RUN_SCRIPT"`
preCloneAppsScriptPath=`find $PWD -name "$EBSVA_APPS_PRECLONE_SCRIPT"`
postCloneAppsPatchScriptPath=`find $PWD -name "$EBSVA_APPS_CLONE_PATCH_SCRIPT"`

if [ "x$postCloneAppsRunScriptPath" == "x" ]; then
    echo -e "[ERROR] could not locate $EBSVA_APPS_CLONE_RUN_SCRIPT"  | tee -a $logfile
    exit 1
fi
chmod 777 $postCloneAppsRunScriptPath

if [ "x$preCloneAppsScriptPath" == "x" ]; then
    echo -e "[ERROR] could not locate $EBSVA_APPS_PRECLONE_SCRIPT"  | tee -a $logfile
    exit 1
fi
chmod 777 $preCloneAppsScriptPath

if [ "x$postCloneAppsPatchScriptPath" == "x" ]; then
    echo -e "[ERROR] could not locate $EBSVA_APPS_CLONE_PATCH_SCRIPT"  | tee -a $logfile
    exit 1
fi
chmod 777 $postCloneAppsPatchScriptPath

#
# update /etc/hosts file with DB host details (in case not configured until control reaches this script)
#
echo -e "[INFO] adding App tier hostname entry in /etc/hosts ($EBSVA_DB_HOSTNAME.$EBSVA_DB_DOMAIN = $EBSVA_DB_IP)" | tee -a $logfile
echo -e "$EBSVA_DB_IP    $EBSVA_DB_HOSTNAME.$EBSVA_DB_DOMAIN    $EBSVA_DB_HOSTNAME" >> /etc/hosts

# 
# check DB connectivity before starting cloning
#
checkDBStatus $EBSVA_DB_HOSTNAME $EBSVA_DB_PORT $EBSVA_DB_CONN_TIMEOUT_MINS

if [ $? != 0 ]; then
    echo -e "[ERROR] DB connectivity test failed, exiting!" | tee -a $logfile
    exit 1
fi

#
# start apps tier cloning
#
echo -e "[INFO] DB connectivity tested successfully, proceeding with App post-cloning" | tee -a $logfile

#
# Run-edition cloning
#
echo -e "[INFO] removing FMW_Home directory structure in run-edition (fs1)"  | tee -a $logfile
rm -fr $EBSVA_APPS_BASE/fs1/FMW_Home

# update minimum host requirement for cloning run edition (see Oracle Support Note: 1588970.1)
echo -e "[INFO] updating refhost.xml (Oracle Support Note: 1588970.1)"  | tee -a $logfile
su $EBSVA_ORACLE_APPS_USER "-c perl -000 -pi.orig -e 's|<VERSION VALUE=\"6\"/>(.*?)<VENDOR VALUE=\"oracle\"/>|<VERSION VALUE=\"6.4\"/>\$1<VENDOR VALUE=\"redhat\"/>|s' $EBSVA_APPS_BASE/fs1/EBSapps/comn/clone/prereq/webtier/Scripts/prereq/linux64/refhost.xml"

echo -e "[INFO] cloning run-edition (fs1)" | tee -a $logfile
su $EBSVA_ORACLE_APPS_USER "-c $postCloneAppsRunScriptPath 2>&1 | tee -a $logfile"

# possible bug: node-manager doesn't listen on port 5556 if started immediately after adcfgclone.pl on run-edition
# workaround: wait for a while and keep trying to restart node-manager

echo -e "[INFO] starting node manager on run-edition (fs1)" | tee -a $logfile
let timeoutSecs=4800
sleepSecs=300
nodeManagerPath="$EBSVA_APPS_BASE/fs1/inst/apps/VIS_$EBSVA_APPS_HOSTNAME/admin/scripts/adnodemgrctl.sh"
nodeManagerStarted=false

while [[ $timeoutSecs -gt 0 ]]
do
    echo -e "[DEBUG] trying to start node manager.." | tee -a $logfile
    su $EBSVA_ORACLE_APPS_USER "-c echo $EBSVA_WLA_PASS | $nodeManagerPath start -nopromptmsg"
    /bin/netstat -an | grep 5556 | grep LISTEN
    if [ $? == 0 ]; then
	echo -e "[INFO] successfully started node manager!" | tee -a $logfile
	nodeManagerStarted=true
	break
    else
	# failed attempt, try to clean up
	su $EBSVA_ORACLE_APPS_USER "-c echo $EBSVA_WLA_PASS | $nodeManagerPath stop -nopromptmsg"
    fi
    echo -e "[DEBUG] failed to start node manager, will try again in $sleepSecs seconds.."  | tee -a $logfile
    sleep $sleepSecs
    let timeoutSecs=timeoutSecs-sleepSecs
done


# start app tier services if node-manager is started
if [ $nodeManagerStarted == false ]; then
    echo -e "[FATAL] couldn't start node-manager; can not continue" | tee -a $logfile
    exit 1
else 
    echo -e "[INFO] starting app tier services in run-edition (fs1)" | tee -a $logfile
    su $EBSVA_ORACLE_APPS_USER "-c { echo 'apps' ; echo $EBSVA_APPS_PASS ; echo $EBSVA_WLA_PASS ; } | $EBSVA_APPS_BASE/fs1/inst/apps/VIS_$EBSVA_APPS_HOSTNAME/admin/scripts/adstrtal.sh @ -nopromptmsg"
fi

#
# patch-edition cloning
#
echo -e "[INFO] pre-cloning run-edition (fs1) to prepare for patch-edition (fs2) cloning" | tee -a $logfile
su $EBSVA_ORACLE_APPS_USER "-c $preCloneAppsScriptPath  2>&1 | tee -a $logfile"

echo -e "[INFO] removing directories under patch-edition and copying EBSapps and inst from run-edition to patch-edition" | tee -a $logfile
rm -fr $EBSVA_APPS_BASE/fs2/*
su $EBSVA_ORACLE_APPS_USER "-c cp -RH $EBSVA_APPS_BASE/fs1/EBSapps $EBSVA_APPS_BASE/fs2/"
su $EBSVA_ORACLE_APPS_USER "-c cp -RH $EBSVA_APPS_BASE/fs1/inst $EBSVA_APPS_BASE/fs2/"

# update minimum host requirement for cloning patch edition (see Oracle Support Note: 1588970.1)
echo -e "[INFO] updating refhost.xml (Oracle Support Note: 1588970.1)"  | tee -a $logfile
su $EBSVA_ORACLE_APPS_USER "-c perl -000 -pi.orig -e 's|<VERSION VALUE=\"6\"/>(.*?)<VENDOR VALUE=\"oracle\"/>|<VERSION VALUE=\"6.4\"/>\$1<VENDOR VALUE=\"redhat\"/>|s' $EBSVA_APPS_BASE/fs2/EBSapps/comn/clone/prereq/webtier/Scripts/prereq/linux64/refhost.xml"

echo -e "[INFO] cloning patch-edition (fs2)" | tee -a $logfile
su $EBSVA_ORACLE_APPS_USER "-c $postCloneAppsPatchScriptPath 2>&1 | tee -a $logfile"

echo -e "[INFO] successfully completed app tier cloning" | tee -a $logfile
