/*
 * IBM Confidential
 * 
 * OCO Source Materials
 * 
 * 5725A15
 * 
 *  Copyright IBM Corp. 2010,2011
 * 
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 */
package com.ibm.casemgmt.sampexterndata.rest;

import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.filenet.api.util.UserContext;
import com.ibm.casemgmt.sampexterndata.api.ExternalSystemConfiguration;
import com.ibm.casemgmt.sampexterndata.api.FileBasedExternalData;
import com.ibm.casemgmt.sampexterndata.api.SEDLogger;

/*
 * service the external data test service
 * 
 * the servlet provides the framework to parse the path and invoke the handler for
 * resources
 * 
 */
public final class SampleServiceServlet extends HttpServlet 
{
    /**
	 * 
	 */
	private static final long serialVersionUID = 5439588989554103422L;

	private static final Log logger = LogFactory.getLog(RESTConstants.LOGGER_SAMPLE_SERVICE_REST); 

    private static final String m_className = "SampleServiceServlet";
    private static final String CONFIG_FILE_PARAM = "ExternalDataFileConfigFile";
    private static final String TEST_AUTHENTICATION_CE_URI_PARAM = "TestAuthenticationCEURI";
    
    private enum SERVLET_METHOD  {METHOD_GET, METHOD_PUT, METHOD_POST, METHOD_DELETE};

    private ExternalSystemConfiguration sysConfig;
    private String testAuthenticationCEURI;
    private boolean testAuthenticationCEURILoaded;
    
    public SampleServiceServlet()
    {
    	SEDLogger.trace(logger, "SampleServiceServlet() called.");
    }
    
    
    /**
     * Servlet init -- called once upon server restart.
     * read the configuration file to get the resource URIs and resource Handlers
     */
    public void init() throws ServletException
    {
    	try
    	{
    	    loadExternalSystemConfiguration();
    	}
    	catch(Exception e)
    	{
    	    // ignore for now.  Will try again with a specific HTTP method
    	    // and return an error response to the caller.
    	}
    }
    
    private void loadTestAuthenticationCEURI()
    {
        if (!testAuthenticationCEURILoaded)
        {
            ServletContext context = getServletContext();
            String ceuri = context.getInitParameter(TEST_AUTHENTICATION_CE_URI_PARAM);
            if (ceuri != null && ceuri.length() > 0)
                testAuthenticationCEURI = ceuri;
            testAuthenticationCEURILoaded = true;
        }
    }
    
    /**
     * Will throw out if there is an error in the config file.
     */
    private void loadExternalSystemConfiguration()
    {
        if (sysConfig == null)
        {
            ServletContext context = getServletContext();
            String configFile = context.getInitParameter(CONFIG_FILE_PARAM);
            InputStream configStream = context.getResourceAsStream("/WEB-INF/"+configFile);

            FileBasedExternalData data = new FileBasedExternalData(configStream);
            sysConfig = data.loadExternalSystem();
        }
    }
    
    public void doGet(HttpServletRequest request, HttpServletResponse response)
    	throws ServletException
    {
    	doMethod(request, response, SERVLET_METHOD.METHOD_GET);
    }

    public void doPost(HttpServletRequest request, HttpServletResponse response)
    	throws ServletException
    {
    	doMethod(request, response, SERVLET_METHOD.METHOD_POST);
    }

    public void doPut(HttpServletRequest request, HttpServletResponse response)
    	throws ServletException
    {
    	doMethod(request, response, SERVLET_METHOD.METHOD_PUT);
    }

    public void doDelete(HttpServletRequest request, HttpServletResponse response)
    	throws ServletException
    {
    	doMethod(request, response, SERVLET_METHOD.METHOD_DELETE);
    }
    
    public void doOptions(HttpServletRequest request, HttpServletResponse response)
    	throws ServletException
    {
    	// don't do anything?
    }
    
    private void doMethod(HttpServletRequest request, HttpServletResponse response,
    						SERVLET_METHOD servletMethod)
    	throws ServletException
    {
    	SEDLogger.trace(logger, "SampleServiceServlet.doMethod(): servletMethod=" + servletMethod);

		String pathInfo = request.getPathInfo();
		SEDLogger.trace(logger, "pathInfo=" + pathInfo);
    	if (pathInfo==null)
    	{
    		// served as "keepAlive"
    		response.setStatus(HttpServletResponse.SC_OK);
    		return;
    	}

        int status = HttpServletResponse.SC_OK;
    	String msg = "OK";
    	String requestId = UUID.randomUUID().toString();
		final String m_method = "doMethod:"+requestId;
		SEDLogger.entering(logger, m_className, m_method);
		
		servletMethod = getRealMethod(request, servletMethod);
		SEDLogger.trace(logger, "Real method: " + servletMethod);
    	StringBuffer responseBuf = new StringBuffer(); // only used when responseLogger is enabled
		SampleServiceHandler tsh = new SampleServiceHandler();
		UserContext uc = new UserContext();
		uc.setLocale(request.getLocale());
        UserContext origUC = UserContext.get();
		UserContext.set(uc);
    	try
    	{
    	    // If loading the config file failed during initialization, this
    	    // will generate an error to return to the caller
    	    loadExternalSystemConfiguration();
    	    loadTestAuthenticationCEURI();
    	    
    		ResourceRequest resourceRequest = new ResourceRequest(request, sysConfig, requestId, 
    		        testAuthenticationCEURI);


    		IResourceResponse resourceResponse = null;
     		if (logger.isTraceEnabled())
    			responseBuf.append("\n------------------------------------------------------\nResponse to Request ID=")
    				.append(requestId).append("\n")
    				.append(request.getMethod()).append(" ").append(request.getRequestURI());
    		
    		switch (servletMethod)
    		{
	    		case METHOD_GET: 	
	    			resourceResponse=tsh.onGet(resourceRequest);
	    			break;
	    		case METHOD_DELETE: 	
	    			resourceResponse=tsh.onDelete(resourceRequest);
	    			break;
	    		case METHOD_POST: 	
	    			resourceResponse=tsh.onPost(resourceRequest);
    				break;
	    		case METHOD_PUT: 	
	    			resourceResponse=tsh.onPut(resourceRequest);
    				break;
    		}
    		
    		generateResponse(request, response, resourceResponse, responseBuf, requestId, m_method, msg);

    	}
    	catch (Exception ex)
    	{
    		SEDLogger.trace(logger, "SampleServiceServlet.doMethod(), exception received");
    		SEDLogger.error(logger, ex.getLocalizedMessage(), ex);
    		
            status = HttpServletResponse.SC_INTERNAL_SERVER_ERROR;
            msg = ex.getLocalizedMessage();
    		boolean handledResponse = false;
		    try
		    {
	            IResourceResponse errorResponse = tsh.handleErrorResponse(ex);
	            status = errorResponse.getStatus();
	            generateResponse(request, response, errorResponse, responseBuf, requestId, m_method, msg);
	            handledResponse = true;
		    }
		    catch(Throwable t)
		    {
		        // Handler shouldn't throw out, just log and treat it as an unhandled
		        // response below.
		        SEDLogger.error(logger, t.getLocalizedMessage(), t);
		    }
    		
    		if (!handledResponse)
    		{
                SEDLogger.error(logger, m_className + ":" + m_method, ex);
                
                if (logger.isTraceEnabled())
                    responseBuf.append("\nStatus=").append(status).
                            append("\n").append(msg);
                finalTracing(responseBuf, m_method, pathInfo, status, msg);

        		throw new ServletException (ex.getMessage(), ex);
    		}
    	}
    	finally
    	{
    	    if (origUC != null)
    	        UserContext.set(origUC);
    		SEDLogger.exiting(logger, m_className, m_method);
    	}
    }
    
    /**
     * Checks for X-HTTP-Method-Override.
     * @param request
     * @param initialMethod
     * @return
     */
    private static SERVLET_METHOD getRealMethod(HttpServletRequest request, SERVLET_METHOD initialMethod)
    {
    	final String m_method = "getRealMethod";
    	SEDLogger.entering(logger, m_className, m_method);
    	
    	try
    	{
	    	// Check for X-HTTP-Method-Override : {PUT,DELETE} to allow PUT or DELETE
	    	String overrideMethod = request.getHeader(HTTPHeaderConstants.X_HTTP_METHOD_OVERRIDE);
	    	if (overrideMethod!=null)
	    	{
	    		if (overrideMethod.equalsIgnoreCase("PUT"))
	    			return SERVLET_METHOD.METHOD_PUT;
	    		if (overrideMethod.equalsIgnoreCase("DELETE"))
	    			return SERVLET_METHOD.METHOD_DELETE;    		
	    	}
	    	// check for the Content-Type : application/x-www-form-urlencoded
	    	// If sent in a POST, need to map to GET
	    	if (initialMethod.equals(SERVLET_METHOD.METHOD_POST))
	    	{
		    	String contentType = request.getHeader(HTTPHeaderConstants.CONTENT_TYPE);
		    	if (contentType!=null 
		    			&&
	    			contentType.contains("application/x-www-form-urlencoded"))
	    		return SERVLET_METHOD.METHOD_GET;
	    	}
    		return initialMethod;
    	}
    	finally
    	{
    		SEDLogger.exiting(logger, m_className, m_method);
    	}
    }
    
    private void generateResponse(HttpServletRequest request, HttpServletResponse response, IResourceResponse resourceResponse, StringBuffer responseBuf,
            String requestId, String m_method, String msg) throws IOException
    {
        // status
        String pathInfo = request.getPathInfo();
        int status = resourceResponse.getStatus();
        response.setStatus(status);
        
        // should set the response locale to be that of the request..
        response.setLocale(request.getLocale());
        
        // header
        writeResponseHeader(resourceResponse, response, responseBuf, requestId);
        // response body -- finally
        if (logger.isTraceEnabled())
        {
            StringBuffer buf = new StringBuffer();
            resourceResponse.write(buf);
            String responseBody = buf.toString();
            responseBuf.append("\nRESPONSE BODY=").append(responseBody);
            finalTracing(responseBuf, m_method, pathInfo, status, msg);
            response.getOutputStream().write(responseBody.getBytes("UTF-8"));
        }
        else
        {
            finalTracing(responseBuf, m_method, pathInfo, status, msg);
            resourceResponse.write(response.getOutputStream());
        }

    }
    
    /**
     * Response time ----
     * write the headers
     * 
     * @param resourceResponse
     * @param response
     */
    private void writeResponseHeader(IResourceResponse resourceResponse, HttpServletResponse response,
    								StringBuffer responseBuf,
    								String requestId)
    {
    	if (responseBuf!=null)
    		responseBuf.append("\nRESPONSE HEADERS:");
    	Map<String, List<String>> responseHeaders = resourceResponse.getResponseHeaders();
    	for (String header:responseHeaders.keySet()){
    		List<String> headerValues = responseHeaders.get(header);
    		Iterator<String> iter = headerValues.iterator();
    		while (iter.hasNext())
    		{
    			String headerValue = iter.next();
    			if (responseBuf!=null)
    				responseBuf.append("\n").append(header).append("=").append(headerValue);
    			response.setHeader(header, headerValue);
    		}
    	}
    }

    private void finalTracing(StringBuffer responseBuf, String m_method,
            String pathInfo,
            int status,
            String msg) {
        if (logger.isTraceEnabled())
        {
            responseBuf.append("\n------------------------------------------------------");
            SEDLogger.trace(logger, m_className, m_method, responseBuf.toString());
        }
        StringBuffer buf = new StringBuffer(pathInfo);
            buf.append("\t").append("\t").append(status).append("\t").append(msg);
    }

    
}
