/*
 * IBM Confidential
 * 
 * OCO Source Materials
 * 
 * 5725A15
 * 
 *  Copyright IBM Corp. 2010,2011
 * 
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 */
package com.ibm.casemgmt.sampexterndata.rest;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.json.JSONArray;
import org.apache.commons.json.JSONException;
import org.apache.commons.json.JSONObject;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.casemgmt.sampexterndata.api.SEDException;
import com.ibm.casemgmt.sampexterndata.api.ExternalSystemConfiguration;
import com.ibm.casemgmt.sampexterndata.api.SEDLogger;


/**
 * utility class to parse HTTP Request
 * or to provide stubs for test program easily
 */
final class ResourceRequest 
{

    private static final Log requestLogger = LogFactory.getLog(RESTConstants.LOGGER_SAMPLE_SERVICE_REST);
    private static final String CLASS_NAME = "ResourceRequest";
    
    private String servletContextPath=null;
    private Map<String, List<String>> requestParams;
    private Object requestBody;
    private StringBuffer requestBuf = new StringBuffer();

    private String requestURI = null;
    private String requestId;
    private Locale requestLocale;
    private HttpServletRequest currRequest;
    private ExternalSystemConfiguration sysConfig;
    private String testAuthenticationCEURI;
	
    public ResourceRequest(HttpServletRequest request, ExternalSystemConfiguration sysConfig, String requestId,
            String testAuthenticationCEURI)
    {
        SEDLogger.trace(requestLogger, "ResourceRequest() called.");
        requestURI = request.getRequestURI();
        this.requestId = requestId;
        requestParams = formulateRequestParams(request);
        requestBody = readRequestStream(request);
        //here's we can formulate the path..from the request..
        servletContextPath = formulateServletContextPath(request);
        currRequest = request;
        requestLocale = request.getLocale();
        this.sysConfig = sysConfig;
        this.testAuthenticationCEURI = testAuthenticationCEURI;
        
        if (requestLogger.isTraceEnabled())
        {
            requestBuf.append("\n------------------------------------------------------");
            SEDLogger.trace(requestLogger, CLASS_NAME, "ctor", requestBuf.toString());
        }
    }
	
	public String getRequestURI()
	{
	    return requestURI;
	}
	
	public ExternalSystemConfiguration getSysConfig()
	{
		return sysConfig;
	}
	
	public String getTestAuthenticationCEURI()
	{
	    return testAuthenticationCEURI;
	}
	
	/*
	 * Returns locale from request object.
	 */
	public Locale getRequestLocale()
	{
	    return requestLocale;
	}

	public String getPathInfo ()
	{
	    return currRequest.getPathInfo();
	}
	
	public String getServletContextPath()
	{
	    return servletContextPath;
	}

	/**
	 * Return the parameters to the request as a keyed map to 
	 * a list of string values.  The map is an unmodifiable
	 * map and each list is an unmodifiable list.
	 */
	public Map<String, List<String>> getRequestParams()
	{
	    return requestParams;
	}

	/**
	 * Returns the list of values for a particular parameter.
	 * The returned list is unmodifiable.
	 */
	public List<String> getRequestParamList(String name)
	{
		if (name!=null && requestParams!=null)
	    	return requestParams.get(name);
	    return null;
	}
	
	public String getRequestParamOne(String name, String defaultParam)
	{
	    String m_method = "getRequestParamOne:"+requestId;
	    String res=defaultParam ;
	    if (name!=null && requestParams!=null)
	    {
		List<String> values = requestParams.get(name);
		if (values!=null && values.size()>0)
		    res= values.get(0);
	    }
	    SEDLogger.trace(requestLogger, CLASS_NAME, m_method, res);
	    return res;
	}
		
    public String getRequiredRequestParameter (String paramName)
    {
        String paramVal = getRequestParamOneIgnoreCase(paramName, null);
        
        if (paramVal == null)
        {
            throw ErrorStatusException.createBadRequestException(RESTConstants.ERROR_MISSING_REQUEST_PARAM, paramName);
        }
        
        return paramVal;
    }

    public String getRequiredRequestParameterNonEmptyValue(String paramName)
    {
        String paramVal = getRequiredRequestParameter(paramName);
        
        if (paramVal.equalsIgnoreCase(""))
        {
        	throw ErrorStatusException.createNotFoundException(RESTConstants.ERROR_INVALID_REQUEST_PARAM_VALUE,paramName,paramVal);
        }
        
        return paramVal;
    }

    
	public String getRequestParamOneIgnoreCase(String name, String defaultParam)
	{
	    String m_method = "getRequestParamOneIgnoreCase:"+requestId;
	    String res=defaultParam ;
	    List<String> values = null;
	    if (name!=null && requestParams!=null)
	    {
		// optimistic case:
		values = requestParams.get(name);
		if (values==null)
		{
		    // must find one with a different case. or should it be all lower case?
		    Set<String> keys = requestParams.keySet();
		    for (String k : keys)
		    {
			if (k.equalsIgnoreCase(name))
			{
			    values = requestParams.get(k);
			    break;
			}
		    }
		}
		if(values!=null&& values.size()>0)
		    res= values.get(0);
	    }
	    SEDLogger.trace(requestLogger, CLASS_NAME, m_method, res);
	    return res;
	}
	
    public String getRequestId()
    {
    	return requestId;
    }
    
    /*
     * Returns parameter from the request object.
     */
    public Object getRequestBodyParameterObject(String paramName)
    {
    	Object paramVal = null;
    	
    	if (requestBody instanceof JSONObject)
    	{
        	JSONObject jo = (JSONObject) requestBody;
        	try {
				paramVal = jo.get(paramName);
			} catch (JSONException e) {
				//thrown if key is not found
				//just ignore and return null
			}
    	}
    	else if (requestBody instanceof JSONArray)
    	{
    		JSONArray ja = (JSONArray) requestBody;
    		Iterator<?> it = ja.iterator();
    		while (it.hasNext())
    		{
    			JSONObject jo = (JSONObject) it.next();
    			try {
					paramVal = jo.get(paramName);
				} catch (JSONException e) {
					//thrown if key is not found
					//just ignore and return null
				}
    			if (paramVal != null)
    				break;
    		}
    	}
    	else
    	{
    		throw SEDException.createException(RESTConstants.ERROR_REQUEST_BODY_NOT_VALID_JSON);
    	}
    	
    	return paramVal;
    }

    public String getRequestBodyParameterString(String paramName)
    {
    	return  (String) getRequestBodyParameterObject(paramName);

    }

    public JSONObject getRequestBodyParameterJSONObject(String paramName)
    {
    	return  (JSONObject) getRequestBodyParameterObject(paramName);

    }

    public Object getRequiredRequestBodyParameterObject(String paramName)
    {
    	Object paramVal = getRequestBodyParameterObject(paramName);

    	if (paramVal == null)
    	{
    	    throw ErrorStatusException.createBadRequestException(RESTConstants.ERROR_MISSING_REQUEST_PARAM, paramName);
    	}
    	
    	return paramVal;
    }

    public String getRequiredRequestBodyParameterString(String paramName)
    {
    	return (String) getRequiredRequestBodyParameterObject(paramName);
    }

    public int getRequiredRequestBodyParameterInt(String paramName)
    {
    	return ((Integer) getRequiredRequestBodyParameterObject(paramName)).intValue();
    }

	public int getRequestBodyParameterInt(String paramName)
    {
    	return ((Integer) getRequestBodyParameterObject(paramName)).intValue();
    }

    public Boolean getRequiredRequestBodyParameterBoolean(String paramName)
    {
    	Object paramObj = getRequiredRequestBodyParameterObject(paramName);
    	
    	return getBooleanValueFromExpectedLongObjectParameter(paramObj, paramName);
    }

	public Boolean getRequestBodyParameterBoolean(String paramName)
    {
    	Object paramObj = getRequestBodyParameterObject(paramName);
    	
    	if (paramObj != null)
    		return getBooleanValueFromExpectedLongObjectParameter(paramObj, paramName);
    	
    	return null;
    }    
    
    private Boolean getBooleanValueFromExpectedLongObjectParameter(Object paramObj, String paramName) 
    {
    	boolean paramBoolean;
    	
    	if (paramObj instanceof Boolean)
    		paramBoolean = (Boolean)paramObj;
    	else
    	{
    	    throw SEDException.createException(RESTConstants.ERROR_INVALID_REQUEST_BODY_TYPE, paramName, paramObj.getClass().getName());
    	}
		return paramBoolean;
	}    
    
    public JSONObject getRequiredRequestBodyParameterJSONObject(String paramName)
    {
    	return (JSONObject) getRequiredRequestBodyParameterObject(paramName);
    }
    
    /**
     * Get the request's headers and parameters into a hash map..
     * This is for ease of integration since the individual handler class
     * can be implemented without dependency on the HttpServletRequest and Response
     * and unit-tested easily..
     * 
     * @param request
     * @return
     */
    private Map<String, List<String>> formulateRequestParams(HttpServletRequest request)
    {
    	final String m_method = "formulateRequestParams:"+requestId;

    	/** need to set the character encoding to UTF-8 for correct decoding of the param.
    	 * This requires that the query string or form POST to be url-encoded of UTF-8 encoded strings.
    	 */
    	try
    	{
    	    request.setCharacterEncoding("UTF-8");
    	}
    	catch (Exception e)
    	{
    		SEDLogger.error(requestLogger, CLASS_NAME +":"+ m_method, e);
    	}
    	Map<String, List<String>> res = new HashMap<String,List<String>>(5);

    	if (requestLogger.isTraceEnabled())
    	{
    	    requestBuf.append("\n------------------------------------------------------\nRequest ID=")
    	    .append(requestId).append("\n")
    	    .append(request.getMethod()).append(" ").append(request.getRequestURI()).append("\nHEADERS:");
    	}
    	
    	// get all the request header?, or just a few..
    	Enumeration<?> headerNamesEnum = request.getHeaderNames();
    	while (headerNamesEnum.hasMoreElements())
    	{
    	    String s = (String)headerNamesEnum.nextElement();


    	    Enumeration<?> valuesEnum = request.getHeaders(s);
    	    List<String> vValues = new ArrayList<String>(1);
    	    while (valuesEnum.hasMoreElements())    		
    		vValues.add((String)valuesEnum.nextElement());    			

    	    if (requestLogger.isTraceEnabled())
    	    {
        		requestBuf.append("\n").append(s).append("=");
        		
        		for ( String val : vValues)
        			requestBuf.append(val);
    	    }
    	    
    	    res.put(s, Collections.unmodifiableList(vValues));    		
    	}  
    	
    	if (requestLogger.isTraceEnabled())
    	{
    		requestBuf.append("\nPARAMETERS:");
    	}
    	// get the other parameters from query String
    	// if the query string is something like this:
    	// opt=opt1,opt2&num=1,2,3
    	//
    	// prefers that opt to have a List<String> with {opt1, opt2}
    	//
    	Enumeration<?> paramNamesEnum = request.getParameterNames();
    	while (paramNamesEnum.hasMoreElements())
    	{
    	    String s = (String)paramNamesEnum.nextElement();
    	    String[] values = request.getParameterValues(s);
    	    int nValues = values==null?0:values.length;
    	    List<String> vValues = new ArrayList<String>(nValues);

    	    for (int i=0; i<nValues; i++)
    	    	vValues.add(values[i]);
    	    res.put(s, Collections.unmodifiableList(vValues));

    	    if (requestLogger.isTraceEnabled())
    	    {
        		requestBuf.append("\n").append(s).append("=");
        		
        		for ( String val : vValues)
        			requestBuf.append(val);
    	    }

    	}

    	return Collections.unmodifiableMap(res);
    }
    
    private String getRequestBody(InputStream inStream)
    	throws IOException
    {
    	BufferedReader reader = new BufferedReader(new InputStreamReader(inStream, "UTF-8"));

    	StringBuffer buf = new StringBuffer();
        char data[] = new char[8196];
        int amtRead = 0;
        amtRead = reader.read(data,0,8196);
        while (amtRead != -1)
        {
            buf.append(data,0,amtRead);
            amtRead = reader.read(data,0,8196);
        }
        String s= buf.toString().trim();
        if (requestLogger.isTraceEnabled())
        	requestBuf.append("\nREQUEST BODY=").append(s);
        return s;
    }

    // read the request body into either JSON or something?
    // should only for doPost or doPut...
    // not doGet, doDelete.
    private Object readRequestStream(HttpServletRequest request)
    {
        final String m_method = "readRequestStream:"+requestId;
        String s = null;
        try
        {
            s=getRequestBody(request.getInputStream());
        }
        catch(IOException e)
        {
            // just ignore if can't read it. and return null;
            //ignore this.
            if (requestLogger.isTraceEnabled())
                SEDLogger.trace(requestLogger, CLASS_NAME, m_method, "Empty Body..");
            return null;
        }
        
        return parse(s);
    }
    
    // read the request body into JSON
    private Object parse(String s)
    {
        if (s==null)
            return null;

        Object res = JSONUtils.getJSONFromString(s);
        if (res==null)
            return s;
        return res;	
    }

    private String formulateServletContextPath(HttpServletRequest request)
    {
        if (request==null)
            return "";
        return request.getContextPath()+request.getServletPath();
    }
    
}
