/*
 * IBM Confidential
 * 
 * OCO Source Materials
 * 
 * 5725A15
 * 
 *  Copyright IBM Corp. 2011
 * 
 * The source code for this program is not published or otherwise divested of
 * its trade secrets, irrespective of what has been deposited with the U.S.
 * Copyright Office.
 */
package com.ibm.casemgmt.sampexterndata.rest;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.json.JSONArray;
import org.apache.commons.json.JSONException;
import org.apache.commons.json.JSONObject;
import org.apache.commons.json.OrderedJSONObject;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.filenet.api.constants.Cardinality;
import com.filenet.api.constants.ChoiceType;
import com.filenet.api.constants.TypeID;
import com.ibm.casemgmt.sampexterndata.api.DisplayMode;
import com.ibm.casemgmt.sampexterndata.api.ExternalChoice;
import com.ibm.casemgmt.sampexterndata.api.ExternalChoiceList;
import com.ibm.casemgmt.sampexterndata.api.ExternalProperty;
import com.ibm.casemgmt.sampexterndata.api.ExternalSystemConfiguration;
import com.ibm.casemgmt.sampexterndata.api.InputProperty;
import com.ibm.casemgmt.sampexterndata.api.PropertyValueHolder;
import com.ibm.casemgmt.sampexterndata.api.SEDConstants;
import com.ibm.casemgmt.sampexterndata.api.SEDException;
import com.ibm.casemgmt.sampexterndata.api.SEDLogger;

final class PropertiesJSONConverter {
	private static final Log logger = LogFactory.getLog(RESTConstants.LOGGER_SAMPLE_SERVICE_REST); 
	
    public static List<InputProperty> processProperties(
            JSONArray array, 
            ExternalSystemConfiguration sysConfig,
            String typeName) 
    {
    	try
    	{
	    	List<InputProperty> inputProps = new ArrayList<InputProperty>();
	    	if (sysConfig.getCaseTypesRepresented().contains(typeName))
	    	{
	            Iterator<?> inputPropsIt = array.iterator();
	            while(inputPropsIt.hasNext())
	            {
	                
	            	JSONObject jo = (JSONObject)inputPropsIt.next();
	                
	                String symName = (String) jo.get(RESTConstants.JSON_KEY_SYMBOLICNAME);
	                if (!sysConfig.getPropertiesRepresented(typeName).contains(symName))
	                    continue;
	                
	                TypeID dataType = sysConfig.getDataType(typeName, symName);
	                Cardinality cardinality = sysConfig.getCardinality(typeName, symName);
	    
	            	boolean requiresUniqueElements = false;
	            	if (cardinality == Cardinality.LIST)
	            	    requiresUniqueElements = sysConfig.getRequiresUniqueElements(typeName, symName);
	            	
	            	InputPropertyImpl inputProp = new InputPropertyImpl(symName, dataType, cardinality, requiresUniqueElements);        	
	              
	                if (jo.containsKey(RESTConstants.JSON_KEY_VALUE))
	                {
	                    Object jsonAbleValue = jo.get(RESTConstants.JSON_KEY_VALUE);
	                    Object propValue = getPropertyValueFromJSONable(
	                    		inputProp.getPropertyType(), jsonAbleValue);
	                    inputProp.setObjectValue(propValue);
	                }
	                else
	                {
	                	inputProp.unspecifyValue();
	                }
	                
	                inputProps.add(inputProp);
	            }
	    	}
	        
	        return inputProps;
    	}
    	catch(JSONException e)
    	{
    		throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED, e);
    	}
    }
    
    private static Object getPropertyValueFromJSONable(TypeID typeId, Object jsonAbleValue)
    {
        Object rtnValue = null;
        if (jsonAbleValue != null)
        {
            PropertyValueHolder singleValHolder = PropertyValueHolder.createHolder(typeId, Cardinality.SINGLE, false);
            // a JSON array, generate a list for a multi-value property
            if (jsonAbleValue instanceof JSONArray)
            {
                List<Object> multiVals = new ArrayList<Object>();
                JSONArray jsonArray = (JSONArray) jsonAbleValue;
                Iterator<?> itemsIt = jsonArray.iterator();
                while (itemsIt.hasNext())
                {
                    Object item = itemsIt.next();
                    singleValHolder.setObjectValue(item);
                    multiVals.add(singleValHolder.getValue());
                }
                
                rtnValue = multiVals;
            }
            else
            {
                singleValHolder.setObjectValue(jsonAbleValue);
                rtnValue = singleValHolder.getValue();
            }
        }
        
        return rtnValue;
    }
    
    public static JSONArray generateJSONProperties(List<ExternalProperty> list) 
    {
    	try
    	{
	    	Iterator<ExternalProperty> it = list.iterator();
	    	JSONArray jaProps = new JSONArray();
	    	while (it.hasNext())
	    	{
	    		OrderedJSONObject joProp = new OrderedJSONObject();
	    		ExternalProperty prop = it.next();
	    		joProp.put(RESTConstants.JSON_KEY_SYMBOLICNAME,prop.getSymbolicName());
	    		if (prop.isValueSpecified())
	    			joProp.put(RESTConstants.JSON_KEY_VALUE,getJSONableFromPropertyValue(prop.getDataType(),prop.getValue()));
	    		if (prop.getDisplayMode()!=null)
	    			joProp.put(RESTConstants.JSON_KEY_DISPLAY_MODE,getDisplayModeString(prop.getDisplayMode()));
	    		if (prop.getCustomValidationError() != null && prop.getCustomValidationError().length() > 0)
	    		    joProp.put(RESTConstants.JSON_KEY_CUSTOM_VALIDATION_ERROR, prop.getCustomValidationError());
	    		List<Integer> invalidItems = prop.getCustomInvalidItems();
	    		if (invalidItems != null && invalidItems.size() > 0)
	    		{
	    		    JSONArray jaItems = new JSONArray();
	    		    Iterator<Integer> itemsIt = invalidItems.iterator();
	    		    while (itemsIt.hasNext())
	    		    {
	    		        jaItems.add(itemsIt.next());
	    		    }
	    		    joProp.put(RESTConstants.JSON_KEY_CUSTOM_INVALID_ITEMS, jaItems);
	    		}
	    		if (prop.getHidden() != null)
	    		    joProp.put(RESTConstants.JSON_KEY_HIDDEN,prop.getHidden());
	    		if (prop.getRequired() != null)
	    			joProp.put(RESTConstants.JSON_KEY_REQUIRED,prop.getRequired());
	    		if (prop.getMaximumValue()!=null)
	        		joProp.put(RESTConstants.JSON_KEY_MAXVALUE,getJSONableFromPropertyValue(prop.getDataType(),prop.getMaximumValue()));
	    		if (prop.getMaximumValue()!=null)
	    			joProp.put(RESTConstants.JSON_KEY_MINVALUE,getJSONableFromPropertyValue(prop.getDataType(),prop.getMinimumValue()));
	    		if (prop.getMaximumLength()!=null)
	    			joProp.put(RESTConstants.JSON_KEY_MAXLENGTH,prop.getMaximumLength());
	    		joProp.put(RESTConstants.JSON_KEY_HAS_DEPENDENT_PROPERTIES,prop.getHasDependentProperties());
	    		if (prop.getChoiceList()!=null)
	    			joProp.put(RESTConstants.JSON_KEY_CHOICELIST,generateJSONChoiceList(prop.getChoiceList()));
	    		jaProps.add(joProp);
	    	}
	    	
	    	return jaProps;
    	}
    	catch(JSONException e)
    	{
    		throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED, e);
    	}
    }
    
    private static String getDisplayModeString(DisplayMode mode)
    {
    	if (mode == DisplayMode.READ_ONLY)
    		return RESTConstants.JSON_VALUE_READONLY;
    	else if (mode == DisplayMode.READ_WRITE)
    		return RESTConstants.JSON_VALUE_READWRITE;
    	else
    		throw new IllegalArgumentException();
    }
    
    private static OrderedJSONObject generateJSONChoiceList(ExternalChoiceList choicelist)
    {
    	try
    	{
	    	OrderedJSONObject joChoiceList = new OrderedJSONObject();
	    	joChoiceList.put(RESTConstants.JSON_KEY_DISPLAYNAME, choicelist.getDisplayName());
	    	List<ExternalChoice> choices = choicelist.getChoices();
	    	JSONArray jaChoices = generateJSONChoices(choices);
	    	joChoiceList.put(RESTConstants.JSON_KEY_CHOICES, jaChoices);
	    	return joChoiceList;
    	}
    	catch(JSONException e)
    	{
    		throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED, e);
    	}
    }

    private static JSONArray generateJSONChoices(List<ExternalChoice> choices)
    {
    	try
    	{
	    	JSONArray jaChoices = new JSONArray();
	    	Iterator<ExternalChoice> it = choices.iterator();
	    	while (it.hasNext())
	    	{
	    		OrderedJSONObject joChoice = new OrderedJSONObject();
	    		ExternalChoice choice = it.next();
	    		joChoice.put(RESTConstants.JSON_KEY_DISPLAYNAME, choice.getDisplayName());
	    		if (choice.getChoiceType()==ChoiceType.INTEGER)
	    			joChoice.put(RESTConstants.JSON_KEY_VALUE, choice.getIntegerChoice());
	    		else if (choice.getChoiceType() == ChoiceType.STRING)
	    			joChoice.put(RESTConstants.JSON_KEY_VALUE, choice.getStringChoice());
	    		else if (choice.getChoiceType() == ChoiceType.MIDNODE_INTEGER || choice.getChoiceType() == ChoiceType.MIDNODE_STRING)
	    			joChoice.put(RESTConstants.JSON_KEY_CHOICES, generateJSONChoices(choice.getGroupOfChoices()));
	    		jaChoices.add(joChoice);
	    	}
	    	
	    	return jaChoices;
    	}
    	catch(JSONException e)
    	{
    		throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED, e);
    	}
    }
    
    private static Object getJSONableFromPropertyValue(TypeID typeId, Object value)
    {
        Object rtnValue = null;
        if (value != null)
        {
            // A multi-value property, generate an array of these values
            if (value instanceof List<?>)
            {
                List<?> values = (List<?>) value;
                JSONArray jsonValues = new JSONArray();
                Iterator<?> valuesIt = values.iterator();
                while (valuesIt.hasNext())
                {
                    Object itemValue = valuesIt.next();
                    jsonValues.add(getJSONableFromPropertyValue(typeId, itemValue));
                }
                rtnValue = jsonValues;
            }
            else if (typeId == TypeID.DATE)
            {
                if (value instanceof java.util.Date)
                {
                    rtnValue = JSONUtils.convertDateToString((java.util.Date) value);
                }
            }
        }
        // Default handling
        if (rtnValue == null)
        {
            // If value is null we want to explicitly include null as the JSON result
            if (value == null 
                    || value instanceof String 
                    || value instanceof Boolean 
                    || value instanceof Number)
                rtnValue = value;
            else
            {
                rtnValue = value.toString();
                SEDLogger.warn(logger, "The JSON property value [" + rtnValue + "] was an urecognized type " + value.getClass().getSimpleName());
            }
        }
        return rtnValue;
    }

    private PropertiesJSONConverter() {}
}
