/*
 * IBM Confidential
 * 
 * OCO Source Materials
 * 
 * 5725A15
 * 
 *  Copyright IBM Corp. 2010,2011
 * 
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 */
package com.ibm.casemgmt.sampexterndata.api;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.SimpleTimeZone;

import com.filenet.api.constants.Cardinality;
import com.filenet.api.constants.TypeID;
import com.filenet.api.util.Id;

public final class PropertyValueHolder 
{
    private final static String DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss'Z'";
    private final static String UTC = "UTC";
    
    private final TypeID dataType;
    private final Cardinality cardinality;
    private final boolean requiresUniqueElements;
    
    private boolean valueSpecified;
    private Object value;
    
    public static PropertyValueHolder createHolder(TypeID dataType, Cardinality cardinality, boolean requiresUniqueElements)
    {
        validateDataTypeAndCardinality(dataType, cardinality);
        return new PropertyValueHolder(dataType, cardinality, requiresUniqueElements);
    }
    
    public static boolean typeSupportsMaximumAndMinimumValues(TypeID dataType)
    {
        boolean supports = true;
        if (dataType != TypeID.LONG 
                && dataType != TypeID.DOUBLE 
                && dataType != TypeID.DATE)
        {
            supports = false;
        }
        return supports;
    }
    
    public static PropertyValueHolder createHolderForMaximumOrMinimumValue(TypeID dataType)
    {
        validateDataTypeAndCardinality(dataType, Cardinality.SINGLE);
        if (!typeSupportsMaximumAndMinimumValues(dataType)) 
        {
            throw new IllegalArgumentException();
        }
        // Use single cardinality always
        return new PropertyValueHolder(dataType, Cardinality.SINGLE, false);
    }
    
    private static void validateDataTypeAndCardinality(TypeID dataType, Cardinality cardinality)
    {
        if (dataType != TypeID.BOOLEAN 
                && dataType != TypeID.LONG 
                && dataType != TypeID.STRING 
                && dataType != TypeID.DOUBLE 
                && dataType != TypeID.DATE
                && dataType != TypeID.GUID)
        {
            throw new UnsupportedOperationException();
        }
        if (cardinality == Cardinality.ENUM)
            throw new UnsupportedOperationException();
    }
    
    private PropertyValueHolder(TypeID dataType, Cardinality cardinality, boolean requiresUniqueElements)
    {
        this.dataType = dataType;
        this.cardinality = cardinality;
        this.requiresUniqueElements = requiresUniqueElements;
    }
    
    public TypeID getDataType()
    {
        return dataType;
    }
    
    public Cardinality getCardinality()
    {
        return cardinality;
    }
    
    public boolean getRequiresUniqueElements()
    {
        return requiresUniqueElements;
    }
    
    public void setObjectValue(Object value)
    {
        if (cardinality == Cardinality.SINGLE)
        {
            this.value = convertSingleValue(value);
        }
        else
        {
            List<Object> multiVals = new ArrayList<Object>();
            if (value != null)
            {
                if (!(value instanceof List<?>))
                    throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
                List<?> inputList = (List<?>) value;
                Iterator<?> it = inputList.iterator();
                while (it.hasNext())
                {
                    Object singleVal = it.next();
                    multiVals.add(convertSingleValue(singleVal));
                }
            }
            this.value = Collections.unmodifiableList(multiVals);
        }
        valueSpecified = true;
    }
    
    private Object convertSingleValue (Object singleValue)
    {
        Object rtnValue = null;
        if (singleValue != null)
        {
            if (dataType == TypeID.BOOLEAN)
            {
                if (singleValue == null || singleValue instanceof Boolean)
                {
                    rtnValue = (Boolean)singleValue;
                }
                else if (singleValue instanceof String)
                {
                    rtnValue = Boolean.valueOf((String) singleValue);
                }
                else
                    throw new IllegalArgumentException();
            }
            else if (dataType == TypeID.LONG)
            {
                if (singleValue == null || singleValue instanceof Integer)
                {
                    rtnValue = (Integer)singleValue;
                }
                else if (singleValue instanceof Number)
                {
                    rtnValue = Integer.valueOf(((Number)singleValue).intValue());
                }
                else if (singleValue instanceof String)
                {
                    rtnValue = Integer.valueOf((String)singleValue);
                }
                else
                    throw new IllegalArgumentException();
            }
            else if (dataType == TypeID.STRING)
            {
                if (singleValue == null || singleValue instanceof String)
                    rtnValue = (String)singleValue;
                else
                    throw new IllegalArgumentException();
            }
            else if (dataType == TypeID.DOUBLE)
            {
                if (singleValue == null || singleValue instanceof Double)
                {
                    rtnValue = (Double)singleValue;
                }
                else if (singleValue instanceof Number)
                {
                    rtnValue = Double.valueOf(((Number)singleValue).doubleValue());
                }
                else if (singleValue instanceof String)
                {
                    rtnValue = Double.valueOf((String)singleValue);
                }
                else
                    throw new IllegalArgumentException();
            }
            else if (dataType == TypeID.DATE)
            {
                if (singleValue == null || singleValue instanceof Date)
                {
                    rtnValue = (Date)singleValue;
                }
                else if (singleValue instanceof String)
                {
                    Date dateVal;
                    try
                    {
                        dateVal = convertStringToDate((String) singleValue);
                    }
                    catch(ParseException e)
                    {
                        throw new IllegalArgumentException(e);
                    }
                    rtnValue = dateVal;
                }
                else
                    throw new IllegalArgumentException();
            }
            else if (dataType == TypeID.GUID)
            {
                if (singleValue == null || singleValue instanceof Id)
                {
                    rtnValue = (Id)singleValue;
                }
                else if (singleValue instanceof String)
                {
                    String strId = (String)singleValue;
                    if (Id.isId((strId)))
                    {
                        rtnValue = new Id(strId);
                    }
                    else
                        throw new IllegalArgumentException();
                }
                else
                    throw new IllegalArgumentException();
            }
            else
                throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
        }
        return rtnValue;
    }
    
    private java.util.Date convertStringToDate(String strDate) throws ParseException
    {
        SimpleDateFormat dateFormatter = new SimpleDateFormat(DATE_FORMAT);
        dateFormatter.setTimeZone(new SimpleTimeZone(0, UTC));
        java.util.Date parsedDate = dateFormatter.parse(strDate);
        return parsedDate;
    }
    
    public Object getValue()
    {
        Object val;
        if (valueSpecified)
        {
            val = value;
        }
        else
        {
            val = null;
        }
        return val;
    }
    
    /**
     * To differentiate between a null value and a value that isn't specified at all
     */
    public boolean isValueSpecified()
    {
        return valueSpecified;
    }
    
    public void unspecifyValue()
    {
        valueSpecified = false;
        value = null;
    }
    
    /**
     * This can be called for a single value or multiple values.  If a single value,
     * this will return true if the value stored in this holder is equal to the other
     * holder.  If multiple values, this will return true if any of the list of values
     * is equal to the other holder.
     */
    public boolean includes(PropertyValueHolder otherHolder)
    {
        validateOtherHolderType(otherHolder);
        // Other holder should always be a single value
        if (otherHolder.getCardinality() != Cardinality.SINGLE)
            throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
        
        boolean rtnIncludes;
        if (valueSpecified == otherHolder.valueSpecified)
        {
            if (valueSpecified)
            {
                if (cardinality == Cardinality.SINGLE)
                    rtnIncludes = equals(otherHolder);
                else
                {
                    rtnIncludes = false;
                    PropertyValueHolder itemHolder = PropertyValueHolder.createHolder(dataType, Cardinality.SINGLE, false);
                    List<?> genericList = getMultiValueList();
                    Iterator<?> it = genericList.iterator();
                    while (it.hasNext())
                    {
                        Object item = it.next();
                        itemHolder.setObjectValue(item);
                        if (otherHolder.equals(itemHolder))
                        {
                            rtnIncludes = true;
                            break;
                        }
                    }
                }
            }
            else
                rtnIncludes = true;
        }
        else
            rtnIncludes = false;
        return rtnIncludes;
    }
    
    private List<?> getMultiValueList()
    {
        if (cardinality != Cardinality.LIST)
            throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
        List<?> genericList = (List<?>) value;
        return genericList;
    }
    
    /**
     * Compare this holder to another holder.  Returns an integer value less than zero if this holder
     * is less than the other holder, using whatever comparison method is appropriate for the type
     * of property.  It returns a value of zero if the two values are equal.  It returns a value
     * greater than zero if this value is greater than the other value.
     * <p>
     * For a multi-value property, the return value is not meaningful unless the two values are equal,
     * returning zero.  So any non-zero value means the 2 multi-value lists are not equal.  A list
     * is considered equal to another list if 1) for a property value where unique elements are not
     * required, the lists have the exact same members in the same order; or 2) if unique elements
     * are required, the lists are the same size and represent the same unique elements.
     */
    public int compare(PropertyValueHolder otherHolder)
    {
        validateOtherHolderTypeAndCardinality(otherHolder);
        int rtn;
        if (valueSpecified == otherHolder.valueSpecified)
        {
            if (valueSpecified)
            {
                if (value == null && otherHolder.value == null)
                    rtn = 0;
                else if (value != null && otherHolder.value != null)
                {
                    if (cardinality == Cardinality.SINGLE)
                    {
                        rtn = singleItemsCompare(value, otherHolder.value);
                    }
                    else
                    {
                        List<?> genericValList = (List<?>)value;
                        List<?> genericOtherValList = (List<?>) otherHolder.value;
                        if (!requiresUniqueElements)
                        {
                            rtn = 0;
                            if (genericValList.size() == genericOtherValList.size())
                            {
                                for (int i = 0; i < genericValList.size(); i++)
                                {
                                    Object itemVal = genericValList.get(i);
                                    Object otherItemVal = genericOtherValList.get(i);
                                    int itemCompare = singleItemsCompare(itemVal, otherItemVal);
                                    if (itemCompare != 0)
                                    {
                                        rtn = itemCompare;
                                        break;
                                    }
                                }
                            }
                            else
                                rtn = (genericValList.size() < genericOtherValList.size() ? -1 : 1);
                        }
                        else
                        {
                            Set<Object> allVals = new HashSet<Object>();
                            allVals.addAll(genericValList);
                            Set<Object> allOtherVals = new HashSet<Object>();
                            allOtherVals.addAll(genericOtherValList);
                            if (allVals.size() == allOtherVals.size())
                            {
                                rtn = (allVals.containsAll(allOtherVals) ? 0 : 1);
                            }
                            else
                                rtn = (allVals.size() < allOtherVals.size() ? -1 : 1);
                        }
                    }
                }
                else
                    rtn = (value == null ? -1 : 1);
            }
            else
                rtn = 0;
        }
        else
            rtn = (valueSpecified ? 1 : -1);
        
        return rtn;
    }

    /**
     * Equivalent to calling compare(holder) == 0.
     */
    public boolean equals(PropertyValueHolder otherHolder)
    {
        return compare(otherHolder) == 0;
    }
    
    private int singleItemsCompare(Object item, Object otherItem)
    {
        int rtnCompare;
        if (dataType == TypeID.BOOLEAN)
            rtnCompare = ((Boolean)item).compareTo((Boolean)otherItem);
        else if (dataType == TypeID.LONG)
            rtnCompare = ((Integer)item).compareTo((Integer)otherItem);
        else if (dataType == TypeID.STRING)
            rtnCompare = ((String)item).compareTo((String)otherItem);
        else if (dataType == TypeID.DOUBLE)
            rtnCompare = ((Double)item).compareTo((Double)otherItem);
        else if (dataType == TypeID.DATE)
            rtnCompare = ((Date)item).compareTo((Date)otherItem);
        else if (dataType == TypeID.GUID)
            rtnCompare = ((Id)item).compareTo((Id)otherItem);
        else
            throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
        return rtnCompare;
    }
    
    /**
     * Only meaningful for single cardinality properties.  Returns true if this holder's value
     * is greater than or equal to the lower holder's value and less than or equal to the upper
     * holder's value.
     */
    public boolean between(PropertyValueHolder lowerValueHolder, PropertyValueHolder upperValueHolder)
    {
        int lowerCompare = compare(lowerValueHolder);
        boolean betweenVal = (lowerCompare > 0 || lowerCompare == 0);
        if (betweenVal)
        {
            int upperCompare = compare(upperValueHolder);
            betweenVal = (upperCompare < 0 || upperCompare == 0);
        }
        return betweenVal;
    }
    
    /**
     * Checks if this property value meets the minimum value constraint from another
     * holder.  The other holder is always a single cardinality property of the same
     * type.  If this holder represents a single cardinality property, this method
     * returns true if the value is greater or equal to the other holder's value.
     * If this holder represents a multi-value cardinality property, this method
     * returns true if all values in the list are greater or equal to the other
     * holder's value.  
     */
    public boolean meetsMinimumValueConstraint(PropertyValueHolder minValHolder)
    {
        return meetsMinimumOrMaximumValueConstraint(minValHolder, false);
    }
    
    /**
     * Checks if this property value meets the maximum value constraint from another
     * holder.  The other holder is always a single cardinality property of the same
     * type.  If this holder represents a single cardinality property, this method
     * returns true if the value is less than or equal to the other holder's value.
     * If this holder represents a multi-value cardinality property, this method
     * returns true if all values in the list are less than or equal to the other
     * holder's value.  
     */
    public boolean meetsMaximumValueConstraint(PropertyValueHolder maxValHolder)
    {
        return meetsMinimumOrMaximumValueConstraint(maxValHolder, true);
    }
    
    /**
     * Private helper method to combine the functionality of checking against
     * a minimum or maximum value constraint.
     */
    private boolean meetsMinimumOrMaximumValueConstraint(PropertyValueHolder minOrMaxHolder, boolean max)
    {
        if (!typeSupportsMaximumAndMinimumValues(dataType))
            throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
        validateOtherHolderType(minOrMaxHolder);
        if (minOrMaxHolder.cardinality != Cardinality.SINGLE)
            throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
        boolean meets = true;
        
        // If no value specified or it is null, then we consider it valid
        if (valueSpecified && value != null)
        {
            if (cardinality == Cardinality.SINGLE)
            {
                int compareVal = compare(minOrMaxHolder);
                meets = (compareVal == 0 
                        || (!max && compareVal > 0)
                        || (max && compareVal < 0));
            }
            else
            {
                PropertyValueHolder itemHolder = minOrMaxHolder.copy();
                List<?> multiVals = (List<?>)value;
                Iterator<?> it = multiVals.iterator();
                while (it.hasNext())
                {
                    Object item = it.next();
                    itemHolder.setObjectValue(item);
                    int compareVal = itemHolder.compare(minOrMaxHolder);
                    if ((!max && compareVal < 0) || (max && compareVal > 0))
                    {
                        meets = false;
                        break;
                    }
                }
            }
        }
        return meets;
    }
    
    /**
     * Checks if this property value meets the maximum length constraint.
     * This call is only valid for a string type property.
     * If this holder represents a single cardinality property, this method
     * returns true if the value's length is less than or equal to maximum.
     * If this holder represents a multi-value cardinality property, this method
     * returns true if all values in the list have lengths less than or equal to the 
     * maximum.  
     */
    public boolean meetsMaximumLengthConstraint(int maxLen)
    {
        if (dataType != TypeID.STRING)
            throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
        boolean meets = true;
        
        // If no value specified or it is null, then we consider it valid
        if (valueSpecified && value != null)
        {
            if (cardinality == Cardinality.SINGLE)
            {
                String singleVal = (String) value;
                meets = singleVal.length() <= maxLen;
            }
            else
            {
                List<?> multiVals = (List<?>)value;
                Iterator<?> it = multiVals.iterator();
                while (it.hasNext())
                {
                    String item = (String)it.next();
                    if (item.length() > maxLen)
                    {
                        meets = false;
                        break;
                    }
                }
            }
        }
        return meets;
    }
    
    private void validateOtherHolderTypeAndCardinality(PropertyValueHolder otherHolder)
    {
        validateOtherHolderType(otherHolder);
        validateOtherHolderCardinality(otherHolder);
    }
    
    private void validateOtherHolderType(PropertyValueHolder otherHolder)
    {
        if (otherHolder.getDataType() != getDataType())
            throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
    }
    
    private void validateOtherHolderCardinality(PropertyValueHolder otherHolder)
    {
        if (otherHolder.getCardinality() != getCardinality())
            throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
    }
    
    public PropertyValueHolder copy()
    {
        PropertyValueHolder copyHolder = new PropertyValueHolder(dataType, cardinality, requiresUniqueElements);
        copyHolder.valueSpecified = valueSpecified;
        if (value != null)
        {
            if (cardinality == Cardinality.SINGLE)
                copyHolder.value = value;
            else
            {
                List<?> listValue = (List<?>)value;
                List<Object> copyList = new ArrayList<Object>();
                copyList.addAll(listValue);
                copyHolder.value = Collections.unmodifiableList(copyList);
            }
        }
        return copyHolder;
    }
    
}
