/*
 * IBM Confidential
 * 
 * OCO Source Materials
 * 
 * 5725A15
 * 
 *  Copyright IBM Corp. 2010,2011
 * 
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 */
package com.ibm.casemgmt.sampexterndata.api;

import java.util.ArrayList;
import java.util.Deque;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.filenet.api.constants.Cardinality;
import com.filenet.api.constants.TypeID;

final class PropertySet 
{
    // Exception message strings.
    private static final String ERROR_NO_STATIC_PROPERTIES_OR_DYNAMIC_PROPERTY_SETS =
        "A property set did not specify a list of static properties or dynamic property sets.";
    private static final String ERROR_PROPERTY_REPRESENTED_MULTIPLE_TIMES =
        "The property {0} is represented more than once in a property set between the list of static properties and the list of dynamic property sets.";
    
    private final List<ExternalPropertyConfiguration> staticProps = new ArrayList<ExternalPropertyConfiguration>();
    private final List<DynamicPropertySet> dynamicPropSets = new ArrayList<DynamicPropertySet>();
    
    static PropertySet createPropertySetWithOnlyStatic(List<ExternalPropertyConfiguration> staticProps)
    {
        if (staticProps == null || staticProps.size() == 0)
            throw SEDException.createException(ERROR_NO_STATIC_PROPERTIES_OR_DYNAMIC_PROPERTY_SETS);

        return createPropertySetHelper(staticProps, null);
    }
    
    static PropertySet createPropertySetWithDynamic(List<ExternalPropertyConfiguration> staticProps,
            List<DynamicPropertySet> dynamicPropSets)
    {
        if (dynamicPropSets == null || dynamicPropSets.size() == 0)
            throw SEDException.createException(ERROR_NO_STATIC_PROPERTIES_OR_DYNAMIC_PROPERTY_SETS);
        List<ExternalPropertyConfiguration> effectiveStaticProps = staticProps;
        if (effectiveStaticProps != null && effectiveStaticProps.size() == 0)
            effectiveStaticProps = null;

        return createPropertySetHelper(effectiveStaticProps, dynamicPropSets);
    }
    
    private static PropertySet createPropertySetHelper(List<ExternalPropertyConfiguration> staticProps,
            List<DynamicPropertySet> dynamicPropSets)
    {
        Set<String> propsRepresented = new HashSet<String>();
        // Make sure static property isn't represented more than once
        if (staticProps != null)
        {
            Iterator<ExternalPropertyConfiguration> staticPropsIt = staticProps.iterator();
            while (staticPropsIt.hasNext())
            {
                ExternalPropertyConfiguration staticProp = staticPropsIt.next();
                if (propsRepresented.contains(staticProp.getSymbolicName()))
                    throw SEDException.createException(ERROR_PROPERTY_REPRESENTED_MULTIPLE_TIMES, staticProp.getSymbolicName());
                propsRepresented.add(staticProp.getSymbolicName());
            }
        }
        
        // Make sure the properties represented by each of the dynamic prop sets don't
        // overlap with the static properties or the other dynamic prop sets
        if (dynamicPropSets != null)
        {
            Iterator<DynamicPropertySet> dynamicPropSetsIt = dynamicPropSets.iterator();
            while (dynamicPropSetsIt.hasNext())
            {
                DynamicPropertySet dynamicPropSet = dynamicPropSetsIt.next();
                Set<String> dynamicPropsRepresented = dynamicPropSet.getPropertiesRepresented();
                Iterator<String> dynamicPropsIt = dynamicPropsRepresented.iterator();
                while (dynamicPropsIt.hasNext())
                {
                    String dynamicProp = dynamicPropsIt.next();
                    if (propsRepresented.contains(dynamicProp))
                        throw SEDException.createException(ERROR_PROPERTY_REPRESENTED_MULTIPLE_TIMES, dynamicProp);
                    propsRepresented.add(dynamicProp);
                }
            }
        }
        
        return new PropertySet(staticProps, dynamicPropSets);
    }
    
    private PropertySet(List<ExternalPropertyConfiguration> staticProps,
            List<DynamicPropertySet> dynamicPropSets)
    {
        if (staticProps != null)
            this.staticProps.addAll(staticProps);
        if (dynamicPropSets != null)
            this.dynamicPropSets.addAll(dynamicPropSets);
    }
    
    Set<String> getPropertiesRepresented()
    {
        Set<String> propsRepresented = new HashSet<String>();
        Iterator<ExternalPropertyConfiguration> staticPropsIt = staticProps.iterator();
        while (staticPropsIt.hasNext())
        {
            ExternalPropertyConfiguration staticProp = staticPropsIt.next();
            propsRepresented.add(staticProp.getSymbolicName());
        }
        Iterator<DynamicPropertySet> dynamicPropSetsIt = dynamicPropSets.iterator();
        while (dynamicPropSetsIt.hasNext())
        {
            DynamicPropertySet dynamicPropSet = dynamicPropSetsIt.next();
            Set<String> dynamicPropsRepresented = dynamicPropSet.getPropertiesRepresented();
            propsRepresented.addAll(dynamicPropsRepresented);
        }
        return propsRepresented;
    }
    
    TypeID getDataType(String propSymName)
    {
        TypeID dataType = null;
        ExternalPropertyConfiguration staticProp = findStaticProperty(propSymName);
        if (staticProp != null)
            dataType = staticProp.getDataType();
        if (dataType == null)
        {
            DynamicPropertySet dynPropSet = findDynamicSetRepresentingProperty(propSymName);
            if (dynPropSet != null)
                dataType = dynPropSet.getDataType(propSymName);
        }
        if (dataType == null)
            throw new IllegalArgumentException();
        return dataType;
    }
    
    Cardinality getCardinality(String propSymName)
    {
        Cardinality cardinality = null;
        ExternalPropertyConfiguration staticProp = findStaticProperty(propSymName);
        if (staticProp != null)
            cardinality = staticProp.getCardinalilty();
        if (cardinality == null)
        {
            DynamicPropertySet dynPropSet = findDynamicSetRepresentingProperty(propSymName);
            if (dynPropSet != null)
                cardinality = dynPropSet.getCardinality(propSymName);
        }
        if (cardinality == null)
            throw new IllegalArgumentException();
        return cardinality;
    }
    
    boolean getRequiresUniqueElements(String propSymName)
    {
        Boolean boolRequires = null; 
        ExternalPropertyConfiguration staticProp = findStaticProperty(propSymName);
        if (staticProp != null)
            boolRequires = Boolean.valueOf(staticProp.getRequiresUniqueElements());
        if (boolRequires == null)
        {
            DynamicPropertySet dynPropSet = findDynamicSetRepresentingProperty(propSymName);
            if (dynPropSet != null)
                boolRequires = Boolean.valueOf(dynPropSet.getRequiresUniqueElements(propSymName));
        }
        if (boolRequires == null)
            throw new IllegalArgumentException();
        return boolRequires.booleanValue();
    }
    
    private ExternalPropertyConfiguration findStaticProperty(String propSymName)
    {
        ExternalPropertyConfiguration foundStaticProp = null;
        Iterator<ExternalPropertyConfiguration> staticPropsIt = staticProps.iterator();
        while (staticPropsIt.hasNext())
        {
            ExternalPropertyConfiguration staticProp = staticPropsIt.next();
            if (staticProp.getSymbolicName().equals(propSymName))
            {
                foundStaticProp = staticProp;
                break;
            }
        }
        return foundStaticProp;
    }
    
    private DynamicPropertySet findDynamicSetRepresentingProperty(String propSymName)
    {
        DynamicPropertySet foundDynPropSet = null;
        Iterator<DynamicPropertySet> dynamicPropSetsIt = dynamicPropSets.iterator();
        while (dynamicPropSetsIt.hasNext())
        {
            DynamicPropertySet dynamicPropSet = dynamicPropSetsIt.next();
            Set<String> dynamicPropsRepresented = dynamicPropSet.getPropertiesRepresented();
            if (dynamicPropsRepresented.contains(propSymName))
            {
                foundDynPropSet = dynamicPropSet;
                break;
            }
        }
        return foundDynPropSet;
    }
    
    List<ExternalPropertyConfiguration> fetchMatchingProperties(
            Map<String, PropertyValueHolder> inputProperties, Deque<Integer> accumIdentifierIndexes)
    {
        return fetchPropertiesFromInputPropsOrIdentifier(inputProperties, accumIdentifierIndexes);
    }
    
    List<ExternalPropertyConfiguration> fetchPropertiesFromIdentifier(
            Deque<Integer> consumeIdentifierIndexes)
    {
        return fetchPropertiesFromInputPropsOrIdentifier(null, consumeIdentifierIndexes);
    }
    
    /**
     * Helper method.  If inputProperties is non-null, accumOrConsumeIdentifierIndexes
     * is taken to be the deque of indexes to accumulate when fetching the properties.
     * If inputProperties is null, accumOrConsumeIdentifierIndexes is taken to be the deque
     * of indexes to consume.
     */
    private List<ExternalPropertyConfiguration> fetchPropertiesFromInputPropsOrIdentifier(
            Map<String, PropertyValueHolder> inputProperties, Deque<Integer> accumOrConsumeIdentifierIndexes)
    {
        Set<String> propNamesSet = new HashSet<String>();
        List<ExternalPropertyConfiguration> accumProps = new ArrayList<ExternalPropertyConfiguration>();
        accumProps.addAll(staticProps);
        Iterator<DynamicPropertySet> dynamicPropSetsIt = dynamicPropSets.iterator();
        while (dynamicPropSetsIt.hasNext())
        {
            DynamicPropertySet dynamicPropSet = dynamicPropSetsIt.next();
            List<ExternalPropertyConfiguration> dynamicProps;
            if (inputProperties != null)
                dynamicProps = dynamicPropSet.fetchMatchingProperties(inputProperties, accumOrConsumeIdentifierIndexes);
            else
                dynamicProps = dynamicPropSet.fetchPropertiesFromIdentifier(accumOrConsumeIdentifierIndexes);
            Iterator<ExternalPropertyConfiguration> dynamicPropsIt = dynamicProps.iterator();
            while (dynamicPropsIt.hasNext())
            {
                ExternalPropertyConfiguration dynamicProp = dynamicPropsIt.next();
                // Sanity check.  When we checked when creating this object we should have verified
                // that there was no overlap in properties
                if (propNamesSet.contains(dynamicProp.getSymbolicName()))
                    throw SEDException.createException(ERROR_PROPERTY_REPRESENTED_MULTIPLE_TIMES, dynamicProp.getSymbolicName());
                propNamesSet.add(dynamicProp.getSymbolicName());
                accumProps.add(dynamicProp);
            }
        }
        
        return accumProps;
    }
}
