/*
 * IBM Confidential
 * 
 * OCO Source Materials
 * 
 * 5725A15
 * 
 *  Copyright IBM Corp. 2010,2011
 * 
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 */
package com.ibm.casemgmt.sampexterndata.api;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.filenet.api.constants.Cardinality;
import com.filenet.api.constants.TypeID;

public final class FileBasedExternalData 
{
	private static final Log logger = LogFactory.getLog(SEDConstants.LOGGER_SAMPLE_SERVICE_API);
			
    private static final String ERROR_NO_PATH = "No path was specified";
    private static final String ERROR_NO_MAIN_SYSTEM_ELEMENT = 
        "The file does not include the main ExternalDataSystem element";
    private static final String ERROR_FAILED_LOADING_FILE = 
        "Failed to load the external data configuration file with error {0}.";
    private static final String ERROR_UNEXPECTED_NUMBER_OF_ELEMENTS =
        "The file includes an unexpected number of {0} elements.  Expected {1} but found {2}.";
    private static final String ERROR_MISSING_ATTRIBUTE =
        "Missing attribute: {0}";
    private static final String ERROR_UNRECOGNIZED_PROPERTY_TYPE =
        "An unrecognized property type was specified: {0}";
    private static final String ERROR_UNRECOGNIZED_CARDINALITY =
        "An unrecognized cardinality was specified: {0}";
    private static final String ERROR_WRONG_NUMBER_OF_CONDITIONAL_CRITERIA_OPERATORS =
        "A conditional property set did not specific exactly one conditional criteria -- Equals, Between or Includes.";
    private static final String ERROR_MISSING_ELEMENT_VALUE =
        "Missing value for element {0}";
    private static final String ERROR_REFERENCED_PROPERTY_IDENTIFIER_MISSING =
        "A configuration for the property with identifier {0} is missing.  The property is referenced elsewhere in the file.";
    private static final String ERROR_REFERENCED_PROPERTY_SYM_NAME_MISSING =
        "A configuration for the property with symbolic name {0} is missing.  The property is referenced elsewhere in the file.";
    private static final String ERROR_REFERENCED_PROPERTY_SET_MISSING =
        "A property set with identifier {0} is missing.  The property set is referenced elsewhere in the file.";
    private static final String ERROR_IDENTIFIER_PRESENT_INLINE_PROPERTY =
        "The identifier attribute is present for an inline Property element definition."
        +"  Properties can only be referenced by identifier if they are standalone.";
    private static final String ERROR_IDENTIFIER_PRESENT_INLINE_PROPERTY_SET =
        "The identifier attribute is present for an inline PropertySet element definition."
        +"  Property sets can only be referenced by identifier if they are standalone.";
    private static final String ERROR_EXACTLY_ONE_PROPERTY_SET_OR_REF =
        "A property set can be specified by reference or an inline definition but not both.";
    private static final String ERROR_DUPLICATE_PROPERTY_IDENTIFIER =
        "A property configuration with identifier {0} is specified more than once.";
    private static final String ERROR_INCONSISTENT_CONFIGURATIONS_SAME_PROPERTY =
        "Multiple configurations for the property {0} have inconsistent data type and/or cardinality";
    private static final String ERROR_DUPLICATE_PROPERTY_SET_IDENTIFIER =
        "A property set configuration with identifier {0} is specified more than once.";
    private static final String ERROR_INVALID_ATTRIBUTE_VALUE =
        "An {0} attribute had an invalid value of {1}.";
    
    private static final String ELEMENT_EXTERNAL_DATA_SYSTEM = "ExternalDataSystem";
    private static final String ELEMENT_PROPERTIES = "Properties";
    private static final String ELEMENT_PROPERTY_SETS = "PropertySets";
    private static final String ELEMENT_CASE_TYPES = "CaseTypes";
    private static final String ELEMENT_PROPERTY = "Property";
    private static final String ATTRIBUTE_IDENTIFIER = "identifier";
    
    private static final String ELEMENT_SYMBOLIC_NAME = "SymbolicName";
    private static final String ELEMENT_PROPERTY_TYPE = "PropertyType";
    private static final String ELEMENT_CARDINALITY = "Cardinality";
    private static final String ELEMENT_REQUIRES_UNIQUE_ELEMENTS = "RequiresUniqueElements";
    private static final String ELEMENT_RENDERED_READ_ONLY_VALUE = "RenderedReadOnlyValue";
    private static final String ELEMENT_HIDDEN = "Hidden";
    private static final String ELEMENT_MAXIMUM_VALUE = "MaximumValue";
    private static final String ELEMENT_MINIMUM_VALUE = "MinimumValue";
    private static final String ELEMENT_MAXIMUM_LENGTH = "MaximumLength";
    private static final String ELEMENT_REQUIRED = "Required";
    private static final String ELEMENT_HAS_DEPENDENT_PROPERTIES = "HasDependentProperties";
    private static final String ELEMENT_CHOICE_LIST = "ChoiceList";
    private static final String ELEMENT_VALUE_IF_NEW_OR_INVALID = "ValueIfNewOrInvalid";
    private static final String ELEMENT_VALUE_IF_NEW = "ValueIfNew";
    private static final String ELEMENT_VALUE_IF_INVALID = "ValueIfInvalid";
    
    private static final String ATTRIBUTE_HANDLING = "handling";
    private static final String VALUE_HANDLING_REPLACE_VALUE = "replaceValue";
    private static final String VALUE_HANDLING_FORCE_ON_CONFIG_CHANGE = "forceOnConfigChange";
    
    private static final String VALUE_HANDLING_RETURN_MESSAGE = "returnMessage";
    private static final String VALUE_HANDLING_RETURN_MESSAGE_NO_INDIVIDUAL_ITEMS = "returnMessageNoIndividualItems";
    
    private static final String ELEMENT_DISPLAY_NAME = "DisplayName";
    private static final String ELEMENT_CHOICES = "Choices";
    private static final String ELEMENT_CHOICE = "Choice";
    private static final String ELEMENT_NAME = "Name";
    private static final String ELEMENT_VALUE = "Value";
    
    private static final String ELEMENT_PROPERTY_SET = "PropertySet";
    private static final String ELEMENT_STATIC_PROPERTIES = "StaticProperties";
    private static final String ELEMENT_PROPERTY_REF = "PropertyRef";
    
    private static final String ELEMENT_DYNAMIC_PROPERTY_SETS = "DynamicPropertySets";
    private static final String ELEMENT_DYNAMIC_PROPERTY_SET = "DynamicPropertySet";
    private static final String ELEMENT_CONDITIONAL_PROPERTY_NAME = "ConditionalPropertyName";
    private static final String ELEMENT_CONDITIONAL_PROPERTY_SETS = "ConditionalPropertySets";
    private static final String ELEMENT_CONDITIONAL_PROPERTY_SET = "ConditionalPropertySet";
    private static final String ELEMENT_EQUALS = "Equals";
    private static final String ELEMENT_BETWEEN = "Between";
    private static final String ELEMENT_INCLUDES = "Includes";
    private static final String ELEMENT_LOWER_VALUE = "LowerValue";
    private static final String ELEMENT_UPPER_VALUE = "UpperValue";
    private static final String ELEMENT_PROPERTY_SET_REF = "PropertySetRef";
    private static final String ELEMENT_DEFAULT_PROPERTY_SET_REF = "DefaultPropertySetRef";
    private static final String ELEMENT_DEFAULT_PROPERTY_SET = "DefaultPropertySet";
    
    private static final String ELEMENT_CASE_TYPE = "CaseType";
    
    private static final String PROPERTY_TYPE_STRING_VALUE = "string";
    private static final String PROPERTY_TYPE_INT_VALUE = "integer";
    private static final String PROPERTY_TYPE_ID_VALUE = "id";
    private static final String PROPERTY_TYPE_BOOLEAN_VALUE = "boolean";
    private static final String PROPERTY_TYPE_FLOAT_VALUE = "float";
    private static final String PROPERTY_TYPE_DATETIME_VALUE = "datetime";
    
    private static final String CARDINALITY_SINGLE = "single";
    private static final String CARDINALITY_MULTI = "multi";
    
    private static final String INLINE_IDENTIFIER_PREFIX = "__inline#";
    
    private final InputStream testDataFis;
    
    private Map<String,ExternalPropertyConfiguration> propertiesByIdentifierMap;
    private Map<String,ExternalPropertyConfiguration> propertiesBySymNameMap;
    private Map<String,PropertySet> propertySetMap;
    private int incrementingConfigIdentity;
    
    private enum ElementDefinitionLocality
    {
        STANDALONE,
        INLINE
    }
    
    public FileBasedExternalData(String path)
    {
        if (path == null || path.length() == 0)
            throw SEDException.createException(ERROR_NO_PATH);
        
        try 
        {
			testDataFis = new FileInputStream(path);
		} 
        catch (FileNotFoundException e)
        {
			throw new IllegalArgumentException(e);
		}
    }
    
    public FileBasedExternalData(InputStream is)
    {
    	if (is == null)
    		throw new IllegalArgumentException();
    	testDataFis = is;
    }
    
    public ExternalSystemConfiguration loadExternalSystem()
    {
        try
        {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder db = dbf.newDocumentBuilder();
            
            org.w3c.dom.Document xmlDoc = db.parse(testDataFis);
            
            Element externSystemEl = xmlDoc.getDocumentElement();
            if (!externSystemEl.getNodeName().equals(ELEMENT_EXTERNAL_DATA_SYSTEM))
                throw SEDException.createException(ERROR_NO_MAIN_SYSTEM_ELEMENT);
            
            ExternalSystemConfiguration sysConfig = processExternalDataSystemElement(externSystemEl);
            
            return sysConfig;
        }
        catch(ParserConfigurationException e)
        {
            throw SEDException.createException(ERROR_FAILED_LOADING_FILE, e, e.getLocalizedMessage());
        }
        catch(SAXException saxe)
        {
            throw SEDException.createException(ERROR_FAILED_LOADING_FILE, saxe, saxe.getLocalizedMessage());
        }
        catch(IOException ioe)
        {
            throw SEDException.createException(ERROR_FAILED_LOADING_FILE, ioe, ioe.getLocalizedMessage());
        }
        finally
        {
            safeClose(testDataFis);
        }
    }
    
    private void safeClose(InputStream is)
    {
        try
        {
            if (is != null)
                is.close();
        }
        catch(Throwable t)
        {
        	SEDLogger.warn(logger, SEDException.getEffectiveExceptionMessage(t));
        }
    }
    
    private ExternalSystemConfiguration processExternalDataSystemElement(Element externalDataSystemElement)
    {
    	//process properties
    	Element propertiesElement = getElement(externalDataSystemElement, ELEMENT_PROPERTIES);
    	if (propertiesElement != null)
    	    processPropertiesElement(propertiesElement);
    	
    	//process property sets
    	Element propertySetsElement = getElement(externalDataSystemElement, ELEMENT_PROPERTY_SETS);
    	if (propertySetsElement != null)
    	    processPropertySetsElement(propertySetsElement);
    	
    	//process case types
    	Element caseTypesElement = getRequiredElement(externalDataSystemElement, ELEMENT_CASE_TYPES);

    	List<ExternalCaseTypeConfiguration> externCaseTypes = processCaseTypesElement(caseTypesElement);
    	
    	ExternalSystemConfiguration sysconfig = ExternalSystemConfiguration.createSystem(externCaseTypes);
    
    	return sysconfig;
    }

    private void processPropertiesElement(Element propertiesElement)
    {
    	List<Element> propertyNodeList = getChildElementsByTagName(propertiesElement, ELEMENT_PROPERTY);
    	
    	propertiesByIdentifierMap = new HashMap<String, ExternalPropertyConfiguration>();
    	propertiesBySymNameMap = new HashMap<String, ExternalPropertyConfiguration>();
    	
    	for (int i=0; i<propertyNodeList.size();i++)
    	{
    		Element propertyElement = propertyNodeList.get(i);
    		processPropertyElement(propertyElement, ElementDefinitionLocality.STANDALONE);
    	}
    }
    
    /**
     * Process a Property element that could show up either inline or standalone under the
     * Properties element.  If standalone the identifier attribute must be specified since
     * its purpose is to reference elsewhere.  If inline, the identifier attribute must not
     * be specified.  Currently we don't allow a property that's defined inline to be referenced
     * elsewhere in the file.
     * <p>
     * @return The identifier used to reference this element, either the one defined as an
     *         attribute or one automatically generated for an inline definition.
     */
    private String processPropertyElement(Element propertyElement, ElementDefinitionLocality defLocality)
    {
        int configIdentity = allocateConfigIdentity();
        String strIdentifier;
        if (defLocality == ElementDefinitionLocality.STANDALONE)
        {
            strIdentifier = getRequiredAttribute(propertyElement, ATTRIBUTE_IDENTIFIER); 
        }
        else
        {
            String assumedMissing = getAttribute(propertyElement, ATTRIBUTE_IDENTIFIER);
            if (assumedMissing != null && assumedMissing.length() > 0)
                throw SEDException.createException(ERROR_IDENTIFIER_PRESENT_INLINE_PROPERTY);
            strIdentifier = INLINE_IDENTIFIER_PREFIX + configIdentity;
        }
        
        //required elements
        Element symbolicNameElement = getRequiredElement(propertyElement, ELEMENT_SYMBOLIC_NAME); 
        String symbolicName = getRequiredElementValue(symbolicNameElement);

        Element propertyTypeElement = getRequiredElement(propertyElement, ELEMENT_PROPERTY_TYPE); 
        String strPropertyType = getRequiredElementValue(propertyTypeElement);
        TypeID propertyType = getTypeIDfromPropertyType(strPropertyType);

        Element cardinalityElement = getRequiredElement(propertyElement, ELEMENT_CARDINALITY); 
        String strCardinality = getRequiredElementValue(cardinalityElement);
        Cardinality cardinality = getCardinalityFromString(strCardinality);

        String hasDependentProperties = null;
        Element hasDependentPropertiesElement = getElement(propertyElement, ELEMENT_HAS_DEPENDENT_PROPERTIES);
        if (hasDependentPropertiesElement != null)
            hasDependentProperties = getElementValue(hasDependentPropertiesElement);

        //optional elements
        boolean requiresUniqueElements = false;
        if (cardinality == Cardinality.LIST)
        {
            Element requiresUniqueElemsElement = getElement(propertyElement, ELEMENT_REQUIRES_UNIQUE_ELEMENTS);
            if (requiresUniqueElemsElement != null)
            {
                String strRequires = getElementValue(requiresUniqueElemsElement);
                if (strRequires != null)
                    requiresUniqueElements = Boolean.valueOf(strRequires);
            }
        }
        
        Object renderedReadOnlyValue = null;
        boolean renderedReadOnlyValueSpecified = false;
        Element renderedReadOnlyValueElement = getElement(propertyElement, ELEMENT_RENDERED_READ_ONLY_VALUE);
        if (renderedReadOnlyValueElement != null)
        {
            renderedReadOnlyValue = getElementValueOfCardinality(renderedReadOnlyValueElement, cardinality);
            renderedReadOnlyValueSpecified = true;
        }
        
        Boolean hidden = null;
        Element hiddenElement = getElement(propertyElement, ELEMENT_HIDDEN);
        if (hiddenElement != null)
        {
            String strHidden = getElementValue(hiddenElement);
            if (strHidden != null && strHidden.length() > 0)
                hidden = Boolean.valueOf(strHidden);
        }

        String maximumValue = null;
        Element maximumValueElement = getElement(propertyElement, ELEMENT_MAXIMUM_VALUE);
        if (maximumValueElement != null)
        {
            maximumValue = getElementValue(maximumValueElement);
        }

        String minimumValue = null;
        Element minimumValueElement = getElement(propertyElement, ELEMENT_MINIMUM_VALUE);
        if (minimumValueElement != null)
        {
            minimumValue = getElementValue(minimumValueElement);
        }

        String maximumLength = null;
        Element maximumLengthElement = getElement(propertyElement, ELEMENT_MAXIMUM_LENGTH);
        if (maximumLengthElement != null)
        {
            maximumLength = getElementValue(maximumLengthElement);
        }

        String required = null;
        Element requiredElement = getElement(propertyElement, ELEMENT_REQUIRED);
        if (requiredElement != null)
        {
            required = getElementValue(requiredElement);
        }
        
        ExternalChoiceList choiceList = null;
        Element choiceListElement = getElement(propertyElement, ELEMENT_CHOICE_LIST);
        if (choiceListElement != null)
        {
            choiceList = processChoiceListElement(choiceListElement,propertyType);
        }

        Object valueIfNew = null;
        boolean valueIfNewSpecified = false;
        ExternalPropertyConfiguration.ValueIfInvalidHandling valueIfInvalidHandling = ExternalPropertyConfiguration.ValueIfInvalidHandling.RETURN_MESSAGE;
        Object valueIfInvalid = null;
        
        Element valueIfNewOrInvalidElement = getElement(propertyElement, ELEMENT_VALUE_IF_NEW_OR_INVALID);
        if (valueIfNewOrInvalidElement != null)
        {
            valueIfNew = getElementValueOfCardinality(valueIfNewOrInvalidElement, cardinality);
            valueIfNewSpecified = true;
            valueIfInvalid = valueIfNew;
            valueIfInvalidHandling = ExternalPropertyConfiguration.ValueIfInvalidHandling.REPLACE_VALUE;
        }
        
        // Let specific ValueIfNew override 
        Element valueIfNewElement = getElement(propertyElement, ELEMENT_VALUE_IF_NEW);
        if (valueIfNewElement != null)
        {
            valueIfNew = getElementValueOfCardinality(valueIfNewElement, cardinality);
            valueIfNewSpecified = true;
        }
        
        // Let specific valueIfInvalid override
        Element valueIfInvalidElement = getElement(propertyElement, ELEMENT_VALUE_IF_INVALID);
        if (valueIfInvalidElement != null)
        {
            valueIfInvalidHandling = ExternalPropertyConfiguration.ValueIfInvalidHandling.RETURN_MESSAGE;
            String handlingVal = getAttribute(valueIfInvalidElement, ATTRIBUTE_HANDLING);
            if (handlingVal != null)
            {
                if (handlingVal.equals(VALUE_HANDLING_RETURN_MESSAGE))
                    valueIfInvalidHandling = ExternalPropertyConfiguration.ValueIfInvalidHandling.RETURN_MESSAGE;
                else if (handlingVal.equals(VALUE_HANDLING_RETURN_MESSAGE_NO_INDIVIDUAL_ITEMS))
                    valueIfInvalidHandling = ExternalPropertyConfiguration.ValueIfInvalidHandling.RETURN_MESSAGE_NO_INDIVIDUAL_ITEMS;
                else if (handlingVal.equals(VALUE_HANDLING_REPLACE_VALUE))
                    valueIfInvalidHandling = ExternalPropertyConfiguration.ValueIfInvalidHandling.REPLACE_VALUE;
                else if (handlingVal.equals(VALUE_HANDLING_FORCE_ON_CONFIG_CHANGE))
                    valueIfInvalidHandling = ExternalPropertyConfiguration.ValueIfInvalidHandling.FORCE_ON_CONFIG_CHANGE;
                else
                    throw SEDException.createException(ERROR_INVALID_ATTRIBUTE_VALUE, ATTRIBUTE_HANDLING, handlingVal);
            }
            if (valueIfInvalidHandling == ExternalPropertyConfiguration.ValueIfInvalidHandling.REPLACE_VALUE)
            {
                valueIfInvalid = getElementValueOfCardinality(valueIfInvalidElement, cardinality);
            }
            if (valueIfInvalidHandling == ExternalPropertyConfiguration.ValueIfInvalidHandling.FORCE_ON_CONFIG_CHANGE)
            {
            	valueIfInvalid = getElementValueOfCardinality(valueIfInvalidElement, cardinality);
            }
            else
            {
            	valueIfInvalid = null;
            }
        }
        
        Integer intMaxLen = (maximumLength!=null?Integer.valueOf(maximumLength):null);
        Boolean boolRequired = required!=null?Boolean.valueOf(required):null;
        boolean boolHasDeps = hasDependentProperties!=null?Boolean.valueOf(hasDependentProperties):false; 
        ExternalPropertyConfiguration prop = ExternalPropertyConfiguration.createProperty(
                configIdentity, 
                symbolicName, 
                propertyType, 
                cardinality, 
                requiresUniqueElements,
                renderedReadOnlyValue, 
                renderedReadOnlyValueSpecified, 
                hidden,
                maximumValue, 
                minimumValue, 
                intMaxLen, 
                boolRequired, 
                boolHasDeps, 
                choiceList, 
                valueIfNew, 
                valueIfNewSpecified,
                valueIfInvalidHandling,
                valueIfInvalid);
        
        if (propertiesByIdentifierMap.containsKey(strIdentifier))
            throw SEDException.createException(ERROR_DUPLICATE_PROPERTY_IDENTIFIER, strIdentifier);
        propertiesByIdentifierMap.put(strIdentifier,prop);
        if (propertiesBySymNameMap.containsKey(symbolicName))
        {
            ExternalPropertyConfiguration otherProp = propertiesBySymNameMap.get(symbolicName);
            if (otherProp.getDataType() != prop.getDataType()
                    || otherProp.getCardinalilty() != prop.getCardinalilty()
                    || otherProp.getRequiresUniqueElements() != prop.getRequiresUniqueElements())
            {
                throw SEDException.createException(ERROR_INCONSISTENT_CONFIGURATIONS_SAME_PROPERTY, symbolicName);
            }
        }
        else
            propertiesBySymNameMap.put(symbolicName, prop);
        
        return strIdentifier;
    }
    
    /**
     * Get optional element.  If present, there can only be a single element
     * with the specified name.
     */
    private Element getElement(Element parentElem, String elemName)
    {
        Element elem = null;
        List<Element> childElems = getChildElementsByTagName(parentElem, elemName);
        if (childElems.size() > 0)
        {
            if (childElems.size() != 1)
                throw SEDException.createException(ERROR_UNEXPECTED_NUMBER_OF_ELEMENTS, elemName, 1, childElems.size());
            elem = childElems.get(0); 
        }
        return elem;
    }
    
    private List<Element> getChildElementsByTagName(Element parentElem, String elemName)
    {
        NodeList childNodes = parentElem.getChildNodes();
        List<Element> childElems = new ArrayList<Element>();
        for (int i = 0; i < childNodes.getLength(); i++)
        {
            Node node = childNodes.item(i);
            if (node instanceof Element)
            {
                Element elem = (Element) node;
                if (elem.getTagName().equals(elemName))
                    childElems.add(elem);
            }
        }
        return childElems;
    }
    
    private Element getRequiredElement(Element parentElem, String elemName)
    {
        Element elem = getElement(parentElem, elemName);
        if (elem == null)
            throw SEDException.createException(ERROR_UNEXPECTED_NUMBER_OF_ELEMENTS, elemName, 1, 0);
        return elem;
    }
    
    private String getAttribute(Element elem, String attrName)
    {
        String attrValue = elem.getAttribute(attrName);
        if (attrValue != null && attrValue.length() == 0)
            attrValue = null;
        return attrValue;
    }
    
    private String getRequiredAttribute(Element elem, String attrName)
    {
        String attrValue = elem.getAttribute(attrName);
        if (attrValue == null || attrValue.length() == 0)
            throw SEDException.createException(ERROR_MISSING_ATTRIBUTE, attrName);
        return attrValue;
    }
    
    private TypeID getTypeIDfromPropertyType(String propertyType)
    {
    	TypeID dataType = TypeID.STRING;
    	if (propertyType.equalsIgnoreCase(PROPERTY_TYPE_INT_VALUE))
    		dataType = TypeID.LONG;
    	else if (propertyType.equalsIgnoreCase(PROPERTY_TYPE_STRING_VALUE))
    		dataType = TypeID.STRING;
    	else if (propertyType.equalsIgnoreCase(PROPERTY_TYPE_ID_VALUE))
    		dataType = TypeID.GUID;
    	else if (propertyType.equalsIgnoreCase(PROPERTY_TYPE_BOOLEAN_VALUE))
    		dataType = TypeID.BOOLEAN;
    	else if (propertyType.equalsIgnoreCase(PROPERTY_TYPE_FLOAT_VALUE))
    		dataType = TypeID.DOUBLE;
    	else if (propertyType.equalsIgnoreCase(PROPERTY_TYPE_DATETIME_VALUE))
    		dataType = TypeID.DATE;
    	else
    		throw SEDException.createException(ERROR_UNRECOGNIZED_PROPERTY_TYPE, propertyType);
    	return dataType;
    }
    
    private Cardinality getCardinalityFromString(String cardinality)
    {
    	if (cardinality.equalsIgnoreCase(CARDINALITY_SINGLE))
    		return Cardinality.SINGLE;
    	else if (cardinality.equalsIgnoreCase(CARDINALITY_MULTI))
    		return Cardinality.LIST;
    	else
    		throw SEDException.createException(ERROR_UNRECOGNIZED_CARDINALITY, cardinality);
    }
    
    private ExternalChoiceList processChoiceListElement(Element choiceListElement, TypeID dataType)
    {
        Element displayNameElem = getRequiredElement(choiceListElement, ELEMENT_DISPLAY_NAME);
    	String displayName = getRequiredElementValue(displayNameElem); 
    	Element choicesElement = getRequiredElement(choiceListElement, ELEMENT_CHOICES); 
    	List<Element> choiceList = getChildElementsByTagName(choicesElement, ELEMENT_CHOICE);
    	List<ExternalChoice> choices = new ArrayList<ExternalChoice>(); 
    	for (int i=0;i<choiceList.size();i++)
    	{
    		ExternalChoice externalChoice = null;
    		Element choice = choiceList.get(i);
    		Element nameElem = getRequiredElement(choice, ELEMENT_NAME);
    		String name = getRequiredElementValue(nameElem);
    		Element valueElem = getRequiredElement(choice, ELEMENT_VALUE);
    		String value = getRequiredElementValue(valueElem); 
    		if (dataType == TypeID.LONG)
    			externalChoice = ExternalChoice.createSingleIntegerChoice(name, Integer.valueOf(value));
    		else if (dataType == TypeID.STRING)
    			externalChoice = ExternalChoice.createSingleStringChoice(name, value);
    		choices.add(externalChoice);
    	}
    	
    	return ExternalChoiceList.createChoiceList(displayName, dataType, choices);
    }
    
    
    private void processPropertySetsElement(Element propertySetsElement)
    {
    	List<Element> propertySetNodeList = getChildElementsByTagName(propertySetsElement, ELEMENT_PROPERTY_SET);
    	
    	propertySetMap = new HashMap<String, PropertySet>();
    	
    	for (int i=0; i<propertySetNodeList.size();i++)
    	{
    		Element propertySetElement = propertySetNodeList.get(i);
    		
    		processPropertySetElement(propertySetElement, ElementDefinitionLocality.STANDALONE);
    	}
    }
    
    /**
     * Process a PropertySet element that could show up either inline or standalone under the
     * PropertySets element.  If standalone the identifier attribute must be specified since
     * its purpose is to reference elsewhere.  If inline, the identifier attribute must not
     * be specified.  Currently we don't allow a property set that's defined inline to be referenced
     * elsewhere in the file.
     * <p>
     * @return The identifier used to reference this element, either the one defined as an
     *         attribute or one automatically generated for an inline definition.
     */
    private String processPropertySetElement(Element propertySetElement, ElementDefinitionLocality defLocality)
    {
        String strIdentifier;
        if (defLocality == ElementDefinitionLocality.STANDALONE)
        {
            strIdentifier = getRequiredAttribute(propertySetElement, ATTRIBUTE_IDENTIFIER); 
        }
        else
        {
            String assumedMissing = getAttribute(propertySetElement, ATTRIBUTE_IDENTIFIER);
            if (assumedMissing != null && assumedMissing.length() > 0)
                throw SEDException.createException(ERROR_IDENTIFIER_PRESENT_INLINE_PROPERTY_SET);
            strIdentifier = INLINE_IDENTIFIER_PREFIX + allocateConfigIdentity();
        }
        List<ExternalPropertyConfiguration> staticPropertyList = processStaticProperties(propertySetElement);           

        List<DynamicPropertySet> dynamicPropertyList = processDynamicProperties(propertySetElement);            
        
        PropertySet propertySet;
        if (dynamicPropertyList != null && dynamicPropertyList.size()>0)
            propertySet = PropertySet.createPropertySetWithDynamic(staticPropertyList, dynamicPropertyList);
        else
            propertySet = PropertySet.createPropertySetWithOnlyStatic(staticPropertyList);
        
        if (propertySetMap.containsKey(strIdentifier))
            throw SEDException.createException(ERROR_DUPLICATE_PROPERTY_SET_IDENTIFIER, strIdentifier);
        propertySetMap.put(strIdentifier, propertySet);
        return strIdentifier;
    }
    
    
    private List<ExternalPropertyConfiguration> processStaticProperties(Element propertySetElement)
    {
    	List<ExternalPropertyConfiguration> staticPropertyList = new ArrayList<ExternalPropertyConfiguration>();
		
    	Element staticPropertiesElement = getElement(propertySetElement, ELEMENT_STATIC_PROPERTIES);
    	if (staticPropertiesElement != null)
		{
    	    List<Element> inlinePropertyList = getChildElementsByTagName(staticPropertiesElement, ELEMENT_PROPERTY);
    	    int i;
    	    for (i = 0; i < inlinePropertyList.size(); i++)
    	    {
    	        Element propElem = inlinePropertyList.get(i);
    	        String ident = processPropertyElement(propElem, ElementDefinitionLocality.INLINE);
    	        ExternalPropertyConfiguration prop = getRequiredReferencedPropertyByIdentifier(ident);
    	        staticPropertyList.add(prop);
    	    }
        	List<Element> propertyRefList = getChildElementsByTagName(staticPropertiesElement, ELEMENT_PROPERTY_REF);
    		for (i=0;i<propertyRefList.size();i++)
    		{
    			String propertyRefId = getRequiredAttribute(propertyRefList.get(i), ATTRIBUTE_IDENTIFIER); 
    			ExternalPropertyConfiguration prop = getRequiredReferencedPropertyByIdentifier(propertyRefId); 
    			staticPropertyList.add(prop);
    		}
		}

		return staticPropertyList;
	}
    
    private ExternalPropertyConfiguration getRequiredReferencedPropertyByIdentifier(String identifier)
    {
        ExternalPropertyConfiguration prop = propertiesByIdentifierMap.get(identifier);
        if (prop == null)
            throw SEDException.createException(ERROR_REFERENCED_PROPERTY_IDENTIFIER_MISSING, identifier);
        return prop;
    }

    private List<DynamicPropertySet> processDynamicProperties(Element propertySetElement)
    {
    	List<DynamicPropertySet> dynamicPropertyList = new ArrayList<DynamicPropertySet>();
		
    	Element dynamicPropertySetsElement = getElement(propertySetElement, ELEMENT_DYNAMIC_PROPERTY_SETS);
    	if (dynamicPropertySetsElement != null)
		{
    		List<Element> dynamicPropertysetList = getChildElementsByTagName(dynamicPropertySetsElement, ELEMENT_DYNAMIC_PROPERTY_SET);
    		for (int i=0;i<dynamicPropertysetList.size();i++)
    		{
    			Element dynamicPropertySetElement = dynamicPropertysetList.get(i);
    			DynamicPropertySet dynamicPropertySet = processDynamicPropertySet(dynamicPropertySetElement);
    			dynamicPropertyList.add(dynamicPropertySet);
    		}
		}

		return dynamicPropertyList;
	}
    
    private DynamicPropertySet processDynamicPropertySet(Element dynamicPropertySetElement)
    {
        Element conditionalPropertyNameElem = getRequiredElement(dynamicPropertySetElement, ELEMENT_CONDITIONAL_PROPERTY_NAME);
    	String conditionalPropertyName = getRequiredElementValue(conditionalPropertyNameElem);

    	Element defaultPropertySetRefElement = getElement(dynamicPropertySetElement, ELEMENT_DEFAULT_PROPERTY_SET_REF);
    	Element defaultPropertySetElement = getElement(dynamicPropertySetElement, ELEMENT_DEFAULT_PROPERTY_SET);
    	if (defaultPropertySetRefElement != null && defaultPropertySetElement != null)
            throw SEDException.createException(ERROR_EXACTLY_ONE_PROPERTY_SET_OR_REF);
    	String strIdentifier;
    	if (defaultPropertySetRefElement != null)
    	{
    	    strIdentifier = getRequiredAttribute(defaultPropertySetRefElement, ATTRIBUTE_IDENTIFIER);
    	}
    	else if (defaultPropertySetElement != null)
    	{
    	    strIdentifier = processPropertySetElement(defaultPropertySetElement, ElementDefinitionLocality.INLINE);
    	}
    	else
    	    throw SEDException.createException(ERROR_EXACTLY_ONE_PROPERTY_SET_OR_REF);
    	PropertySet defaultPropertySet = getRequiredReferencedPropertySet(strIdentifier); 
    	
    	Element conditionalPropertySetsElement = getRequiredElement(dynamicPropertySetElement, ELEMENT_CONDITIONAL_PROPERTY_SETS);
    	ExternalPropertyConfiguration prop = getRequiredReferencedPropertyBySymName(conditionalPropertyName);
    	TypeID dataType = prop.getDataType();
    	Cardinality cardinality = prop.getCardinalilty();
    	boolean requiresUniqueElements = prop.getRequiresUniqueElements();
    	List<ConditionalPropertySet> conditionalPropertySetList = processConditionalPropertySet(
    	        conditionalPropertySetsElement,dataType,cardinality, requiresUniqueElements);
    	
    	return DynamicPropertySet.createDynamicPropertySet(conditionalPropertyName, conditionalPropertySetList, defaultPropertySet);
    }
    
    private PropertySet getRequiredReferencedPropertySet(String identifier)
    {
        PropertySet propSet = propertySetMap.get(identifier);
        if (propSet == null)
            throw SEDException.createException(ERROR_REFERENCED_PROPERTY_SET_MISSING, identifier);
        return propSet;
    }
    
    private ExternalPropertyConfiguration getRequiredReferencedPropertyBySymName(String symName)
    {
        ExternalPropertyConfiguration prop = propertiesBySymNameMap.get(symName);
        if (prop == null)
            throw SEDException.createException(ERROR_REFERENCED_PROPERTY_SYM_NAME_MISSING, symName);
        return prop;
    }
    
    
    private List<ConditionalPropertySet> processConditionalPropertySet (
            Element conditionalPropertySetsElement,TypeID dataType, Cardinality cardinality,
            boolean requiresUniqueElements)
    {
    	List<ConditionalPropertySet> conditionalPropertySetList = new ArrayList<ConditionalPropertySet>();
    	
    	List<Element> conditionalPropertySetNodeList = getChildElementsByTagName(conditionalPropertySetsElement, ELEMENT_CONDITIONAL_PROPERTY_SET);
    	for (int i=0; i<conditionalPropertySetNodeList.size();i++)
    	{
    		Element conditionalPropertySetElement = conditionalPropertySetNodeList.get(i);
    		
    		ConditionalCriteria criteria;
    		
    		// Equals, for a single or muti-value property
    		int criteriaCount = 0;
    		Element equalsElement = getElement(conditionalPropertySetElement, ELEMENT_EQUALS);
    		if (equalsElement != null)
    		    criteriaCount++;
            // Between, only a single-value property
    		Element betweenElement = getElement(conditionalPropertySetElement, ELEMENT_BETWEEN);
    		if (betweenElement != null)
    		    criteriaCount++;
            // Includes, only a multi-value property
    		Element includesElement = getElement(conditionalPropertySetElement, ELEMENT_INCLUDES);
    		if (includesElement != null)
    		    criteriaCount++;
    		if (criteriaCount != 1)
                throw SEDException.createException(ERROR_WRONG_NUMBER_OF_CONDITIONAL_CRITERIA_OPERATORS);
    		
    		if (equalsElement != null)
    		{
    			Object equalsValue = getElementValueOfCardinality(equalsElement, cardinality);
    			PropertyValueHolder pvh = PropertyValueHolder.createHolder(dataType, cardinality, requiresUniqueElements);
    			pvh.setObjectValue(equalsValue);
    			criteria = ConditionalCriteria.createEqualsCriteria(pvh);
    		}
    		// Between, only a single-value property
    		else if (betweenElement != null)
    		{
    		    Element lowerValueElem = getRequiredElement(betweenElement, ELEMENT_LOWER_VALUE);
    			String lowerValue = getRequiredElementValue(lowerValueElem);
    			Element upperValueElem = getRequiredElement(betweenElement, ELEMENT_UPPER_VALUE);
    			String upperValue = getRequiredElementValue(upperValueElem); 
    			PropertyValueHolder lowerValHolder = PropertyValueHolder.createHolder(dataType, Cardinality.SINGLE, false);
    			PropertyValueHolder upperValHolder = PropertyValueHolder.createHolder(dataType, Cardinality.SINGLE, false);
    			lowerValHolder.setObjectValue(lowerValue);
    			upperValHolder.setObjectValue(upperValue);
    			criteria = ConditionalCriteria.createBetweenCriteria(lowerValHolder, upperValHolder);
    		}
    		// Includes, only a multi-value property
            else if (includesElement != null)
            {
                String incValue = getElementValue(includesElement);
                PropertyValueHolder incValHolder = PropertyValueHolder.createHolder(dataType, Cardinality.SINGLE, false);
                incValHolder.setObjectValue(incValue);
                criteria = ConditionalCriteria.createIncludesCriteria(incValHolder);
            }
            else
                throw SEDException.createException(ERROR_WRONG_NUMBER_OF_CONDITIONAL_CRITERIA_OPERATORS);
    		
    		Element propertySetRefElement = getElement(conditionalPropertySetElement, ELEMENT_PROPERTY_SET_REF);
    		Element propertySetElement = getElement(conditionalPropertySetElement, ELEMENT_PROPERTY_SET);
    		if (propertySetRefElement != null && propertySetElement != null)
    		    throw SEDException.createException(ERROR_EXACTLY_ONE_PROPERTY_SET_OR_REF);
    		String strIdentifier;
    		if (propertySetRefElement != null)
    		{
    		    strIdentifier = getRequiredAttribute(propertySetRefElement, ATTRIBUTE_IDENTIFIER);
    		}
    		else if (propertySetElement != null)
    		{
    		    strIdentifier = processPropertySetElement(propertySetElement, ElementDefinitionLocality.INLINE);
    		}
    		else
    		    throw SEDException.createException(ERROR_EXACTLY_ONE_PROPERTY_SET_OR_REF);
    		PropertySet propSet = getRequiredReferencedPropertySet(strIdentifier); 
    		
    		ConditionalPropertySet conditionalPropertySet = new ConditionalPropertySet(criteria, propSet);
    		conditionalPropertySetList.add(conditionalPropertySet);	
    	}
    	
    	return conditionalPropertySetList;
    }
    
    private List<ExternalCaseTypeConfiguration> processCaseTypesElement(Element caseTypesElement)
    {
    	List<ExternalCaseTypeConfiguration> caseTypeList = new ArrayList<ExternalCaseTypeConfiguration>();
    	List<Element> caseTypeNodeList = getChildElementsByTagName(caseTypesElement, ELEMENT_CASE_TYPE);
    	for (int i=0; i<caseTypeNodeList.size();i++)
    	{
    		Element caseTypeElement = caseTypeNodeList.get(i);
    		Element nameElem = getRequiredElement(caseTypeElement, ELEMENT_NAME);
    		String name = getRequiredElementValue(nameElem); 
    		Element propertySetRefElement = getElement(caseTypeElement, ELEMENT_PROPERTY_SET_REF);
    		Element propertySetElement = getElement(caseTypeElement, ELEMENT_PROPERTY_SET);
            if (propertySetRefElement != null && propertySetElement != null)
                throw SEDException.createException(ERROR_EXACTLY_ONE_PROPERTY_SET_OR_REF);
            String strIdentifier;
            if (propertySetRefElement != null)
            {
                strIdentifier = getRequiredAttribute(propertySetRefElement, ATTRIBUTE_IDENTIFIER);
            }
            else if (propertySetElement != null)
            {
                strIdentifier = processPropertySetElement(propertySetElement, ElementDefinitionLocality.INLINE);
            }
            else
                throw SEDException.createException(ERROR_EXACTLY_ONE_PROPERTY_SET_OR_REF);
    		PropertySet propertySet = getRequiredReferencedPropertySet(strIdentifier); 
    		ExternalCaseTypeConfiguration caseType = new ExternalCaseTypeConfiguration(name, propertySet);
    		caseTypeList.add(caseType);
    	}
    	
    	return caseTypeList;
    	
    }
    
    private String getElementValue(Element element)
    {
    	String value = null;
    	
    	if (element.getChildNodes()!=null && element.getChildNodes().getLength()>0)
    	{
    		if (element.getChildNodes().item(0).getNodeType() == Node.TEXT_NODE)
    			value = element.getChildNodes().item(0).getNodeValue();
    	}
    	
    	return value;
    }
    
    private String getRequiredElementValue(Element element)
    {
        String value = getElementValue(element);
        if (value == null || value.length() == 0)
            throw SEDException.createException(SEDException.formatErrorMessage(ERROR_MISSING_ELEMENT_VALUE, element.getNodeName()));
        return value;
    }
    
    /**
     * Get either a single value or a list of values depending on the cardinality.  If a
     * single cardinality property, this call just calls getElementValue(Element).  
     * <p>
     * If a multiple
     * cardinality property, if the element contains Value sub-elements, getElementValue(Element)
     * is called on each of the Value sub-elements and a list is returned.  If the element contains
     * no Value sub-elements, getElementValue(Element) is called on this element itself and if
     * any content is returned, a 1 item list is returned with that value.  If there is no
     * content, an empty list is returned.
     * <p>
     * The values returned, either a single value or a list of values, are each String objects.
     * This class relies on the downstream objects (i.e. PropertyValueHolder) to convert from
     * String to the appropriate type. 
     */
    private Object getElementValueOfCardinality(Element element, Cardinality cardinality)
    {
        Object elemValue;
        if (cardinality == Cardinality.SINGLE)
        {
            elemValue = getElementValue(element);
        }
        else if (cardinality == Cardinality.LIST)
        {
            List<Object> multiVals = new ArrayList<Object>();
            List<Element> subElems = getChildElementsByTagName(element, ELEMENT_VALUE);
            if (subElems.size() > 0)
            {
                for (int i = 0; i < subElems.size(); i++)
                {
                    Element subElem = subElems.get(i);
                    multiVals.add(getElementValue(subElem));
                }
            }
            else
            {
                String singleVal = getElementValue(element);
                if (singleVal != null)
                    multiVals.add(singleVal);
            }
            elemValue = multiVals;
        }
        else
            // Shouldn't happen
            throw SEDException.createException(ERROR_UNRECOGNIZED_CARDINALITY, cardinality.toString());
        return elemValue;
    }
    
    private int allocateConfigIdentity()
    {
        return ++incrementingConfigIdentity;
    }
        
}
