/*
 * IBM Confidential
 * 
 * OCO Source Materials
 * 
 * 5725A15
 * 
 *  Copyright IBM Corp. 2010,2011
 * 
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 */
package com.ibm.casemgmt.sampexterndata.api;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.filenet.api.constants.Cardinality;
import com.filenet.api.constants.TypeID;

public final class ExternalSystemConfiguration 
{
    private static final String ERROR_NO_CASE_TYPES =
        "The system configuration does not specify any case types.";
    private static final String ERROR_DUPLICATE_CASE_TYPE =
        "The case type {0} is specified more than once in the system configuration.";
    private static final String ERROR_CIRCULAR_DEPENDENCIES_DETECTED =
        "Some circular dependencies exist in the system configuration.  Unable to determine a stable list of property configurations.";
    private static final String ERROR_INCONSISTENT_PROPERTY_SET_BETWEEN_CALLS =
        "The system did not respond with a consistent set of properties between two different calls.  "
        + "The property {0} was not returned in a previous call.";
    
    private static final String VALIDATION_ERROR_SINGLE_VALUE_MINIMUM =
        "The value is less than the externally specified minimum value.";
    private static final String VALIDATION_ERROR_SINGLE_VALUE_MAXIMUM =
        "The value is greater than the externally specified maximum value.";
    private static final String VALIDATION_ERROR_SINGLE_VALUE_MAXLEN =
        "The length is greater than the externally specified maximum length.";
    private static final String VALIDATION_ERROR_SINGLE_VALUE_CHOICE_LIST =
        "The value is not a member of the externally specified choice list.";
    private static final String VALIDATION_ERROR_MULTI_VALUES_MINIMUM =
        "One or more values are less than the externally specified minimum value.";
    private static final String VALIDATION_ERROR_MULTI_VALUES_MAXIMUM =
        "One or more values are greater than the externally specified maximum value.";
    private static final String VALIDATION_ERROR_MULTI_VALUES_MAXLEN =
        "One or more values have lengths greater than the externally specified maximum length.";
    private static final String VALIDATION_ERROR_MULTI_VALUES_CHOICE_LIST =
        "One or more values are not members of the externally specified choice list.";
    private static final String VALIDATION_ERROR_MULTI_VALUES_VARYING =
        "There are varying externally specified constraint violations with multiple values.";
    private static final String VALIDATION_ERROR_MULTI_VALUES_INCLUDE_ITEMS_POSTFIX =
        "  The values include: ";
    private static final String COMMA = ",";
    
    private final Map<String, ExternalCaseTypeConfiguration> externCaseTypesMap;
    
    public static class FetchExternalPropertiesData
    {
        final private List<ExternalProperty> externProps;
        final private String opaqueIdentifier;
        
        private FetchExternalPropertiesData(List<ExternalProperty> externProps, String opaqueIdentifier)
        {
            this.externProps = externProps;
            this.opaqueIdentifier = opaqueIdentifier;
        }
        
        public List<ExternalProperty> getExternalProperties()
        {
            return externProps;
        }
        
        public String getOpaqueIdentifer()
        {
            return opaqueIdentifier;
        }
    }
    
    static ExternalSystemConfiguration createSystem(List<ExternalCaseTypeConfiguration> externCaseTypes)
    {
        if (externCaseTypes == null || externCaseTypes.size() == 0)
            throw SEDException.createException(ERROR_NO_CASE_TYPES);
        Map<String, ExternalCaseTypeConfiguration> externCaseTypesMap = 
            new HashMap<String, ExternalCaseTypeConfiguration>();
        Iterator<ExternalCaseTypeConfiguration> it = externCaseTypes.iterator();
        while (it.hasNext())
        {
            ExternalCaseTypeConfiguration externCaseType = it.next();
            if (externCaseTypesMap.containsKey(externCaseType.getCaseTypeName()))
                throw SEDException.createException(ERROR_DUPLICATE_CASE_TYPE, externCaseType.getCaseTypeName());
            externCaseTypesMap.put(externCaseType.getCaseTypeName(), externCaseType);
        }
        return new ExternalSystemConfiguration(externCaseTypesMap);
    }
    
    private ExternalSystemConfiguration(Map<String, ExternalCaseTypeConfiguration> externCaseTypesMap)
    {
        this.externCaseTypesMap = externCaseTypesMap;
    }
    
    public FetchExternalPropertiesData fetchExternalProperties(
            String caseTypeName, 
            ExternalRequestMode requestMode,
            List<InputProperty> currentProperties,
            String currentOpaqueIdentifier)
    {
        List<String> currentPropNames = new ArrayList<String>();
        Map<String, PropertyInfo> workingPropsMap = new HashMap<String, PropertyInfo>();
        Iterator<InputProperty> inPropsIt = currentProperties.iterator();
        while (inPropsIt.hasNext())
        {
            InputProperty inProp = inPropsIt.next();
            PropertyValueHolder holder = PropertyValueHolder.createHolder(
                    inProp.getPropertyType(), inProp.getCardinality(), inProp.getRequiresUniqueElements());
            if (inProp.isValueSpecified())
                holder.setObjectValue(inProp.getValue());
            workingPropsMap.put(inProp.getSymbolicName(), new PropertyInfo(inProp.getSymbolicName(), holder));
            currentPropNames.add(inProp.getSymbolicName());
        }
        
        Deque<Integer> consumeCurrentIdentifierIndexes = null;
        if (requestMode == ExternalRequestMode.IN_PROGRESS_CHANGES && currentOpaqueIdentifier != null)
            consumeCurrentIdentifierIndexes = identifierIndexesFromOpaqueString(currentOpaqueIdentifier);
        Deque<Integer> accumNewIdentifierIndexes = new LinkedList<Integer>();

        List<ExternalProperty> externProps = 
            fetchExternalPropertiesFromConfigurableData(
                caseTypeName, requestMode, currentPropNames, workingPropsMap, 
                consumeCurrentIdentifierIndexes, accumNewIdentifierIndexes);
        
        String newOpaqueIdentifier = opaqueIdentifierStringFromIndexes(accumNewIdentifierIndexes);
        return new FetchExternalPropertiesData(externProps, newOpaqueIdentifier);
    }
    
    private Deque<Integer> identifierIndexesFromOpaqueString(String opaqueIdentifier)
    {
        Deque<Integer> identifierIndexes = new LinkedList<Integer>();
        if (opaqueIdentifier.length() > 0)
        {
            String[] strIndexes = opaqueIdentifier.split(COMMA);
            for (int i = 0; i < strIndexes.length; i++)
            {
                identifierIndexes.addLast(Integer.valueOf(strIndexes[i]));
            }
        }
        return identifierIndexes;
    }
    
    private String opaqueIdentifierStringFromIndexes(Deque<Integer> identifierIndexes)
    {
        StringBuffer sb = new StringBuffer();
        Iterator<Integer> it = identifierIndexes.iterator();
        while (it.hasNext())
        {
            Integer index = it.next();
            if (sb.length() > 0)
                sb.append(COMMA);
            sb.append(index.intValue());
        }
        return sb.toString();
    }
    
    private List<ExternalProperty> fetchExternalPropertiesFromConfigurableData(
            String caseTypeName,
            ExternalRequestMode requestMode,
            List<String> currentPropNames,
            Map<String, PropertyInfo> workingPropsMap,
            Deque<Integer> consumeCurrentIdentifierIndexes,
            Deque<Integer> accumNewIdentifierIndexes)
    {
        List<ExternalProperty> externProps;
        ExternalCaseTypeConfiguration externConfigCaseType = externCaseTypesMap.get(caseTypeName);
        if (externConfigCaseType == null)
        {
            externProps = new ArrayList<ExternalProperty>();
        }
        else
        {
            PropertySet propSet = externConfigCaseType.getPropertySet();
            externProps = fetchExternalPropertiesFromPropertySet(
                    requestMode, propSet,
                    currentPropNames, workingPropsMap, 
                    consumeCurrentIdentifierIndexes, accumNewIdentifierIndexes);
        }
        return externProps;
    }
    
    private List<ExternalProperty> fetchExternalPropertiesFromPropertySet(
            ExternalRequestMode requestMode,
            PropertySet propSet,
            List<String> currentPropNames,
            Map<String, PropertyInfo> workingPropsMap,
            Deque<Integer> consumeCurrentIdentifierIndexes,
            Deque<Integer> accumNewIdentifierIndexes)
    {
        Map<String, ExternalPropertyConfiguration> baseConfigPropsMap = null;
        
        int i = 0;
        
        if (consumeCurrentIdentifierIndexes != null)
        {
            baseConfigPropsMap = new HashMap<String, ExternalPropertyConfiguration>();
            List<ExternalPropertyConfiguration> baseExternProps = propSet.fetchPropertiesFromIdentifier(consumeCurrentIdentifierIndexes);
            Iterator<ExternalPropertyConfiguration> it = baseExternProps.iterator();
            while (it.hasNext())
            {
                ExternalPropertyConfiguration configProp = it.next();
                baseConfigPropsMap.put(configProp.getSymbolicName(), configProp);
            }
        }
        
        fetchPropertiesFromPropertySetWithFixupsIfNecessary(
                requestMode, propSet, currentPropNames, baseConfigPropsMap, 
                workingPropsMap, accumNewIdentifierIndexes);
        
        // Collect copies of all the ExternalProperty objects from the 
        // ConfigurableExternalProperty objects
        Map<String, ExternalProperty> externPropCopies = new HashMap<String, ExternalProperty>();
        Iterator<Map.Entry<String, PropertyInfo>> workingPropsIt = workingPropsMap.entrySet().iterator();
        while (workingPropsIt.hasNext())
        {
            Map.Entry<String, PropertyInfo> entry = workingPropsIt.next();
            PropertyInfo propInfo = entry.getValue();
            if (propInfo.getConfigurationProperty() != null)
                externPropCopies.put(propInfo.getConfigurationProperty().getSymbolicName(), propInfo.getConfigurationProperty().getExternalPropertyCopy());
        }
        
        // Set the values for the modified properties into the
        // ExternalProperty objects.
        Iterator<Map.Entry<String, ExternalProperty>> externPropsIt = externPropCopies.entrySet().iterator();
        while (externPropsIt.hasNext())
        {
            Map.Entry<String, ExternalProperty> entry = externPropsIt.next();
            ExternalProperty externProp = entry.getValue();
            PropertyInfo propInfo = workingPropsMap.get(entry.getKey());
            if (propInfo.getModifiedValue() != null)
            {
                externProp.setObjectValue(propInfo.getModifiedValue().getValue());
            }
            if (propInfo.getValidationError() != null)
            {
                String msg = propInfo.getValidationError().getErrorMessage();
                externProp.setCustomValidationError(msg);
                if (propInfo.getConfigurationProperty().getValueIfInvalidHandling() != ExternalPropertyConfiguration.ValueIfInvalidHandling.RETURN_MESSAGE_NO_INDIVIDUAL_ITEMS)
                {
                    List<Integer> invalidItems = propInfo.getValidationError().getMultiValueItems();
                    if (invalidItems != null && invalidItems.size() > 0)
                        externProp.setCustomInvalidItems(invalidItems);
                }
            }
        }
        
        // Now, if only returning the differences, determine
        // those differences based on whether the property has been modified or whether the
        // specific identifier differs from the base set.  Then include those differences
        // in the returned set.
        // If not returning differences, include all the ExternalProperty objects, in the 
        // same order as they were passed to us
        List<ExternalProperty> rtnProps = new ArrayList<ExternalProperty>();
        Iterator<String> currentPropsIt = currentPropNames.iterator();
        while (currentPropsIt.hasNext())
        {
            String propName = currentPropsIt.next();
            PropertyInfo propInfo = workingPropsMap.get(propName);
            // Continue if we don't have a configuration for this property
            if (propInfo.getConfigurationProperty() == null)
                continue;
            
            ExternalProperty externProp = externPropCopies.get(propName); 

            addExternalPropertyToReturnPropertiesIfAppropriate(externProp, baseConfigPropsMap, propInfo, rtnProps);
        }
        
        return rtnProps;
    }
    
    /**
     * Fetch the properties from the top PropertySet object.  May require fixups to current values
     * and multiple calls to the PropertySet object.
     * <p>
     * Some of the passed in collections are modified upon return.
     * <p>
     * @param workingPropsMap  If any property values need to be modified, the values are updated
     *                         in the map.
     */
    private void fetchPropertiesFromPropertySetWithFixupsIfNecessary(
            ExternalRequestMode requestMode,
            PropertySet topPropertySet,
            List<String> currentPropNames,
            Map<String, ExternalPropertyConfiguration> baseConfigProps,
            Map<String, PropertyInfo> workingPropsMap,
            Deque<Integer> accumNewIdentifierIndexes)
    {
        int propsFetchCount = 0;
        boolean refetchRequired = false;
        do
        {
            // If refetches required, clear out to accumulate new indexes
            accumNewIdentifierIndexes.clear();
            
            Map<String, PropertyValueHolder> workingValsMap = new HashMap<String, PropertyValueHolder>();
            Iterator<String> propNamesIt = workingPropsMap.keySet().iterator();
            while (propNamesIt.hasNext())
            {
                String propName = propNamesIt.next();
                PropertyValueHolder workingVal = workingPropsMap.get(propName).getEffectiveValue();
                if (workingVal != null)
                    workingValsMap.put(propName, workingVal);
            }
            List<ExternalPropertyConfiguration> configProps = topPropertySet.fetchMatchingProperties(
                    workingValsMap, accumNewIdentifierIndexes);

            refetchRequired = checkWorkingPropertiesAgainstConfigurations(
                    requestMode, currentPropNames, baseConfigProps, configProps, workingPropsMap);
            propsFetchCount++;
        } while (refetchRequired && propsFetchCount < 1000);
        // We seemed to fall into an infinite loop
        if (refetchRequired)
            throw SEDException.createException(ERROR_CIRCULAR_DEPENDENCIES_DETECTED);
    }
    
    /**
     * Add the ExternalProperty from the given ConfigurationProperty to the list of properties
     * to return.  If baseConfigPropsMap is not null it means we are determining the differences
     * based on some initial stable set of configuration properties.  The PropertyInfo structure
     * tells us whether the value has been modified.
     */
    private void addExternalPropertyToReturnPropertiesIfAppropriate(
            ExternalProperty externProp,
            Map<String, ExternalPropertyConfiguration> baseConfigPropsMap,
            PropertyInfo propInfo,
            List<ExternalProperty> rtnProps)
    {
        boolean include = true;
        if (baseConfigPropsMap != null)
        {
            String propName = propInfo.getPropertyName();
            include = false;
            ExternalPropertyConfiguration baseConfigProp = baseConfigPropsMap.get(propName);
            // This is a configuration error.  We should return the same set of properties
            // each time.
            if (baseConfigProp == null)
                throw SEDException.createException(ERROR_INCONSISTENT_PROPERTY_SET_BETWEEN_CALLS, propName);
            if (propInfo.getModifiedValue() != null
                    || propInfo.getValidationError() != null
                    || baseConfigProp.getIdentifier() != propInfo.getConfigurationProperty().getIdentifier())
            {
                include = true;
            }
        }
        if (include)
            rtnProps.add(externProp);
    }
    
    private boolean checkWorkingPropertiesAgainstConfigurations(
            ExternalRequestMode requestMode,
            List<String> currentPropNames,
            Map<String, ExternalPropertyConfiguration> baseConfigProps,
            List<ExternalPropertyConfiguration> configProps,
            Map<String, PropertyInfo> workingPropsMap)
    {
        Map<String, ExternalPropertyConfiguration> configPropsMap = 
            new HashMap<String, ExternalPropertyConfiguration>();
        Iterator<ExternalPropertyConfiguration> configPropsIt = configProps.iterator();
        while (configPropsIt.hasNext())
        {
            ExternalPropertyConfiguration configProp = configPropsIt.next();
            configPropsMap.put(configProp.getSymbolicName(), configProp);
        }
        boolean newPropertyConfigurationsRequired = false;
        for (int iname = 0; iname < currentPropNames.size(); iname++)
        {
            String propName = currentPropNames.get(iname);
            ExternalPropertyConfiguration configProp = configPropsMap.get(propName);
            // If no configuration property for this property, nothing to check
            if (configProp == null)
                continue;
            
            PropertyInfo propInfo = workingPropsMap.get(propName);
            // We already checked this property before and the configuration has not changed,
            // so go on to the next.
            if (propInfo.getConfigurationProperty() != null && propInfo.getConfigurationProperty().getIdentifier() == configProp.getIdentifier())
                continue;
            
            // If value is currently modified and the current configuration has dependents,
            // then refetch is required since we are reverting to original value.
            if (propInfo.getModifiedValue() != null && propInfo.getConfigurationProperty().hasDependentProperties())
                newPropertyConfigurationsRequired = true;
            
            propInfo.setConfigurationProperty(configProp);
            
            // Check various cases if we need to modify the current value
            
            boolean valueHandled = false;
            // Check rendered read-only value first
            if (configProp.isRenderedReadOnlyValueSpecified())
            {
                valueHandled = true;
                PropertyValueHolder readOnlyValHolder = propInfo.getInputValue().copy();
                readOnlyValHolder.setObjectValue(configProp.getRenderedReadOnlyValue());
                if (!propInfo.getInputValue().equals(readOnlyValHolder))
                {
                    propInfo.setModifiedValue(readOnlyValHolder);
                }
            }

            // Check new object value
            if (!valueHandled && requestMode == ExternalRequestMode.INITIAL_NEW_OBJECT && configProp.isValueIfNewSpecified())
            {
                valueHandled = true;
                PropertyValueHolder newValHolder = propInfo.getInputValue().copy();
                newValHolder.setObjectValue(configProp.getValueIfNew());
                
                if (!propInfo.getInputValue().equals(newValHolder))
                {
                    propInfo.setModifiedValue(newValHolder);
                }
            }

            if (!valueHandled && requestMode == ExternalRequestMode.IN_PROGRESS_CHANGES && 
            	(configProp.getValueIfInvalidHandling() == ExternalPropertyConfiguration.ValueIfInvalidHandling.FORCE_ON_CONFIG_CHANGE) &&
            	(baseConfigProps != null))
            
            {
            	ExternalPropertyConfiguration epc = baseConfigProps.get(configProp.getSymbolicName());
            	if (epc.getIdentifier() != configProp.getIdentifier())
            	{
                    valueHandled = true;
                    PropertyValueHolder forceOnConfigChange = propInfo.getInputValue().copy();
                    forceOnConfigChange.setObjectValue(configProp.getValueIfInvalid());
                    
                    if (!propInfo.getInputValue().equals(forceOnConfigChange))
                    {
                        propInfo.setModifiedValue(forceOnConfigChange);
                    }
            	}
            }

			// Check if current value is valid based on other constraints 
            if (!valueHandled && propInfo.getInputValue().isValueSpecified())
            {
                PropertyValueHolder copyHolder = propInfo.getInputValue().copy();
                List<Integer> collectMultiValuesWithViolations = null;
                // Collect the invalid multi-value items even if we aren't returning
                // the individual items in the payload.  If returning a custom error
                // message we need the multi-value items to format the message.
                if (configProp.getCardinalilty() == Cardinality.LIST 
                        && (configProp.getValueIfInvalidHandling() == ExternalPropertyConfiguration.ValueIfInvalidHandling.RETURN_MESSAGE
                                || configProp.getValueIfInvalidHandling() == ExternalPropertyConfiguration.ValueIfInvalidHandling.RETURN_MESSAGE_NO_INDIVIDUAL_ITEMS))
                {
                    collectMultiValuesWithViolations = new ArrayList<Integer>();
                }

                ConstraintViolationType violation = configProp.removeValuesViolatingConstraints(
                        copyHolder, collectMultiValuesWithViolations);
                if (violation != null)
                {
                    if ((configProp.getValueIfInvalidHandling() == ExternalPropertyConfiguration.ValueIfInvalidHandling.REPLACE_VALUE) ||
               		   (configProp.getValueIfInvalidHandling() == ExternalPropertyConfiguration.ValueIfInvalidHandling.FORCE_ON_CONFIG_CHANGE))
                    {
                        // Value may only be partially modified if a multi-value
                        if (!copyHolder.isValueSpecified())
                        {
                            copyHolder.setObjectValue(configProp.getValueIfInvalid());
                        }
                        // If for some reason the value isn't any different from the
                        // input value, leave it alone.
                        // This may happen if the value specified in the config
                        // file is itself invalid (null for example)
                        if (!propInfo.getInputValue().equals(copyHolder))
                            propInfo.setModifiedValue(copyHolder);
                    }
                    
                    else
                    {
                        String validationMessage = formatValidationErrorMessage(
                                violation, propInfo.getInputValue(), collectMultiValuesWithViolations);
                        if (validationMessage != null && validationMessage.length() > 0)
                            propInfo.setValidationError(new ValidationErrorInfo(validationMessage, collectMultiValuesWithViolations));
                    }
                }
            }
            
            if (propInfo.getModifiedValue() != null && configProp.hasDependentProperties())
            {
                newPropertyConfigurationsRequired = true;
            }
            
            if (newPropertyConfigurationsRequired)
                break;
        }
        return newPropertyConfigurationsRequired;
    }
    
    private String formatValidationErrorMessage(ConstraintViolationType violation, 
            PropertyValueHolder valueHolder, List<Integer> multiValuesWithViolations)
    {
        StringBuffer sb = new StringBuffer();
        boolean appendMultiValues = false;
        switch(violation)
        {
        case MINIMUM_VALUE:
            if (multiValuesWithViolations == null)
                sb.append(VALIDATION_ERROR_SINGLE_VALUE_MINIMUM);
            else
            {
                sb.append(VALIDATION_ERROR_MULTI_VALUES_MINIMUM);
                appendMultiValues = true;
            }
            break;
        case MAXIMUM_VALUE:
            if (multiValuesWithViolations == null)
                sb.append(VALIDATION_ERROR_SINGLE_VALUE_MAXIMUM);
            else
            {
                sb.append(VALIDATION_ERROR_MULTI_VALUES_MAXIMUM);
                appendMultiValues = true;
            }
            break;
        case MAXIMUM_LENGTH:
            if (multiValuesWithViolations == null)
                sb.append(VALIDATION_ERROR_SINGLE_VALUE_MAXLEN);
            else
            {
                sb.append(VALIDATION_ERROR_MULTI_VALUES_MAXLEN);
                appendMultiValues = true;
            }
            break;
        case CHOICE_LIST:
            if (multiValuesWithViolations == null)
                sb.append(VALIDATION_ERROR_SINGLE_VALUE_CHOICE_LIST);
            else
            {
                sb.append(VALIDATION_ERROR_MULTI_VALUES_CHOICE_LIST);
                appendMultiValues = true;
            }
            break;
        case REQUIRED_VALUE:
            // Ignore this one, an error message will not be returned
            break;
        case VARYING_VIOLATIONS_MULTI_VALUES:
            sb.append(VALIDATION_ERROR_MULTI_VALUES_VARYING);
            appendMultiValues = true;
            break;
        default:
            // Shouldn't happen
            throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
        }
        
        if (appendMultiValues)
        {
            List<?> multiVals = (List<?>) valueHolder.getValue();
            sb.append(VALIDATION_ERROR_MULTI_VALUES_INCLUDE_ITEMS_POSTFIX);
            // Max of 3 to append to end of message
            for (int i = 0; i < multiValuesWithViolations.size() && i < 3; i++)
            {
                Integer idx = multiValuesWithViolations.get(i);
                Object val = multiVals.get(idx.intValue());
                if (i > 0)
                    sb.append(COMMA);
                sb.append(val.toString());
            }
        }
        
        return sb.toString();
    }

	public Set<String> getCaseTypesRepresented() 
	{
	    Set<String> typesRepresented = new HashSet<String>();
	    typesRepresented.addAll(externCaseTypesMap.keySet());
	    return typesRepresented;
	}
	
	public Set<String> getPropertiesRepresented(String caseTypeName)
	{
	    ExternalCaseTypeConfiguration caseTypeConfig = externCaseTypesMap.get(caseTypeName);
	    if (caseTypeConfig == null)
	        throw new IllegalArgumentException();
	    return caseTypeConfig.getPropertySet().getPropertiesRepresented();
	}
	
	public TypeID getDataType(String caseTypeName, String propSymName)
	{
        ExternalCaseTypeConfiguration caseTypeConfig = externCaseTypesMap.get(caseTypeName);
        if (caseTypeConfig == null)
            throw new IllegalArgumentException();
        return caseTypeConfig.getPropertySet().getDataType(propSymName);
	}
	
	public Cardinality getCardinality(String caseTypeName, String propSymName)
	{
        ExternalCaseTypeConfiguration caseTypeConfig = externCaseTypesMap.get(caseTypeName);
        if (caseTypeConfig == null)
            throw new IllegalArgumentException();
        return caseTypeConfig.getPropertySet().getCardinality(propSymName);
	}
	
	public boolean getRequiresUniqueElements(String caseTypeName, String propSymName)
	{
        ExternalCaseTypeConfiguration caseTypeConfig = externCaseTypesMap.get(caseTypeName);
        if (caseTypeConfig == null)
            throw new IllegalArgumentException();
        return caseTypeConfig.getPropertySet().getRequiresUniqueElements(propSymName);
	}
	
	private static class ValidationErrorInfo
	{
	    private final String errorMsg;
        private final List<Integer> multiValueItems;
	    
	    ValidationErrorInfo(String errorMsg, List<Integer> multiValueItems)
	    {
	        this.errorMsg = errorMsg;
	        if (multiValueItems != null)
	            this.multiValueItems = Collections.unmodifiableList(multiValueItems);
	        else
	            this.multiValueItems = null;
	    }
	    
	    String getErrorMessage()
	    {
	        return errorMsg;
	    }
	    
	    List<Integer> getMultiValueItems()
	    {
	        return multiValueItems;
	    }
	}
    
	private static class PropertyInfo
	{
	    private final String propName;
	    private final PropertyValueHolder inputValue;
	    
	    private PropertyValueHolder modifiedValue;
	    private ExternalPropertyConfiguration configProp;
	    private ValidationErrorInfo validationError;
	    
	    PropertyInfo(String propName, PropertyValueHolder inputValue)
	    {
	        this.propName = propName;
	        this.inputValue = inputValue;
	    }
	    
	    String getPropertyName()
	    {
	        return propName;
	    }
	    
	    PropertyValueHolder getInputValue()
	    {
	        return inputValue;
	    }
	    
	    void setConfigurationProperty(ExternalPropertyConfiguration configProp)
	    {
	        this.configProp = configProp;
	        // Reset modified state when set a new configuration prop
	        modifiedValue = null;
	        validationError = null;
	    }
	    
	    ExternalPropertyConfiguration getConfigurationProperty()
	    {
	        return configProp;
	    }
	    
	    PropertyValueHolder getEffectiveValue()
	    {
	        PropertyValueHolder effectiveValue;
	        if (modifiedValue != null)
	            effectiveValue = modifiedValue;
	        else
	            effectiveValue = inputValue;
	        return effectiveValue;
	    }
	    
	    PropertyValueHolder getModifiedValue()
	    {
	        return modifiedValue;
	    }
	    
	    void setModifiedValue(PropertyValueHolder modifiedValue)
	    {
	        this.modifiedValue = modifiedValue;
	    }
	    
	    void setValidationError(ValidationErrorInfo validationError)
	    {
	        this.validationError = validationError;
	    }
	    
	    ValidationErrorInfo getValidationError()
	    {
	        return validationError;
	    }
	}
	
}
