/*
 * IBM Confidential
 * 
 * OCO Source Materials
 * 
 * 5725A15
 * 
 *  Copyright IBM Corp. 2010,2011
 * 
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 */
package com.ibm.casemgmt.sampexterndata.api;

import java.util.List;

import com.filenet.api.constants.Cardinality;
import com.filenet.api.constants.TypeID;

final class ExternalPropertyConfiguration 
{
    private final int identifier;
    private final ExternalProperty externalProperty;
    private final PropertyValueHolder valueIfNew;
    private final PropertyValueHolder valueIfInvalid;
    private final ValueIfInvalidHandling valueIfInvalidHandling;
    
    /**
     * Indicates the type of handling to perform if a property value is
     * determined to be invalid based on other external data constraints.
     */
    enum ValueIfInvalidHandling
    {
        /**
         * The default handling, leave the value as-is and return a
         * custom validation error message for this property.
         * If this is a multi-value property, also return the
         * indices of the individual items that are invalid.
         */
        RETURN_MESSAGE,
        /**
         * Return a custom validation error message but for a multi-value
         * property, don't return the individual invalid items.
         */
        RETURN_MESSAGE_NO_INDIVIDUAL_ITEMS,
        /**
         * Replace the value with the specified value.
         */
        REPLACE_VALUE,
        /**
         * Replace the value from a different property set
         */
        FORCE_ON_CONFIG_CHANGE
        
    }
    
    static ExternalPropertyConfiguration createProperty(int identifier, String propName, 
            TypeID propType, Cardinality cardinality, boolean requiresUniqueElements,
            Object renderedReadOnlyValue, boolean renderedReadOnlyValueSpecified,
            Boolean hidden,
            Object maximumValue, Object minimumValue, Integer maxLength, 
            Boolean required, boolean hasDependentProperties,
            ExternalChoiceList choiceList,
            Object valueIfNew, boolean valueIfNewSpecified,
            ValueIfInvalidHandling valueIfInvalidHandling, 
            Object valueIfInvalid)
    {
        ExternalProperty extProp = ExternalProperty.createProperty(propName, propType, cardinality, requiresUniqueElements);
        if (renderedReadOnlyValueSpecified)
        {
            extProp.setObjectValue(renderedReadOnlyValue);
            extProp.setDisplayMode(DisplayMode.READ_ONLY);
        }
        if (hidden != null)
        {
            extProp.setHidden(hidden.booleanValue());
        }
        if (maximumValue != null)
        {
            extProp.setMaximumObjectValue(maximumValue);
        }
        if (minimumValue != null)
        {
            extProp.setMinimumObjectValue(minimumValue);
        }
        if (required != null)
        {
            extProp.setRequired(required.booleanValue());
        }
        if (maxLength != null)
        {
            extProp.setMaximumLength(maxLength.intValue());
        }
        extProp.setHasDependentProperties(hasDependentProperties);
        if (choiceList != null)
        {
            extProp.setChoiceList(choiceList);
        }
        
        PropertyValueHolder valueIfNewHolder = null;
        if (valueIfNewSpecified)
        {
            valueIfNewHolder = PropertyValueHolder.createHolder(propType, cardinality, requiresUniqueElements);
            valueIfNewHolder.setObjectValue(valueIfNew);
        }
        
        PropertyValueHolder valueIfInvalidHolder = null;
        if ((valueIfInvalidHandling == ValueIfInvalidHandling.REPLACE_VALUE) ||
            (valueIfInvalidHandling == ValueIfInvalidHandling.FORCE_ON_CONFIG_CHANGE))
        {
            valueIfInvalidHolder = PropertyValueHolder.createHolder(propType, cardinality, requiresUniqueElements);
            valueIfInvalidHolder.setObjectValue(valueIfInvalid);
        }
        
        return new ExternalPropertyConfiguration(identifier, extProp, 
        										 valueIfNewHolder, valueIfInvalidHolder, 
        										 valueIfInvalidHandling);
    }
    
    private ExternalPropertyConfiguration(
    		int identifier, 
            ExternalProperty externalProperty, 
            PropertyValueHolder valueIfNew,
            PropertyValueHolder valueIfInvalid,
            ValueIfInvalidHandling valueIfInvalidHandling)
    {
        this.identifier = identifier;
        this.externalProperty = externalProperty;
        this.valueIfNew = valueIfNew;
        this.valueIfInvalid = valueIfInvalid;
        this.valueIfInvalidHandling = valueIfInvalidHandling;
    }
    
    int getIdentifier()
    {
        return identifier;
    }
    
    String getSymbolicName()
    {
        return externalProperty.getSymbolicName();
    }
    
    TypeID getDataType()
    {
        return externalProperty.getDataType();
    }
    
    Cardinality getCardinalilty()
    {
        return externalProperty.getCardinality();
    }
    
    boolean getRequiresUniqueElements()
    {
        return externalProperty.getRequiresUniqueElements();
    }
    
    ExternalProperty getExternalPropertyCopy()
    {
        return externalProperty.copy();
    }
    
    ConstraintViolationType removeValuesViolatingConstraints(
            PropertyValueHolder holder,
            List<Integer> collectMultiValuesWithViolations)
    {
        return externalProperty.removeValuesViolatingConstraints(
                holder, collectMultiValuesWithViolations);
    }
    
    boolean isRenderedReadOnlyValueSpecified()
    {
        return (externalProperty.getDisplayMode() == DisplayMode.READ_ONLY);
    }
    
    Object getRenderedReadOnlyValue()
    {
        Object val = null;
        if (externalProperty.getDisplayMode() == DisplayMode.READ_ONLY)
            val = externalProperty.getValue();
        return val;
    }
    
    boolean isValueIfNewSpecified()
    {
        return (valueIfNew != null);
    }
    
    Object getValueIfNew()
    {
        Object val = null;
        if (valueIfNew != null)
            val = valueIfNew.getValue();
        return val;
    }
    
    ValueIfInvalidHandling getValueIfInvalidHandling()
    {
        return valueIfInvalidHandling;
    }
    
    Object getValueIfInvalid()
    {
        Object val = null;
        if (valueIfInvalid != null)
            val = valueIfInvalid.getValue();
        return val;
    }
    
    boolean hasDependentProperties()
    {
        return externalProperty.getHasDependentProperties().booleanValue();
    }
    
    ExternalPropertyConfiguration copy()
    {
        PropertyValueHolder copyValueIfNew = null;
        if (valueIfNew != null)
            copyValueIfNew = valueIfNew.copy();
        PropertyValueHolder copyValueIfInvalid = null;
        if (valueIfInvalid != null)
            copyValueIfInvalid = valueIfInvalid.copy();
        
        return new ExternalPropertyConfiguration(identifier, externalProperty.copy(), 
        		copyValueIfNew, copyValueIfInvalid, 
        		valueIfInvalidHandling);
    }
}
