/*
 * IBM Confidential
 * 
 * OCO Source Materials
 * 
 * 5725A15
 * 
 *  Copyright IBM Corp. 2010,2011
 * 
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 */
package com.ibm.casemgmt.sampexterndata.api;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.filenet.api.constants.Cardinality;
import com.filenet.api.constants.TypeID;

public final class ExternalProperty 
{
    private final String symbolicName;
    private final PropertyValueHolder valueHolder;
    private final PropertyValueHolder maximumValueHolder;
    private final PropertyValueHolder minimumValueHolder;
    
    private String customValidationError;
    private List<Integer> customInvalidItems;
    private DisplayMode displayMode;
    private Boolean hidden;
    private Boolean required;
    private Integer maxLength;
    private Boolean hasDependentProperties;
    private ExternalChoiceList choiceList;
    
    public static Map<String, ExternalProperty> getMap(List<ExternalProperty> externProps)
    {
        Map<String, ExternalProperty> externPropsMap = new HashMap<String, ExternalProperty>();
        Iterator<ExternalProperty> externPropsIt = externProps.iterator();
        while (externPropsIt.hasNext())
        {
            ExternalProperty externProp = externPropsIt.next();
            externPropsMap.put(externProp.getSymbolicName(), externProp);
        }
        return externPropsMap;
    }
    
    public static List<ExternalProperty> copyList(List<ExternalProperty> externProps)
    {
        List<ExternalProperty> copyExternProps = new ArrayList<ExternalProperty>();
        Iterator<ExternalProperty> it = externProps.iterator();
        while (it.hasNext())
        {
            ExternalProperty externProp = it.next();
            copyExternProps.add(externProp.copy());
        }
        return copyExternProps;
    }
    
    static ExternalProperty createProperty(String symbolicName, TypeID dataType, Cardinality cardinality, boolean requiresUniqueElements)
    {
        PropertyValueHolder holder = PropertyValueHolder.createHolder(dataType, cardinality, requiresUniqueElements);
        PropertyValueHolder maxHolder = null;
        PropertyValueHolder minHolder = null;
        if (PropertyValueHolder.typeSupportsMaximumAndMinimumValues(dataType))
        {
            maxHolder = PropertyValueHolder.createHolderForMaximumOrMinimumValue(dataType);
            minHolder = PropertyValueHolder.createHolderForMaximumOrMinimumValue(dataType);
        }
        return new ExternalProperty(symbolicName, holder, maxHolder, minHolder);
    }
    
    private ExternalProperty(String symbolicName, 
            PropertyValueHolder holder, 
            PropertyValueHolder maximumHolder,
            PropertyValueHolder minimumHolder)
    {
        this.symbolicName = symbolicName;
        valueHolder = holder;
        maximumValueHolder = maximumHolder;
        minimumValueHolder = minimumHolder;
    }
    
    public String getSymbolicName()
    {
        return symbolicName;
    }
    
    public TypeID getDataType()
    {
        return valueHolder.getDataType();
    }
    
    public Cardinality getCardinality()
    {
        return valueHolder.getCardinality();
    }
    
    public boolean getRequiresUniqueElements()
    {
        return valueHolder.getRequiresUniqueElements();
    }
    
    public Object getValue()
    {
        return valueHolder.getValue();
    }
    
    /**
     * To differentiate between a null value and a value that isn't specified at all
     */
    public boolean isValueSpecified()
    {
        return valueHolder.isValueSpecified();
    }
    
    void setObjectValue(Object value)
    {
        valueHolder.setObjectValue(value);
    }
    
    /**
     * Returns null if no custom validation error has been specified
     */
    public String getCustomValidationError()
    {
        return customValidationError;
    }
    
    void setCustomValidationError(String customValidationError)
    {
        this.customValidationError = customValidationError;
    }
    
    /**
     * Returns null if individual invalid items of a multi-value
     * property were not specified or multi-value items are not
     * applicable.
     */
    public List<Integer> getCustomInvalidItems()
    {
        return customInvalidItems;
    }
    
    void setCustomInvalidItems(List<Integer> invalidItems)
    {
        customInvalidItems = Collections.unmodifiableList(invalidItems);
    }
    
    /**
     * Returns null if display mode is not specified
     */
    public DisplayMode getDisplayMode()
    {
        return displayMode;
    }
    
    void setDisplayMode(DisplayMode mode)
    {
        displayMode = mode;
    }
    
    /**
     * Returns null if hidden is not specified
     */
    public Boolean getHidden()
    {
        return hidden;
    }
    
    void setHidden(boolean hidden)
    {
        this.hidden = Boolean.valueOf(hidden);
    }
    
    /**
     * Returns null if required is not specified
     */
    public Boolean getRequired()
    {
        return required;
    }
    
    void setRequired(boolean required)
    {
        this.required = Boolean.valueOf(required);
    }

    /**
     * Returns null if maximum value is not specified
     */
    public Object getMaximumValue()
    {
        Object val = null;
        if (maximumValueHolder != null)
            val = maximumValueHolder.getValue();
        return val;
    }
    
    void setMaximumObjectValue(Object maxValue)
    {
        if (maximumValueHolder != null)
            maximumValueHolder.setObjectValue(maxValue);
    }
    
    /**
     * Returns null if minimum value is not specified
     */
    public Object getMinimumValue()
    {
        Object val = null;
        if (minimumValueHolder != null)
            val = minimumValueHolder.getValue(); 
        return val;
    }
    
    void setMinimumObjectValue(Object minValue)
    {
        if (minimumValueHolder != null)
            minimumValueHolder.setObjectValue(minValue);
    }
    
    /**
     * Returns null if maximum length is not specified or not applicable
     */
    public Integer getMaximumLength()
    {
        return maxLength;
    }
    
    void setMaximumLength(int maxLength)
    {
        if (getDataType() != TypeID.STRING)
            throw new UnsupportedOperationException();
        this.maxLength = Integer.valueOf(maxLength);
    }
    
    /**
     * Returns null if hasDependentProperties is not specified
     */
    public Boolean getHasDependentProperties()
    {
        return hasDependentProperties;
    }
    
    void setHasDependentProperties(boolean hasDependents)
    {
        hasDependentProperties = Boolean.valueOf(hasDependents);
    }
    
    /**
     * Returns null if no choice list has been specified
     */
    public ExternalChoiceList getChoiceList()
    {
        return choiceList;
    }
    
    void setChoiceList(ExternalChoiceList choiceList)
    {
        this.choiceList = choiceList;
    }
    
    public ExternalProperty copy()
    {
        PropertyValueHolder valueHolderCopy = valueHolder.copy();
        PropertyValueHolder maximumValueHolderCopy = null;
        if (maximumValueHolder != null)
            maximumValueHolderCopy = maximumValueHolder.copy();
        PropertyValueHolder minimumValueHolderCopy = null;
        if (minimumValueHolder != null)
            minimumValueHolderCopy = minimumValueHolder.copy();
        ExternalProperty copyProperty = new ExternalProperty(symbolicName, valueHolderCopy, maximumValueHolderCopy, minimumValueHolderCopy);
        copyProperty.displayMode = displayMode;
        copyProperty.hidden = hidden;
        copyProperty.required = required;
        copyProperty.maxLength = maxLength;
        copyProperty.hasDependentProperties = hasDependentProperties;
        copyProperty.choiceList = choiceList;
        return copyProperty;
    }
    
    /**
     * If a single value property, unspecifies the value on the holder if the value is invalid.
     * <p>
     * If a multi-value property and the list is empty, leave as an empty list.
     * <p>
     * If a multi-value property with a non-empty list, removes any invalid values.
     * If all items were removed then unspecifies the value.
     * <p>
     * If the value or any of the multiple values were invalid, returns the violation type.
     * <p>
     * Returns null (no reason) if the value in the holder was already valid and no 
     * value(s) were removed. 
     */
    ConstraintViolationType removeValuesViolatingConstraints(PropertyValueHolder holder, 
            List<Integer> collectMultiValuesWithViolations)
    {
        validateOtherHolderType(holder);
        
        ConstraintViolationType currentValuesViolation = null;
        
        // Checks to make against each individual item if this is a multi-value list
        if (valueHolder.getCardinality() == Cardinality.SINGLE)
        {
            currentValuesViolation = checkConstraintsSingleValue(holder);
            if (currentValuesViolation != null)
                holder.unspecifyValue();
        }
        else
        {
            PropertyValueHolder singleItemHolder = PropertyValueHolder.createHolder(valueHolder.getDataType(), Cardinality.SINGLE, false);
            List<Object> itemsList = new ArrayList<Object>();
            itemsList.addAll((List<?>) holder.getValue());
            if (itemsList.size() > 0)
            {
                int origIndex = 0;
                int index = 0;
                while (index < itemsList.size())
                {
                    Object item = itemsList.get(index);
                    singleItemHolder.setObjectValue(item);
                    ConstraintViolationType itemViolation = checkConstraintsSingleValue(singleItemHolder);
                    if (itemViolation != null)
                    {
                        itemsList.remove(index);
                        if (currentValuesViolation == null)
                            currentValuesViolation = itemViolation;
                        if (currentValuesViolation != itemViolation)
                            currentValuesViolation = ConstraintViolationType.VARYING_VIOLATIONS_MULTI_VALUES;
                        if (collectMultiValuesWithViolations != null)
                            collectMultiValuesWithViolations.add(Integer.valueOf(origIndex));
                    }
                    else
                        index++;
                    origIndex++;
                }
                if (currentValuesViolation != null)
                {
                    if (itemsList.size() == 0)
                        holder.unspecifyValue();
                    else
                        holder.setObjectValue(itemsList);
                }
            }
            else
            {
                // If list starts out empty, leave as an empty list unless
                // the external setting for required is true, in which case
                // consider it a constraint violation
                if (required != null && required.booleanValue())
                {
                    currentValuesViolation = ConstraintViolationType.REQUIRED_VALUE;
                    holder.unspecifyValue();
                }
            }
        }
        
        return currentValuesViolation;
    }
    
    private ConstraintViolationType checkConstraintsSingleValue(PropertyValueHolder singleValueHolder)
    {
        ConstraintViolationType violationType = null;
        boolean valid;
        if (singleValueHolder.getValue() != null)
        {
            // Check choice if present
            if (choiceList != null)
            {
                Object singleValue = singleValueHolder.getValue();
                valid = choiceList.isValueValid(singleValue);
                if (!valid)
                    violationType = ConstraintViolationType.CHOICE_LIST;
            }
            
            // Check min/max if present
            if (violationType == null && minimumValueHolder != null && minimumValueHolder.isValueSpecified())
            {
                valid = singleValueHolder.meetsMinimumValueConstraint(minimumValueHolder);
                if (!valid)
                    violationType = ConstraintViolationType.MINIMUM_VALUE;
            }
            if (violationType == null && maximumValueHolder != null && maximumValueHolder.isValueSpecified())
            {
                valid = singleValueHolder.meetsMaximumValueConstraint(maximumValueHolder);
                if (!valid)
                    violationType = ConstraintViolationType.MAXIMUM_VALUE;
            }
            
            // Check max length if present
            if (violationType == null && maxLength != null)
            {
                valid = singleValueHolder.meetsMaximumLengthConstraint(maxLength.intValue());
                if (!valid)
                    violationType = ConstraintViolationType.MAXIMUM_LENGTH;
            }
        }
        else
        {
            // Consider a null value invalid only if the external required
            // setting is true
            if (required != null && required.booleanValue())
            {
                violationType = ConstraintViolationType.REQUIRED_VALUE;
            }
        }

        return violationType;
    }
    
    private void validateOtherHolderType(PropertyValueHolder otherHolder)
    {
        if (valueHolder.getDataType() != otherHolder.getDataType()
                || valueHolder.getCardinality() != otherHolder.getCardinality()
                || valueHolder.getRequiresUniqueElements() != otherHolder.getRequiresUniqueElements())
        {
            throw SEDException.createException(SEDConstants.ERROR_UNEXPECTED);
        }
    }
    
}
