/*
 * IBM Confidential
 * 
 * OCO Source Materials
 * 
 * 5725A15
 * 
 *  Copyright IBM Corp. 2010,2011
 * 
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has
 * been deposited with the U.S. Copyright Office.
 */
package com.ibm.casemgmt.sampexterndata.api;

import java.util.ArrayList;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.filenet.api.constants.Cardinality;
import com.filenet.api.constants.TypeID;
import com.ibm.casemgmt.sampexterndata.api.PropertyValueHolder;

final class DynamicPropertySet 
{
    private static final String ERROR_MISSING_PARAMETERS = 
        "A dynamic property set is missing one or more required parameters"
        + " -- the conditional property name, one or more conditional property sets and a default property set.";
    private static final String ERROR_SAME_PROPERTIES_NOT_REPRESENTED =
        "In a dynamic property set, all of the conditional property sets must contain configurations for the same set of properties.";
    private static final String ERROR_MISSING_INPUT_PROPERTY =
        "The property {0} is missing from the list of input properties.  The property is needed to evaluate a dynamic property set.";
    private static final String ERROR_PROPERTY_NOT_CONSISTENT =
        "In a dynamic property set, the {0} property is not consistent in all of the conditional property sets."
        + "  The data type, cardinality and requires unique elements flag must be consistent in all conditions.";
    
    private final String conditionalPropertyName;
    private final List<ConditionalPropertySet> conditionalPropertySets = new ArrayList<ConditionalPropertySet>();
    private final PropertySet defaultPropertySet;
    
    static DynamicPropertySet createDynamicPropertySet(String conditionalPropertyName,
            List<ConditionalPropertySet> conditionalPropertySets,
            PropertySet defaultPropertySet)
    {
        if (conditionalPropertyName == null || conditionalPropertyName.length() == 0
                || conditionalPropertySets == null || conditionalPropertySets.size() == 0
                || defaultPropertySet == null)
        {
            throw SEDException.createException(ERROR_MISSING_PARAMETERS);
        }
        
        // all property sets should represent the same properties.  Get the set for the default 
        // property set first
        Set<String> propsRepresented = defaultPropertySet.getPropertiesRepresented();
        
        // Make sure each of the conditional ones have the same represented properties
        Iterator<ConditionalPropertySet> condPropSetsIt = conditionalPropertySets.iterator();
        while (condPropSetsIt.hasNext())
        {
            ConditionalPropertySet condPropSet = condPropSetsIt.next();
            Set<String> condPropsRepresented = condPropSet.getPropertySet().getPropertiesRepresented();
            if (!propsRepresented.containsAll(condPropsRepresented) 
                    || !condPropsRepresented.containsAll(propsRepresented))
            {
                throw SEDException.createException(ERROR_SAME_PROPERTIES_NOT_REPRESENTED);
            }
            
            // Make sure data type, cardinality and requires unique
            // flags are consistent between all sets, all properties
            Iterator<String> propsIt = propsRepresented.iterator();
            while (propsIt.hasNext())
            {
                String prop = propsIt.next();
                TypeID defaultDataType = defaultPropertySet.getDataType(prop);
                Cardinality defaultCardinality = defaultPropertySet.getCardinality(prop);
                boolean defaultRequiresUniqueElems = defaultPropertySet.getRequiresUniqueElements(prop);
                
                TypeID dataType = condPropSet.getPropertySet().getDataType(prop);
                Cardinality cardinality = condPropSet.getPropertySet().getCardinality(prop);
                boolean requiresUniqueElems = condPropSet.getPropertySet().getRequiresUniqueElements(prop);
                
                if (defaultDataType != dataType
                        || defaultCardinality != cardinality
                        || defaultRequiresUniqueElems != requiresUniqueElems)
                {
                    throw SEDException.createException(ERROR_PROPERTY_NOT_CONSISTENT, prop);
                }
            }
            
        }
        
        return new DynamicPropertySet(conditionalPropertyName, 
                conditionalPropertySets, defaultPropertySet);
    }
    
    private DynamicPropertySet(String conditionalPropertyName,
            List<ConditionalPropertySet> conditionalPropertySets,
            PropertySet defaultPropertySet)
    {
        this.conditionalPropertyName = conditionalPropertyName;
        this.conditionalPropertySets.addAll(conditionalPropertySets);
        this.defaultPropertySet = defaultPropertySet;
    }
    
    Set<String> getPropertiesRepresented()
    {
        // All dynamic prop sets and default prop set represent the same properties.
        // Return a set from the default prop set.
        return defaultPropertySet.getPropertiesRepresented();
    }
    
    TypeID getDataType(String propSymName)
    {
        // All dynamic prop sets and default prop set represent the same properties.
        // Return data type from the default prop set.
        return defaultPropertySet.getDataType(propSymName);
    }
    
    Cardinality getCardinality(String propSymName)
    {
        // All dynamic prop sets and default prop set represent the same properties.
        // Return cardinality from the default prop set.
        return defaultPropertySet.getCardinality(propSymName);
    }
    
    boolean getRequiresUniqueElements(String propSymName)
    {
        // All dynamic prop sets and default prop set represent the same properties.
        // Return flag from the default prop set.
        return defaultPropertySet.getRequiresUniqueElements(propSymName);
    }
    
    List<ExternalPropertyConfiguration> fetchMatchingProperties(
            Map<String, PropertyValueHolder> inputProperties, Deque<Integer> accumIdentifierIndexes)
    {
        PropertyValueHolder valueHolder = inputProperties.get(conditionalPropertyName);
        // A property needed for the dynamic criteria must be specified in the input properties
        if (valueHolder == null)
            throw SEDException.createException(ERROR_MISSING_INPUT_PROPERTY, conditionalPropertyName);
        
        int foundIndex = -1;
        PropertySet foundPropSet = null;
        for (int index = 0; index < conditionalPropertySets.size(); index++)
        {
            ConditionalPropertySet condPropSet = conditionalPropertySets.get(index);
            ConditionalCriteria crit = condPropSet.getCriteria();
            if (crit.matchesCriteria(valueHolder))
            {
                foundPropSet = condPropSet.getPropertySet();
                foundIndex = index;
                break;
            }
        }
        if (foundPropSet == null)
            foundPropSet = defaultPropertySet;
        accumIdentifierIndexes.addLast(Integer.valueOf(foundIndex));
        return foundPropSet.fetchMatchingProperties(inputProperties, accumIdentifierIndexes);
    }
    
    List<ExternalPropertyConfiguration> fetchPropertiesFromIdentifier(
            Deque<Integer> consumeIdentifierIndexes)
    {
        int identifierIndex = consumeIdentifierIndexes.removeFirst();
        PropertySet identifiedPropSet;
        if (identifierIndex >= 0)
        {
            ConditionalPropertySet condPropSet = conditionalPropertySets.get(identifierIndex);
            identifiedPropSet = condPropSet.getPropertySet();
        }
        else
            identifiedPropSet = defaultPropertySet;
        return identifiedPropSet.fetchPropertiesFromIdentifier(consumeIdentifierIndexes);
    }
    
}
