/*
    IBM grants you a nonexclusive copyright license to use all programming code 
	examples from which you can generate similar function tailored to your own 
	specific needs.

	All sample code is provided by IBM for illustrative purposes only.
	These examples have not been thoroughly tested under all conditions.  IBM, 
	therefore cannot guarantee or imply reliability, serviceability, or function of 
	these programs.

	All Programs or code component contained herein are provided to you AS IS  
	without any warranties of any kind.
	The implied warranties of non-infringement, merchantability and fitness for a 
	particular purpose are expressly disclaimed.

	 Copyright IBM Corporation 2008, ALL RIGHTS RESERVED.
 */

using System;
using System.IO;
using System.Text;
using System.Collections;
using FileNet.Api.Core;
using FileNet.Api.Constants;
using FileNet.Api.Collection;
using FileNet.Api.Exception;
using FileNet.Api.Property;

using SampDitaTool;

//
// this class will parse and store a DITA document to a P8 server
// using a compound document structure in P8.
//
public class StoreDITADoc
{
	internal IObjectStore os;
	
    //
    // Construcor
    //
	public StoreDITADoc(IObjectStore os)
	{
		this.os = os;
	}
	
    //
    // create the P8 document with the given title
    //
	public static IDocument CreateDitaMap(IObjectStore os, string title, IContentTransfer ct)
	{
		IDocument ditaMap = Factory.Document.CreateInstance(os, "DitaMap");
		
        ditaMap.CompoundDocumentState = CompoundDocumentState.COMPOUND_DOCUMENT;
		ditaMap.Properties["DitaTitle"] = title;
        ditaMap.Properties["DocumentTitle"] = title;

		// Add content	
		if (ct != null)
		{
			IContentElementList cel = Factory.ContentElement.CreateList();
            cel.Add(ct);
            ditaMap.ContentElements = cel;
		}
		
        ditaMap.Checkin(AutoClassify.DO_NOT_AUTO_CLASSIFY, CheckinType.MAJOR_VERSION);
        ditaMap.Save(RefreshMode.REFRESH);
        DitaToolForm.Out.WriteLine("    created: " + title);
		
		return ditaMap;
	}
	
	//
    // create the P8 document with the title in the given folder
    //
	public static IDocument CreateDitaMap(IObjectStore os, string title, IContentTransfer ct, IFolder folder)
	{
		IDocument ditaMap = Factory.Document.CreateInstance(os, "DitaMap");
		
		ditaMap.CompoundDocumentState = CompoundDocumentState.COMPOUND_DOCUMENT;
		ditaMap.Properties["DitaTitle"] = title;
		ditaMap.Properties["DocumentTitle"] = title;
		
		string name = folder.PathName + "/" + title;
		IContentElementList cel = Factory.ContentElement.CreateList();
		cel.Add(ct);
		ditaMap.ContentElements = cel;
		ditaMap.Checkin(AutoClassify.DO_NOT_AUTO_CLASSIFY, CheckinType.MAJOR_VERSION);
		ditaMap.Save(RefreshMode.REFRESH);
		
		IReferentialContainmentRelationship rcr = folder.File(ditaMap, AutoUniqueName.NOT_AUTO_UNIQUE, title, DefineSecurityParentage.DO_NOT_DEFINE_SECURITY_PARENTAGE);
		rcr.Save(RefreshMode.NO_REFRESH);

        DitaToolForm.Out.WriteLine("    created ditamap: " + title + " in folder: " + folder.PathName);
		return ditaMap;
	}
	
    //
    // create the transfer class
    //
	public static IContentTransfer CreateContentTransfer(FileInfo filesysDoc, string name, string type)
	{
		IContentTransfer ctNew = null;
		
		try
		{
			ctNew = Factory.ContentTransfer.CreateInstance();
			Stream input_stream = new FileStream(filesysDoc.FullName, FileMode.Open, FileAccess.Read);
			ctNew.SetCaptureSource(input_stream);
			ctNew.RetrievalName = name;
			ctNew.ContentType = type;
		}
		catch (Exception t)
		{
            DitaToolForm.Out.WriteLine(t.StackTrace);
            throw t;
		}
		return ctNew;
	}

    //
    // Add a topic to the P8 server
    //
    public static void AddDitaTopic(IObjectStore os, IDocument parent, IDocument ditaProject, string label)
	{
		IDocument topic = null;
		
		topic = CreateDitaTopic(os, label + "-topic-", "topic-id");
		CreateDitaTopicref(os, ditaProject, parent, topic, "dita-topicref");
	}

    //
    // Add a list of topics to the P8 server
    //
    public static void AddDitaTopics(IObjectStore os, IDocument parent, IDocument ditaProject, string label, int count)
	{
		IDocument topic = null;
		
		for (int ix = 0; ix < count; ix++)
		{
			topic = CreateDitaTopic(os, label + "-topic-" + System.Convert.ToString(ix), "topic-id");
			CreateDitaTopicref(os, ditaProject, parent, topic, "dita-topicref-" + System.Convert.ToString(ix));
		}
	}

    //
    // create a topic reference between two documents
    //
    public static void CreateDitaTopicref(IObjectStore os, IDocument ditaProject, IDocument parent, IDocument child, string label)
	{
		CreateDitaRelationship(os, ditaProject, parent, child, "DitaTopicref");
	}

    //
    // create a map reference between two documents
    //
    public static void CreateDitaMapref(IObjectStore os, IDocument ditaProject, IDocument parent, IDocument child, string label)
	{
		CreateDitaRelationship(os, ditaProject, parent, child, "DitaMapref");
	}

    //
    // createa  relationship between two documents
    //
    public static void CreateDitaRelationship(IObjectStore os, IDocument ditaProject, IDocument parent, IDocument child, string className)
	{
		IComponentRelationship cr = Factory.ComponentRelationship.CreateInstance(os, className, ComponentRelationshipType.DYNAMIC_CR, parent, "cas-DYNAMIC-CR-LATEST-VERSION");
		
		cr.ChildComponent = child;
		cr.VersionBindType = VersionBindType.LATEST_VERSION;
		cr.ComponentCascadeDelete = ComponentCascadeDeleteAction.CASCADE_DELETE;
		cr.Properties["DitaProject"] = ditaProject.VersionSeries.Id;
		cr.Save(RefreshMode.REFRESH);
	}

    //
    // create a dita topic on the P8 server
    //
    public static IDocument CreateDitaTopic(IObjectStore os, string title, string id)
	{
		string TOPIC_AUTHOR1 = "Jane Author";
		string PLATFORM1 = "windows";
		string KEYWORD1 = "keyword1";
		
		IDocument topic = Factory.Document.CreateInstance(os, "DitaTopic");
		topic.CompoundDocumentState = CompoundDocumentState.COMPOUND_DOCUMENT;
		
		topic.Properties["DocumentTitle"] = title;
		topic.Properties["DitaTitle"] = title;
		topic.Properties["DitaId"] = id;
		
		// create some multi-valued string properties for demo queries
		IStringList authors = Factory.StringList.CreateList();
		IStringList keywords = Factory.StringList.CreateList();
		IStringList platforms = Factory.StringList.CreateList();
		
		authors.Add(TOPIC_AUTHOR1);
		keywords.Add(KEYWORD1);
		platforms.Add(PLATFORM1);
		
		topic.Properties["DitaAuthors"] = authors;
		topic.Properties["DitaPlatforms"] = platforms;
		topic.Properties["DitaKeywords"] = keywords;
		
		topic.Checkin(AutoClassify.DO_NOT_AUTO_CLASSIFY, CheckinType.MAJOR_VERSION);
		topic.Save(RefreshMode.REFRESH);
        DitaToolForm.Out.WriteLine("    created: " + title);
        DitaToolForm.Out.WriteLine("        author: " + TOPIC_AUTHOR1 + ", keyword = " + KEYWORD1 + ", platform = " + PLATFORM1);
		
		return topic;
	}

    //
    // create a dita topic using the transfer object
    //
    public static IDocument CreateDitaTopic(IObjectStore os, string title, string id, IContentTransfer ct)
	{
		IDocument topic = Factory.Document.CreateInstance(os, "DitaTopic");
		topic.CompoundDocumentState = CompoundDocumentState.COMPOUND_DOCUMENT;
		
		topic.Properties["DocumentTitle"] = title;
		topic.Properties["DitaTitle"] = title;
		topic.Properties["DitaId"] = id;
		
		// Add content	
		if (ct != null)
		{
			IContentElementList cel = Factory.ContentElement.CreateList();
			cel.Add(ct);
			topic.ContentElements = cel;
		}
		
		topic.Checkin(AutoClassify.DO_NOT_AUTO_CLASSIFY, CheckinType.MAJOR_VERSION);
		topic.Save(RefreshMode.REFRESH);
        DitaToolForm.Out.WriteLine("    created: " + title);
		
		return topic;
	}

    //
    // create a dita topic using the transfer object and specified folder
    //
    public static IDocument CreateDitaTopic(IObjectStore os, string title, string id, IContentTransfer ct, IFolder folder)
	{
		IDocument topic = Factory.Document.CreateInstance(os, "DitaTopic");
		topic.CompoundDocumentState = CompoundDocumentState.COMPOUND_DOCUMENT;
		
		topic.Properties["DocumentTitle"] = title;
		topic.Properties["DitaTitle"] = title;
		topic.Properties["DitaId"] =  id;
		
		// Add content	
		IContentElementList cel = Factory.ContentElement.CreateList();
		cel.Add(ct);
		topic.ContentElements = cel;
		
		topic.Checkin(AutoClassify.DO_NOT_AUTO_CLASSIFY, CheckinType.MAJOR_VERSION);
		topic.Save(RefreshMode.REFRESH);
		
		IReferentialContainmentRelationship rcr = folder.File(topic, AutoUniqueName.NOT_AUTO_UNIQUE, title, DefineSecurityParentage.DO_NOT_DEFINE_SECURITY_PARENTAGE);
		rcr.Save(RefreshMode.NO_REFRESH);
        DitaToolForm.Out.WriteLine("    created topic: " + title + " in folder: " + folder.PathName);
		
		return topic;
	}
	
	//
    // retrieves a folder or creates it if it doesn't exist
    //
	public static IFolder GetOrCreateFolder(IObjectStore os, IFolder parentFolder, string subFolderName)
	{
		IFolder f = null;
		
		try
		{
			string path = null;
			
			if (parentFolder == null)
			{
				path = "/" + subFolderName;
			}
			else
			{
				path = parentFolder.PathName + "/" + subFolderName;
			}
			
			f = Factory.Folder.FetchInstance(os, path, null);
		}
		catch (EngineRuntimeException e)
		{
            if (e.GetExceptionCode() == ExceptionCode.E_OBJECT_NOT_FOUND)
			{
				if (parentFolder == null)
				{
					parentFolder = os.RootFolder;
				}
				
				f = parentFolder.CreateSubFolder(subFolderName);
				f.Save(RefreshMode.REFRESH);
			}
			else
			{
                DitaToolForm.Out.WriteLine(e.StackTrace);
				throw e;
			}
		}
		return f;
	}

    //
    // look for ".ditamap" files in the specified directory
    //
	public static string[] FindDitaMaps(string currentDir)
	{
		//FileInfo f = new FileInfo(currentDir);
		//FileInfo[] files = SupportClass.FileSupport.GetFiles(f);
        DirectoryInfo f = new DirectoryInfo(currentDir);
        FileInfo[] files = f.GetFiles();

		string[] ditamapNames = null;
		string[] ditamapTmpNames = null;
		int ditamapInx = 0;
		
		ditamapTmpNames = new string[files.Length];
		
		for (int i = 0; i < files.Length; i++)
		{
			if (System.IO.File.Exists(files[i].FullName))
			{
				// determine ditamap file
				if (files[i].Name.ToUpper().IndexOf(".DITAMAP") > 0)
				{
					ditamapTmpNames[ditamapInx] = files[i].Name;
					ditamapInx++;
				}
			}
		}
		
		if (ditamapInx > 0)
		{
			ditamapNames = new string[ditamapInx];
			
			int i = 0;
			while (ditamapTmpNames[i] != null)
			{
				ditamapNames[i] = ditamapTmpNames[i];
				i++;
			}
		}
		
		return ditamapNames;
	}
	
    //
    // find the first file in the director with a .DITAMAP extension
    //
	public static string FindDitaMap(string currentDir)
	{
		//FileInfo f = new FileInfo(currentDir);
		//FileInfo[] files = SupportClass.FileSupport.GetFiles(f);
        DirectoryInfo f = new DirectoryInfo(currentDir);
        FileInfo[] files = f.GetFiles();

		string ditaMapName = null;
		
		for (int i = 0; i < files.Length; i++)
		{
			if (File.Exists(files[i].FullName))
			{
				if (files[i].Name.ToUpper().IndexOf(".DITAMAP") > 0)
				{
					return files[i].Name;
				}
			}
		}
		
		return ditaMapName;
	}
	
	//
    // The main call to create a P8 compound document from a ditamap
    //
	public static IDocument CreateP8DITAStructureFromFilesystem(IObjectStore os, string p8BaseDir, string baseDir, string filename)
	{
		IFolder DITAP8BaseFolder = Factory.Folder.FetchInstance(os, p8BaseDir, null);
		FileInfo FileSystemBaseFolder = new FileInfo(baseDir);
		
		System.Collections.IDictionary ditaDocs = new System.Collections.Hashtable();
		System.Collections.IList ditaOrderedDocs = new System.Collections.ArrayList();
		
		FindDITAComponents(baseDir, filename, ditaDocs, ditaOrderedDocs, "ditamap","   ");

        DitaToolForm.Out.WriteLine("List All Associated Dita Files");
		int count = ditaOrderedDocs.Count;
		string path = null;
		for (int ix = 0; ix < count; ix++)
		{
			path = ((string) ditaOrderedDocs[ix]);
            DitaToolForm.Out.WriteLine(path);
		}
        DitaToolForm.Out.WriteLine("Listing Complete");

		string relType = null;
		IDocument ditamap = null;

        if (DitaToolForm.simulateUL)
        {
            DitaToolForm.Out.WriteLine("Simulate Upload");
        }
        else
        {
            DitaToolForm.Out.WriteLine("Begin Upload");
        }
		for (int ix = count - 1; ix >= 0; ix--)
		{
			path = ((string) ditaOrderedDocs[ix]);
			relType = ((string) ditaDocs[path]);
			bool autoClassify = !NonDITARelationship(relType);
			
			FileInfo doc = new FileInfo(path);
			
			IDocument parentDoc = PersistDocument(os, doc, "Document", DITAP8BaseFolder, FileSystemBaseFolder, autoClassify);
			
			if (relType.Equals("ditamap"))
			{
				ditamap = parentDoc;
			}
		}
        if (DitaToolForm.simulateUL)
        {
            DitaToolForm.Out.WriteLine("Simulate Upload Complete");
        }
        else
        {
            DitaToolForm.Out.WriteLine("Upload Complete");
        }
        
		return ditamap;
	}

    //
    // find all the files associated with the ditamap
    //
    private static void FindDITAComponents(string baseDir, string filename, System.Collections.IDictionary ditaDocs, System.Collections.IList ditaOrderedDocs, string relType, string inset)
	{
		
		try
		{
            if (!baseDir.EndsWith("/"))
            {
                baseDir = baseDir + "/";
            }

			string newDocPath = ResolveRelativePath(baseDir, filename);
			
			if (ditaDocs.Contains(newDocPath))
			{
				return ;
			}
			
			FileInfo newDoc = new FileInfo(newDocPath);
			
			string newBaseDir = newDoc.FullName;
			
			int index = newBaseDir.LastIndexOf(System.IO.Path.DirectorySeparatorChar.ToString());
			
			if (index == - 1)
			{
				throw new System.SystemException("No separator found: " + newBaseDir);
			}
			newBaseDir = newBaseDir.Substring(0, (index + 1) - (0));
			
			DitaParser dp = new DitaParser(newDoc);

			AddRelationship(ditaDocs, ditaOrderedDocs, newDocPath, relType,inset);
			
			IDictionary holder = dp.ComponentRelationships;
			
			IDictionary relationships = new System.Collections.Hashtable(holder);
			
			dp = null;

            //IEnumerator it = new SupportClass.HashSetSupport(relationships.Keys).GetEnumerator();
            IEnumerator it = relationships.Keys.GetEnumerator();
			
			while (it.MoveNext())
			{
				string childFilePath = (string) it.Current;
				relType = ((string) relationships[childFilePath]);
				
				string newPath = ResolveRelativePath(newBaseDir, childFilePath);
				
				if (!ditaDocs.Contains(newPath))
				{
					if (NonDITARelationship(relType))
					{
						// Non DITA file (image, pdf, etc.)
						AddRelationship(ditaDocs, ditaOrderedDocs, newPath, relType,inset);
					}
					else
					{
						FindDITAComponents(newBaseDir, childFilePath, ditaDocs, ditaOrderedDocs, relType, inset+"    ");
					}
				}
			}
		}
		catch (Exception t)
		{
            DitaToolForm.Out.WriteLine(t.StackTrace);
			throw new System.SystemException(t.Message);
		}
	}
	
    //
    // add a relationship to the list of ditaDocs
    //
	private static void AddRelationship(System.Collections.IDictionary ditaDocs, System.Collections.IList ditaOrderedDocs, string path, string relType, string inset)
	{
		if (!ditaDocs.Contains(path))
		{
			ditaDocs[path] = relType;
			ditaOrderedDocs.Add(path);
            DitaToolForm.Out.WriteLine(inset+ path + "(" + relType + ")");
		}
	}
	
    //
    // see if this is a non-dita (image, other) document reference
    //
	private static bool NonDITARelationship(string relType)
	{
		return (relType.Equals("DitaImageref", StringComparison.OrdinalIgnoreCase)) || relType.Equals("DitRef", StringComparison.OrdinalIgnoreCase);
	}
	
    //
    // convert the relative path to a full path
    //
	private static string ResolveRelativePath(string currentBaseDir, string childFilePath)
	{
		// Shouldn't need to use both delimiters... but
        string[] dirs = childFilePath.Split(new Char[] {'/', '\\'});
		
		int count = 0;
		
		foreach(string s in dirs)
		{
			if (s.Equals(".."))
			{
				count++;
			}
		}
		
		for (int ix = 0; ix < count; ix++)
		{
			int index = currentBaseDir.LastIndexOf(System.IO.Path.DirectorySeparatorChar.ToString());
			
			int len = currentBaseDir.Length;
			
			if (index + 1 == len)
			{
				currentBaseDir = currentBaseDir.Substring(0, (len - 1) - (0));
				
				index = currentBaseDir.LastIndexOf(System.IO.Path.DirectorySeparatorChar.ToString());
			}
			
			if (index == - 1)
			{
				throw new System.SystemException("File separator not found: " + currentBaseDir);
			}
			
			currentBaseDir = currentBaseDir.Substring(0, (index + 1) - (0));
			
			childFilePath = childFilePath.Substring(3, (childFilePath.Length) - (3));
		}
		string path = currentBaseDir + childFilePath;
		FileInfo f = new FileInfo(path);
		bool tmpBool;
		if (File.Exists(f.FullName))
			tmpBool = true;
		else
			tmpBool = System.IO.Directory.Exists(f.FullName);
		if (!tmpBool)
		{
			throw new System.SystemException("files not not exist: " + path);
		}
		try
		{
			path = f.FullName;
		}
		catch (System.Exception t)
		{
			throw new System.SystemException(t.Message);
		}
		return path;
	}
	
    //
    // Add or update the document to the P8 object store
    //
	private static IDocument PersistDocument(IObjectStore os, FileInfo filesysDoc, string p8ClassName, IFolder DITABaseFolder, FileInfo FileSystemBaseFolder, bool autoClassify)
	{
		bool tmpBool;
		if (File.Exists(filesysDoc.FullName))
			tmpBool = true;
		else
			tmpBool = System.IO.Directory.Exists(filesysDoc.FullName);
		if (!tmpBool)
		{
			throw new System.SystemException("Filesystem file does not exist: " + filesysDoc.FullName);
		}
		
		IDocument ditacomponent = null;
		IFolder folder = null;
		
		string name = filesysDoc.Name;
		
		try
		{
			folder = GetSubFolder(os, DITABaseFolder, FileSystemBaseFolder, filesysDoc);
			
			ditacomponent = GetExistingP8Document(os, folder, name);

            if (DitaToolForm.simulateUL)
            {
                if (ditacomponent!=null)
                    DitaToolForm.Out.WriteLine("SIMULATE update " + filesysDoc);
                else
                    DitaToolForm.Out.WriteLine("SIMULATE create " + filesysDoc);
            }
			else if (ditacomponent != null)
			{
                DitaToolForm.Out.WriteLine("update existing: " + filesysDoc);
                ditacomponent = UpdateExistingP8Document(ditacomponent, name, filesysDoc, folder);
			}
			else
			{
                DitaToolForm.Out.WriteLine("create document: " + filesysDoc);
                ditacomponent = CreateNewP8Document(os, p8ClassName, name, filesysDoc, folder, autoClassify);
			}
		}
		catch (EngineRuntimeException t)
		{
            DitaToolForm.Out.WriteLine("*** E_NOT_UNIQUE exception thrown, most likey stub document created by DITA Classifier.");
            DitaToolForm.Out.WriteLine("***     Check if document now exists and update.");
			if (t.GetExceptionCode() == ExceptionCode.E_NOT_UNIQUE)
			{
				ditacomponent = GetExistingP8Document(os, folder, name);
				
				if (ditacomponent != null)
				{
					ditacomponent = UpdateExistingP8Document(ditacomponent, name, filesysDoc, folder);
					return ditacomponent;
				}
			}
			//throw new RuntimeException(t.GetLocalizedMessage);
            throw new EngineRuntimeException(t, ExceptionCode.E_UNEXPECTED_EXCEPTION, null);
		}
		
		return ditacomponent;
	}
	
	//
    // Update the document in the P8 object store
    //
	private static IDocument UpdateExistingP8Document(IDocument ditacomponent, string name, FileInfo filesysDoc, IFolder f)
	{
		if (ditacomponent.Properties.IsPropertyPresent(PropertyNames.CONTENT_SIZE))
		{
			Double size = (Double) ditacomponent.Properties.GetFloat64Value(PropertyNames.CONTENT_SIZE);
			
			bool hasContent = false;
			
			if (size != 0)
			{
				hasContent = (int) size > 0;
			}
			
			if (!hasContent)
			{
				ditacomponent.Checkout(ReservationType.EXCLUSIVE, null, null, null);
				ditacomponent.Save(RefreshMode.REFRESH);
				
				IDocument reservation = (IDocument) ditacomponent.Reservation;
				
				reservation.CompoundDocumentState = CompoundDocumentState.COMPOUND_DOCUMENT;
				
				string mimeType = "application/dita+xml";
				if (name.IndexOf(".gif") != - 1)
				{
					mimeType = "image/gif";
				}
				else if (name.IndexOf(".jpg") != - 1)
				{
					mimeType = "image/jpg";
				}
				else if (name.IndexOf(".jpeg") != - 1)
				{
					mimeType = "image/jpeg";
				}
				
				reservation.Checkin(mimeType.Equals("application/dita+xml", StringComparison.OrdinalIgnoreCase)?AutoClassify.AUTO_CLASSIFY:AutoClassify.DO_NOT_AUTO_CLASSIFY, CheckinType.MAJOR_VERSION);
				
				reservation.Properties["DocumentTitle"] = name;
				IContentTransfer ct = CreateContentTransfer(filesysDoc, name, mimeType);
				IContentElementList cel = Factory.ContentElement.CreateList();
				cel.Add(ct);
				reservation.ContentElements = cel;
				reservation.Save(RefreshMode.REFRESH);
				ditacomponent = reservation;

                DitaToolForm.Out.WriteLine("   update existing P8 document: " + name + ", in folder: " + f.PathName);
			}
		}
		
		return ditacomponent;
	}
	
	//
    // Create a new document in the P8 object store
    //
	private static IDocument CreateNewP8Document(IObjectStore os, string p8ClassName, string name, FileInfo filesysDoc, IFolder f, bool autoClassify)
	{
		
		UpdatingBatch ub = UpdatingBatch.CreateUpdatingBatchInstance(os.Domain, RefreshMode.REFRESH);
		
		//string title = null;
		
		IDocument ditacomponent = Factory.Document.CreateInstance(os, p8ClassName);
		ditacomponent.CompoundDocumentState = CompoundDocumentState.COMPOUND_DOCUMENT;
		ditacomponent.Checkin(autoClassify?AutoClassify.AUTO_CLASSIFY:AutoClassify.DO_NOT_AUTO_CLASSIFY, CheckinType.MAJOR_VERSION);
		
		string mimeType = "application/dita+xml";
		if (name.IndexOf(".gif") != - 1)
		{
			mimeType = "image/gif";
		}
		else if (name.IndexOf(".jpg") != - 1)
		{
			mimeType = "image/jpg";
		}
		else if (name.IndexOf(".jpeg") != - 1)
		{
			mimeType = "image/jpeg";
		}
		ditacomponent.Properties["DocumentTitle"] = name;
		IContentTransfer ct = CreateContentTransfer(filesysDoc, name, mimeType);
		IContentElementList cel = Factory.ContentElement.CreateList();
		cel.Add(ct);
		ditacomponent.ContentElements = cel;
		
		ub.Add(ditacomponent, null);
		
		
		IReferentialContainmentRelationship rcr = f.File(ditacomponent, AutoUniqueName.NOT_AUTO_UNIQUE, name, DefineSecurityParentage.DO_NOT_DEFINE_SECURITY_PARENTAGE);
		ub.Add(rcr, null);
		
		ub.UpdateBatch();

        DitaToolForm.Out.WriteLine("   persist new P8 document: " + name + ", in folder: " + f.PathName);
		
		return ditacomponent;
	}
	
	//
    // retrieve a document from the P8 store if it exists, return null if it doesn't exist
    //
	private static IDocument GetExistingP8Document(IObjectStore os, IFolder folder, string name)
	{
		IDocument doc = null;
		try
		{
			PropertyFilter pf = new PropertyFilter();
			pf.AddIncludeProperty(1, null, false, PropertyNames.CONTENT_SIZE, null);
			string filepath = folder.PathName + "/" + name;
			doc = Factory.Document.FetchInstance(os, filepath, pf);
            DitaToolForm.Out.WriteLine("*** P8 Document Exists: " + filepath);
		}
        catch (System.Exception )
		{
			// ignore assume doesn't exists - expensive way to do this
			doc = null;
		}
		return doc;
	}
	
    //
    // retrieve or create the given folder structure
    //
	private static IFolder GetSubFolder(IObjectStore os, IFolder DITABaseFolder, FileInfo FileSystemBaseFolder, FileInfo filesysDoc)
	{
		IFolder f = DITABaseFolder;
		
		System.Collections.IList subfolders = DiffFolders(FileSystemBaseFolder, filesysDoc);
		
		System.Collections.IEnumerator it = subfolders.GetEnumerator();
		
		while (it.MoveNext())
		{
			string subfolder = (string) it.Current;
			f = GetOrCreateFolder(os, f, subfolder);
		}
		
		return f;
	}
	
    //
    // Check for folder differences
    //
	private static System.Collections.IList DiffFolders(FileInfo FileSystemBaseFolder, FileInfo filesysDoc)
	{
		System.Collections.ArrayList subfolderList = new System.Collections.ArrayList();
		
		string baseDir = FileSystemBaseFolder.FullName;
		string filePath = filesysDoc.FullName;
		
		int index = filePath.LastIndexOf(System.IO.Path.DirectorySeparatorChar.ToString());
		
		if (index == - 1)
		{
			throw new System.SystemException("No file separator: " + filePath);
		}
		
		// remove file name part
		filePath = filePath.Substring(0, (index + 1) - (0));
		
		if (!filePath.StartsWith(baseDir))
		{
			throw new System.SystemException("File path does not contain staed base dir: " + filePath);
		}
		
		string subFolders = filePath.Substring(baseDir.Length);
		
		string[] dirs = subFolders.Split(System.IO.Path.DirectorySeparatorChar);
		
		foreach(string s in dirs)
		{
			subfolderList.Add(s);
		}
		
		return subfolderList;
	}
	
    //
    // the main entry point to create a compound DITA document
    //
	public static IDocument CreateDitaSourceDoc(IObjectStore os, string ditaP8BaseDir, string baseDir, string ditamapName)
	{
		IDocument srcDoc = null;
		
		srcDoc = CreateP8DITAStructureFromFilesystem(os, ditaP8BaseDir, baseDir, ditamapName);
		
		return srcDoc;
	}
}
