/// <summary>IBM grants you a nonexclusive copyright license to use all programming code 
/// examples from which you can generate similar function tailored to your own 
/// specific needs.
/// All sample code is provided by IBM for illustrative purposes only.
/// These examples have not been thoroughly tested under all conditions.  IBM, 
/// therefore cannot guarantee or imply reliability, serviceability, or function of 
/// these programs.
/// All Programs or code component contained herein are provided to you AS IS  
/// without any warranties of any kind.
/// The implied warranties of non-infringement, merchantability and fitness for a 
/// particular purpose are expressly disclaimed.
///  Copyright IBM Corporation 2008, ALL RIGHTS RESERVED.
/// </summary>
using System;
using System.IO;
using System.Text;
using System.Collections;
using FileNet.Api.Core;
using FileNet.Api.Constants;
using FileNet.Api.Collection;
using FileNet.Api.Exception;
using FileNet.Api.Property;
using FileNet.Api.Util;

using SampDitaTool;

	
	/// <summary> This class will download a DITA document from a P8 server. A DITA
	/// document is a compound document and may consist of many files. All
	/// the files in the compound document structure will be downloaded.
	/// 
	/// </summary>
	public class GetDITADoc
	{
		
		private static System.Collections.ArrayList stubFiles;
		
		/// <summary> Given the ditamap document, download the compound document structure
		/// to the workingDirectory.
		/// 
		/// </summary>
		/// <param name="ditamap">
		/// </param>
		/// <param name="workingDirectory">
		/// </param>
		/// <returns>
		/// </returns>
		public static System.IO.FileInfo copyCompoundDocumentToFilesystem(IDocument ditamap, System.String workingDirectory)
		{
			stubFiles = System.Collections.ArrayList.Synchronized(new System.Collections.ArrayList(10));
			
			// verify working directory
			System.IO.FileInfo workingDir = new System.IO.FileInfo(workingDirectory);
			
			bool tmpBool;
			if (System.IO.File.Exists(workingDir.FullName))
				tmpBool = true;
			else
				tmpBool = System.IO.Directory.Exists(workingDir.FullName);
			if (!tmpBool)
			{
				DitaToolForm.Out.WriteLine("Creating download directory " + workingDir);
				if ((!DitaToolForm.simulateDL) && (System.IO.Directory.CreateDirectory(workingDir.FullName)!=null))
				{
					throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION, "Cannot create DITA working directory in filesystem: " + workingDirectory);
				}
			}
			
			System.DateTime now = System.DateTime.Now;
			
			// refresh the properties we will need in the download process
			ditamap.Refresh(new System.String[]{"DocumentTitle", PropertyNames.CHILD_DOCUMENTS, PropertyNames.ID, PropertyNames.OBJECT_STORE, PropertyNames.CONTENT_SIZE});
			
			System.Collections.IDictionary ditaDocs = new System.Collections.Hashtable();
			
			// map the structure of the compound document
            DitaToolForm.Out.WriteLine("Ditamap structure:");
			retrieveCompoundDocumentStructure(ditamap, ditaDocs, "ditamap", "");
            DitaToolForm.Out.WriteLine("Ditamap structure complete.");
			
			// copy the document structure to the file system
            return doCopyCompoundDocumentToFilesystem(ditamap.GetObjectStore(), ditaDocs, workingDirectory, ref now);
        }
		
		/// <summary> Create a map of documents associated with the ditacomponent.
		/// Store the list of documents in the ditaDocs map.
		/// Since this method is recursive and outputs the files it retrieves
		/// the indent is kept with the number of spaces in the current structure.
		/// 
		/// </summary>
		/// <param name="ditacomponent">
		/// </param>
		/// <param name="ditaDocs">
		/// </param>
		/// <param name="type">
		/// </param>
		/// <param name="indent">
		/// </param>
		private static void  retrieveCompoundDocumentStructure(IDocument ditacomponent, System.Collections.IDictionary ditaDocs, System.String type, System.String indent)
		{
			Id id = ditacomponent.Id;
			
			// only retrieve a document if it has not yet been referenced.
			if (!ditaDocs.Contains(id))
			{
				// don't descend on this document again
				ditaDocs[id] = type;
				
				// get the name of the document
				System.String name = null;
				if (ditacomponent.Properties.IsPropertyPresent("DocumentTitle"))
				{
					name = ditacomponent.Properties.GetStringValue("DocumentTitle");
				}
				
				if (name == null)
				{
                    DitaToolForm.Out.WriteLine("ERROR - DITA component filename is null");
					throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION, "DITA component filename is null");
				}
				
				// check the size of the document, looking for 0 length stub files
				System.Double size = (Double) ditacomponent.Properties.GetFloat64Value(PropertyNames.CONTENT_SIZE);

                DitaToolForm.Out.WriteLine(indent + ditacomponent.Properties.GetStringValue("DocumentTitle") + " size:(" + (int)size + ")");
				
				// find all the compound document associations to this document
				IDocumentSet ds = ditacomponent.ChildDocuments;
				
				// iterate through all the compound documents
				System.Collections.IEnumerator it = ds.GetEnumerator();
				while (it.MoveNext())
				{
					IDocument child = (IDocument) it.Current;
					
					// recursive call to map each document structure
					retrieveCompoundDocumentStructure(child, ditaDocs, "ditacomponent", indent + "   ");
				}
			}
		}
		
		
		/// <summary> Do the actual copy of the documents in the ditaDocs map. The map
		/// is created with the retrieveCompoundDocumentStructure method.
		/// 
		/// </summary>
		/// <param name="objStore">
		/// </param>
		/// <param name="ditaDocs">
		/// </param>
		/// <param name="workingDirectory">
		/// </param>
		/// <param name="startTime">
		/// </param>
		/// <returns>
		/// </returns>
		//UPGRADE_NOTE: ref keyword was added to struct-type parameters. "ms-help://MS.VSCC.v80/dv_commoner/local/redirect.htm?index='!DefaultContextWindowIndex'&keyword='jlca1303'"
		private static System.IO.FileInfo doCopyCompoundDocumentToFilesystem(IObjectStore objStore, System.Collections.IDictionary ditaDocs, string workingDirectory, ref System.DateTime startTime)
		{
			System.IO.FileInfo ditamap = null;
			
			PropertyFilter pf = new PropertyFilter();
			pf.AddIncludeProperty(0, null, false, "DocumentTitle " + PropertyNames.FOLDERS_FILED_IN + " " + PropertyNames.CONTENT_ELEMENTS + " " + PropertyNames.CONTENT_SIZE, null);
			
			System.String type = null;

            System.Collections.IEnumerator it = ditaDocs.Keys.GetEnumerator();
			
			while (it.MoveNext())
			{
				Id id = (Id) it.Current;
				type = ((System.String) ditaDocs[id]);
				
				IDocument ditacomponent = Factory.Document.FetchInstance(objStore, id, pf);
				
				System.String name = null;
				
				if (ditacomponent.Properties.IsPropertyPresent("DocumentTitle"))
				{
					name = ditacomponent.Properties.GetStringValue("DocumentTitle");
				}
				
				if (name == null)
				{
					throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION, "DITA component filename is null");
				}
				
				IFolderSet fs = ditacomponent.FoldersFiledIn;
				
				System.Collections.IEnumerator iy = fs.GetEnumerator();
				System.String folderPath = null;
				int i = 0;
				
				while (iy.MoveNext())
				{
					// constraint: ditacomponent may only be filed in a single folder, and it must be filed
					// this matches file system
					IFolder f = (IFolder) iy.Current;
					
					f.Refresh(new System.String[]{PropertyNames.PATH_NAME});
					
					folderPath = f.PathName;
					if (i > 0)
					{
						throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION, "DITA component filed in more than one folder: " + name);
					}
					i++;
				}
				
				if (i == 0)
				{
					// throw exception, diatcomponent not filed in any folder
					throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION, "DITA component not filed in any folder: " + name);
				}
				
				System.IO.FileInfo newDir = new System.IO.FileInfo(workingDirectory + folderPath);
				
				// create filesystem folder for current ditacomponent if it does not already exist
				bool tmpBool;
				if (System.IO.File.Exists(newDir.FullName))
					tmpBool = true;
				else
					tmpBool = System.IO.Directory.Exists(newDir.FullName);
				if (!tmpBool)
				{
                    string prepend = "";
					if (DitaToolForm.simulateDL)
                        prepend = "Simulate ";
                    DitaToolForm.Out.WriteLine(prepend+"Creating directory " + newDir);

                    if (!DitaToolForm.simulateDL)
					{
                        System.IO.Directory.CreateDirectory(newDir.FullName);
					}
				}
				
				System.IO.FileInfo parentFile = new System.IO.FileInfo(newDir.FullName + "\\" + name);
				bool alreadyExistsOrOld = false;
				
				// if the ditacomponent already exists on the filesystem (means the component is being reused within DITA structure)
				// determine if was created for this publishing operation, if so, do not output it to filesystem again
				// if created before this publishing operation started, delete it and download current ditacomponent
				bool tmpBool2;
				if (System.IO.File.Exists(parentFile.FullName))
					tmpBool2 = true;
				else
					tmpBool2 = System.IO.Directory.Exists(parentFile.FullName);
				if (tmpBool2)
				{
					System.DateTime lastModified = new System.DateTime(((parentFile.LastWriteTime.Ticks - 621355968000000000) / 10000));
					
					// check if existing file was created after start of pull operation, if not overwrite it
					alreadyExistsOrOld = (lastModified > startTime);

                    string prepend = "";
					if (DitaToolForm.simulateDL)
                        prepend = "Simulate ";
                    DitaToolForm.Out.WriteLine(prepend+"Deleting " + parentFile);
					if ((DitaToolForm.simulateDL) && (!alreadyExistsOrOld))
					{
						// The file was create before the current operation started, delete it
						bool tmpBool3;
						if (System.IO.File.Exists(parentFile.FullName))
						{
							System.IO.File.Delete(parentFile.FullName);
							tmpBool3 = true;
						}
						else if (System.IO.Directory.Exists(parentFile.FullName))
						{
							System.IO.Directory.Delete(parentFile.FullName);
							tmpBool3 = true;
						}
						else
							tmpBool3 = false;
						bool generatedAux2 = tmpBool3;
					}
				}
				
				System.String path = parentFile.FullName;
				System.Double size = (Double) ditacomponent.Properties.GetFloat64Value(PropertyNames.CONTENT_SIZE);
				
				if (!alreadyExistsOrOld)
				{
                    string prepend = "";
					if (DitaToolForm.simulateDL)
                        prepend = "Simulate ";
					
					System.String sz = " size:(" + (int) size + ")";
					if (size == 0.0)
					{
						sz = sz + " -- STUB";
					}
                    DitaToolForm.Out.WriteLine(prepend+"  Downloading DITA component: " + path + sz);
					
					if (!DitaToolForm.simulateDL)
					{
						writeToFileSystem(ditacomponent, parentFile);
					}
					
					if (type.ToUpper().Equals("ditamap".ToUpper()))
					{
						ditamap = parentFile;
					}
				}
			}
			
			return ditamap;
		}
		
		/// <summary> Write the P8 docuemnt to the local file system
		/// 
		/// </summary>
		/// <param name="parent">
		/// </param>
		/// <param name="parentFile">
		/// </param>
		public static void  writeToFileSystem(IDocument parent, System.IO.FileInfo parentFile)
        {
            FileStream fsIn = null;
            Stream sOut = null;
			try
			{
				bool hasContent = false;
				
				if (parent.Properties.IsPropertyPresent(PropertyNames.CONTENT_SIZE))
				{
					System.Double size = (Double) parent.Properties.GetFloat64Value(PropertyNames.CONTENT_SIZE);
					
					if (size != 0)
					{
						hasContent = (int) size > 0;
					}
				}
				
				if (hasContent)
				{
                    try
                    {
                        fsIn = new FileStream(parentFile.FullName, FileMode.CreateNew);
                        BinaryWriter bw = new BinaryWriter(fsIn);
                        sOut = parent.AccessContentStream(0);
                        byte[] data = new byte[sOut.Length];
                        sOut.Read(data, 0, data.Length);
                        sOut.Close();
                        bw.Write(data);
                        bw.Close();
                        fsIn.Close();
                    }
                    catch (Exception e)
                    {
                        System.Console.WriteLine(e.StackTrace);
                    }
				}
				else
				{
					// no content just create the file
                    DitaToolForm.Out.WriteLine("   Zero length DITA component: " + parentFile.FullName);
                    parentFile.Create();
				}
			}
			catch (System.IO.IOException io)
			{
				throw new EngineRuntimeException(io, ExceptionCode.E_UNEXPECTED_EXCEPTION, null);
			}
			finally
			{
				if (fsIn != null)
				{
					try
					{
						fsIn.Close();
					}
					catch (System.Exception )
					{
					}
				}
				if (sOut != null)
				{
					try
					{
						sOut.Flush();
						sOut.Close();
					}
					catch (System.Exception )
					{
					}
				}
			}
		}
	}