/*
    IBM grants you a nonexclusive copyright license to use all programming code 
	examples from which you can generate similar function tailored to your own 
	specific needs.

	All sample code is provided by IBM for illustrative purposes only.
	These examples have not been thoroughly tested under all conditions.  IBM, 
	therefore cannot guarantee or imply reliability, serviceability, or function of 
	these programs.

	All Programs or code component contained herein are provided to you AS IS  
	without any warranties of any kind.
	The implied warranties of non-infringement, merchantability and fitness for a 
	particular purpose are expressly disclaimed.

	 Copyright IBM Corporation 2008, ALL RIGHTS RESERVED.
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Text;
using System.Windows.Forms;
using FileNet.Api.Exception;
using FileNet.Api.Constants;
using FileNet.Api.Core;

//
// The sample tool form that gets information about the
// upload or download task request.
//
namespace SampDitaTool
{
    public partial class DitaToolForm : Form
    {
        CEConnection ce;
        IObjectStore os;
        IFolder ditaP8BaseFolder;
        IDocument srcDoc;

		// properties loaded from the xml configuration file
        private static string configFilename;
		private static System.Collections.Specialized.NameValueCollection props = new System.Collections.Specialized.NameValueCollection();

        // form extraction data

        public static String stURI;
        public static String stUsername;
        public static String stPassword;
        public static bool savePass;
        public static bool downloadNoGui;
        public static bool uploadNoGui;
        private static String stDomain;
        private static String stObjectStore;
        private static String stBaseDir;
        private static String stUploadDir;
        private static String stDownloadDir;
        private static String stDitamap;
        public static bool simulateUL;
        public static bool simulateDL;


        // which tab is active upload=true, download=false
        bool bUpload;

        // output stream which forks to both Console.Out and
        // the gui panel
        public static ForkIO Out = null;

        // defaults
        private const String URI = "http://localhost:9080/wsi/FNCEWS40MTOM/";
        //private const String URI = "http://hqdaphnesun1.usca.ibm.com:9080/wsi/FNCEWS40MTOM/";
        //private const String URI = "http://9.39.20.102:9080/wsi/FNCEWS40MTOM/";
        //private static String USER_NAME = "test1";
        //private static String PASSWORD = "test1";
        private static String USER_NAME = "daphnetest";
        private static String PASSWORD = "daphnetest";
        private const String DOMAIN_NAME = "mydomain";
        private const String OBJECT_STORE_NAME = "OS1";
        private const String P8_BASE_DIR = "/DITAProjects";
        private const String UPLOAD_DIR = "C:/ADM/dita/RegressionSuite/testdata/";
        private const String DOWNLOAD_DIR = "C:/ADM/dita/DlTest2";
        private const String DITAMAP = "articles/DITA-articles.ditamap";

        //
        // constructor given an existing connection to a P8 server
        public DitaToolForm(CEConnection ceInit)
        {
            ce = ceInit;
            InitializeComponent();

            // create the fork output stream to fork output to
            // Console.Out and the gui form
            Out = new ForkIO(textBoxStatusUL, textBoxStatusDL);

            // replace the out and error streams with our fork stream
            // to capture output to our gui and send it to stdout/err
            //StreamWriter swStatus = new StreamWriter(statusOutput);
            //System.Console.SetOut(statusOutput);
            //System.Console.SetError(statusOutput);

            SetInputs();

            bUpload = true;
        }

        //
        // setup the default
        //
        private static void SetDefaultInputs()
        {
            stURI = URI;
            stUsername = USER_NAME;
            stPassword = PASSWORD;
            stDomain = DOMAIN_NAME;
            stObjectStore = OBJECT_STORE_NAME;
            stBaseDir = P8_BASE_DIR;
            stUploadDir = UPLOAD_DIR;
            stDownloadDir = DOWNLOAD_DIR;
            stDitamap = DITAMAP;
            simulateUL = true;
            simulateDL = true;
        }

        //
        // set the current settings into the gui elements
        //
        private void SetInputs()
        {
            textBoxP8DomainUL.Text = stDomain;
            textBoxP8DomainDL.Text = stDomain;
            textBoxObjectStoreUL.Text = stObjectStore;
            textBoxObjectStoreDL.Text = stObjectStore;
            textBoxP8BaseDirUL.Text = stBaseDir;
            textBoxP8BaseDirDL.Text = stBaseDir;
            textBoxLocalDirUL.Text = stUploadDir;
            textBoxDirDL.Text = stDownloadDir;
            textBoxDitamapUL.Text = stDitamap;
            textBoxDitamapDL.Text = stDitamap;
            checkBoxSimulateUL.Checked = simulateUL;
            checkBoxSimulateDL.Checked = simulateDL;
        }

        //
        // extract the current gui elements into the local class vars
        //
        private void ExtractInput()
        {
            if (bUpload)
            {
                stDomain = textBoxP8DomainUL.Text;
                stObjectStore = textBoxObjectStoreUL.Text;
                stBaseDir = textBoxP8BaseDirUL.Text;
                stUploadDir = textBoxLocalDirUL.Text;
                stDitamap = textBoxDitamapUL.Text;
                simulateUL = checkBoxSimulateUL.Checked;
            }
            else
            {
                stDomain = textBoxP8DomainDL.Text;
                stObjectStore = textBoxObjectStoreDL.Text;
                stBaseDir = textBoxP8BaseDirDL.Text;
                stDownloadDir = textBoxDirDL.Text;
                stDitamap = textBoxDitamapDL.Text;
                simulateDL = checkBoxSimulateDL.Checked;
            }
            // set the values back since these are shared
            // between two panes
            SetInputs();
        }

        //
        // upload button clicked event
        //
        private void buttonUpload_Click(object sender, EventArgs e)
        {
            // get the input from the gui forms
            ExtractInput();

            // output informational messages
            if (simulateUL)
            {
                DitaToolForm.Out.WriteLine("SIMULATE upload started");
                DitaToolForm.Out.WriteLine("    NO UPLOAD WILL ACTUALLY OCCUR.");
            }
            else
                DitaToolForm.Out.WriteLine("Upload Started");
            DitaToolForm.Out.WriteLine("Domain: " + stDomain);
            DitaToolForm.Out.WriteLine("Object Store: " + stObjectStore);
            DitaToolForm.Out.WriteLine("P8 Base Dir: " + stBaseDir);
            DitaToolForm.Out.WriteLine("Upload Dir: " + stUploadDir);
            DitaToolForm.Out.WriteLine("Ditamap Dir: " + stDitamap);

            // fetch the domain and object store
            ce.FetchDomain(stDomain);
            os = ce.FetchOS(stObjectStore);

            // get the p8 base folder where the files will be uploaded
            ditaP8BaseFolder = null;
            try
            {
                ditaP8BaseFolder = Factory.Folder.FetchInstance(os, stBaseDir, null);
            }
            catch (EngineRuntimeException ex)
            {
                if (ex.GetExceptionCode() == ExceptionCode.E_OBJECT_NOT_FOUND)
                {
                    IFolder fldrRoot = (IFolder)os.RootFolder;

                    String folderName = stBaseDir;

                    if (folderName.StartsWith("/"))
                    {
                        folderName = folderName.Substring(1);
                    }

                    ditaP8BaseFolder = fldrRoot.CreateSubFolder(folderName);
                    ditaP8BaseFolder.Save(RefreshMode.REFRESH);
                }
                else
                {
                    Console.Error.WriteLine(ex.StackTrace);
                    throw ex;
                }
            }

            // do the upload
            DitaToolForm.Out.WriteLine("..Creating dita structure - " + "/" + stUploadDir + "/" + stDitamap);
            srcDoc = StoreDITADoc.CreateDitaSourceDoc(os, stBaseDir, stUploadDir, stDitamap);

            // output completion message
            if (simulateUL)
                DitaToolForm.Out.WriteLine("SIMULATE upload complete");
            else
                DitaToolForm.Out.WriteLine("Upload complete");

        }

        //
        // download button clicked event
        //
        private void buttonDownload_Click(object sender, EventArgs e)
        {
            // get the gui form data
            ExtractInput();

            // output informational messages
            if (simulateDL)
            {
                DitaToolForm.Out.WriteLine("SIMULATE download started\n");
                DitaToolForm.Out.WriteLine("    NO DOWNLOAD WILL ACTUALLY OCCUR.\n\n");
            }
            else
                DitaToolForm.Out.WriteLine("Download Started\n");
            DitaToolForm.Out.WriteLine("Domain: " + stDomain + "\n");
            DitaToolForm.Out.WriteLine("Object Store: " + stObjectStore + "\n");
            DitaToolForm.Out.WriteLine("P8 Base Dir: " + stBaseDir + "\n");
            DitaToolForm.Out.WriteLine("Download Dir: " + stDownloadDir + "\n");
            DitaToolForm.Out.WriteLine("Ditamap Dir: " + stDitamap + "\n");

            // fetch the domain and object store
            ce.FetchDomain(stDomain);
            os = ce.FetchOS(stObjectStore);
            
            // open the P8 document
            String ditaDocPath = null;

            if (stDitamap.StartsWith("/"))
            {
                ditaDocPath = stBaseDir + stDitamap;
            }
            else
            {
                ditaDocPath = stBaseDir + "/" + stDitamap;
            }

            IDocument ditaDoc = Factory.Document.FetchInstance(os, ditaDocPath, null);

            // get the needed document properties
            ditaDoc.Refresh(new string[] { "DocumentTitle", PropertyNames.FOLDERS_FILED_IN, PropertyNames.CONTENT_ELEMENTS, PropertyNames.CONTENT_SIZE });

            // copy the document and component relationships to the local disk
            GetDITADoc.copyCompoundDocumentToFilesystem(ditaDoc, stDownloadDir);

            // output completion message
            if (simulateDL)
                DitaToolForm.Out.WriteLine("SIMULATE download complete");
            else
                DitaToolForm.Out.WriteLine("Download complete");

        }

        //
        // tab change event handling
        //
        private void TabControl1_Selected(Object sender, TabControlEventArgs e)
        {
            if (e.TabPageIndex == 0)
                bUpload = true;
            else
                bUpload = false;
        }

        /// <summary> Set the default property values into the property object
        /// 
        /// </summary>
        static private void fillDefaultProperties()
        {
            SetDefaultInputs();
            props["uri"] = stURI;
            props["domain"] = stDomain;
            props["os"] = stObjectStore;
            props["user"] = stUsername;
            props["basedir"] = stBaseDir;
            props["uploaddir"] = stUploadDir;
            props["ditamap"] = stDitamap;
            if (simulateUL)
            {
                props["simulateUL"] = "true";
            }
            else
            {
                props["simulateUL"] = "false";
            }
            props["DownloadDir"] = stDownloadDir;
            if (simulateDL)
            {
                props["simulateDL"] = "true";
            }
            else
            {
                props["simulateDL"] = "false";
            }
        }

		/// <summary> Set the current property values into the property object
		/// 
		/// </summary>
		public static void  fillProperties()
		{
			props["uri"] = stURI;
			props["domain"] = stDomain;
			props["os"] = stObjectStore;
			props["user"] = stUsername;
            if (savePass)
            {
                props["pass"] = stPassword;
            }
            else
            {
                props["pass"] = "";
            }
            props["basedir"] = stBaseDir;
            props["uploaddir"] = stUploadDir;
			props["ditamap"] = stDitamap;
			if (simulateUL)
			{
				props["simulateUL"] = "true";
			}
			else
			{
				props["simulateUL"] = "false";
			}
			props["DownloadDir"] = stDownloadDir;
			if (simulateDL)
			{
				props["simulateDL"] = "true";
			}
			else
			{
				props["simulateDL"] = "false";
			}
		}
		
		/// <summary> Save the current property object into the config file
		/// 
		/// </summary>
		public static void  saveProperties()
		{
			try
			{
				System.IO.FileStream fos = null;
				if (configFilename == null)
				{
					fos = new System.IO.FileStream("ditatool.dat", System.IO.FileMode.Create);
				}
				else
				{
					fos = new System.IO.FileStream(configFilename, System.IO.FileMode.Create);
				}
                StreamWriter sw = new StreamWriter(fos);

                String[] propkeys = props.AllKeys;
                int i;
                for (i=0;i<propkeys.Length;i++)
                {
                    String cur = props.Get(propkeys[i]);
                    sw.WriteLine(propkeys[i] + "=" + cur);
                    Console.Out.WriteLine(propkeys[i] + "=" + cur);
                }
                sw.Close();
                fos.Close();
			}
			catch (System.IO.IOException e)
			{
				Console.Out.WriteLine(e.StackTrace);
			}
		}

    public static void loadProperties(string configFilenameInit)
    {
            configFilename = configFilenameInit;

			// try to load the properties from a file in the local directory
			try
			{
				System.IO.FileStream fis = null;
				if (configFilename == null)
				{
					Console.Out.WriteLine("Loading settings from ditatool.dat");
					fis = new System.IO.FileStream("ditatool.dat", System.IO.FileMode.Open, System.IO.FileAccess.Read);
				}
				else
				{
                    Console.Out.WriteLine("Loading settings from " + configFilename);
					fis = new System.IO.FileStream(configFilename, System.IO.FileMode.Open, System.IO.FileAccess.Read);
				}

                Console.Out.WriteLine("Starting prefs read");
                StreamReader sr = new StreamReader(fis);
                String line;
                while ((line = sr.ReadLine()) != null) 
                {
                    Console.Out.WriteLine(line);

                    int eqPos = line.IndexOf("=");
                    if (eqPos > 0)
                    {
                        String key = line.Substring(0, eqPos);
                        String val = line.Substring(eqPos + 1, line.Length - (eqPos+1));
                        props[key] = val;
                    }
                }
				
				// extract the properties into local variables
				stURI = props.Get("uri");
				stDomain = props.Get("domain");
				stObjectStore = props.Get("os");
				stUsername = props.Get("user");

				//p8BaseDir = props.Get("p8base");
				if ((props["pass"]!=null) && (props["pass"].Length>0))
				{
					savePass = true;
					stPassword = props.Get("pass");
				}
				else
				{
					savePass = false;
					stPassword = "";
				}
                stUploadDir = props.Get("uploaddir");
				stBaseDir = props.Get("basedir");
				stDitamap = props.Get("ditamap");
				if ((props["simulateUL"]!=null) && (props["simulateUL"].Equals("true")))
					simulateUL = true;
				else
					simulateUL = false;
				
				if ((props["simulateDL"]!=null) && (props["simulateDL"].Equals("true")))
					simulateDL = true;
				else
					simulateDL = false;
				stDownloadDir = props.Get("DownloadDir");
                sr.Close();
                fis.Close();
			}
			catch (System.IO.IOException e)
			{
				if (configFilename != null)
				{
					System.Console.Out.WriteLine("Error: config file you specified: " + configFilename + " does not exist");
					System.Console.Out.WriteLine("       You can generate one by running this application in a simulated upload");
					return ;
				}
				
				// set default properties
				fillDefaultProperties();
			}
        }
    }

    //
    // Fork IO class to fork stream output to the gui and Console.Out
    //
    public class ForkIO : TextWriter
    {
        TextBox statusUL;
        TextBox statusDL;

        TextWriter stderr;
        TextWriter stdout;

        //
        // Constructor wit the two status text boxes
        //
        public ForkIO(TextBox statusULInit, TextBox statusDLInit)
        {
            statusUL = statusULInit;
            statusDL = statusDLInit;
            stderr = Console.Error;
            stdout = Console.Out;
        }

        //
        // overloaded WriteLine method to fork the stream data
        //
        public override void WriteLine(String st)
        {
            statusUL.AppendText(st+"\r\n");
            statusDL.AppendText(st+"\r\n");

            stdout.WriteLine(st);
        }

        //
        // Required interface method. No implementation.
        //
        public override Encoding Encoding
        {
            get { throw new Exception("The method or operation is not implemented."); }
        }
    }
}