/*
    IBM grants you a nonexclusive copyright license to use all programming code 
	examples from which you can generate similar function tailored to your own 
	specific needs.

	All sample code is provided by IBM for illustrative purposes only.
	These examples have not been thoroughly tested under all conditions.  IBM, 
	therefore cannot guarantee or imply reliability, serviceability, or function of 
	these programs.

	All Programs or code component contained herein are provided to you AS IS  
	without any warranties of any kind.
	The implied warranties of non-infringement, merchantability and fitness for a 
	particular purpose are expressly disclaimed.

	 Copyright IBM Corporation 2008, ALL RIGHTS RESERVED.
 */
using System;
using System.Collections;
using System.IO;
using System.Text;
using System.Xml;
using System.Xml.XPath;
using FileNet.Api.Exception;

using SampDitaTool;

//
// class to parse DITA documents and create P8 compound document
// structures.
//
public class DitaParser
{
	public string ComponentP8DitaClass
	{
		// Return P8 document class name for ditacomponent
		// Use this document class when creating the ditacomponent in P8
        get
        {
            if (componentP8DitaClass == null)
            {
                try
                {
                    string rootElement = ((XmlElement)domDoc.DocumentElement).Name;

                    if (ditatypeMap.Contains(rootElement))
                    {
                        componentP8DitaClass = ((string)ditatypeMap[rootElement]);
                        rootNode = (XmlElement)domDoc.DocumentElement;
                    }
                    else
                    {
                        throw new SystemException("Error: unsupported DITA class: " + rootElement);
                    }
                }
                catch (System.Exception t)
                {
                    throw new SystemException(t.Message);
                }
            }

            return componentP8DitaClass;
        }
	}

    // Returns mapping of relationships to other file found in ditacomponent
    // The Map will have the following key:value pairs:
    //
    //      Key:   String value that specifies filesystem path to related file
    //      Value: String value that specifies the P8 Component Relationhip class name 
    //             that should be used to create the relationship in P8
    public IDictionary ComponentRelationships
	{	
		get
		{
			if (fileRelationshipMap == null)
			{
				try
				{
					string generatedAux = ComponentP8DitaClass;
					
					fileRelationshipMap = new System.Collections.Hashtable();
					
					FindRelationships(fileRelationshipMap, "//*[@href]", DITATOPICREF_CLASS);
					
					FindRelationships(fileRelationshipMap, "//@conref", DITACONTENTREF_CLASS);
				}
				catch (System.Exception t)
				{
					throw new SystemException(t.Message);
				}
			}
			return fileRelationshipMap;
		}
		
	}
	
    //
    // get the properties of a component
    //
    public IDictionary ComponentProperties
	{
		get
		{
			return propertiesMap;
		}
		
	}

    // document, map and xml parse information
	private FileInfo ditacomponent = null;
	private string componentP8DitaClass = null;
	private IDictionary fileRelationshipMap = null;
	private IDictionary propertiesMap = null;
	private XmlNode rootNode = null;
	private XmlDocument domDoc = null;
	
	// From org.apache.xerces.impl.Constants
	public const string XERCES_FEATURE_PREFIX = "http://apache.org/xml/features/";
	public const string LOAD_EXTERNAL_DTD_FEATURE = "nonvalidating/load-external-dtd";
	
    // P8 classes
	private const string DITAMAP_CLASS = "DitaMap";
	private const string DITABOOKMAP_CLASS = "DitaBookMap";
	private const string DITACONCEPT_CLASS = "DitaConcept";
	private const string DITAREFERENCE_CLASS = "DitaReference";
	private const string DITATASK_CLASS = "DitaTask";
	private const string DITATOPIC_CLASS = "DitaTopic";
	private const string DITABASE_CLASS = "DitaBase";
	
	private const string DITABASEREF_CLASS = "DitaBaseref";
	private const string DITATOPICREF_CLASS = "DitaTopicref";
	private const string DITAMAPREF_CLASS = "DitaMapref";
	private const string DITAIMAGEREF_CLASS = "DitaImageref";
	private const string DITACONTENTREF_CLASS = "DitaContentref";
	
	private const bool REQUIRED = true;
	private const bool NOT_REQUIRED = false;
	
	private static IDictionary ditatypeMap = null;
	private static IDictionary hrefMap = null;
	private static IDictionary basicLinkTypeMap = null;
	private static IDictionary xrefLinkTypeMap = null;
	private static IDictionary inheritAttributeParents = null;
	
	// constructor
	protected internal DitaParser()
	{
	}
	

	//
    // constructor which creates the parse with a given local file
    //
	public DitaParser(FileInfo ditacomponent)
	{
		bool tmpBool;
		if (File.Exists(ditacomponent.FullName))
			tmpBool = true;
		else
			tmpBool = Directory.Exists(ditacomponent.FullName);
		if (!tmpBool)
		{
			throw new SystemException("DITA component does not exist: " + ditacomponent.FullName);
		}
		
		this.ditacomponent = ditacomponent;
		
		try
		{
            // setup the parser and parse into dom document
            XmlTextReader txtReader = new XmlTextReader(ditacomponent.FullName);
            XmlValidatingReader reader = new XmlValidatingReader(txtReader);
            reader.ValidationType = ValidationType.None;
            reader.XmlResolver = null;
            domDoc = new XmlDocument();
            domDoc.Load(reader);
		}
		catch (System.Exception t)
		{
            DitaToolForm.Out.WriteLine(t.StackTrace);
			throw new SystemException(t.Message);
		}
	}
	
    //
    // get the value of a given xml attribute
    //
	private string GetNodeValue(System.Xml.XmlNamedNodeMap attribMap, string name, bool required)
	{
		System.Xml.XmlNode n = attribMap.GetNamedItem(name);
		string value_Renamed = null;
		if (n == null)
		{
			if (required)
			{
				throw new SystemException("Required value not found: " + name);
			}
		}
		else
		{
			value_Renamed = n.Value;
		}
		return value_Renamed;
	}
	
    //
    // get the node value of the xml node based on the query
    //
	private string GetNodeValue(string queryStr, bool required)
	{
		string value_Renamed = null;
		
		try
		{
			// XmlNodeList nodes = (XmlNodeList) xPath.Evaluate(queryStr, domDoc, XPathConstants.NODESET);
            XmlNodeList nodes = domDoc.SelectNodes(queryStr);
			
			int count = nodes.Count;
			
			if (count == 1)
			{
                System.Xml.XmlNode n = nodes.Item(0);
				string name = n.Name;
				
				value_Renamed = n.Value;
			}
			else if (count > 1)
			{
				throw new SystemException("two manu value found: " + queryStr + ", count = " + System.Convert.ToString(count));
			}
		}
		catch (System.Exception t)
		{
			throw new SystemException(t.Message);
		}
		
		if (value_Renamed == null && required)
		{
			throw new SystemException("required value not found: " + queryStr);
		}
		return value_Renamed;
	}
	
    //
    // get a list of node values based on a query string
    //
	private System.Collections.IList GetNodeMultiValue(string queryStr, bool required)
	{
		IList list = new ArrayList();
		
		try
		{
            XmlNodeList nodes = domDoc.SelectNodes(queryStr);

			int count = nodes.Count;
			
			for (int ix = 0; ix < count; ix++)
			{
				System.Xml.XmlNode n = nodes.Item(ix);
				
				string name = n.Name;
				string val = n.Value;
				
				if (val != null && val.Length > 0)
				{
					list.Add(val);
				}
			}
		}
		catch (Exception t)
		{
			throw new SystemException(t.Message);
		}
		
		if (list.Count == 0 && required)
		{
			throw new SystemException("required value not found: " + queryStr);
		}
		
		return list;
	}
	
	//
    // add a property to the map
    //
	private void  AddProperty(IDictionary map, string name, Object value_Renamed)
	{
		if (value_Renamed != null)
		{
			if (value_Renamed.GetType().FullName.Equals("String", StringComparison.OrdinalIgnoreCase))
			{
				string val = (string) value_Renamed;
				if (val.Length == 0)
				{
					return ;
				}
			}
			map[name] = value_Renamed;
		}
	}
	
    //
    // parse the map looking for related documents
    //
	private void  FindRelationships(IDictionary relationshipMap, string queryStr, string defaultRelType)
	{
		try
		{
			// XmlNodeList nodes = (XmlNodeList) xPath.evaluate(queryStr, domDoc, XPathConstants.NODESET);
            XmlNodeList nodes = domDoc.SelectNodes(queryStr);

			int count = nodes.Count;
			
			for (int ix = 0; ix < count; ix++)
			{
				System.Xml.XmlNode n = nodes.Item(ix);
				
				string filePath = null;
				string relType = defaultRelType;
				bool externalNode = false;
				
				if (defaultRelType.Equals(DITACONTENTREF_CLASS, StringComparison.OrdinalIgnoreCase))
				{
					// "//@conref"
					filePath = n.Value;
				}
				else
				{
					// "//@href"
					
					relType = DITATOPICREF_CLASS;
					
					System.Xml.XmlNode parentNode = n;
					
					if (parentNode != null)
					{
						IDictionary values = ExtractNodeValues(parentNode);
						string parentName = (string) values["parentName"];
						
					
						if (hrefMap[parentName] != null)
						{
							relType = ((string) hrefMap[parentName]);
						}
						
						filePath = ((string) values["href"]);
						
						// if "href" not present in topicref (legal!) ignore it
						if (filePath != null)
						{
							string format = (string) values["format"];
							
							if (format != null)
							{
								if (format.Equals("ditamap", StringComparison.OrdinalIgnoreCase))
								{
									relType = DITAMAPREF_CLASS;
								}
								else if (!format.Equals("dita", StringComparison.OrdinalIgnoreCase))
								{
									externalNode = !IsFilesystemBased(filePath);
									relType = DITABASEREF_CLASS;
								}
							}
							else
							{
								string type = (string) values["type"];
								
								if (type != null)
								{
									string linkType = (string) basicLinkTypeMap[type];
									if (linkType == null)
									{
										if (parentName.Equals("xref", StringComparison.OrdinalIgnoreCase))
										{
											linkType = ((string) xrefLinkTypeMap[type]);
										}
										
										if (linkType == null)
										{
											externalNode = !IsFilesystemBased(filePath);
											// the "type" attribute references a non-standard type
											// modify relationship type to basic non-DITA
											relType = DITABASEREF_CLASS;
										}
									}
								}
								else if (parentName.Equals("link", StringComparison.OrdinalIgnoreCase))
								{
									externalNode = IsLinkExternal(parentNode);
								}
							}
							string scope = (string) values["scope"];
							if (scope != null)
							{
								if (scope.Equals("external", StringComparison.OrdinalIgnoreCase))
								{
									// "scope=external" overrides format=dita
									externalNode = !IsFilesystemBased(filePath);
									relType = DITABASEREF_CLASS;
								}
							}
						}
						else
						{
							filePath = null;
						}
					}
				}
				if (!externalNode && filePath != null)
				{
					int index = filePath.IndexOf("#");
					
					// index == 0, conref to within current file so ignore it
					if (index != 0)
					{
						if (index != - 1)
						{
							relType = DITACONTENTREF_CLASS;
							filePath = filePath.Substring(0, (index) - (0));
						}
						
						//System.out.println("relationship type = " + relType + ", file = " + filePath);
						relationshipMap[filePath] = relType;
					}
				}
			}
		}
		catch (Exception t)
		{
			throw new SystemException(t.Message);
		}
	}
	
    //
    // check to see if the link is an external file link
    //
	private bool IsLinkExternal(System.Xml.XmlNode parentNode)
	{
		bool external = false;
		
		System.Xml.XmlNode grandparent = parentNode.ParentNode;
		
		if (grandparent != null)
		{
			string name = grandparent.Name;
			
			if (name != null && inheritAttributeParents.Contains(name))
			{
				System.Collections.IDictionary values = ExtractNodeValues(grandparent);
				
				string format = (string) values["format"];
				
				if (format != null && !format.Equals("dita", StringComparison.OrdinalIgnoreCase))
				{
					external = true;
				}
				
				string scope = (string) values["scope"];
				
				if (scope != null && scope.Equals("external", StringComparison.OrdinalIgnoreCase))
				{
					external = true;
				}
			}
		}
		
		return external;
	}

    //
    // get the values of a given node
    //
	private IDictionary ExtractNodeValues(XmlNode n)
	{
		IDictionary values = new Hashtable();
		
		XmlNamedNodeMap nMap = (XmlAttributeCollection) n.Attributes;
		
		values["parentName"] = n.Name;
		
		if (nMap != null)
		{
			XmlNode nFilePath = nMap.GetNamedItem("href");
			string filePath = nFilePath != null?nFilePath.Value:null;
			
			XmlNode nFormat = nMap.GetNamedItem("format");
			string format = nFormat != null?nFormat.Value:null;
			
			XmlNode nScope = nMap.GetNamedItem("scope");
			string scope = nScope != null?nScope.Value:null;
			
			XmlNode nType = nMap.GetNamedItem("type");
			string type = nType != null?nType.Value:null;
			
			values["href"] = filePath;
			values["format"] = format;
			values["scope"] = scope;
			values["type"] = type;
		}
		else
        {
			//string data = n.GetTextContent();
           	string data = n.Value;
	
			if (data != null)
			{
				bool pairFound = true;
				data = data.Trim();
                DitaToolForm.Out.WriteLine(data);

                string[] st = data.Split('='); 
                for (int i=0; i<st.Length; i++)
                {
                    pairFound = false;
                    string key = st[i].Trim();
                    for (int j = i+1; j < st.Length; j++)
                    {
                        string val = st[j].Trim();
                        if (val != null && val.Length > 0)
                        {
                            values[key] = val;
                            pairFound = true;
                            break;
                        }
                    }

                }

				if (!pairFound)
				{
					throw new SystemException();
				}
			}
		}
		
		return values;
	}

    //
    // see if a path is a file on the local filesystem
    //
	private bool IsFilesystemBased(string path)
	{
		bool isFileBased = false;
		
		if (path != null && path.Length > 0)
		{
			try
			{
				System.Uri url = new System.Uri(path);
			}
			catch (System.UriFormatException )
			{
				if (path.IndexOf(":") <= 0)
				{
					// crude but appears to work
					// If ":" not in path, UNIX based
					// If ":" present but index > 0, then URL with unknown protocol like news://...
					// NOTE: This works because only hrefs with a unknown format get here, 
					//       so no relative paths to file system do get here.
					isFileBased = true;
				}
			}
		}
		return isFileBased;
	}

    //
    // create a static version of the parser
    //
	static DitaParser()
	{
		{
			ditatypeMap = new System.Collections.Hashtable();
			ditatypeMap["map"] = DITAMAP_CLASS;
			ditatypeMap["bookmap"] = DITABOOKMAP_CLASS;
			ditatypeMap["concept"] = DITACONCEPT_CLASS;
			ditatypeMap["reference"] = DITAREFERENCE_CLASS;
			ditatypeMap["task"] = DITATASK_CLASS;
			ditatypeMap["topic"] = DITATOPIC_CLASS;
			ditatypeMap["dita"] = DITABASE_CLASS;
			
			hrefMap = new System.Collections.Hashtable();
			
			hrefMap["topic"] = "DitaTopicref";
			hrefMap["xref"] = "DitaXref";
			hrefMap["image"] = "DitaImageref";
			hrefMap["conref"] = "DitaContentref";
			
			basicLinkTypeMap = new System.Collections.Hashtable();
			
			basicLinkTypeMap["concept"] = "concept";
			basicLinkTypeMap["task"] = "task";
			basicLinkTypeMap["reference"] = "reference";
			basicLinkTypeMap["topic"] = "topic";
			
			xrefLinkTypeMap = new System.Collections.Hashtable();
			
			xrefLinkTypeMap["fig"] = "fig";
			xrefLinkTypeMap["table"] = "table";
			xrefLinkTypeMap["li"] = "li";
			xrefLinkTypeMap["fn"] = "fn";
			xrefLinkTypeMap["section"] = "section";
			
			inheritAttributeParents = new System.Collections.Hashtable();
			
			inheritAttributeParents["linklist"] = "linklist";
			inheritAttributeParents["linkpool"] = "linkpool";
		}
	}
}
