/**
	IBM grants you a nonexclusive copyright license to use all programming code 
	examples from which you can generate similar function tailored to your own 
	specific needs.

	All sample code is provided by IBM for illustrative purposes only.
	These examples have not been thoroughly tested under all conditions.  IBM, 
	therefore cannot guarantee or imply reliability, serviceability, or function of 
	these programs.

	All Programs or code component contained herein are provided to you AS IS  
	without any warranties of any kind.
	The implied warranties of non-infringement, merchantability and fitness for a 
	particular purpose are expressly disclaimed.

	 Copyright IBM Corporation 2008, ALL RIGHTS RESERVED.
 */

package sampdita;

import java.io.*;
import java.util.*;


import com.filenet.api.constants.*;
import com.filenet.api.util.*;
import com.filenet.api.collection.*;
import com.filenet.api.property.*;
import com.filenet.api.exception.*;
import com.filenet.api.core.*;

public class StoreDITADoc 
{
	private static Domain	   d  = null;
	private static ObjectStore os = null;
	
    /**
     * Constructor for TestApp
     */
    public StoreDITADoc(Domain domain, ObjectStore objStore)
    {
    	d = domain;
    	os = objStore;
   }
   

    /**
     * Using the baseDir and file name, place the ditamap (filename) into
     * P8 as a compound document structure under P8BaseDir.
     * 
     * @param p8BaseDir
     * @param baseDir
     * @param filename
     * @return
     */
	public Document testCreateP8DITAStructureFromFilesystem(String p8BaseDir, String baseDir, String filename)
	{
		SampDITATool.out.println("Create DITA Structure From Filesystem");

		// p8BaseDir must exist in P8 object store
		Folder DITAP8BaseFolder = null;
		
		if (!SampDITATool.simulate)
		{
			DITAP8BaseFolder = Factory.Folder.fetchInstance(os, p8BaseDir, null);
		}
		SampDITATool.out.println("p8 Base Directory: "+p8BaseDir);
		SampDITATool.out.println("Local Directory:"+baseDir);
		
		File   FileSystemBaseFolder = new File(baseDir);
		
		Map ditaDocs = new HashMap();
		List ditaOrderedDocs = new ArrayList();

		// create a map of all the associated documents
		findDITAComponents(baseDir, filename, ditaDocs, ditaOrderedDocs, "ditamap",null);

		int count = ditaOrderedDocs.size();
		
		String path = null;
		String relType = null;
		Document ditamap = null;
		
		// iterate through all the documents
		for(int ix = count - 1; ix >= 0; ix--)
		{
			// get the path, and relationship for the local file
			path = (String) ditaOrderedDocs.get(ix);
		    relType = (String) ditaDocs.get(path);
			boolean autoClassify = !nonDITARelationship(relType);
		    
			// open the local file
			File doc = new File(path);

			SampDITATool.out.println("Persist document: "+doc);
			if (!SampDITATool.simulate)
			{
				Document parentDoc = persistDocument(doc,
	                    "Document",
						DITAP8BaseFolder,
						FileSystemBaseFolder,
						autoClassify);

				if(relType.equals("ditamap"))
				{
					ditamap = parentDoc;
				}
			}
		}
		SampDITATool.out.println("Create DITA Structure From Filesystem - Done");
		
		return ditamap;
	}
	
	/**
	 * Find the all the related documents to ditamap specified by filename.
	 * All the files and their relationships are stored in ditaDocs and
	 * ditaOrderedDocs.
	 * 
	 * Since this is recursive, the indent string expands with each
	 * recursive call so files output to the log will be indented
	 * together depending on their parental relationships.
	 * 
	 * @param baseDir
	 * @param filename
	 * @param ditaDocs
	 * @param ditaOrderedDocs
	 * @param relType
	 * @param indent
	 */

	private void findDITAComponents(String baseDir, 
			String filename,
			Map ditaDocs,
			List ditaOrderedDocs, 
			String relType,
			String indent)
	{
		Document parentDoc = null;
	
		if (indent == null)
			indent = "";
		else
			indent = indent+"    ";
		
		try
		{
			String newDocPath = resolveRelativePath(baseDir, filename);
			
			if(ditaDocs.containsKey(newDocPath))
			{
				return;
			}
			
			// open the current document
			File newDoc = new File(newDocPath);
			
			// get the path to that document
			String newBaseDir = newDoc.getPath();
			
			// extract the final directory from the path
			int index = newBaseDir.lastIndexOf(File.separator);
			if(index == -1)
			{
				throw new RuntimeException("No separator found: " + newBaseDir);
			}
			newBaseDir = newBaseDir.substring(0, index + 1);
			
			// create a parser for the current document
			DITAParser dp = new DITAParser(newDoc);
			SampDITATool.out.println(indent + newDoc.getPath());
			
			// find the relationships to this document
			addRelationship(ditaDocs,ditaOrderedDocs, newDocPath, relType);

			// get a map of all the documents related to this one
			Map holder = dp.getComponentRelationships();
			Map relationships = new HashMap(holder);
			
			dp = null;
			
			// iterate through all the child documents
			Iterator it = relationships.keySet().iterator();
			
			while(it.hasNext())
			{
				String childFilePath = (String) it.next();
				relType       = (String) relationships.get(childFilePath);
				
				String newPath = resolveRelativePath(newBaseDir, childFilePath);
					
				// only process if this document has not already been parsed
				if(!ditaDocs.containsKey(newPath))
				{
					// just add non-dita documents to the list
					if(nonDITARelationship(relType))
					{
						// Non DITA file (image, pdf, etc.)
						addRelationship(ditaDocs,ditaOrderedDocs, newPath, relType);
						SampDITATool.out.println(indent+"    relationship: " + newPath +  "(" + relType + ")");	

					}
					else
					{
						// recursively parse child dita documents
						findDITAComponents(newBaseDir,
								childFilePath,
								ditaDocs,
								ditaOrderedDocs, 
								relType,
								indent);
					}
				}
			}
		}
		catch(Throwable t)
		{
	        SampDITATool.out.println(t.toString());
			throw new RuntimeException(t.getLocalizedMessage());
		}
	}
	
	/**
	 * Add a relationship to path
	 * 
	 * @param ditaDocs
	 * @param ditaOrderedDocs
	 * @param path
	 * @param relType
	 */
	private void addRelationship(Map ditaDocs, List ditaOrderedDocs, String path, String relType)
	{
		if(!ditaDocs.containsKey(path))
		{
			ditaDocs.put(path, relType);
			ditaOrderedDocs.add(path);
		}
	}
	
	/**
	 * Check to see if a document type is a dita document.
	 * 
	 * @param relType
	 * @return
	 */
	private boolean nonDITARelationship(String relType)
	{
		return (relType.equalsIgnoreCase("DitaImageref") || relType.equalsIgnoreCase("DitaBaseref"));
	}
	
	/**
	 * check to s
	 * @param currentBaseDir
	 * @param childFilePath
	 * @return
	 */
	private String resolveRelativePath(String currentBaseDir, String childFilePath)
	{
		// Shouldn't need to use both delimiters... but
		StringTokenizer st = new StringTokenizer(childFilePath, "/" + "\\");
		
		int count = 0;
		
		// count the number of .. relative paths
		while(st.hasMoreTokens())
		{
			String token = st.nextToken();
			
			if(token.equals(".."))
			{
				count++;
			}
		}
		
		// extract the .. relative paths
		for(int ix = 0; ix < count; ix++)
		{
			int index = currentBaseDir.lastIndexOf(File.separator);
			int len = currentBaseDir.length();
			
			if(index + 1 == len)
			{
				currentBaseDir = currentBaseDir.substring(0, len - 1);
				
				index = currentBaseDir.lastIndexOf(File.separator);
			}
			
			if(index == -1)
			{
				throw new RuntimeException("File separator not found: " + currentBaseDir);
			}
			
			currentBaseDir = currentBaseDir.substring(0, index + 1);
			
			childFilePath = childFilePath.substring(3, childFilePath.length());
		}
		// make sure the file/directory still exists
		String path = currentBaseDir + childFilePath;
		File f = new File(path);
		if(!f.exists())
		{
			throw new RuntimeException("files not not exist: " + path);
		}
		try
		{
			path = f.getCanonicalPath();
		}
		catch(Throwable t)
		{
			throw new RuntimeException(t.getLocalizedMessage());
		}
		return path;	
	}
 
	/**
	 * Upload a document to P8 that already exists.
	 * 
	 * @param ditacomponent
	 * @param name
	 * @param filesysDoc
	 * @param f
	 * @return
	 */
	private Document updateExistingP8Document(Document ditacomponent, String name, File filesysDoc, Folder f)
	{
		// make sure the files is not zero length.
		if(ditacomponent.getProperties().isPropertyPresent(PropertyNames.CONTENT_SIZE))
		{
			Double size = (Double) ditacomponent.getProperties().getFloat64Value(PropertyNames.CONTENT_SIZE);
			
			boolean hasContent = false;
			
			if(size != null)
			{
				hasContent = size.intValue() > 0;
			}
			
			// write files that have actual content in them.
			if(!hasContent)
			{
				ditacomponent.checkout(ReservationType.EXCLUSIVE, null, null, null);
				ditacomponent.save(RefreshMode.REFRESH);
				
				com.filenet.api.core.Document reservation = 						
						(com.filenet.api.core.Document) ditacomponent.get_Reservation();
				
				reservation.getProperties().removeFromCache(PropertyNames.CONTENT_SIZE); 
				reservation.set_CompoundDocumentState(CompoundDocumentState.COMPOUND_DOCUMENT);    
								    	
				String mimeType = "application/dita+xml";
				if(name.indexOf(".gif") != -1)
				{
					mimeType = "image/gif";
				}
				else if(name.indexOf(".jpg") != -1)
				{
					mimeType = "image/jpg";
				}
				else if(name.indexOf(".jpeg") != -1)
				{
					mimeType = "image/jpeg";
				}
				
				reservation.checkin(mimeType.equalsIgnoreCase("application/dita+xml") ? AutoClassify.AUTO_CLASSIFY : AutoClassify.DO_NOT_AUTO_CLASSIFY,
			              CheckinType.MAJOR_VERSION);
	
				reservation.getProperties().putValue("DocumentTitle", name);
				ContentTransfer ct = createContentTransfer(filesysDoc, name, mimeType);
				ContentElementList cel = Factory.ContentElement.createList();
		        cel.add(ct);
		        reservation.set_ContentElements(cel);
		        reservation.save(RefreshMode.REFRESH);  
		        ditacomponent = reservation;
		        
		        SampDITATool.out.println("   update existing P8 document: " + name + ", in folder: " + f.get_PathName());

			}
		}
		
		return ditacomponent;

	}
	
	/**
	 * Create a new document on the P8 server.
	 *  
	 * @param p8ClassName
	 * @param name
	 * @param filesysDoc
	 * @param f
	 * @param autoClassify
	 * @return
	 */
	private Document createNewP8Document(String p8ClassName,
			                             String name,
										 File filesysDoc,
										 Folder f,
										 boolean autoClassify)
	{

        UpdatingBatch ub = UpdatingBatch.createUpdatingBatchInstance(os.get_Domain(), RefreshMode.REFRESH);

		String title = null;
		
    	Document ditacomponent = Factory.Document.createInstance(os, p8ClassName);
    	ditacomponent.set_CompoundDocumentState(CompoundDocumentState.COMPOUND_DOCUMENT);    	
    	ditacomponent.checkin(autoClassify ? AutoClassify.AUTO_CLASSIFY : AutoClassify.DO_NOT_AUTO_CLASSIFY,
    			              CheckinType.MAJOR_VERSION);
    	
    	
		String mimeType = "application/dita+xml";
		if(name.indexOf(".gif") != -1)
		{
			mimeType = "image/gif";
		}
		else if(name.indexOf(".jpg") != -1)
		{
			mimeType = "image/jpg";
		}
		else if(name.indexOf(".jpeg") != -1)
		{
			mimeType = "image/jpeg";
		}
    	ditacomponent.getProperties().putValue("DocumentTitle", name);
		ContentTransfer ct = createContentTransfer(filesysDoc, name, mimeType);
		ContentElementList cel = Factory.ContentElement.createList();
        cel.add(ct);
        ditacomponent.set_ContentElements(cel);
        
        ub.add(ditacomponent, null);
        
        
		ReferentialContainmentRelationship rcr = f.file(ditacomponent,
	    			                                    AutoUniqueName.NOT_AUTO_UNIQUE,
														name,
														DefineSecurityParentage.DO_NOT_DEFINE_SECURITY_PARENTAGE);
	    ub.add(rcr, null);
	    
	    ub.updateBatch();
	    
	    SampDITATool.out.println("   persist new P8 document: " + name + ", in folder: " + f.get_PathName());
    	
    	return ditacomponent;
	}
	
	/**
	 * Upload the file to the P8 server.
	 * 
	 * @param filesysDoc
	 * @param p8ClassName
	 * @param DITABaseFolder
	 * @param FileSystemBaseFolder
	 * @param autoClassify
	 * @return
	 */
	private Document persistDocument(File filesysDoc,
			                         String p8ClassName,
									 Folder DITABaseFolder,
									 File FileSystemBaseFolder,
									 boolean autoClassify)
	{
		if(!filesysDoc.exists())
		{
			throw new RuntimeException("Filesystem file does not exist: " + filesysDoc.getPath());
		}

		Document ditacomponent = null;
		Folder folder = null;
		
		String name = filesysDoc.getName();

		try
		{	
			folder = getSubFolder(DITABaseFolder, FileSystemBaseFolder, filesysDoc);
	
			ditacomponent = getExistingP8Document(folder, name);
			
			if(ditacomponent != null)
			{
				ditacomponent = updateExistingP8Document(ditacomponent, name, filesysDoc, folder);
			}
			else
			{
				ditacomponent = createNewP8Document(p8ClassName, name, filesysDoc, folder, autoClassify);
			}
		}
		catch(EngineRuntimeException t)
		{
			if(t.getExceptionCode() == ExceptionCode.E_NOT_UNIQUE)
			{
				SampDITATool.out.println("*** E_NOT_UNIQUE exception thrown, most likey stub document created by DITA Classifier.");
				SampDITATool.out.println("***     Check if document now exists and update.");

				ditacomponent = getExistingP8Document(folder, name);
	
				if(ditacomponent != null)
				{
					ditacomponent = updateExistingP8Document(ditacomponent, name, filesysDoc, folder);	
					return ditacomponent;
				}
			}
			t.printStackTrace();
			throw new RuntimeException(t.getLocalizedMessage());
		}
			
    	return ditacomponent;
	}
	
	/**
	 * See if the file already exists on the server.
	 * 
	 * @param folder
	 * @param name
	 * @return
	 */
	private Document getExistingP8Document(Folder folder, String name)
	{
		Document doc = null;
		try
		{
			PropertyFilter pf = new PropertyFilter();
			pf.addIncludeProperty(1, null, Boolean.FALSE, PropertyNames.CONTENT_SIZE, null);
			String filepath = folder.get_PathName() + "/" + name;
			doc = Factory.Document.fetchInstance(os, filepath,pf);
			SampDITATool.out.println("*** P8 Document Exists: " + filepath);
			
		}
		catch(Throwable t)
		{
			// ignore assume doesn't exists - expensive way to do this
			doc = null;
		}
		return doc;
	}
	
	/**
	 * Gets the P8 subfolder matching the local folder.
	 * 
	 * @param DITABaseFolder
	 * @param FileSystemBaseFolder
	 * @param filesysDoc
	 * @return
	 */
	private Folder getSubFolder(Folder DITABaseFolder, File FileSystemBaseFolder, File filesysDoc)
	{
		Folder f = DITABaseFolder;
		
		List subfolders = diffFolders(FileSystemBaseFolder, filesysDoc);
		
		Iterator it = subfolders.iterator();
		
		while(it.hasNext())
		{
			String subfolder = (String) it.next();
			f = getOrCreateFolder(f, subfolder);
		}
		
		return f;
	}

	/**
	 * Given the folder and the document compare their paths
	 * and return a list of equivalent directories.
	 * 
	 * @param FileSystemBaseFolder
	 * @param filesysDoc
	 * @return
	 */
	private List diffFolders(File FileSystemBaseFolder, File filesysDoc)
	{
		ArrayList subfolderList = new ArrayList();
		
		String baseDir  = FileSystemBaseFolder.getPath();
		String filePath = filesysDoc.getPath(); 
		
		int index = filePath.lastIndexOf(File.separator);
		
		if(index == -1)
		{
			throw new RuntimeException("No file separator: " + filePath);
		}
		
		// remove file name part
		filePath = filePath.substring(0, index + 1);
		
		if(!filePath.startsWith(baseDir))
		{
			throw new RuntimeException("File path does not contain staed base dir: " + filePath);
		}
		
		String subFolders = filePath.substring(baseDir.length());
		
		StringTokenizer st = new StringTokenizer(subFolders, File.separator);

		while(st.hasMoreTokens())
		{
			String token = st.nextToken();
			subfolderList.add(token);
		}
	
		return subfolderList;
	}

	/**
	 * Either create or return the specified P8 subfolder.
	 * 
	 * @param parentFolder
	 * @param subFolderName
	 * @return
	 */
    private Folder getOrCreateFolder(Folder parentFolder, String subFolderName)
    {
    	Folder f = null;
    	
    	try
		{
    		String path = null;
    		
    		if(parentFolder == null)
    		{
    			path = "/" + subFolderName;
    		}
    		else
    		{
    			path = parentFolder.get_PathName() + "/" + subFolderName;
    		}
    		
    		f = Factory.Folder.fetchInstance(os, path, null);
		}
    	catch(EngineRuntimeException e)
		{
            if (e.getExceptionCode() == ExceptionCode.E_OBJECT_NOT_FOUND)
            {
            	if(parentFolder == null)
            	{
            		parentFolder = os.get_RootFolder();
            	}
                
                f = parentFolder.createSubFolder(subFolderName);
                f.save(RefreshMode.REFRESH);
            }
            else
            {
                  throw e;
            }
		}
    	return f;
    }
    
	/**
	 * Setup a content transfer for filesysDoc.
	 * 
	 * @param filesysDoc
	 * @param name
	 * @param type
	 * @return
	 */
    public static ContentTransfer createContentTransfer(File filesysDoc, String name, String type)
    {
        ContentTransfer ctNew = null;
        
        try
		{
        	ctNew = Factory.ContentTransfer.createInstance();
            InputStream is = new FileInputStream(filesysDoc);
            ctNew.setCaptureSource(is);
            ctNew.set_RetrievalName(name);
            ctNew.set_ContentType(type);
		}
        catch(Throwable t)
		{
        	throw new RuntimeException(t.getLocalizedMessage());
		}
        return ctNew;
    }

    /**
     * Setup a content transfer.
     * 
     * @param ct
     * @param id
     * @return
     */
    public static ContentTransfer createContentTransfer(String[] ct, Id id)
    {
        ContentTransfer ctNew = Factory.ContentTransfer.createInstance();
        String content = ct[2].replaceFirst("OUTPUT_FOLDER_ID", id.toString());
        ByteArrayInputStream is = new ByteArrayInputStream(content.getBytes());
        ctNew.setCaptureSource(is);
        ctNew.set_RetrievalName(ct[0]);
        ctNew.set_ContentType(ct[1]);
        return ctNew;
    }

    public void testCleanup()
    {
    	SampDITATool.out.println("Cleanup - Done");
    }

}
