/**
	IBM grants you a nonexclusive copyright license to use all programming code 
	examples from which you can generate similar function tailored to your own 
	specific needs.

	All sample code is provided by IBM for illustrative purposes only.
	These examples have not been thoroughly tested under all conditions.  IBM, 
	therefore cannot guarantee or imply reliability, serviceability, or function of 
	these programs.

	All Programs or code component contained herein are provided to you AS IS  
	without any warranties of any kind.
	The implied warranties of non-infringement, merchantability and fitness for a 
	particular purpose are expressly disclaimed.

	 Copyright IBM Corporation 2008, ALL RIGHTS RESERVED.
 */

package sampdita;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.security.PrivilegedAction;
import java.util.Properties;
import javax.security.auth.Subject;
import javax.swing.*;


import com.filenet.api.property.PropertyFilter;
import com.filenet.api.security.User;
import com.filenet.api.constants.PropertyNames;
import com.filenet.api.core.*;
import com.filenet.api.util.UserContext;

/**
 *
 * This is a sample that shows how one might use P8 to store
 * DITA documents using compound document repationships.
 * 
 *  By default the tool starts up a gui and stores previous
 *  data input in the file ditatool.xml.
 *  
 *  The configuration file can be specified using the -f option
 *  on the command line.
 *
 *  A -u and -d option on the command line can also be used to bypass
 *  the gui and run an upload or download sequence. The flag for "simulate"
 *  or "simulateDl" will be used and a simple simulation with no upload
 *  or download will occur if that flag is set in teh ditatool.xml file.
 *
 */
	public class SampDITATool implements  PrivilegedAction
	{
		CEConnection ce;
		public static SampDITATool sdtinstance;
		
		// to domain, user and object store
		Domain domain = null;
		User u = null;
		ObjectStore os = null;
			
		private static int uniqueDirId = -1;
		
		// default values for the tool
		public static String URI = "http://localhost:9080/wsi/FNCEWS40DIME/";
		public static String DOMAIN_NAME = "mydomain";
		public static String OBJECT_STORE_NAME = "OS1";
		public static String USER_NAME = "CEAdmin";
		public static String PASSWORD = "test";
		public static String JAAS_WS_STANZA = "WSI";
        //private static String JAAS_WS_STANZA = "FileNetP8";
		public static boolean SavePassword=false;

        // default values for the ditamap to upload
		public static String p8BaseDir = "/DITAProjects";
		public static String baseDir = "C:/temp/filenet/samples/ce/sampditatool/testdata/";
		public static String filename = "articles/DITA-articles.ditamap";
		
		// default values for the location to download
		public static String baseDlDir = "C:/temp/filenet/samples/ce/sampditatool/DlTest";

		// simulate upload or download flags
		public static boolean simulate = false;
        public static boolean simulateDl = false;
        
        // ignore gui flags - set from the command line using -u or -d
        public static boolean uploadNoGui = false;
        public static boolean downloadNoGui = false;
        
        // the name of the file to use for configuration
        // set from the command line using -f
        public static String configFilename = null;
        
        // when running, flag tells if this is a upload or download event
        public static boolean bUpload = true;
        
        // properties loaded from the xml configuration file
        private static Properties props;
        
        // an output stream to send log messages to
        // this gets split into standard output and the log panel in the gui
        public static PrintStream out;
        
        /**
         * Run an DITA document upload sequence.
         * 
         * @param domain
         * @param objStore
         */
        private void runUpload(Domain domain, ObjectStore objStore)
        {

         	StoreDITADoc storeDITA = new StoreDITADoc(domain, objStore);

        	Document newDITADocument = null;
        	
          	try
			{
          		newDITADocument = storeDITA.testCreateP8DITAStructureFromFilesystem(p8BaseDir, baseDir, filename);
			}
        	catch(Throwable t)
			{
        		t.printStackTrace();
        		throw new RuntimeException(t.getLocalizedMessage());
			}    	
        	
        	try
			{
        		storeDITA.testCleanup();
			}
        	catch(Throwable t)
			{
        		throw new RuntimeException(t.getLocalizedMessage());
			}
	        SampDITATool.out.println("    GOOD, all passed.");
        }

        /**
         * Run an DITA document download sequence.
         * 
         * @param domain
         * @param objStore
         */
       private void runDownload(Domain domain, ObjectStore objStore)
        {         	
    	   // open the P8 document
    	   // open the P8 document
           Document ditaDoc = Factory.Document.getInstance(objStore, "DitaMap", p8BaseDir+"/"+filename);
           
           // get the needed document properties
           ditaDoc.refresh(new String [] {"DocumentTitle", PropertyNames.FOLDERS_FILED_IN, PropertyNames.CONTENT_ELEMENTS, PropertyNames.CONTENT_SIZE});
            
           // copy the document and component relationships to the local disk
           GetDITADoc.copyCompoundDocumentToFilesystem(ditaDoc, baseDlDir);
        }
        
       /**
        * Get the domain and the object store
        * 
        */
        public void getDomainAndOS()
        {
    		SampDITATool.out.println("Connecting to Domain: "+DOMAIN_NAME);
    		
            // fetch the domain
	        domain = ce.fetchDomain(DOMAIN_NAME);

	        // fetch the object store
	        SampDITATool.out.println("Connecting to Object Store: " + OBJECT_STORE_NAME);
        	os = ce.fetchOS(OBJECT_STORE_NAME);
        	SampDITATool.out.println("Connection complete");

        }
        
        /**
         * action.run target for kicking off the upload or download
         */
		public Object run()
	    {
			if (bUpload)
			{
				SampDITATool.out.println("Connected to server at " + URI);
			
				// connection not needed to simulate an upload
				if (!SampDITATool.simulate)
				{
					getDomainAndOS();
				}
				
		        runUpload(domain, os);
			}
			else
			{
				getDomainAndOS();
	        	
	        	runDownload(domain,os);
			}   
	        return null;
	    }
		
		/**
		 * create the frame for the tool
		 * 
		 */
		public void ShowMainWindow()
		{
			
			SwingUtilities.invokeLater(new Runnable()
			{
				public void run()
				{
					ce = new CEConnection();
					MainFrame thisClass = new MainFrame(SampDITATool.sdtinstance,ce);
					thisClass.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
					thisClass.setVisible(true);
				}
			});
		}
		
		/**
		 * create the frame for the tool
		 * 
		 */
		public void ShowWindow()
		{
			DITAToolFrame tf = new DITAToolFrame(this,ce);
		}

	    /**
	     * Calls the Content Engine (used to be in a JAAS context)
	     * This model (a specific method to call the engine) is used because we used to need
	     * it in order to have the proper JAAS context.
	     * Now, it's just convenient because it provides cursor & status bar updates. It's not
	     * used in all the existing code, only where the operation is likely to be particularly
	     * long. 
	     */
	    static public Object callEngine(PrivilegedAction action)
	    {
	    	// save the properties for the next run
	    	fillProperties();
	    	saveProperties();

	        return action.run();
	    }

	    /**
	     * Set the current property values into the property object
	     *
	     */
	    static public void fillProperties()
	    {
	    	props.setProperty("uri", URI);
	    	props.setProperty("stanza", JAAS_WS_STANZA);
	    	props.setProperty("domain", DOMAIN_NAME);
	    	props.setProperty("os", OBJECT_STORE_NAME);
	    	props.setProperty("user", USER_NAME);
	    	props.setProperty("p8base", p8BaseDir);
	    	if (SavePassword)
	    	{
	    		props.setProperty("pass", PASSWORD);
	    	}
	    	props.setProperty("basedir",baseDir);
	    	props.setProperty("filename",filename);	
	    	if (simulate)
	    		props.setProperty("simulatePub","true");
	    	else
	    		props.setProperty("simulatePub","false");
	    	props.setProperty("DownloadDir",baseDlDir);
	    	if (simulateDl)
	    		props.setProperty("simulateDl","true");
	    	else
	    		props.setProperty("simulateDl","false");	   
	    }
	    
	    /**
	     * Save the current property object into the config file
	     * 
	     */
	    static public void saveProperties()
	    {
	    	try 
	    	{
	    		FileOutputStream fos = null;
	    		if (configFilename==null)
	    		{
	    		    fos = new FileOutputStream("ditatool.xml");
	    		}
	    		else
	    		{
	    		    fos = new FileOutputStream(configFilename);	    			
	    		}
	    	    props.storeToXML(fos, "DitaTool Settings");
	    	}
	    	catch (IOException e)
	    	{
	    		e.printStackTrace();
	    	}
	    }

	    /**
	     * Main method
	     * 
	     * @param args
	     * @throws Exception
	     */
	    static public void main(final String[] args) throws Exception
	    {
	    	out = System.out;
	    	
	    	// create a self instance of this class
 	        SampDITATool thisDE = new SampDITATool();
 	        sdtinstance = thisDE;
 	        
 	        int i;
 	        for (i=0;i<args.length;i++)
 	        {
 	        	// check for arguments.
 	        	if (args[i].toUpperCase().equals("-H"))
 	        	{
 	        		// -H = HELP
 	        		System.out.println("SampDITATool [-h -u -d -f <filename>]");
 	        		System.out.println(" If given no -u or -d option a gui will be shown.");
 	        		System.out.println(" -h                Help");
 	        		System.out.println(" -u                Non-Gui upload/publish");
 	        		System.out.println(" -d                Non-Gui download");
 	        		System.out.println(" -f <filename>     Configuration xml file");
 	        		System.out.println("                   This file is generated by the gui if it doesn't exist");
 	        		return;
 	        	}
 	        	else if (args[i].toUpperCase().equals("-F"))
 	        	{
 	        		// -F specify a config file name
 	        		// the next entry is the config file name
 	        		if (args.length>i+1)
 	        		{
 	        			configFilename = args[i+1];
 	        			i++;
 	        		}
 	        		else
 	        		{
 	        			// error, no second argument given
 	        			System.out.println("Error in command line argument -f");
 	        			System.out.println(" -f requires a second argument identifying");
 	        			System.out.println("    the xml configuration file to use.");
 	        			System.out.println("    This file is automatically generated when running with");
 	        			System.out.println("    the standard gui and upload with simulate turned on.");
 	        			return;
 	        		}
 	        	}
 	        	else if (args[i].toUpperCase().equals("-U"))
 	        	{
 	        		// -U forces an upload event without showing the gui
 	        		if (downloadNoGui)
 	        		{
 	        			System.out.println("Error: Cannot have both -u and -d flags set.");
 	        			System.out.println("       You can either upload or download on a single run. Not both.");
 	        			return;
 	        		}
 	        		uploadNoGui = true;
 	        	}
 	        	else if (args[i].toUpperCase().equals("-D"))
 	        	{
 	        		// -D forces a download event without showing the gui
 	        		if (uploadNoGui)
 	        		{
 	        			System.out.println("Error: Cannot have both -u and -d flags set.");
 	        			System.out.println("       You can either upload or download on a single run. Not both.");
 	        			return;
 	        		}
 	        		downloadNoGui = true;
 	        	}
 	        }

 	        // create an empty properties object
	    	props = new Properties();
	    	
	    	// try to load the properties from a file in the local directory
	    	try
	    	{
	    		FileInputStream fis =null;
	    		if (configFilename==null)
	    		{
		    		SampDITATool.out.println("Loading settings from ditatool.xml");
	    			fis = new FileInputStream("ditatool.xml");
	    		}
	    		else
	    		{
		    		SampDITATool.out.println("Loading settings from "+configFilename);
	    			fis = new FileInputStream(configFilename);
	    		}
	    		props.loadFromXML(fis);
	    		
	    		// extract the properties into local variables
		    	URI = props.getProperty("uri");
		    	JAAS_WS_STANZA = props.getProperty("stanza");
		    	DOMAIN_NAME = props.getProperty("domain");
		    	OBJECT_STORE_NAME = props.getProperty("os");
		    	USER_NAME = props.getProperty("user");
		    	p8BaseDir = props.getProperty("p8base");
	    		if (props.containsKey("pass"))
	    		{
	    			SavePassword = true;
		    		PASSWORD = props.getProperty("pass");
	    		}
	    		else
	    		{
	    			SavePassword = false;
	    			PASSWORD = "";
	    		}
		    	baseDir = props.getProperty("basedir");
		    	filename = props.getProperty("filename");
		    	if ((props.getProperty("simulatePub")!=null) && (props.getProperty("simulatePub").equals("true")))
		    		simulate = true;
		    	else
		    		simulate = false;
		    	
		    	if ((props.getProperty("simulateDl")!=null) && (props.getProperty("simulateDl").equals("true")))
		    		simulateDl = true;
		    	else
		    		simulateDl = false;
		    	baseDlDir = props.getProperty("DownloadDir");
		    	}
	    	catch (IOException e)
	    	{
	    		if (configFilename!=null)
	    		{
	    			System.out.println("Error: config file you specified: "+configFilename+" does not exist");
	    			System.out.println("       You can generate one by running this application in a simulated upload");
	    			return;
	    		}
	    		
	    		// set default properties
	    		fillProperties();
		    }
	    	
	    	// no gui options
	    	if (downloadNoGui)
	    	{
    	        bUpload = false;
    	        if (simulateDl)
    	        	out.println("Starting Download Simulation");
    	        else
    	        	out.println("Starting Download");	    		
    	        thisDE.callEngine(thisDE);
    	        if (simulateDl)
    	        	out.println("Download Simulation Complete");
    	        else
    	        	out.println("Download Complete");
	    	}
	    	else if (uploadNoGui)
	    	{
    	        bUpload = true;
    	        if (simulate)
    	        	out.println("Staring Upload Simulation");
    	        else
    	        	out.println("Starting Upload");
    	        thisDE.callEngine(thisDE);
    	        if (simulate)
    	        	out.println("Publish Simulation Comple");
    	        else
    	        	out.println("Publish Complete");	    		
	    	}
	    	else
	    	{
	    		// run with the standard gui
	    		thisDE.ShowMainWindow();
	    	}
	    }

	}
	
