/**
	IBM grants you a nonexclusive copyright license to use all programming code 
	examples from which you can generate similar function tailored to your own 
	specific needs.

	All sample code is provided by IBM for illustrative purposes only.
	These examples have not been thoroughly tested under all conditions.  IBM, 
	therefore cannot guarantee or imply reliability, serviceability, or function of 
	these programs.

	All Programs or code component contained herein are provided to you AS IS  
	without any warranties of any kind.
	The implied warranties of non-infringement, merchantability and fitness for a 
	particular purpose are expressly disclaimed.

	 Copyright IBM Corporation 2008, ALL RIGHTS RESERVED.
 */

package sampdita;


import java.io.*;
import java.util.*;
import com.filenet.api.constants.*;
import com.filenet.api.collection.*;
import com.filenet.api.core.*;
import com.filenet.api.exception.*;
import com.filenet.api.property.*;
import com.filenet.api.util.*;

/**
 * This class will download a DITA document from a P8 server. A DITA
 * document is a compound document and may consist of many files. All
 * the files in the compound document structure will be downloaded.
 * 
 */
public class GetDITADoc 
{

	private static Vector stubFiles;
	
	/**
	 * Given the ditamap document, download the compound document structure
	 * to the workingDirectory.
	 * 
	 * @param ditamap
	 * @param workingDirectory
	 * @return
	 */
	   public static File copyCompoundDocumentToFilesystem(Document ditamap, String workingDirectory)
	    {
		    stubFiles = new Vector();
		    
	    	// verify working directory
	    	File workingDir = new File(workingDirectory);
	    	
	    	if(!workingDir.exists())
	    	{
	    		SampDITATool.out.println("Creating download directory "+workingDir);
	        	if ((!SampDITATool.simulateDl) && (!workingDir.mkdirs()))
	        	{
	                throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION,
	                        "Cannot create DITA working directory in filesystem: "
	                        + workingDirectory);
	        	}
	    	}
	 	
	    	Date now = new Date();
	  
	    	// refresh the properties we will need in the download process
	        ditamap.refresh(new String [] {"DocumentTitle",
	        		                       PropertyNames.CHILD_DOCUMENTS,
	        		                       PropertyNames.ID,
										   PropertyNames.OBJECT_STORE,
										   PropertyNames.CONTENT_SIZE});
	        
	        Map ditaDocs = new HashMap();
	        
	        // map the structure of the compound document
	        SampDITATool.out.println("Ditamap structure:");
	        retrieveCompoundDocumentStructure(ditamap, ditaDocs, "ditamap","");
	        SampDITATool.out.println("Ditamap structure complete.");
	        
	        // copy the document structure to the file system
	    	return doCopyCompoundDocumentToFilesystem(ditamap.getObjectStore(), ditaDocs, workingDirectory, now);
	    }
	    
	   /**
	    * Create a map of documents associated with the ditacomponent.
	    * Store the list of documents in the ditaDocs map.
	    * Since this method is recursive and outputs the files it retrieves
	    * the indent is kept with the number of spaces in the current structure.
	    * 
	    * @param ditacomponent
	    * @param ditaDocs
	    * @param type
	    * @param indent
	    */
	    private static void retrieveCompoundDocumentStructure(Document ditacomponent, Map ditaDocs, String type, String indent)
	    {
	        Id id = ditacomponent.get_Id(); 

	        // only retrieve a document if it has not yet been referenced.
	        if(!ditaDocs.containsKey(id))
	        {
	        	// don't descend on this document again
	        	ditaDocs.put(id, type);
	        	
	        	// get the name of the document
	        	String name = null;
	    		if(ditacomponent.getProperties().isPropertyPresent("DocumentTitle"))
	    		{
	    			name = ditacomponent.getProperties().getStringValue("DocumentTitle");
	    		}
	    		
	    		if(name == null)
	    		{
	    			SampDITATool.out.println("ERROR - DITA component filename is null");
	    			throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION, "DITA component filename is null");
	    		}
	    		
	    		// check the size of the document, looking for 0 length stub files
    			Double size = ditacomponent.getProperties().getFloat64Value(PropertyNames.CONTENT_SIZE);

    			SampDITATool.out.println(indent+ditacomponent.getProperties().getStringValue("DocumentTitle")+" size:("+size.intValue()+")");
	    		
    			// find all the compound document associations to this document
	           	DocumentSet ds = ditacomponent.get_ChildDocuments();

	           	// iterate through all the compound documents
		    	Iterator it = ds.iterator();
		    	while(it.hasNext())
		    	{
		    		Document child = (Document) it.next();
		    		
		    		// recursive call to map each document structure
		    		retrieveCompoundDocumentStructure(child, ditaDocs, "ditacomponent", indent+"   ");
		    	}
	        }
	    }


	    /**
	     * Do the actual copy of the documents in the ditaDocs map. The map
	     * is created with the retrieveCompoundDocumentStructure method.
	     * 
	     * @param objStore
	     * @param ditaDocs
	     * @param workingDirectory
	     * @param startTime
	     * @return
	     */
	    private static File doCopyCompoundDocumentToFilesystem(ObjectStore objStore,
	    		                                               Map ditaDocs,
	    		                                               String workingDirectory,
															   Date startTime)
	    {
	    	File ditamap = null;
	    	
	    	PropertyFilter pf = new PropertyFilter();
	    	pf.addIncludeProperty(0, null, Boolean.FALSE,
	    			"DocumentTitle " + PropertyNames.FOLDERS_FILED_IN + " " + PropertyNames.CONTENT_ELEMENTS + " " + PropertyNames.CONTENT_SIZE,
					null);
	    	
	    	String type = null;
	    	Iterator it = ditaDocs.keySet().iterator();
	    	
	    	while(it.hasNext())
	    	{
	    		Id id = (Id) it.next();
	    		type = (String) ditaDocs.get(id);
	    		
	    		Document ditacomponent = Factory.Document.fetchInstance(objStore, id, pf);
	    		
	    		String name = null;
	    		
	    		if(ditacomponent.getProperties().isPropertyPresent("DocumentTitle"))
	    		{
	    			name = ditacomponent.getProperties().getStringValue("DocumentTitle");
	    		}
	    		
	    		if(name == null)
	    		{
	    			throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION, "DITA component filename is null");
	    		}
	    		
	    		FolderSet fs = ditacomponent.get_FoldersFiledIn();
	    		
	    		Iterator iy = fs.iterator();
	    		String folderPath = null;
	    		int i = 0;
	    		
	    		while(iy.hasNext())
	    		{
	    			// constraint: ditacomponent may only be filed in a single folder, and it must be filed
	    			// this matches file system
	    			Folder f = (Folder) iy.next();
	    			
	    			f.refresh(new String [] {PropertyNames.PATH_NAME});
	    			
	    			folderPath = f.get_PathName();
	    			if(i > 0)
	    			{
	    				throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION,
	    						"DITA component filed in more than one folder: "
	    						+  name);
	    			}
	    			i++;
	    		}
	    		
	    		if(i == 0)
	    		{
	    			// throw exception, diatcomponent not filed in any folder
	    			throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION,
	    					"DITA component not filed in any folder: "
	    					+ name);
	    		}
	    		
	    		File newDir = new File(workingDirectory + folderPath);
	    		
	    		// create filesystem folder for current ditacomponent if it does not already exist
	    		if (!newDir.exists())
	    		{
	    			if (SampDITATool.simulateDl)
	    				SampDITATool.out.print("Simulate ");
	    			SampDITATool.out.println("Creating directory "+newDir);
	    			if ((!SampDITATool.simulateDl) && (!newDir.mkdirs()))
	    			{
	    				throw new EngineRuntimeException(ExceptionCode.E_UNEXPECTED_EXCEPTION,
	    						"Cannot create filesystem directory: "
	    						+ workingDirectory + folderPath);
	    			}
	    		}
	    		
	    		File parentFile = new File(newDir, name);
	    		boolean alreadyExistsOrOld = false;
	    		
	    		// if the ditacomponent already exists on the filesystem (means the component is being reused within DITA structure)
	    		// determine if was created for this publishing operation, if so, do not output it to filesystem again
	    		// if created before this publishing operation started, delete it and download current ditacomponent
	    		if (parentFile.exists())
	    		{
	    			Date lastModified = new Date(parentFile.lastModified());
	    			
	    			// check if existing file was created after start of pull operation, if not overwrite it
	    			alreadyExistsOrOld = lastModified.after(startTime);
	    			
	    			if (SampDITATool.simulateDl)
	    				SampDITATool.out.print("Simulate ");
	    			SampDITATool.out.println("Deleting "+parentFile);
	    			if ((SampDITATool.simulateDl) && (!alreadyExistsOrOld))
	    			{
	    				// The file was create before the current operation started, delete it
	    				parentFile.delete();
	    			}
	    		}
	    		
	    		String path = parentFile.getPath();
    			Double size = ditacomponent.getProperties().getFloat64Value(PropertyNames.CONTENT_SIZE);
	    		
	    		if(!alreadyExistsOrOld)
	    		{
	    			if (SampDITATool.simulateDl)
	    				SampDITATool.out.print("Simulate ");
	    			
	    			String sz = " size:("+size.intValue()+")";
	    			if (size.doubleValue()==0.0)
	    			{
	    				sz = sz + " -- STUB";
	    			}
	    			SampDITATool.out.println("  Downloading DITA component: " + path + sz);

	    			if (!SampDITATool.simulateDl)
	    			{
	    			  writeToFileSystem(ditacomponent, parentFile);
	    			}
	    			
	    			if(type.equalsIgnoreCase("ditamap"))
	    			{
	    				ditamap = parentFile;
	    			}
	    		}
	    	}
	    	
	    	return ditamap;
	     }
	    
	    /**
	     * Write the P8 docuemnt to the local file system
	     * 
	     * @param parent
	     * @param parentFile
	     */
	    public static void writeToFileSystem(Document parent, File parentFile) 
	    {
	    	InputStream in =  null;
	    	OutputStream out = null;

	    	try 
			{
	    		boolean hasContent = false;
	    		
	    		if(parent.getProperties().isPropertyPresent(PropertyNames.CONTENT_SIZE))
	    		{
	    			Double size = parent.getProperties().getFloat64Value(PropertyNames.CONTENT_SIZE);
	    			
	    			if(size != null)
	    			{
	    				hasContent = size.intValue() > 0;
	    			}    			
	    		}
	    		
				if(hasContent)
				{
			    	in =  parent.accessContentStream(0);	
			    	out = new FileOutputStream(parentFile);
			    	long millis = System.currentTimeMillis();
			    	byte[] buffer = new byte[2048];
			    	int bytesRead;
			    	while ((bytesRead = in.read(buffer)) >= 0) {
			    		out.write(buffer, 0, bytesRead);
			    	}
				}
				else
				{
					// no content just create the file
					SampDITATool.out.println("   Zero length DITA component: " + parentFile.getPath());
					parentFile.createNewFile();
				}
			} 
	    	catch(IOException io)
			{
	            throw new EngineRuntimeException(io, ExceptionCode.E_UNEXPECTED_EXCEPTION, null);
			}
	    	finally 
			{
	    		if (in != null) 
	    		{
	    			try
					{
	    				in.close();
					}
	    			catch(Throwable t) 
					{
	    	            throw new EngineRuntimeException(t, ExceptionCode.E_UNEXPECTED_EXCEPTION, null);
					};
	    		}
	    		if (out != null) 
	    		{
	    			try
					{
	    				out.flush();
	    				out.close();
					}
	    			catch(Throwable t) 
					{
	    	            throw new EngineRuntimeException(t, ExceptionCode.E_UNEXPECTED_EXCEPTION, null);
					};
	    		}
			}
	    }

}
