/*
*
* Sample program for use with the Product
* Licensed Materials  - Property of IBM
* 5724-I66
* (c) Copyright IBM Corp.  2009
*
*/
package com.ibm.wbit.tel.client.jsf.query;

import java.util.ArrayList;
import java.util.List;

import com.ibm.bpc.clientcore.ClientException;
import com.ibm.bpc.clientcore.GenericBPCQuery;
import com.ibm.task.api.QueryResultSet;
import com.ibm.task.clientmodel.HTMConnection;
import com.ibm.wbit.tel.client.jsf.bean.ToDoInstance;
import com.ibm.wbit.tel.client.jsf.infrastructure.FacesUtils;
import com.ibm.wbit.tel.client.jsf.infrastructure.SharedConstants;



/**
 * This class is to query the data base for sub ToDos.
 * The result will be shown in a list, refer to faces-config to see the definition of that query.
 */
public class SubToDosQuery extends GenericBPCQuery implements SharedConstants {

    /**
     * Connection to Human Task Manager to execute the query
     */
    private HTMConnection connection;

    /**
     * Where clause for all sub ToDos the generated client is applicable for.
     */
    private String whereClauseSubToDos = null;
    
    /**
     * The task instance ID of the main ToDo 
     */
    private String mainToDoID = null;
    
    /**
     * Executes the query against the HumanTaskManagerService
     * The HumanTaskManagerService is determined by the connection property.
     *
     * @see com.ibm.bpc.clientcore.Query#execute()
     * @throws ClientException: Any exception is caught,nested and rethrown as a ClientException
     */
    public List<ToDoInstance> execute() throws ClientException {
        ArrayList<ToDoInstance> result;

        try {
            // Connect to the HTM API
            HTMConnection htm = getConnection();
            QueryResultSet apiResult = htm.getHumanTaskManagerService().query(getSelectClause(), getWhereClause(),
                    getOrderClause(), getThreshold(), null);

            // Iterate over the result returned by the API and
            // and transfrom each row to a bean and add it to the
            // result list.
            result = new ArrayList<ToDoInstance>(apiResult.size());
            while (apiResult.next()) {
                result.add(transform(apiResult));
            }
        } catch (Exception e) {
            throw new ClientException(NLS_CATALOG, QUERY_FAILED_SUB_TODOS, null, e);
        }
        return result;
    }

    /**
     * Transforms a single row of the result set into a <code>ToDoInstance</code>
     *
     * @param apiResult the result set as returned by the query
     * @return <code>ToDoInstance</code>
     */
    private ToDoInstance transform(QueryResultSet apiResult) {

        return new ToDoInstance(apiResult, connection);
    }

    /**
     * Returns the where clause as specified in the faces-config.xml,
     * augmented with all H-Task and standalone O-Tasks the client is applicable for.
     */
    public String getWhereClause() {
    	StringBuilder ret = new StringBuilder();
    	
		if (whereClauseSubToDos == null) {
			StringBuffer buffer = new StringBuffer();

			// get access to ToDosWhereExtension
			ToDosWhereExtension extension = (ToDosWhereExtension) FacesUtils.getManagedBeanInstance("subToDosWhereExtension", ToDosWhereExtension.class);

			buffer.append(super.getWhereClause());
			buffer.append(" AND (");
			buffer.append(extension.getWhereClauseExtension());
			buffer.append(")");
			whereClauseSubToDos = buffer.toString();
		}
		ret.append(whereClauseSubToDos);
		ret.append(" AND TASK.PARENT_CONTEXT_ID = ID('");
		ret.append( getMainToDoID() );
		ret.append("')");
		return ret.toString();  
	}
    

    /**
	 * Indicates the type of objects returned by this query
	 * 
	 * @return
	 */
    public String getType() {
        return ToDoInstance.class.getName();
    }

    /**
     * @return Returns the connection.
     */
    public HTMConnection getConnection() {
        return connection;
    }

    /**
     * @param connection The connection to set.
     */
    public void setConnection(HTMConnection connection) {
        this.connection = connection;
    }

    /**
     * Get the ID of main ToDo
     * 
     * @return the ID of main ToDo
     */
	public String getMainToDoID() {
		return mainToDoID;
	}

	/**
	 * Set the ID of main ToDo
	 * 
	 * @param mainToDoID the ID of main ToDo
	 */
	public void setMainToDoID(String mainToDoID) {
		this.mainToDoID = mainToDoID;
	}
    
}
