/*
 *
 * Sample program for use with the Product
 * Licensed Materials  - Property of IBM
 * 5724-I66
 * (c) Copyright IBM Corp.  2009
 *
 */
package com.ibm.wbit.tel.client.jsf.handler;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import com.ibm.bpc.clientcore.ClientException;
import com.ibm.bpc.clientcore.MessageWrapper;
import com.ibm.bpe.jsf.handler.BPCListHandler;
import com.ibm.bpe.jsf.handler.ItemListener;
import com.ibm.bpe.jsf.handler.ItemProvider;
import com.ibm.wbit.tel.client.jsf.bean.ToDoInstance;
import com.ibm.wbit.tel.client.jsf.infrastructure.FacesUtils;
import com.ibm.wbit.tel.client.jsf.infrastructure.Messages;
import com.ibm.wbit.tel.client.jsf.infrastructure.SharedConstants;

/**
 * SubToDoMessageHandler is an item listener for the sub task list. It also
 * implements the ItemProvider interface and can therefore be used as a model in
 * the command bar component. Furthermore it extends SubviewHandler to bring up
 * a page which shows the input message and output message for the selected sub
 * task.
 */
public class SubToDoMessageHandler extends SubviewHandler implements ItemListener, ItemProvider, SharedConstants {

	/**
	 * The sub ToDo selected in the list
	 */
	private ToDoInstance subToDoInstance = null;
	
	/**
	 * HashMap to initialize output message of mainToDo with output message of subToDo 
	 */
	private HashMap<String, Object> initValuesMainOut = null; 

	/**
	 * Notification, when a sub ToDo was selected in the list. Set input and
	 * output message of ToDoInstance and provide subview IDs to render the sub
	 * ToDo messages.
	 * 
	 * @see com.ibm.bpe.jsf.handler.ItemListener#itemChanged(java.lang.Object)
	 */
	public void itemChanged(Object item) throws ClientException {
		nextSubviewInput((String) null);
		nextSubviewOutput((String) null);
		initValuesMainOut = null; 

		if (item != null && item instanceof ToDoInstance) {
			subToDoInstance = (ToDoInstance) item;

			// get input message and convert to HashMap
			Object object = subToDoInstance.getInputMessageWrapper().getMessage();
			subToDoInstance.setInputValuesAll( object );
			
			nextSubviewInput(subToDoInstance);

			if (isOutputMessageDefined() && isSubToDoFinished()) {
				nextSubviewOutput(subToDoInstance);
				MessageWrapper outputMessage = subToDoInstance.getOutputMessageWrapper();
				// outputmessage correctly set
				if (outputMessage != null && outputMessage.getMessage() != null) {
					object = outputMessage.getMessage();
					subToDoInstance.setOutputValuesAll( object );
				}
				Object mainToDoMessageHandler = FacesUtils.getManagedBeanInstance("toDoMessageHandler", ToDoMessageHandler.class);
				ToDoInstance mainToDoInstance = ((ToDoMessageHandler) mainToDoMessageHandler).getToDoInstance();
				
				initValuesMainOut = FacesUtils.getValueMapSubOutMainOut(subToDoInstance, mainToDoInstance);
			}
			setNavigationInfo();
		}
	}

	/**
	 * Refreshes list of sub ToDos.
	 * 
	 * @return navigtion rule as of faces config for list with open ToDos
	 */
	public String refreshList() {
		Object listHandler = FacesUtils.getManagedBeanInstance("subToDosListHandler", BPCListHandler.class);
		((BPCListHandler) listHandler).refreshList();

		return DETAILS_TODO_UNDER_WORK;
	}

	/**
	 * Is sub ToDo in state finished.
	 * 
	 * @return boolean to indicate whether sub ToDo is in state finished
	 */
	public boolean isSubToDoFinished() {
		return subToDoInstance.getState() == ToDoInstance.STATE_FINISHED ? true : false;
	}

	/**
	 * Is output message definded for sub ToDo.
	 * 
	 * @return boolean to indicate whether output message definded for sub ToDo.
	 */
	public boolean isOutputMessageDefined() {
		return subToDoInstance.getOutputMessageTypeName() != null ? true : false;
	}

	/**
	 * Used to initialize managed bean via faces context. With
	 * FacesContext.getCurrentInstance().getApplication().createValueBinding(name);
	 * the managed properties are not initialized.
	 */
	public boolean isInit() {
		return false;
	}

	/**
	 * Indicate whether or not to show the not finished info.
	 * 
	 * @return boolean to indicate whether or not to show the not finished info.
	 */
	public boolean isShowNotFinishedInfo() {
		return !isSubToDoFinished() && isOutputMessageDefined();
	}

	/**
	 * Indicate whether or not to show the UseOutputButton.
	 * 
	 * @return boolean to indicate whether or not to show the UseOutputButton.
	 */
	public boolean isShowUseOutputButton() {
		return isSubToDoFinished() && isOutputMessageDefined() && (initValuesMainOut != null);
	}

	/**
	 * Set navigation info to be shown in the JSF pages.
	 */
	private void setNavigationInfo() {
		String preNavigation;

		preNavigation = Messages.getString("LINE_TODOS_CLAIMED_SUBTODO", FacesUtils.getLocale());

		setNavigationInfo(preNavigation + " > " + subToDoInstance.getName());
	}

	/**
	 * Provide the sub ToDo selected in the list, e.g. to the command bar
	 * component.
	 * 
	 * @see com.ibm.bpe.jsf.handler.ItemProvider#getSelectedItems()
	 */
	public List<ToDoInstance> getSelectedItems() { 
		List<ToDoInstance> list = Collections.emptyList();
		if (subToDoInstance != null) {
			list = new ArrayList<ToDoInstance>(1);
			list.add(subToDoInstance);
		}
		return list;
	}

	/**
	 * Get the sub ToDo selected in the list
	 * 
	 * @return sub ToDo selected in the list
	 */
	public ToDoInstance getSubToDoInstance() {
		return subToDoInstance;
	}

	
	/**
	 * Get HashMap to initialize output message of mainToDo with output message of subToDo. 
	 * If initialization is not possible null is returned. 
	 * 
	 * @return HashMap to initialize output message of mainToDo with output message of subToDo.
	 */
	public HashMap<String, Object> getInitValuesMainOut() {
		return initValuesMainOut;
	}
	

	/**
	 * Boolean indicating if client type for passed toDoInstance is JSF.
	 * 
	 * @return boolean indicating if client type for passed toDoInstance is JSF.
	 */
	public boolean isJSFClientType() {
		return isJSFClientType(subToDoInstance);
	}
}
