/*
*
* Sample program for use with the Product
* Licensed Materials  - Property of IBM
* 5724-I66
* (c) Copyright IBM Corp.  2006, 2009
*   
*/
package com.ibm.wbit.tel.client.jsf.handler;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import javax.faces.event.ActionEvent;
import javax.faces.event.ValueChangeEvent;

import com.ibm.bpc.clientcore.ClientException;
import com.ibm.bpe.jsf.handler.ItemProvider;
import com.ibm.wbit.tel.client.jsf.bean.BCCreateInstance;
import com.ibm.wbit.tel.client.jsf.bean.ToDoInstance;
import com.ibm.wbit.tel.client.jsf.infrastructure.FacesUtils;
import com.ibm.wbit.tel.client.jsf.infrastructure.Messages;
import com.ibm.wbit.tel.client.jsf.infrastructure.SharedConstants;


/** 
 * BCCreateMessageHandler acts as listener for command links.
 * It extends  SubviewHandler to bring up a page which shows the 
 * input message and and custom properties for the selected action.   
 * Furthermore it is an ItemProvider and can therefore be used as a model 
 * in the command bar component. 
 */
public class BCCreateMessageHandler extends SubviewHandler implements ItemProvider, SharedConstants {

    /**
	 * Hashmap to get the namespace of a component, for which the client has been generated for. 
	 * key: component name, value: component namespace
	 */	
    private HashMap<String, String> componentNamespaces;
    
    /**
	 * Hashmap to get the taskIDs of a task (according to template based approach),
	 * for which the client has been generated for. 
	 * key: QName of Message + component name, value: task ID
	 */	
    private HashMap<String, String> taskIDs;    
	    
    /**
     * The Business Case that should be created on server.
     */
    private BCCreateInstance bcCreateInstance = new BCCreateInstance();
    
    /**
     * Value of last selected item in the list of sub ToDo's.
     */
    private String lastSelectedSubToDo;
    
    /**
     * The main ToDo for which to create a sub ToDo
     */
    private ToDoInstance mainToDoInstance;
    
    /**
     * Initialize BCCreateMessageHandler for creation of sub ToDo.
     * 
     * @param  ToDoInstance the main ToDo for which to create sub ToDo
     */
    public void initSubToDoCreation(ToDoInstance mainToDo) {
       	bcCreateInstance.clear();
    	
    	nextSubviewInput((String) null);
    	lastSelectedSubToDo = null;
    	this.mainToDoInstance = mainToDo;
	        	
    	String preNavigation = Messages.getString("LINE_TODOS_CLAIMED", FacesUtils.getLocale())+" > ";
    	String sufNavigation = Messages.getString("LINE_SUBTODO_CREATE", FacesUtils.getLocale());
        	
        setNavigationInfo(preNavigation + mainToDo.getName() + " > "+ sufNavigation);
    }
    
    /**
     * Action event of a command link.
     * Provide the next subviewIDs and decorate an instance of BCCreateInstance 
     * to create a new business case on server. 
     * 
     * @param actionEvent of command link
     */
    public void itemSelected(ActionEvent actionEvent) {

     	String componentName, QNameInputMessage, componentNameStart;
     	
    	componentName = (String) actionEvent.getComponent().getAttributes().get("componentName");
    	QNameInputMessage = (String) actionEvent.getComponent().getAttributes().get("inputMessage");
    	componentNameStart = (String) actionEvent.getComponent().getAttributes().get("startName");
    	
        nextSubviewCustomProperties(componentName);
        nextSubviewInput(QNameInputMessage);

        bcCreateInstance.clear();
        bcCreateInstance.setComponentName(componentName);
        bcCreateInstance.setComponentNameStart(componentNameStart);
        bcCreateInstance.setComponentNamespace((String) componentNamespaces.get(componentName));
        bcCreateInstance.setTaskID((String) taskIDs.get(QNameInputMessage + componentName));
        bcCreateInstance.setQNameInputMessage(QNameInputMessage);
        
        String preNavigation = Messages.getString("LINE_BCCREATE", FacesUtils.getLocale())+" > ";
        if (componentNameStart == null) {
            bcCreateInstance.setType(BCCreateInstance.TYPE_HUMAN_TASK);
            setNavigationInfo(preNavigation + componentName);
        } else {
            bcCreateInstance.setType(BCCreateInstance.TYPE_BUSINESS_PROCESS);
            setNavigationInfo(preNavigation + componentName + " > "+ componentNameStart);
        }
    }
    
    
    /**
     * Value Change event of drop down list to select sub ToDo to create.
     * Provide the next subviewID and decorate an instance of BCCreateInstance 
     * to create a new business case on server. 
     * 
     * @param changeEvent of drop down list
     */
    public void valueChange(ValueChangeEvent changeEvent) throws ClientException {
    	String componentName, componentNamespace, QNameInputMessage, taskID;
    	HashMap<String, Object> initValues = null; 
	    	
    	setLastSelectedSubToDo((String) changeEvent.getNewValue()); 
    	
    	componentName = (String) changeEvent.getComponent().getAttributes().get("componentName"+getLastSelectedSubToDo());
    	QNameInputMessage = (String) changeEvent.getComponent().getAttributes().get("inputMessage"+getLastSelectedSubToDo());
    	componentNamespace = (String) componentNamespaces.get(componentName);
    	taskID = (String) taskIDs.get(QNameInputMessage + componentName);
    	initValues = FacesUtils.getValueMapMainInSubIn(componentNamespace + componentName, mainToDoInstance);
        
        nextSubviewInput(QNameInputMessage);

        bcCreateInstance.clear();
        bcCreateInstance.setMainToDoInstance( mainToDoInstance );
        bcCreateInstance.setComponentName(componentName);
        bcCreateInstance.setComponentNamespace(componentNamespace);
        bcCreateInstance.setTaskID(taskID);
        bcCreateInstance.setQNameInputMessage(QNameInputMessage);
        if ( initValues != null ) {
        	bcCreateInstance.setInputValuesAll( initValues );	
        }
    }
    
    
    /**
     * Provide the BCCreateInstance selected in the list, e.g. to the command bar component.
     * 
     * @see com.ibm.bpe.jsf.handler.ItemProvider#getSelectedItems()
     */
    public List<BCCreateInstance> getSelectedItems() {
		
		List<BCCreateInstance> list = Collections.emptyList();
		if(bcCreateInstance != null)
		{
			list = new ArrayList<BCCreateInstance>(1);
			list.add(bcCreateInstance);
		}
		return list;
	}
	
	
	/**
	 * Get the <code>BCCreateInstance</code> selected in list.
	 * 
	 * @return
	 */
    public BCCreateInstance getBcCreateInstance() {
        return bcCreateInstance;
    }
    
	/**
	 * Set the <code>BCCreateInstance</code> selected in list.
	 * 
	 * @param
	 */
    public void setBcCreateInstance(final BCCreateInstance bcCreateInstance) {
        this.bcCreateInstance = bcCreateInstance;
    }

	/**
	 * Set Hashmap to get the namespace of a component, for which the client has been generated for. 
	 * key: component name, value: component namespace
	 * 
	 * @param componentNamespaces Hashmap to get the namespace of a component
	 */	
    public void setComponentNamespaces(HashMap<String, String> componentNamespaces) {
        this.componentNamespaces = componentNamespaces;
    }
    
    /**
	 * Set Hashmap to get the ID of a task (according to template based approach), 
	 * for which the client has been generated for. 
	 * key: QName of Message + component name, value: task ID
	 * 
	 * @param taskIDs Hashmap to get the ID of a task
	 */	
    public void setTaskIDs(HashMap<String, String> taskIDs) {
        this.taskIDs = taskIDs;
    }    
    
    /**    
     * Get the last selected item from the list of available sub ToDos to create. 
     * 
     * @return the last selected item from the list of available sub ToDos to create.
     */
    public String getLastSelectedSubToDo() {
		return lastSelectedSubToDo;
	}
    
    /**    
     * Set the last selected item from the list of available sub ToDos to create. 
     * 
     * @param lastSelectedSubToDo the last selected item from the list of available sub ToDos to create.
     */
    public void setLastSelectedSubToDo(final String lastSelectedSubToDo) {
		this.lastSelectedSubToDo = lastSelectedSubToDo;
	}

	/**
     * Defines whether or not a bpe:commandbar is needed.
     * 
     * @return boolean indicating whether or not a bpe:commandbar is needed.
     */
    public boolean isCommandBarNeeded() {
    	boolean ret = false;
    	
    	if (this.bcCreateInstance != null && getSubviewIDInput() != null) {
    		if ( this.bcCreateInstance.getQNameInputMessage().endsWith(CLIENT_TYPE_JSF) ) {
    			ret = true;
    		}
    	}
    	return ret;
    }
       
}
