#*******************************************************************************
#*
#* Copyright (C) Capgemini Engineering ACT S.A.S. 2017-2025.  All rights reserved.
#*
#*******************************************************************************
#*******************************************************************************
#*
#*  Copyright (C) Altran ACT S.A.S. 2018, 2020, 2021, 2022.  All rights reserved.
#*
#*******************************************************************************
#
#======================================================================
# Name: Update-CDWindows.ps1
# Description: 
#   Update an installed IBM Connect:Direct for Microsoft Windows
#   instance with a fix pack or iFix
#======================================================================
#Requires -RunAsAdministrator
Param(
    [Parameter(Mandatory=$True,Position=1)][string]$FixPackExe
)

$ScriptVersion = "1.0.0.11"


#----------------------------------------------------------------------
# Function: Get the Connect:Direct version from a file, typically
# CDNT.exe or the installer executable, and returns an object
# with version specific attributes, like service names or defaults.
#----------------------------------------------------------------------
function Get-CDObject {
    param([Parameter(Mandatory=$True)][string]$FileName)

    # Get file
    if (Test-Path -Path "$FileName" -PathType Leaf) {
        $file = Get-Item $FileName
    } else {
        Throw "File not found: $FileName"
    }

    # Determine version from file
    if ($file.VersionInfo.ProductVersion) {
        if ($file.VersionInfo.ProductVersion.StartsWith("6.0.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.0.0"
                "ShortVersion"   = "600"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.0.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.0.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.0.0"
                "IAService"      = "InstallAgentv6.0.0"
		"ICDWSService"   = ""
		"RemoveFeatures" = "Requester,Symbols"
            }
        } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.1.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.1.0"
                "ShortVersion"   = "610"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.1.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.1.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.1.0"
                "IAService"      = "InstallAgentv6.1.0"
		"ICDWSService"   = ""
		"RemoveFeatures" = "Requester,Symbols"
            }
        } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.2.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.2.0"
                "ShortVersion"   = "620"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.2.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.2.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.2.0"
                "IAService"      = "InstallAgentv6.2.0"
		"ICDWSService"   = ""
		"RemoveFeatures" = "Requester,Symbols,FileAgent"
            }
       } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.3.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.3.0"
                "ShortVersion"   = "630"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.3.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.3.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.3.0"
                "IAService"      = "InstallAgentv6.3.0"
		"ICDWSService"   = "Web_Services_for_Connect_Direct_v6.3.0"
		"RemoveFeatures" = "FileAgent"
            }
       } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.4.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.4.0"
                "ShortVersion"   = "640"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.4.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.4.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.4.0"
                "IAService"      = "InstallAgentv6.4.0"
		"ICDWSService"   = "Web_Services_for_Connect_Direct_v6.4.0"
		"RemoveFeatures" = "FileAgent"
            }
        } else {
            Throw "Unsupported version $file.VersionInfo.ProductVersion"
        }
        Return $CDObject
    } else {
        Throw "Unable to determine Connect:Direct version from $FileName"
    }
}


#----------------------------------------------------------------------
# Function: Write a heading for a new section
#----------------------------------------------------------------------
function Write-Heading {
    param([string]$Heading = "Heading")

    Write-Output "" `
        "--------------------------------------------------------------------------------" `
        "- $Heading" `
        "--------------------------------------------------------------------------------" `
}


#======================================================================
# Main
#======================================================================


#----------------------------------------------------------------------
# Show script name and version
#----------------------------------------------------------------------
Write-Heading "Update-CDWindows.ps1 version $ScriptVersion"

$ComputerName = $env:COMPUTERNAME
$UserName     = $env:USERNAME

# Set installation log file
$Loglile = "CDWinFixPack_$ComputerName.log"

# Set command line arguments
$SetupArgs = "/v""SEPATCH_ONLY_FLAG=1 REBOOT=ReallySuppress /l*v \""$Loglile\"" /qn"" /s /w /clone_wait"

# Verify and show settings
Get-Variable -Name ComputerName, UserName, FixPackExe, SetupArgs, Loglile -ErrorAction Stop | Format-Table -Wrap


#----------------------------------------------------------------------
# Determine Connect:Direct version
#----------------------------------------------------------------------
Write-Heading "Determine Connect:Direct version from installer"
Write-Output "File: $FixPackExe"

# Get CD object and show
$CDObject = Get-CDObject($FixPackExe)
$CDObject

# Get CD install dir
$CDServerPath = Get-ItemProperty -Path "HKLM:$($CDObject.Registry)\*\Initialization Parameters\Miscellaneous Command" -Name "server.path"
$CDInstallDir= $CDServerPath.'server.path' -replace "\\Server\\$", ""
Write-Output "CDInstallDir: $CDInstallDir"

# Set ICDW shortcuts
$WebServicesShortcut = "$CDInstallDir\WebServices\webservice.url"
$SwaggerShortcut = "$CDInstallDir\WebServices\swagger.url"


#----------------------------------------------------------------------
# Pre-processing
#----------------------------------------------------------------------
Write-Heading "Pre-processing"

# Stop local services
Write-Output "Stop local services (warnings may appear)"
Stop-Service -Name $CDObject.Service -Force -Verbose
$svc = Get-Service -Name $CDObject.DBService
if ($svc) {
    # Stop database service
    Stop-Service -Name $CDObject.DBService -Force -Verbose
}
$svc = Get-Service -Name $CDObject.IAService
if ($svc) {
    # Stop InstalAgent service
    Stop-Service -Name $CDObject.IAService -Force -Verbose
}
if ($CDObject.ICDWSService) {
    $svc = Get-Service -Name $CDObject.ICDWSService
    if ($svc) {
        # Stop ICDWS service
        Stop-Service -Name $CDObject.ICDWSService -Force -Verbose
        # Backup ICDWS shortcuts
        Write-Output "Backup Integrated CD Web Services shortcuts"
        Copy-Item -Path "$WebServicesShortcut" -Destination "$WebServicesShortcut.bak"
        Copy-Item -Path "$SwaggerShortcut" -Destination "$SwaggerShortcut.bak"
    }
}


#----------------------------------------------------------------------
# Run Connect:Direct fix pack installer
#----------------------------------------------------------------------
Write-Heading "Start update installation"
Write-Output "File: $FixPackExe"

# Start installer, wait for it to complete and get its exit code
Write-Output "Run the installer. This may take a while..."
$SetupProc = Start-Process "$FixPackExe" -Wait -NoNewWindow -PassThru -ArgumentList $SetupArgs
if (!$SetupProc.HasExited) {
    Throw "Update has not completed"
}
if ($SetupProc.ExitCode -eq 0) {
    Write-Output "Update completed successfully"
} elseif ($SetupProc.ExitCode -eq 3010) {
    Write-Warning "Update completed, but the system requires a reboot"
} else {
    $rc = $SetupProc.ExitCode
    Throw "Update has failed with rc=$rc. Review log file $LogFile."
}


#----------------------------------------------------------------------
# Post-processing
#----------------------------------------------------------------------
Write-Heading "Post-processing"

# Stop local services
Start-Sleep -Seconds 3
Write-Output "Stop local services (warnings may appear)"
Stop-Service -Name $CDObject.Service -Force -Verbose
$svc = Get-Service -Name $CDObject.DBService
if ($svc) {
    # Stop default database
    Stop-Service -Name $CDObject.DbService -Force -Verbose
}
$svc = Get-Service -Name $CDObject.IAService
if ($svc) {
    # Stop and disable InstalAgent service
    Write-Output "Disable InstallAgent service"
    Stop-Service -Name $CDObject.IAService -Force -Verbose
    Set-Service -Name $CDObject.IAService -StartupType Disabled -Verbose
}
if ($CDObject.ICDWSService) {
    $svc = Get-Service -Name $CDObject.ICDWSService
    if ($svc) {
        # Stop ICDWS service
        Stop-Service -Name $CDObject.ICDWSService -Force -Verbose
        # Restore ICDWS shortcuts
        Write-Output "Restore Integrated CD Web Services shortcuts"
        Copy-Item -Path "$WebServicesShortcut.bak" -Destination "$WebServicesShortcut"
        Copy-Item -Path "$SwaggerShortcut.bak" -Destination "$SwaggerShortcut"
    }
}


#----------------------------------------------------------------------
# Completed
#----------------------------------------------------------------------
Write-Heading "Update completed"
Write-Output "Local services have been stopped."
if ($SetupProc.ExitCode -eq 3010) {
    Write-Warning "A reboot is required. Please reboot your system as soon as possible!!!"
}
