#*******************************************************************************
#*
#* Copyright (C) Capgemini Engineering ACT S.A.S. 2017-2025.  All rights reserved.
#*
#*******************************************************************************
#*******************************************************************************
#*
#*  Copyright (C) Altran ACT S.A.S. 2018, 2020, 2022.  All rights reserved.
#*
#*******************************************************************************
#
#======================================================================
# Name: Install-CDWindows.ps
# Description: 
#   Install IBM Connect:Direct for Microsoft Windows in a Microsoft
#   Windows Failover Cluster
#======================================================================
#Requires -RunAsAdministrator
Param(
    [Parameter(Position=1)][string]$IniFile = "cd_cluster.ini",
    [switch]$Interactive = $false,
    [switch]$RemoveSecurePlus = $false
)

$ScriptVersion = "1.0.0.11"


#----------------------------------------------------------------------
# Function: Get the content of an INI file and return it as a hashtable
#----------------------------------------------------------------------
Function Get-IniContent (
    [Parameter(Mandatory=$True,Position=1)][string]$IniFile)
{
    $ini = @{}
    Switch -regex -file $IniFile { 
       "^\[(.+)\]$" { # New section header
                $section = $matches[1] 
                $ini[$section] = @{} 
            }
        "^(;.*)$" { } # Comment
        "(.+?)=(.*)" { # Name=Value
                if (!($section)) {
                    $section = "No-Section" 
                    $ini[$section] = @{} 
                } 
                $name,$value = $matches[1..2] 
                $ini[$section][$name] = $value 
            }
    }
    Return $ini
}


#----------------------------------------------------------------------
# Function: Get the Connect:Direct version from a file, typically
# CDNT.exe or the installer executable, and returns an object
# with version specific attributes, like service names or defaults.
#----------------------------------------------------------------------
function Get-CDObject {
    param([string]$FileName)

    # Get file
    if (Test-Path -Path "$FileName" -PathType Leaf) {
        $file = Get-Item $FileName
    } else {
        Throw "File not found: $FileName"
    }

    # Determine version from file
    if ($file.VersionInfo.ProductVersion) {
        if ($file.VersionInfo.ProductVersion.StartsWith("6.0.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.0.0"
                "ShortVersion"   = "600"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.0.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.0.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.0.0"
                "IAService"      = "InstallAgentv6.0.0"
                "ICDWSService"   = ""
                "RemoveFeatures" = "Requester,Symbols"
            }
        } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.1.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.1.0"
                "ShortVersion"   = "610"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.1.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.1.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.1.0"
                "IAService"      = "InstallAgentv6.1.0"
                "ICDWSService"   = ""
                "RemoveFeatures" = "Requester,Symbols"
            }
        } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.2.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.2.0"
                "ShortVersion"   = "620"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.2.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.2.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.2.0"
                "IAService"      = "InstallAgentv6.2.0"
                "ICDWSService"   = ""
                "RemoveFeatures" = "Requester,Symbols,FileAgent"
            }
       } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.3.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.3.0"
                "ShortVersion"   = "630"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.3.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.3.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.3.0"
                "IAService"      = "InstallAgentv6.3.0"
                "ICDWSService"   = "Web_Services_for_Connect_Direct_v6.3.0"
                "RemoveFeatures" = "FileAgent"
            }
       } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.4.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.4.0"
                "ShortVersion"   = "640"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.4.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.4.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.4.0"
                "IAService"      = "InstallAgentv6.4.0"
                "ICDWSService"   = "Web_Services_for_Connect_Direct_v6.4.0"
                "RemoveFeatures" = "FileAgent"
            }
        } else {
            Throw "Unsupported version $file.VersionInfo.ProductVersion"
        }
        Return $CDObject
    } else {
        Throw "Unable to determine Connect:Direct version from $FileName"
    }
}


#----------------------------------------------------------------------
# Function: Write a heading for a new section
#----------------------------------------------------------------------
function Write-Heading {
    param([string]$Heading)

    Write-Output "" `
        "--------------------------------------------------------------------------------" `
        "- $Heading" `
        "--------------------------------------------------------------------------------" `
}


#======================================================================
# Main
#======================================================================


#----------------------------------------------------------------------
# Show script name and version
#----------------------------------------------------------------------
Write-Heading "Install-CDWindows.ps1 version $ScriptVersion"

$ComputerName = $env:COMPUTERNAME
$UserName     = $env:USERNAME

Get-Variable -Name ComputerName, UserName, Interactive, IniFile, RemoveSecurePlus -ErrorAction Stop | Format-Table -Wrap


#----------------------------------------------------------------------
# Get properties from CD cluster property file (cd_cluster.ini)
#----------------------------------------------------------------------
Write-Heading "Read Connect:Direct cluster property file"
Write-Output "File: $IniFile"

# Read file
if (Test-Path -Path "$IniFile" -PathType Leaf) {
    $CDClusterIni = Get-IniContent "$IniFile" 
} else {
    Throw "Connect:Direct cluster property file not found"
}
if ($CDClusterIni -eq $null) {
    Throw "Failed to read Connect:Direct cluster property file $IniFile"
}

# Get properties and show
$CDInstallDir  = $CDClusterIni["Setup"]["InstallDir"]
$CDSetupExe    = $CDClusterIni["Setup"]["SetupExe"]
$CDSrvrIniFile = $CDClusterIni["Setup"]["CD_SRVR_INI"]
Get-Variable -Name CDInstallDir, CDSetupExe, CDSrvrIniFile -ErrorAction Stop | Format-Table -Wrap


#----------------------------------------------------------------------
# Determine Connect:Direct version
#----------------------------------------------------------------------
Write-Heading "Determine Connect:Direct version from installer"
Write-Output "File: $CDSetupExe"

# Get CD object and show
$CDObject = Get-CDObject($CDSetupExe)
$CDObject


#----------------------------------------------------------------------
# Get properties from CD server poperty file (cd_srvr.ini)
#----------------------------------------------------------------------
Write-Heading "Read Connect:Direct server property file"
Write-Output "File: $CDSrvrIniFile"

# Read file
if (Test-Path -Path "$CDSrvrIniFile" -PathType Leaf) {
    $CDSrvrIni = Get-IniContent "$CDSrvrIniFile" 
} else {
    Throw "Connect:Direct server property file not found"
}
if ($CDSrvrIni -eq $null) {
    Throw "Failed to read Connect:Direct server property file $CDSrvrIniFile"
}

# Get properties and show
$CDNodeName       = $CDSrvrIni["Server"]["CD_NODENAME"]
$CDDatabaseType   = $CDSrvrIni["Server"]["CD_DATABASE_TYPE"]
$CDSpeDisableFlag = $CDSrvrIni["Server"]["CD_SPE_DISABLE_FLAG"]
Get-Variable -Name CDNodeName, CDDatabaseType, CDSpeDisableFlag -ErrorAction Stop | Format-Table -Wrap


#----------------------------------------------------------------------
# Verify properties and apply defaults where needed
#----------------------------------------------------------------------
Write-Heading "Verify properties and apply defaults"
if ($CDSrvrIni["Server"]["CD_SETUP_TYPE"] -eq "Upgrade") {
    Throw "Upgrade is not supported"
}
if (!$CDDatabaseType) {
    # Set default database
    $CDDatabaseType = $CDObject.DbType
    Write-Output "Database type not specified. Default is $CDDatabaseType."
}
if ($RemoveSecurePlus) {
    # Secure+ is mandatory for version 6.2.0 and later
    if([int]($CDObject.ShortVersion) -ge 620) {
        Throw "Parameter -RemoveSecurePlus is no longer supported for Connect:Direct version 6.2.0 and later."
    }
    # Secure+ is recommended for earlier versions
    Write-Warning "Secure+ will not be installed (not recommended)."
} elseif (!$CDSpeDisableFlag) {
    # SPE was not disabled, but should be
    $CDSpeDisableFlag = "0"
    Write-Warning "Strong Password Encryption (SPE) not disabled. Will be disabled later."
}

# Show variables
Get-Variable -Name CDDatabaseType, CDSpeDisableFlag | Format-Table -Wrap


#----------------------------------------------------------------------
# Set install features and arguments
#----------------------------------------------------------------------
Write-Heading "Prepare installation"

# Set features that will not be installed
$CDFeaturesRemove = $CDObject.RemoveFeatures
if ($CDDatabaseType -ne $CDObject.DbType) {
    # Remove the database feature when not using the default database
    if ($CDFeaturesRemove) {
        $CDFeaturesRemove += ","
    }
    $CDFeaturesRemove += $CDObject.DbFeature
}
if ($RemoveSecurePlus) {
    # Remove Secure+ feature
    if ($CDFeaturesRemove) {
        $CDFeaturesRemove += ","
    }
    $CDFeaturesRemove += "SecurePlus"
}

# Set installation log file
$LogFile = "CDWinInstall_$ComputerName.log"

# Set command line arguments
$SetupArgs = "/v""INSTALLDIR=\""$CDInstallDir\"""
$SetupArgs += " ADDLOCAL=ALL"
if ($CDFeaturesRemove) {
	$SetupArgs += " REMOVE=$CDFeaturesRemove"
}
$SetupArgs += " CD_SRVR_INI_FILE=\""$CDSrvrIniFile\"""
$SetupArgs += " /l*v \""$LogFile\"""
$SetupArgs += " REBOOT=ReallySuppress"
if ($Interactive) {
    $SetupArgs += " """ 
} else {
    $SetupArgs += " /qn"" /s"
}
$SetupArgs += " /w /clone_wait"

# Show variables
Get-Variable -Name CDSetupExe, CDFeaturesRemove, LogFile, SetupArgs -ErrorAction Stop | Format-Table -Wrap


#----------------------------------------------------------------------
# Installation
#----------------------------------------------------------------------
Write-Heading "Start installation"

# Start installer, wait for it to complete and get its exit code
Write-Output "Run the installer. This may take a while..."
$SetupProc = Start-Process "$CDSetupExe" -Wait -NoNewWindow -PassThru -ArgumentList $SetupArgs
if (!$SetupProc.HasExited) {
    Throw "Installer has not completed"
}
if ($SetupProc.ExitCode -eq 0) {
    Write-Output "Installation completed successfully"
} elseif ($SetupProc.ExitCode -eq 3010) {
    Write-Warning "Installation completed, but the system requires a reboot"
} else {
    $rc = $SetupProc.ExitCode
    Throw "Installation has failed with rc=$rc. Review log file $LogFile."
}


#----------------------------------------------------------------------
# Post-processing
#----------------------------------------------------------------------
Write-Heading "Post-processing"

# Disable SPE now, if not done already during the installation
if ((!$RemoveSecurePlus) -and ($CDSpeDisableFlag -ne "1")) {
    Write-Output "Disable Strong Password Encryption (SPE)"
    if (Test-Path -Path "$CDInstallDir\Server\Secure+\SPCli.cmd" -PathType Leaf) {
        # Disable SPE through SPCli
        Write-Output "Update Password SpeEnable=n;" | & "$CDInstallDir\Server\Secure+\SPCli.cmd" -li y
    
        # Restart Connect:Direct service
        Start-Sleep -Seconds 3
        Write-Output "" "Restart Connect:Direct service for SPE changes to take affect"
        Restart-Service -Name $CDObject.Service -Verbose
    } else {
        Write-Warning "SPCli.cmd not found"
    }
}

# Stop local services
Start-Sleep -Seconds 3
Write-Output "Stop local services (warnings may appear)"
Stop-Service -Name $CDObject.Service -Force -Verbose
if ($CDDatabaseType -eq $CDObject.DbType) {
    # Stop default database
    Stop-Service -Name $CDObject.DbService -Force -Verbose
}
$svc = Get-Service -Name $CDObject.IAService
if ($svc) {
    # Stop and disable InstalAgent service
    Write-Output "Disable InstallAgent service"
    Stop-Service -Name $CDObject.IAService -Force -Verbose
    Set-Service -Name $CDObject.IAService -StartupType Disabled -Verbose
}
if ($CDObject.ICDWSService) {
    $svc = Get-Service -Name $CDObject.ICDWSService
    if ($svc) {
        # Stop ICDWS service
        Stop-Service -Name $CDObject.ICDWSService -Force -Verbose
    }
}


#----------------------------------------------------------------------
# Completed
#----------------------------------------------------------------------
Write-Heading "Installation completed"
Write-Output "Local services have been stopped."
if ($SetupProc.ExitCode -eq 3010) {
    Write-Warning "A reboot is required. Please reboot your system as soon as possible!!!"
}
