#*******************************************************************************
#*
#* Copyright (C) Capgemini Engineering ACT S.A.S. 2017-2025.  All rights reserved.
#*
#*******************************************************************************
#*******************************************************************************
#*
#*  Copyright (C) Altran ACT S.A.S. 2018, 2020, 2021, 2022.  All rights reserved.
#*
#*******************************************************************************
#
#======================================================================
# Name: Add-ClusterCDRole.ps1
# Description: 
#   Deploy IBM Connect:Direct for Microsoft Windows as a new
#   Generic Service Role in a Microsoft Windows Failover Cluster
#======================================================================
#Requires -RunAsAdministrator
Param(
    [Parameter(Position=1)][string]$IniFile = "cd_cluster.ini"
)

$ScriptVersion = "1.0.0.11"


#----------------------------------------------------------------------
# Import modules
#----------------------------------------------------------------------
Import-Module FailoverClusters


#----------------------------------------------------------------------
# Function: Get the content of an INI file and return it as a hashtable
#----------------------------------------------------------------------
Function Get-IniContent (
    [Parameter(Mandatory=$True,Position=1)][string]$IniFile)
{
    $ini = @{}
    Switch -regex -file $IniFile { 
       "^\[(.+)\]$" { # New section header
                $section = $matches[1] 
                $ini[$section] = @{} 
            }
        "^(;.*)$" { } # Comment
        "(.+?)=(.*)" { # Name=Value
                if (!($section)) {
                    $section = "No-Section" 
                    $ini[$section] = @{} 
                } 
                $name,$value = $matches[1..2] 
                $ini[$section][$name] = $value 
            }
    }
    Return $ini
}


#----------------------------------------------------------------------
# Function: Get the Connect:Direct version from a file, typically
# CDNT.exe or the installer executable, and returns an object
# with version specific attributes, like service names or defaults.
#----------------------------------------------------------------------
function Get-CDObject {
    param([Parameter(Mandatory=$True)][string]$FileName)

    # Get file
    if (Test-Path -Path "$FileName" -PathType Leaf) {
        $file = Get-Item $FileName
    } else {
        Throw "File not found: $FileName"
    }

    # Determine version from file
    if ($file.VersionInfo.ProductVersion) {
        if ($file.VersionInfo.ProductVersion.StartsWith("6.0.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.0.0"
                "ShortVersion"   = "600"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.0.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.0.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.0.0"
                "IAService"      = "InstallAgentv6.0.0"
                "ICDWSService"   = ""
                "RemoveFeatures" = "Requester,Symbols"
            }
        } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.1.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.1.0"
                "ShortVersion"   = "610"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.1.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.1.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.1.0"
                "IAService"      = "InstallAgentv6.1.0"
                "ICDWSService"   = ""
                "RemoveFeatures" = "Requester,Symbols"
            }
        } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.2.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.2.0"
                "ShortVersion"   = "620"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.2.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.2.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.2.0"
                "IAService"      = "InstallAgentv6.2.0"
                "ICDWSService"   = ""
                "RemoveFeatures" = "Requester,Symbols,FileAgent"
            }
       } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.3.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.3.0"
                "ShortVersion"   = "630"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.3.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.3.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.3.0"
                "IAService"      = "InstallAgentv6.3.0"
                "ICDWSService"   = "Web_Services_for_Connect_Direct_v6.3.0"
                "RemoveFeatures" = "FileAgent"
            }
       } elseif ($file.VersionInfo.ProductVersion.StartsWith("6.4.0")) {
            $CDObject = [PSCustomObject]@{
                "Version"        = "6.4.0"
                "ShortVersion"   = "640"
                "ProductVersion" = $file.VersionInfo.ProductVersion.Trim()
                "Service"        = "Connect Direct v6.4.0"
                "DbService"      = "PostgreSQL - Connect Direct v6.4.0"
                "DbType"         = "POSTGRESQL"
                "DbFeature"      = "PostgreSQL"
                "Registry"       = "SOFTWARE\Sterling Commerce\Connect:Direct for Windows NT\v6.4.0"
                "IAService"      = "InstallAgentv6.4.0"
                "ICDWSService"   = "Web_Services_for_Connect_Direct_v6.4.0"
                "RemoveFeatures" = "FileAgent"
            }
        } else {
            Throw "Unsupported version $file.VersionInfo.ProductVersion"
        }
        Return $CDObject
    } else {
        Throw "Unable to determine Connect:Direct version from $FileName"
    }
}


#----------------------------------------------------------------------
# Function: Write a heading for a new section
#----------------------------------------------------------------------
function Write-Heading {
    param([string]$Heading = "Heading")

    Write-Output "" `
        "--------------------------------------------------------------------------------" `
        "- $Heading" `
        "--------------------------------------------------------------------------------" `
}


#----------------------------------------------------------------------
# Function: Update ICDWS shortcut
#----------------------------------------------------------------------
function Update-ShortcutFile {
    param (
        [Parameter(Mandatory=$True)][string]$ShortcutFile,
        [Parameter(Mandatory=$True)][string]$NewHost
    )

    Write-Output "File: $ShortcutFile"
    if (Test-Path -Path "$ShortcutFile" -PathType Leaf) {
    	# Read shortcut
	    $Shortcut = Get-Content -Path "$ShortcutFile"
	
    	# Replace host in URL (2nd line)
	    $pattern = "(?<=URL=https://)(.*?)(?=:.*)"
	    $Shortcut[1] = $Shortcut[1] -replace $pattern, $NewHost
        Write-Output $Shortcut[1]

    	# Write shortcut
	    $Shortcut | Out-File -FilePath "$ShortcutFile"
    } else {
        Write-Warning "Shortcut file not found: $ShortcutFile"
    }
}



#======================================================================
# Main
#======================================================================


#----------------------------------------------------------------------
# Show script name and version
#----------------------------------------------------------------------
Write-Heading "Add-ClusterCDRole.ps1 version $ScriptVersion"

$ComputerName = $env:COMPUTERNAME
$UserName     = $env:USERNAME

Get-Variable -Name ComputerName, UserName, IniFile -ErrorAction Stop | Format-Table -Wrap


#----------------------------------------------------------------------
# Get properties from CD cluster property file (cd_cluster.ini)
#----------------------------------------------------------------------
Write-Heading "Read Connect:Direct cluster property file"
Write-Output "File: $IniFile"

# Read file
if (Test-Path -Path "$IniFile" -PathType Leaf) {
    $CDClusterIni = Get-IniContent "$IniFile" 
} else {
    Throw "Connect:Direct cluster property file not found"
}
if ($CDClusterIni -eq $null) {
    Throw "Failed to read Connect:Direct cluster property file $IniFile"
}

# Get settings
$ClusterRole    = $CDClusterIni["Cluster"]["ClusterRole"]
$ClusterDisk     = $CDClusterIni["Cluster"]["ClusterDisk"]
$ClusterStaticIP = $CDClusterIni["Cluster"]["ClusterStaticIP"]
$CDInstallDir    = $CDClusterIni["Setup"]["InstallDir"]
$CDSrvrIniFile   = $CDClusterIni["Setup"]["CD_SRVR_INI"]

$CDNTExe         = $CDInstallDir + "\Server\CDNT.exe"

# Verify and show settings
Get-Variable -Name ClusterRole, ClusterDisk, ClusterStaticIP, CDSrvrIniFile, CDInstallDir, CDNTExe -ErrorAction Stop | Format-Table -Wrap


#----------------------------------------------------------------------
# Determine Connect:Direct version
#----------------------------------------------------------------------
Write-Heading "Determine Connect:Direct version from CDNT.exe"
Write-Output "File: $CDNTExe"

# Get CD object and show
$CDObject = Get-CDObject($CDNTExe)
$CDObject


#----------------------------------------------------------------------
# Get properties from CD server poperty file (cd_srvr.ini)
#----------------------------------------------------------------------
Write-Heading "Read Connect:Direct server property file"
Write-Output "File: $CDSrvrIniFile"

# Read file
if (Test-Path -Path "$CDSrvrIniFile" -PathType Leaf) {
    $CDSrvrIni = Get-IniContent "$CDSrvrIniFile" 
} else {
    Throw "Connect:Direct server property file not found"
}
if ($CDSrvrIni -eq $null) {
    Throw "Failed to read Connect:Direct server property file $CDSrvrIniFile"
}

# Get properties and show
$CDNodeName       = $CDSrvrIni["Server"]["CD_NODENAME"]
$CDDatabaseType   = $CDSrvrIni["Server"]["CD_DATABASE_TYPE"]
Get-Variable -Name CDNodeName, CDDatabaseType -ErrorAction Stop | Format-Table -Wrap

if (!$CDDatabaseType) {
    # Set default database
    $CDDatabaseType = $CDObject.DbType
    Write-Output "Database type not specified. Default is $CDDatabaseType."
}


#----------------------------------------------------------------------
# Stop local services
#----------------------------------------------------------------------
Write-Heading "Stop local services (warnings may appear)"
Stop-Service -Name $CDObject.Service -Force -Verbose
$svc = Get-Service -Name $CDObject.DBService
if ($svc) {
    # Stop database service
    Stop-Service -Name $CDObject.DBService -Force -Verbose
}
$svc = Get-Service -Name $CDObject.IAService
if ($svc) {
    # Stop InstalAgent service
    Stop-Service -Name $CDObject.IAService -Force -Verbose
}
if ($CDObject.ICDWSService) {
    $svc = Get-Service -Name $CDObject.ICDWSService
    if ($svc) {
        # Stop ICDWS service
        Stop-Service -Name $CDObject.ICDWSService -Force -Verbose
    }
}


#----------------------------------------------------------------------
# Update ICDWS shortcuts to use the cluster role's address
#----------------------------------------------------------------------
if ($CDObject.ICDWSService) {
    $svc = Get-Service -Name $CDObject.ICDWSService
    if ($svc) {
        # This needs to be done before stopping the cluster role, which takes the disk offline
        Write-Heading "Update Integrated CD Web Services shortcuts"
        Update-ShortcutFile -ShortcutFile "$CDInstallDir\WebServices\webservice.url" -NewHost $ClusterRole
        Update-ShortcutFile -ShortcutFile "$CDInstallDir\WebServices\swagger.url" -NewHost $ClusterRole
    }
}


#----------------------------------------------------------------------
# Add Connect:Direct as a generic service cluster role
#----------------------------------------------------------------------
Write-Heading "Add Connect:Direct as a new cluster role"
if ($ClusterStaticIP) {
    Add-ClusterGenericServiceRole -ServiceName $CDObject.Service -CheckpointKey $CDObject.Registry -Name $ClusterRole -StaticAddress $ClusterStaticIP -Storage $ClusterDisk -ErrorAction Stop -Verbose
} else {
    Add-ClusterGenericServiceRole -ServiceName $CDObject.Service -CheckpointKey $CDObject.Registry -Name $ClusterRole -Storage $ClusterDisk -ErrorAction Stop -Verbose
}
  
Write-Output "" "Stop the Connect:Direct cluster role"
Stop-ClusterGroup -name $ClusterRole -Verbose


#----------------------------------------------------------------------
# Add database as a cluster resource (if enabled)
#----------------------------------------------------------------------
Write-Heading "Add database to the cluster role"
if ($CDDatabaseType -eq $CDObject.DbType) {
    # Using the default database

    # Add database resource
    Write-Output "Add database service to the new cluster role"
    Add-ClusterResource -Name $CDObject.DbService -ResourceType "Generic Service" -Group $ClusterRole -Verbose
    $SvcRegistry = "HKLM:\SYSTEM\CurrentControlSet\Services\" + $CDObject.DbService
    $SvcImagePath = (Get-Item $SvcRegistry).GetValue("ImagePath")
    if ($SvcImagePath -eq $null) {
	    Throw "Unable to determine ImagePath for database service"
    }
    $SvcResource = Get-ClusterResource -Name $CDObject.DbService
    $SvcResource | Set-ClusterParameter -Name ServiceName -Value $CDObject.DbService -Verbose
    $SvcResource | Set-ClusterParameter -Name StartupParameters -Value $SvcImagePath -Verbose

    # Add dependencies
    Write-Output "Add dependency: database depends on the cluster disk"
    Add-ClusterResourceDependency -Provider $ClusterDisk -Resource $CDObject.DbService -Verbose
    Write-Output "Add dependency: Connect:Direct depends on the database"
    $CDServiceFull = $CDObject.Service + " - " + $CDNodeName
    Add-ClusterResourceDependency -Provider $CDObject.DbService -Resource $CDServiceFull -Verbose
} else {
    # Using external database
    $WarnText = "A database other than the default had been selected and no dependency was set" 
    $WarnText += " on the database service here. If you require your database to run on the same"
    $WarnText += " cluster node as Connect:Direct, add a dependency manually."
    Write-Warning $WarnText
}


#----------------------------------------------------------------------
# Add ICDWS as a cluster resource (if enabled)
#----------------------------------------------------------------------
if ($CDObject.ICDWSService) {
    Write-Heading "Add Integrated Web Services to the cluster role"

    # Add ICDWS resource
    Write-Output "Add Integrated Web Services service to the new cluster role"
    Add-ClusterResource -Name $CDObject.ICDWSService -ResourceType "Generic Service" -Group $ClusterRole -Verbose
    $SvcRegistry = "HKLM:\SYSTEM\CurrentControlSet\Services\" + $CDObject.ICDWSService
    $SvcImagePath = (Get-Item $SvcRegistry).GetValue("ImagePath")
    if ($SvcImagePath -eq $null) {
	    Throw "Unable to determine ImagePath for Integrated Web Services service"
    }
    $SvcResource = Get-ClusterResource -Name $CDObject.ICDWSService
    $SvcResource | Set-ClusterParameter -Name ServiceName -Value $CDObject.ICDWSService -Verbose
    $SvcResource | Set-ClusterParameter -Name StartupParameters -Value $SvcImagePath -Verbose

    # Add dependencies
    Write-Output "Add dependency: Integrated Web Services depends on the cluster disk"
    Add-ClusterResourceDependency -Provider $ClusterDisk -Resource $CDObject.ICDWSService -Verbose
    Write-Output "Add dependency: Connect:Direct depends on the Integrated Web Services"
    $CDServiceFull = $CDObject.Service + " - " + $CDNodeName
    Add-ClusterResourceDependency -Provider $CDObject.ICDWSService -Resource $CDServiceFull -Verbose
}

#----------------------------------------------------------------------
# Completed
#----------------------------------------------------------------------
Write-Heading "Cluster role $ClusterRole added"
Write-Output "The cluster role is currently offline."
Write-Output "Use the following commands to start or stop it:" 
Write-Output "  Start-ClusterGroup -Name $ClusterRole"
Write-Output "  Stop-ClusterGroup -Name $ClusterRole"
