/*
    IBM grants you a nonexclusive copyright license to use all programming code 
	examples from which you can generate similar function tailored to your own 
	specific needs.

	All sample code is provided by IBM for illustrative purposes only.
	These examples have not been thoroughly tested under all conditions.  IBM, 
	therefore cannot guarantee or imply reliability, serviceability, or function of 
	these programs.

	All Programs or code component contained herein are provided to you AS IS  
	without any warranties of any kind.
	The implied warranties of non-infringement, merchantability and fitness for a 
	particular purpose are expressly disclaimed.

	 Copyright IBM Corporation 2007, ALL RIGHTS RESERVED.
 */

using System;
using System.Collections.Generic;
using System.Text;
using FileNet.Api.Core;
using FileNet.Api.Collection;
using System.IO;
using FileNet.Api.Constants;

namespace BulkLoader
{
    //
    // This class provides static methods to make API calls
    // to Content Engine.
    //
    class BulkUtil
    {
        private static IObjectStore os;
        private static int batchCount;
        private static int batchSize;

        //
        // Provides getter and setter for currently
        // selected ObjectStore by the User
        //
        public static IObjectStore OS
        {
            set
            {
                os = value;
            }
            get
            {
                return os;
            }
        }

        //
        // Provides getter and setter for batchCount
        //
        public static int BatchCount
        {
            set
            {
                batchCount = value;
            }
            get
            {
                return batchCount;
            }
        }

        //
        // Provides getter and setter for batchSize
        //
        public static int BatchSize
        {
            set
            {
                batchSize = value;
            }
            get
            {
                return batchSize;
            }
        }

        //
        // Calculates the batchCount based on the count of files in supplied
        // file system, whether subdirectories is included or not and batchSize
        //
        public static void CalculateBatchCount(String path, bool includeSubDirectories)
        {
            String[] dirs = Directory.GetDirectories(path);
            String[] files = Directory.GetFiles(path);
            int count = 0;
            count++;
            for (int i = 0; i < files.Length; i++)
            {
                if (count < BatchSize)
                    count++;
                if (count == BatchSize || i == files.Length - 1)
                {
                    ++batchCount;
                    count = 0;
                }
            }
            if (includeSubDirectories)
            {
                foreach (String s in dirs)
                    CalculateBatchCount(s, includeSubDirectories);
            }
        }

        //
        // Retrieves the name of the directory on a filesystem
        // from its path.
        //
        public static String GetDirectoryName(String directoryPath)
        {
            String[] split = directoryPath.Split('\\');
            return split[split.Length - 1];
        }

        //
        // Reads the content from a file and stores it
        // in byte array. The byte array will later be
        // used to create ContentTransfer object.
        //
        public static byte[] ReadContentFromFile(String filePath)
        {
            FileInfo fi = new FileInfo(filePath);
            long numBytes = fi.Length;
            byte[] buffer = null;
            if (numBytes > 0)
            {
                try
                {
                    FileStream fs = new FileStream(filePath, FileMode.Open);
                    BinaryReader br = new BinaryReader(fs);
                    buffer = br.ReadBytes((int)numBytes);
                    br.Close();
                    fs.Close();
                }
                catch (Exception e)
                {
                    System.Console.WriteLine(e.StackTrace);
                }
            }
            return buffer;
        }

        //
        // Creates the ContentTransfer object from supplied file's
        // content.
        //
        public static IContentTransfer CreateContentTransfer(String filePath,String fileName)
        {
            IContentTransfer ct = null;
            if (ReadContentFromFile(filePath) != null)
            {
                ct = Factory.ContentTransfer.CreateInstance();
                Stream s = new MemoryStream(ReadContentFromFile(filePath));
                ct.SetCaptureSource(s);
                ct.RetrievalName = fileName;
            }
            return ct;
        }

        //
        // Creates the ContentElementList from ContentTransfer object.
        //
        public static IContentElementList CreateContentElementList(String filePath,String fileName)
        {
            IContentElementList cel = null;
            if (CreateContentTransfer(filePath,fileName) != null)
            {
                cel = Factory.ContentElement.CreateList();
                IContentTransfer ct = CreateContentTransfer(filePath,fileName);
                cel.Add(ct);
            }
            return cel;
        }

        //
        // Creates the Document with/without content.
        //
        public static IDocument CreateDocument(String filePath, String fileName, IObjectStore os)
        {
            IDocument doc = Factory.Document.CreateInstance(os, null);
            doc.Properties["DocumentTitle"] = fileName;
            doc.Checkin(AutoClassify.DO_NOT_AUTO_CLASSIFY, CheckinType.MAJOR_VERSION);
            if (CreateContentElementList(filePath,fileName) != null)
            {
                IContentElementList cel = CreateContentElementList(filePath,fileName);
                doc.ContentElements = cel;
            }
            return doc;
        }

        //
        // API call to CE to fetch Folder instance from supplied ObjectStore
        // at the supplied path.
        //
        public static IFolder FetchFolder(IObjectStore os, String fPath)
        {
            IFolder f = Factory.Folder.FetchInstance(os, fPath, null);
            return f;
        }

        //
        // API call to CE to retrieve sub Folders of the given
        // Folder instance.
        //
        public static IFolderSet GetSubFolders(IFolder f)
        {
            IFolderSet fs = f.SubFolders;
            return fs;
        }

        //
        // API call to CE to retrieve Name property of the given
        // Folder instance.
        //
        public static String GetFolderName(IFolder f)
        {
            String name = f.Name;
            return name;
        }

        //
        // Creates the Folder instance at specified path using specified name.
        //
        public static IFolder CreateFolder(IObjectStore os, String fPath, String fName)
        {
            IFolder f = Factory.Folder.FetchInstance(os, fPath, null);
            IFolder nf = f.CreateSubFolder(fName);
            return nf;
        }

        //
        // API call to CE to retrieve containees of give Folder instance. 
        // It returns ReferentialContainmentRelationshipSet which can be iterated through 
        // to get ReferentialContainmentRelationship objects from which Documents,
        // CustomObjects can be retrieved. 
        //
        public static IReferentialContainmentRelationshipSet GetFolderContainees(IFolder f)
        {
            IReferentialContainmentRelationshipSet rcrSet = f.Containees;
            return rcrSet;
        }

        //
        // API call to Content Engine to retrieve ContainmentName property of 
        // given ReferentialContainmentRelationship object.
        //
        public static String GetContainmentName(IReferentialContainmentRelationship rcr)
        {
            String name = rcr.ContainmentName;
            return name;
        }

        //
        // Files the Containable object (i.e. Document,CustomObject) in
        // specified Folder.
        //
        public static IReferentialContainmentRelationship FileDocument(IDocument doc, String docName, IFolder f)
        {
            IReferentialContainmentRelationship rcr = null;
            rcr = f.File(doc, AutoUniqueName.AUTO_UNIQUE, docName, DefineSecurityParentage.DO_NOT_DEFINE_SECURITY_PARENTAGE);
            return rcr;
        }

        //
        // Creates the UpdatingBatch object
        //
        public static UpdatingBatch CreateBatch(IDomain dom)
        {
            UpdatingBatch batch = UpdatingBatch.CreateUpdatingBatchInstance(dom, RefreshMode.REFRESH);
            return batch;
        }

        //
        // Populates the UpdatingBatching object with the supplied 
        // IndependentlyPersistableObject instance
        //
        public static void PopulateBatch(UpdatingBatch ub, IIndependentlyPersistableObject o)
        {
            ub.Add(o, null);
        }
    }
}
