/*
       IBM grants you a nonexclusive copyright license to use all programming code 
	examples from which you can generate similar function tailored to your own 
	specific needs.

	All sample code is provided by IBM for illustrative purposes only.
	These examples have not been thoroughly tested under all conditions.  IBM, 
	therefore cannot guarantee or imply reliability, serviceability, or function of 
	these programs.

	All Programs or code component contained herein are provided to you AS IS  
	without any warranties of any kind.
	The implied warranties of non-infringement, merchantability and fitness for a 
	particular purpose are expressly disclaimed.

	 Copyright IBM Corporation 2007, ALL RIGHTS RESERVED.
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Collections;
using FileNet.Api.Core;
using FileNet.Api.Collection;
using FileNet.Api.Exception;
using System.Threading;
using System.Runtime.Remoting.Messaging;

namespace BulkLoader
{
     //
    // Form with UI components to collect necessary information
    // for loading documents in an ObjectStore.
    //
    public partial class BulkLoaderForm : Form
    {
        // delegates
        delegate void PerformStepCallback();
        delegate void AppendTextCallback(String text);

        private CEConnection ce;
        private bool includeSubDirectories;
        private static int batchNumber = 0;
        private LoadStatusForm loadStatusForm;
        private delegate void DelegateLoadDocuments(String d, String f);
        
        //
        // Constructor
        //
        public BulkLoaderForm(CEConnection c)
        {
            ce = c;
            InitializeComponent();
        }

        //
        // Populates the 'osComboBox' with names of the available
        // object stores.
        //
        private void PopulateOSComboBox()
        {
            ArrayList osNames = ce.GetOSNames();
            osComboBox.Items.Clear();
            foreach (String s in osNames)
            {
                osComboBox.Items.Add(s);
            }
        }

        //
        // Populates the 'dirComboBox' with names of the available
        // drives (i.e. C:\,D:\ etc) on a file system.
        //
        private void PopulateDirComboBox()
        {
            String[] drives = Directory.GetLogicalDrives();
            foreach (String s in drives)
            {
                dirComboBox.Items.Add(s);
            }
        }

        //
        // Handles the event generated when BulkLoader form is loaded.
        // Calls methods to populate dirComboBox and osComboBox.
        //
        private void BulkLoaderForm_Load(object sender, EventArgs e)
        {
            PopulateDirComboBox();
            PopulateOSComboBox();
        }

        //
        // Handles the event generated when item is selected in 'dirComboBox'.
        // Clears the existing tree from the TreeView, then intializes tree with
        // selected drive.
        //
        private void dirComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            String drive = (String)dirComboBox.SelectedItem;
            dirTreeView.Nodes.Clear();
            dirTreeView.Nodes.Add(new CustomTreeNode(drive,true));
        }

        //
        // Handles the event generated when some node is selected in the dirTreeView.
        // Fetches the subdirectories and files of the selected directory if they have not been fetched already.
        //
        private void dirTreeView_AfterSelect(object sender, TreeViewEventArgs e)
        {
            CustomTreeNode node = (CustomTreeNode)dirTreeView.SelectedNode;
            String path = node.FullPath;
            if (!(node.Nodes.Count > 0) && node.AllowChildren)
            {
                    String[] dirs = Directory.GetDirectories(path);
                    String[] files = Directory.GetFiles(path);
                    foreach (String s in dirs)
                    {
                        CustomTreeNode child = new CustomTreeNode(BulkUtil.GetDirectoryName(s),true);
                        node.Nodes.Add(child);
                    }
                    foreach (String s in files)
                    {
                        CustomTreeNode child = new CustomTreeNode(BulkUtil.GetDirectoryName(s),1,1,false);
                        node.Nodes.Add(child);
                    }
             }
       }

       //
       // Handles the event generated when item is selected in 'osComboBox'.
       // Clears the existing tree from the TreeView. Then intializes tree with root folder.
       //
        private void osComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            String osName = (String)osComboBox.SelectedItem;
            IObjectStore os = ce.FetchOS(osName);
            BulkUtil.OS = os;
            osTreeView.Nodes.Clear();
            osTreeView.Nodes.Add(new CustomTreeNode("/",true));
        }

        //
        // Handles the event generated when some node is selected in the osTreeView.
        // Fetches the subfolders of the selected Folder and its containees, such as  
        // Document and CustomObject, if they have not been fetched already.
        //
        private void osTreeView_AfterSelect(object sender, TreeViewEventArgs e)
        {
            CustomTreeNode node = (CustomTreeNode)osTreeView.SelectedNode;
            String path = node.FullPath;
            try
            {
                if (!(node.Nodes.Count > 0) && node.AllowChildren)
                {
                    IFolder root = BulkUtil.FetchFolder(BulkUtil.OS, path);
                    IReferentialContainmentRelationshipSet rcrSet = BulkUtil.GetFolderContainees(root);
                    IEnumerator ie = rcrSet.GetEnumerator();
                    
                    while (ie.MoveNext())
                    {
                        IReferentialContainmentRelationship rcr = (IReferentialContainmentRelationship)ie.Current;
                        CustomTreeNode child = new CustomTreeNode(BulkUtil.GetContainmentName(rcr),1,1,false);
                        node.Nodes.Add(child);
                    }

                    IFolderSet fs = BulkUtil.GetSubFolders(root);
                    IEnumerator ie1 = fs.GetEnumerator();
                    while (ie1.MoveNext())
                    {
                        IFolder f = (IFolder)ie1.Current;
                        node.Nodes.Add(new CustomTreeNode(BulkUtil.GetFolderName(f),true));
                    }
                }
            }
            catch (EngineRuntimeException ere)
            {
                exceptionTextBox.Text = ere.Message;
                System.Console.WriteLine(ere.StackTrace);
            }
        }

        //
        // Loads the Documents created from the files in the selected source directory on the file system
        // into the ObjectStore in the selected destination folder. If includeSubDirectories checkbox is 
        // selected, the method also loads the Documents created from files in those subdirectories, too.
        // It files the Documents in the Folders created from the corresponding 
        // directories on the file system. The Documents are loaded in batches with the batch size 
        // of 10 (default), unless the user specifies a different size.
        //
        private void LoadDocuments(String selectedDirPath, String selectedFolderPath)
        {
            String[] dirs = Directory.GetDirectories(selectedDirPath);
            String[] files = Directory.GetFiles(selectedDirPath);
            String selectedDirectoryName = BulkUtil.GetDirectoryName(selectedDirPath);
            int j = 0;
            UpdatingBatch batch = BulkUtil.CreateBatch(ce.GetDomain());
            IFolder f = BulkUtil.CreateFolder(BulkUtil.OS, selectedFolderPath, selectedDirectoryName);
            BulkUtil.PopulateBatch(batch, f);
            j++;
            String createdFolderPath;
            if (selectedFolderPath.Equals("/"))
                createdFolderPath = selectedFolderPath + selectedDirectoryName;
            else
                createdFolderPath = selectedFolderPath + "/" + selectedDirectoryName;
            if (files.Length > 0)
            {
                for (int i = 0; i < files.Length; i++)
                {
                    if (j == BulkUtil.BatchSize)
                    {
                        batch.UpdateBatch();
                        j = 0;
                        ++batchNumber;
                        PerformStep();
                        AppendText("Batch " + batchNumber + " done." + '\n');
                    }
                    if (j <= BulkUtil.BatchSize)
                    {
                        String fileName = BulkUtil.GetDirectoryName(files[i]);
                        IDocument d = BulkUtil.CreateDocument(files[i], fileName, BulkUtil.OS);
                        BulkUtil.PopulateBatch(batch, d);
                        IReferentialContainmentRelationship rcr = BulkUtil.FileDocument(d, fileName, f);
                        BulkUtil.PopulateBatch(batch, rcr);
                        j++;
                    }
                    if (i == files.Length - 1)
                    {
                        batch.UpdateBatch();
                        j = 0;
                        ++batchNumber;                                                
                       
                        PerformStep();
                        AppendText("Batch " + batchNumber + " done." + '\n');                        
                    }
                }
            }
            else
            {
                batch.UpdateBatch();
            }
            if (includeSubDirectories)
            {
                for (int k = 0; k < dirs.Length; k++)
                {
                    LoadDocuments(dirs[k],createdFolderPath);
                }
            }
        }

        private void PerformStep()
        {
            // InvokeRequired required compares the thread ID of the
            // calling thread to the thread ID of the creating thread.
            // If these threads are different, it returns true.
            if(loadStatusForm.LoadProgressBar.InvokeRequired)            
            {
                PerformStepCallback d = new PerformStepCallback(PerformStep);
                this.Invoke(d);
            }
            else
            {
                loadStatusForm.LoadProgressBar.PerformStep();
            }
        }

        private void AppendText(String text)
        {
            // InvokeRequired required compares the thread ID of the
            // calling thread to the thread ID of the creating thread.
            // If these threads are different, it returns true.
            if (loadStatusForm.StatusTextBox.InvokeRequired)
            {
                AppendTextCallback d = new AppendTextCallback(AppendText);
                this.Invoke(d, new object[] { text });
            }
            else
            {
                loadStatusForm.StatusTextBox.AppendText(text);
            }
        }

        //
        // This method is invoked by the delegate (invoker of LoadDocuments method)
        // after it completes the execution of LoadDocuments method. It shuts down
        // the thread used by the delegate and resets variables such as batchNumber,
        // batchCount, and includeSubDirectories to default values.
        //
        private void LoadDocumentsCallack(IAsyncResult ar)
        {
            AppendText("Done with all the batches");
            includeSubDirectories = false;
            batchNumber = 0;
            BulkUtil.BatchCount = 0;
            AsyncResult result = (AsyncResult)ar;
            DelegateLoadDocuments dld = (DelegateLoadDocuments)result.AsyncDelegate;
            try
            {
                dld.EndInvoke(ar);
            }
            catch (EngineRuntimeException ere)
            {
                exceptionTextBox.Text = ere.Message;
                System.Console.WriteLine(ere.StackTrace);
                loadStatusForm.Close();
            }
       }

        //
        // Handles the event generated when 'Load' button is clicked.
        // It gethers all the required information such as selected directory
        // on the file system, selected destination folder on the ObjectStore,
        // batch size, and whether user wants the subdirectories included or not. 
        // It then displays the Windows form to display the progress of the loading
        // documents. It creates a seperate thread for Document loading 
        // operation. This thread calls the loadDocuments method, which loads all 
        // the files as Documents into the ObjectStore.
        //
        private void loadButton_Click(object sender, EventArgs e)
        {
            TreeNode dirNode = dirTreeView.SelectedNode;
            String selectedDirPath = dirNode.FullPath;
            CustomTreeNode folderNode = (CustomTreeNode)osTreeView.SelectedNode;
            String selectedFolderPath = folderNode.FullPath;
            if (batchCheckBox.Checked)
            {
                String BatchSize = batchTextBox.Text;
                if(BatchSize.Equals(""))
                    BulkUtil.BatchSize = 10;
                else
                    BulkUtil.BatchSize = Int32.Parse(BatchSize);
            }
            else
                BulkUtil.BatchSize = 10;
            if (includeCheckBox.Checked)
                includeSubDirectories = true;
            else
                includeSubDirectories = false;
            BulkUtil.CalculateBatchCount(selectedDirPath, includeSubDirectories);
            loadStatusForm = new LoadStatusForm();
            ProgressBar pb = loadStatusForm.LoadProgressBar;
            pb.Minimum = 0;
            pb.Maximum = BulkUtil.BatchCount;
            pb.Step = 1;
            RichTextBox statusTextBox = loadStatusForm.StatusTextBox;
            statusTextBox.Text = "";
            loadStatusForm.Show();
            try
            {
                DelegateLoadDocuments dld = new DelegateLoadDocuments(LoadDocuments);
                AsyncCallback ac = new AsyncCallback(LoadDocumentsCallack);
                dld.BeginInvoke(selectedDirPath, selectedFolderPath, ac, null);
            }
            catch (EngineRuntimeException ere)
            {
                exceptionTextBox.Text = ere.Message;
                System.Console.WriteLine(ere.StackTrace);
            }
        }

        //
        // Handles the event generated when 'Refresh' button is clicked.
        // Rebuilds the TreeView for selected ObjectStore.
        //
        private void refreshButton_Click(object sender, EventArgs e)
        {
            String osName = (String)osComboBox.SelectedItem;
            IObjectStore os = ce.FetchOS(osName);
            BulkUtil.OS = os;
            osTreeView.Nodes.Clear();
            osTreeView.Nodes.Add(new CustomTreeNode("/", true));
        }

        //
        // Handles the event generated when 'Close' button is clicked.
        // Closes this Form.
        //
        private void closeButton_Click(object sender, EventArgs e)
        {
            base.Dispose();
        }
    }
}
