/**
	IBM grants you a nonexclusive copyright license to use all programming code 
	examples from which you can generate similar function tailored to your own 
	specific needs.

	All sample code is provided by IBM for illustrative purposes only.
	These examples have not been thoroughly tested under all conditions.  IBM, 
	therefore cannot guarantee or imply reliability, serviceability, or function of 
	these programs.

	All Programs or code component contained herein are provided to you AS IS  
	without any warranties of any kind.
	The implied warranties of non-infringement, merchantability and fitness for a 
	particular purpose are expressly disclaimed.

	 Copyright IBM Corporation 2007, ALL RIGHTS RESERVED.
 */

package bulkloader;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.io.File;
import java.util.Iterator;
import java.util.Vector;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.border.TitledBorder;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import com.filenet.api.collection.FolderSet;
import com.filenet.api.collection.ReferentialContainmentRelationshipSet;
import com.filenet.api.core.Document;
import com.filenet.api.core.Folder;
import com.filenet.api.core.ReferentialContainmentRelationship;
import com.filenet.api.core.UpdatingBatch;
import com.filenet.api.exception.EngineRuntimeException;
import com.filenet.api.util.UserContext;

/**
 * This is the UI class for loading documents
 * in an ObjectStore. It has three panels. The sourcePanel 
 * allows a user to select a folder on a file system from which
 * to get files. The destinationPanel allows a user to select 
 * an ObjectStore and folder into which to load documents. The OptionPanel
 * allows users to specify batch size to take advantage of
 * batching facility of Content Engine (instead of committing documents
 * one by one in an ObjectStore, commit them in batches).
 */
public class LoaderFrame extends JFrame
{
	private static final long serialVersionUID = 1L;
	private JPanel jContentPane = null;
	private JPanel sourcePanel = null;
	private JSplitPane inputSplitPane = null;
	private JSplitPane optionSplitPane = null;
	private CEConnection ce = null;
	private JPanel destinationPanel = null;
	private JPanel optionPanel = null;
	private JComboBox directoryComboBox = null;
	private JCheckBox includeCheckBox = null;
	private JTree directoryTree = null;
	private JScrollPane treeScrollPane = null;
	private JComboBox osComboBox = null;
	private JButton refreshButton = null;
	private JScrollPane osScrollPane = null;
	private JTree osTree = null;
	private JPanel buttonPanel = null;
	private JPanel loadButtonPanel = null;
	private JButton loadButton = null;
	private JButton closeButton = null;
	private JPanel optPanel = null;
	private Vector osnames = null;
	private JLabel batchLabel = null;
	private JTextField batchTextField = null;
	private JDialog loadDialog = null;
	private JPanel dialogContentPane = null;
	private JPanel progressPanel = null;
	private JScrollPane dialogScrollPane = null;
	private JTextArea dialogTextArea = null;
	private JLabel progressLabel = null;
	private JRadioButton batchRadioButton = null;
	private JButton closeDialogButton = null;
	private JScrollPane exceptionScrollPane = null;
	private JTextArea exceptionTextArea = null;
	private JProgressBar progressBar = null;
	private static int batchNumber = 0;
	private boolean includeSubDirectories = false;
	
	/**
	 * This is the default constructor.
	 */
	public LoaderFrame(CEConnection c)
	{
		super();
		ce = c;
		osnames = ce.getOSNames();
		initialize();
	}

	/**
	 * This method initializes this class.
	 * 
	 * @return void
	 */
	private void initialize()
	{
		this.setSize(758, 360);
		this.setContentPane(getJContentPane());
		loadDirectoryComboBox();
		loadOSComboBox();
		this.setTitle("Bulk Loader");
	}

	/**
	 * This method initializes jContentPane.
	 * 
	 * @return javax.swing.JPanel
	 */
	private JPanel getJContentPane()
	{
		if (jContentPane == null)
		{
			jContentPane = new JPanel();
			jContentPane.setBorder(BorderFactory.createTitledBorder(null, "Bulk Loader Inputs", TitledBorder.DEFAULT_JUSTIFICATION, TitledBorder.DEFAULT_POSITION, new Font("Dialog", Font.BOLD, 12), Color.black));
			jContentPane.setLayout(new BorderLayout());
			jContentPane.add(getInputSplitPane(), BorderLayout.CENTER);
		}
		return jContentPane;
	}
	
	/**
	 * This method initializes inputSplitPane.	
	 * 	
	 * @return javax.swing.JSplitPane	
	 */
	private JSplitPane getInputSplitPane()
	{
		if (inputSplitPane == null)
		{
			inputSplitPane = new JSplitPane();
			inputSplitPane.setDividerLocation(250);
			inputSplitPane.setLeftComponent(getSourcePanel());
			inputSplitPane.setRightComponent(getOptionSplitPane());
		}
		return inputSplitPane;
	}

	/**
	 * This method initializes sourcePanel.	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getSourcePanel()
	{
		if (sourcePanel == null)
		{
			sourcePanel = new JPanel();
			sourcePanel.setLayout(new BorderLayout());
			sourcePanel.setBorder(BorderFactory.createTitledBorder(null, "Choose Source Directory", TitledBorder.DEFAULT_JUSTIFICATION, TitledBorder.DEFAULT_POSITION, new Font("Dialog", Font.BOLD, 12), Color.black));
			sourcePanel.add(getDirectoryComboBox(), BorderLayout.NORTH);
			sourcePanel.add(getIncludeCheckBox(), BorderLayout.SOUTH);
			sourcePanel.add(getTreeScrollPane(), BorderLayout.CENTER);
		}
		return sourcePanel;
	}
	
	/**
	 * This method initializes directoryComboBox.	
	 * 	
	 * @return javax.swing.JComboBox	
	 */
	private JComboBox getDirectoryComboBox()
	{
		if (directoryComboBox == null)
		{
			directoryComboBox = new JComboBox();
			directoryComboBox.addItemListener(new java.awt.event.ItemListener()
				{
					public void itemStateChanged(java.awt.event.ItemEvent e)
					{
						if (!(e.getStateChange() == java.awt.event.ItemEvent.SELECTED))
						{
							directoryComboboxItemSelectionActionPerformed(e);
						}
					}
				});
		}
		return directoryComboBox;
	}
	
	/**
	 * This method initializes includeCheckBox.
	 * 	
	 * @return javax.swing.JCheckBox	
	 */
	private JCheckBox getIncludeCheckBox()
	{
		if (includeCheckBox == null)
		{
			includeCheckBox = new JCheckBox();
			includeCheckBox.setText("Include sub directories");
		}
		return includeCheckBox;
	}
	
	/**
	 * This method initializes treeScrollPane.	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getTreeScrollPane()
	{
		if (treeScrollPane == null)
		{
			treeScrollPane = new JScrollPane();
			treeScrollPane.setViewportView(getDirectoryTree());
		}
		return treeScrollPane;
	}

	/**
	 * This method initializes directoryTree.	
	 * 	
	 * @return javax.swing.JTree	
	 */
	private JTree getDirectoryTree()
	{
		if (directoryTree == null)
		{
			directoryTree = new JTree();
			//Set the icon for nodes in JTree.
			directoryTree.setCellRenderer(new CustomCellRenderer());
			directoryTree.addTreeSelectionListener(new javax.swing.event.TreeSelectionListener()
			{
				public void valueChanged(javax.swing.event.TreeSelectionEvent e)
				{
					directoryTreeSelectionEventPerformed(e);
				}
			});
		}
		return directoryTree;
	}
	
	/**
	 * This method initializes optionSplitPane.	
	 * 	
	 * @return javax.swing.JSplitPane	
	 */
	private JSplitPane getOptionSplitPane()
	{
		if (optionSplitPane == null)
		{
			optionSplitPane = new JSplitPane();
			optionSplitPane.setDividerLocation(260);
			optionSplitPane.setLeftComponent(getDestinationPanel());
			optionSplitPane.setRightComponent(getOptionPanel());
		}
		return optionSplitPane;
	}

	/**
	 * This method initializes destinationPanel.	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getDestinationPanel()
	{
		if (destinationPanel == null)
		{
			destinationPanel = new JPanel();
			destinationPanel.setLayout(new BorderLayout());
			destinationPanel.setBorder(BorderFactory.createTitledBorder(null, "Choose Destination ObjectStore and Folder", TitledBorder.DEFAULT_JUSTIFICATION, TitledBorder.DEFAULT_POSITION, new Font("Dialog", Font.BOLD, 12), Color.black));
			destinationPanel.add(getOsComboBox(), BorderLayout.NORTH);
			destinationPanel.add(getOsScrollPane(), BorderLayout.CENTER);
			destinationPanel.add(getButtonPanel(), BorderLayout.SOUTH);
		}
		return destinationPanel;
	}

	/**
	 * This method initializes osComboBox.	
	 * 	
	 * @return javax.swing.JComboBox	
	 */
	private JComboBox getOsComboBox()
	{
		if (osComboBox == null)
		{
			osComboBox = new JComboBox();
		    osComboBox.addItemListener(new java.awt.event.ItemListener()
			{
				public void itemStateChanged(java.awt.event.ItemEvent e)
				{
					if (!(e.getStateChange() == java.awt.event.ItemEvent.SELECTED))
					{
						osComboBoxItemSelectionActionPerformed(e);
					}
				}
			});
		}
		return osComboBox;
	}
	
	/**
	 * This method initializes osScrollPane.	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getOsScrollPane()
	{
		if (osScrollPane == null)
		{
			osScrollPane = new JScrollPane();
			osScrollPane.setViewportView(getOsTree());
		}
		return osScrollPane;
	}

	/**
	 * This method initializes osTree.	
	 * 	
	 * @return javax.swing.JTree	
	 */
	private JTree getOsTree()
	{
		if (osTree == null)
		{
			osTree = new JTree();
			//Set the icon for nodes in JTree.
			osTree.setCellRenderer(new CustomCellRenderer());
			osTree.addTreeSelectionListener(new javax.swing.event.TreeSelectionListener()
			{
				public void valueChanged(javax.swing.event.TreeSelectionEvent e)
				{
					osTreeSelectionEventPerformed(e);
				}
			});
		}
		return osTree;
	}
	
	/**
	 * This method initializes buttonPanel.	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getButtonPanel()
	{
		if (buttonPanel == null)
		{
			buttonPanel = new JPanel();
			buttonPanel.setLayout(new GridBagLayout());
			buttonPanel.add(getRefreshButton(), new GridBagConstraints());
		}
		return buttonPanel;
	}

	/**
	 * This method initializes refreshButton.	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getRefreshButton()
	{
		if (refreshButton == null)
		{
			refreshButton = new JButton();
			refreshButton.setText("Refresh ObjectStore");
			refreshButton.addActionListener(new java.awt.event.ActionListener()
				{
					public void actionPerformed(java.awt.event.ActionEvent e)
					{
						refreshButtonActionPerformed(e);
					}
				});
		}
		return refreshButton;
	}
	
	/**
	 * This method initializes optionPanel.	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getOptionPanel()
	{
		if (optionPanel == null)
		{
			optionPanel = new JPanel();
			optionPanel.setLayout(new BorderLayout());
			optionPanel.setBorder(BorderFactory.createTitledBorder(null, "Choose Options", TitledBorder.DEFAULT_JUSTIFICATION, TitledBorder.DEFAULT_POSITION, new Font("Dialog", Font.BOLD, 12), Color.black));
			optionPanel.add(getLoadButtonPanel(), BorderLayout.SOUTH);
			optionPanel.add(getOptPanel(), BorderLayout.CENTER);
		}
		return optionPanel;
	}

	/**
	 * This method initializes optPanel.	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getOptPanel()
	{
		if (optPanel == null)
		{
			GridBagConstraints gridBagConstraints3 = new GridBagConstraints();
			gridBagConstraints3.fill = GridBagConstraints.BOTH;
			gridBagConstraints3.gridy = 4;
			gridBagConstraints3.weightx = 1.0;
			gridBagConstraints3.weighty = 1.0;
			gridBagConstraints3.gridx = 0;
			GridBagConstraints gridBagConstraints2 = new GridBagConstraints();
			gridBagConstraints2.gridx = 0;
			gridBagConstraints2.anchor = GridBagConstraints.WEST;
			gridBagConstraints2.fill = GridBagConstraints.HORIZONTAL;
			gridBagConstraints2.ipadx = 10;
			gridBagConstraints2.ipady = 10;
			gridBagConstraints2.gridheight = 1;
			gridBagConstraints2.gridy = 2;
			GridBagConstraints gridBagConstraints11 = new GridBagConstraints();
			gridBagConstraints11.fill = GridBagConstraints.VERTICAL;
			gridBagConstraints11.gridy = 3;
			gridBagConstraints11.weightx = 0.0;
			gridBagConstraints11.anchor = GridBagConstraints.WEST;
			gridBagConstraints11.gridwidth = 1;
			gridBagConstraints11.ipadx = 60;
			gridBagConstraints11.ipady = 10;
			gridBagConstraints11.gridx = 0;
			GridBagConstraints gridBagConstraints1 = new GridBagConstraints();
			gridBagConstraints1.gridx = 0;
			gridBagConstraints1.anchor = GridBagConstraints.WEST;
			gridBagConstraints1.fill = GridBagConstraints.HORIZONTAL;
			gridBagConstraints1.ipadx = 10;
			gridBagConstraints1.ipady = 10;
			gridBagConstraints1.gridy = 1;
			batchLabel = new JLabel();
			batchLabel.setText("Batch size (Default is 10)");
			optPanel = new JPanel();
			optPanel.setLayout(new GridBagLayout());
			optPanel.add(batchLabel, gridBagConstraints1);
			optPanel.add(getBatchTextField(), gridBagConstraints11);
			optPanel.add(getBatchRadioButton(), gridBagConstraints2);
			optPanel.add(getExceptionScrollPane(), gridBagConstraints3);
		}
		return optPanel;
	}
	
	/**
	 * This method initializes batchRadioButton.
	 * 	
	 * @return javax.swing.JRadioButton	
	 */
	private JRadioButton getBatchRadioButton()
	{
		if (batchRadioButton == null)
		{
			batchRadioButton = new JRadioButton();
			batchRadioButton.setText("Change Batch Size (<100):");
		}
		return batchRadioButton;
	}
	
	/**
	 * This method initializes batchTextField.	
	 * 	
	 * @return javax.swing.JTextField	
	 */
	private JTextField getBatchTextField()
	{
		if (batchTextField == null)
		{
			batchTextField = new JTextField();
			batchTextField.setText("");
		}
		return batchTextField;
	}
	
	/**
	 * This method initializes exceptionScrollPane.
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getExceptionScrollPane()
	{
		if (exceptionScrollPane == null)
		{
			exceptionScrollPane = new JScrollPane();
			exceptionScrollPane.setBorder(BorderFactory.createTitledBorder(null, "Exception Detail", TitledBorder.DEFAULT_JUSTIFICATION, TitledBorder.DEFAULT_POSITION, new Font("Dialog", Font.BOLD, 12), new Color(51, 51, 51)));
			exceptionScrollPane.setViewportView(getExceptionTextArea());
		}
		return exceptionScrollPane;
	}

	/**
	 * This method initializes exceptionTextArea.	
	 * 	
	 * @return javax.swing.JTextArea	
	 */
	private JTextArea getExceptionTextArea()
	{
		if (exceptionTextArea == null)
		{
			exceptionTextArea = new JTextArea();
			exceptionTextArea.setLineWrap(true);
			exceptionTextArea.setEditable(false);
		}
		return exceptionTextArea;
	}
	
	/**
	 * This method initializes loadButtonPanel.	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getLoadButtonPanel()
	{
		if (loadButtonPanel == null)
		{
			loadButtonPanel = new JPanel();
			loadButtonPanel.setLayout(new FlowLayout());
			loadButtonPanel.add(getLoadButton(), null);
			loadButtonPanel.add(getCloseButton(), null);
		}
		return loadButtonPanel;
	}

	/**
	 * This method initializes loadButton.	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getLoadButton()
	{
		if (loadButton == null)
		{
			loadButton = new JButton();
			loadButton.setText("Load");
			loadButton.addActionListener(new java.awt.event.ActionListener()
				{
					public void actionPerformed(java.awt.event.ActionEvent e)
					{
						loadButtonActionPerformed(e);
					}
				});
		}
		return loadButton;
	}
	
	/**
	 * This method initializes closeButton.	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getCloseButton()
	{
		if (closeButton == null)
		{
			closeButton = new JButton();
			closeButton.setText("Close");
			closeButton.addActionListener(new java.awt.event.ActionListener()
				{
					public void actionPerformed(java.awt.event.ActionEvent e)
					{
						closeButtonActionPerformed(e);
					}
				});
		}
		return closeButton;
	}
	
	/*
	 * Creates the JDialog to display the progress of 
	 * Document loading operation.
	 */
	private JDialog createLoadDialog()
	{
		loadDialog = new JDialog(LoaderFrame.this,"Loading Documents");
		loadDialog.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
		loadDialog.setContentPane(getDialogContentPane());
		loadDialog.setLocationRelativeTo(null);
		loadDialog.setSize(300, 200);
		return loadDialog;
	}

	/**
	 * This method initializes dialogContentPane1.	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getDialogContentPane()
	{
		if (dialogContentPane == null)
		{
			dialogContentPane = new JPanel();
			dialogContentPane.setLayout(new BorderLayout());
			dialogContentPane.add(getProgressPanel(), BorderLayout.NORTH);
			dialogContentPane.add(getDialogScrollPane(), BorderLayout.CENTER);
			dialogContentPane.add(getCloseDialogButton(), BorderLayout.SOUTH);
		}
		return dialogContentPane;
	}

	/**
	 * This method initializes progressPanel.	
	 * 	
	 * @return javax.swing.JPanel	
	 */
	private JPanel getProgressPanel()
	{
		if (progressPanel == null)
		{
			progressLabel = new JLabel();
			progressLabel.setText("Progress...");
			progressPanel = new JPanel();
			progressPanel.setLayout(new BorderLayout());
			progressPanel.add(progressLabel, BorderLayout.NORTH);
			progressPanel.add(getProgressBar(), BorderLayout.CENTER);
		}
		return progressPanel;
	}
	
	/**
	 * This method initializes progressBar.	
	 * 	
	 * @return javax.swing.JProgressBar	
	 */
	private JProgressBar getProgressBar()
	{
		if (progressBar == null)
		{
			progressBar = new JProgressBar();
			progressBar.setStringPainted(true);
		}
		return progressBar;
	}

	/**
	 * This method initializes dialogScrollPane.	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getDialogScrollPane()
	{
		if (dialogScrollPane == null)
		{
			dialogScrollPane = new JScrollPane();
			dialogScrollPane.setViewportView(getDialogTextArea());
		}
		return dialogScrollPane;
	}

	/**
	 * This method initializes dialogTextArea.	
	 * 	
	 * @return javax.swing.JTextArea	
	 */
	private JTextArea getDialogTextArea()
	{
		if (dialogTextArea == null)
		{
			dialogTextArea = new JTextArea();
		}
		return dialogTextArea;
	}
	
	/**
	 * This method initializes closeDialogButton.	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getCloseDialogButton()
	{
		if (closeDialogButton == null)
		{
			closeDialogButton = new JButton();
			closeDialogButton.setText("Close");
			closeDialogButton.addActionListener(new java.awt.event.ActionListener()
				{
					public void actionPerformed(java.awt.event.ActionEvent e)
					{
						loadDialog.dispose();
					}
				});
		}
		return closeDialogButton;
	}
	
	/*
	 * Returns the path string of the selected node
	 * on a JTree.
	 */
	private String getSelectedNodePath(TreePath tp)
	{
		String path = "";
		for(int i = 0; i < tp.getPathCount(); i++)
		{
			if( i == 0 || i == (tp.getPathCount() - 1))
				path = path + tp.getPathComponent(i);
			else
				path = path +  tp.getPathComponent(i) +  "/";
		}
		return path;
	}
	
	/*
	 * Loads the Documents created from the files in the selected directory on the file system
	 * into the selected ObjectStore folder. If includeSubFolder checkbox is 
	 * selected, the method also loads the Documents created from files in those subdirectories, too.
	 * It files the Documents in the folders created from the corresponding subdirectories on 
	 * the file system. The Documents are loaded in batches with the batch size of 10 (default), 
	 * unless the user specifies a different size.
	 */
	private void loadDocuments(String folderPath, String selectedOSFolderPath)
	{
		File f = new File(folderPath);
		File[] files = BulkUtil.getFileList(f);
		File[] dirs = BulkUtil.getDirList(f);
		int j = 0;
		UpdatingBatch ub = BulkUtil.createBatch(ce.fetchDomain());
		Folder fol = BulkUtil.createFolder(BulkUtil.getOs(), selectedOSFolderPath, f.getName());
		BulkUtil.populateBatch(ub, fol);
		j++;
		String createdFolderPath;
		if (selectedOSFolderPath.equals("/"))
			createdFolderPath = selectedOSFolderPath + f.getName();
		else
			createdFolderPath = selectedOSFolderPath + "/" + f.getName();
		if(files.length > 0)
		{
			for(int i = 0; i < files.length; i++)
			{
				if (j == BulkUtil.getBatchSize())
				{
					ub.updateBatch();
					j = 0;
					++batchNumber;
					getProgressBar().setValue(batchNumber);
					getDialogTextArea().append("Batch " + batchNumber + " done." + '\n');
				}
				if (j <= BulkUtil.getBatchSize())
				{
					Document d = BulkUtil.createDocument(files[i], BulkUtil.getOs(), files[i].getName());
					BulkUtil.populateBatch(ub, d);
					ReferentialContainmentRelationship rcr = BulkUtil.fileDocument(d, files[i].getName(),fol);
					BulkUtil.populateBatch(ub, rcr);
					j++;
				}
				if (i == files.length - 1)
				{
					ub.updateBatch();
					j = 0;
					++batchNumber;
					getProgressBar().setValue(batchNumber);
					getDialogTextArea().append("Batch " + batchNumber + " done." + '\n');
				}
			}
		}
		else
		{
			ub.updateBatch();
		}
		if(includeSubDirectories)
		{
			for(int k = 0; k < dirs.length; k++)
			{
				loadDocuments(dirs[k].getPath(),createdFolderPath);
			}
		}
	}

	/*
	 * Populates the directoryComboBox using available roots from the
	 * file system.
	 */
	private void loadDirectoryComboBox()
	{
		File[] list = File.listRoots();
		directoryComboBox.addItem("Select Root");
		for(int i = 0; i < list.length; i++)
		{
			directoryComboBox.addItem((String)list[i].getAbsolutePath());
		}
	}
	
	/*
	 * Based on selected root of the file system, calls createDirectoryTree method to build
	 * new JTree.
	 */
	private void directoryComboboxItemSelectionActionPerformed(java.awt.event.ItemEvent e)
	{
		DefaultTreeModel m = (DefaultTreeModel) directoryTree.getModel();
		m.setRoot(null);
		createDirectoryTree(directoryTree);
	}
	
	/*
	 * Builds new JTree with selected file system root as 
	 * top node.
	 */
	private void createDirectoryTree(JTree tree)
	{
		String node = (String)directoryComboBox.getSelectedItem();
		DefaultMutableTreeNode top = new DefaultMutableTreeNode(node);
		DefaultTreeModel m = (DefaultTreeModel) tree.getModel();
		m.setRoot(top);
	}
	
	/*
	 * Action performed while selecting a node from the directoryTree. 
	 * On selection, it fetches the subdirectories and files
	 * from the directory represented by selected node.
	 */
	private void directoryTreeSelectionEventPerformed(javax.swing.event.TreeSelectionEvent e)
	{
		try
		{
			DefaultTreeModel m = (DefaultTreeModel) directoryTree.getModel();
			if(m.getRoot() != null)
			{
				DefaultMutableTreeNode top = (DefaultMutableTreeNode) directoryTree.getLastSelectedPathComponent();
				if(top.isLeaf() && top.getAllowsChildren())
				{
					TreePath tp = directoryTree.getSelectionPath();
					String path = getSelectedNodePath(tp);
					File f = new File(path);
					File[] list = f.listFiles();
					for(int j = 0; j < list.length; j++)
					{
						if(list[j].isDirectory())
						{
							top.add(new DefaultMutableTreeNode(list[j].getName()));
						}
						else
						{
							DefaultMutableTreeNode child = new DefaultMutableTreeNode(list[j].getName(),false);
							top.add(child);
						}
					}
					if(top.isRoot())
						directoryTree.expandPath(new TreePath(top.getPath()));
				}
			}
		} 
		catch (RuntimeException e1)
		{
			e1.printStackTrace();
		}
	}
	
	/*
	 * Populates the osComboBox with available object stores.
	 */
	private void loadOSComboBox()
	{
		osComboBox.addItem("Select OS");
	    for(int i = 0; i < osnames.size(); i++)
	    {
	        osComboBox.addItem(osnames.get(i));
	    }
	}
	
	/*
	 * Based on selected object store, calls createOSTree method to build
	 * new JTree.
	 */
	private void osComboBoxItemSelectionActionPerformed(java.awt.event.ItemEvent e)
	{
		DefaultTreeModel m = (DefaultTreeModel) osTree.getModel();
		m.setRoot(null);
		createOSTree(osTree);
	}
	
	/*
	 * Builds new JTree with root folder from selected object store as 
	 * top node.
	 */
	private void createOSTree(JTree tree)
	{
		DefaultMutableTreeNode top = new DefaultMutableTreeNode("/");
		DefaultTreeModel m = (DefaultTreeModel) tree.getModel();
		m.setRoot(top);
		BulkUtil.setOs(ce.fetchOS((String) osComboBox.getSelectedItem()));
	}
	
	/*
	 * Action performed while selecting a node from the tree. 
	 * On selection, it fetches the Documents and CustomObjects
	 * from the folder represented by selected node. If  
	 * subfolders of that folder have not been fetched yet, it 
	 * fetches them, too.
	 */
	private void osTreeSelectionEventPerformed(javax.swing.event.TreeSelectionEvent e)
	{
		try
		{
			DefaultTreeModel m = (DefaultTreeModel) osTree.getModel();
			if(m.getRoot() != null)
			{
				DefaultMutableTreeNode top = (DefaultMutableTreeNode) osTree.getLastSelectedPathComponent();
				if(top.isLeaf()&& top.getAllowsChildren())
				{
					TreePath tp = osTree.getSelectionPath();
					String path = getSelectedNodePath(tp);
					Folder root = BulkUtil.fetchFolder(BulkUtil.getOs(), path);
					FolderSet fs = BulkUtil.getSubFolders(root);
					Iterator it = fs.iterator();
					while(it.hasNext())
					{
						Folder f = (Folder) it.next();
						top.add(new DefaultMutableTreeNode(BulkUtil.getFolderName(f)));
					}
					ReferentialContainmentRelationshipSet rcrSet = BulkUtil.getFolderContainees(root);
					Iterator i = rcrSet.iterator();
					while(i.hasNext())
					{
						ReferentialContainmentRelationship rcr = (ReferentialContainmentRelationship) i.next();
						DefaultMutableTreeNode child = new DefaultMutableTreeNode(BulkUtil.getContainmentName(rcr),false);
						top.add(child);
					}
					if(top.isRoot())
						osTree.expandPath(new TreePath(top.getPath()));
				}
			}
		} 
		catch (EngineRuntimeException e1)
		{
			e1.printStackTrace();
		}
		catch (RuntimeException e2)
		{
			e2.printStackTrace();
		}
	}
	
	/*
	 * Action performed on clicking Refresh button.
	 * Re-creates the object store folder tree by calling createOSTree method.
	 */
	private void refreshButtonActionPerformed(java.awt.event.ActionEvent e)
	{
		DefaultTreeModel m = (DefaultTreeModel) osTree.getModel();
		m.setRoot(null);
		createOSTree(osTree);
	}
	
	/*
	 * Action performed on clicking Load button.
	 * It gathers all the required information such as selected directory
	 * on the file system, selected destination folder on the object store,
	 * batch size, and whether user wants the subdirectories included or not. 
	 * It then displays the JDialog that shows the progress of the load
	 * operation. It creates a separate thread for document loading 
	 * operation. This thread does the JAAS login and calls the loadDocuments
	 * method, which will load all the files as Documents into the object store.
	 */
	private void loadButtonActionPerformed(java.awt.event.ActionEvent e)
	{
		final String selectedFolderPath = getSelectedNodePath(directoryTree.getSelectionPath());
		final String selectedOSFolderPath = getSelectedNodePath(osTree.getSelectionPath());
		exceptionTextArea.setText("");
		if (batchRadioButton.isSelected())
		{
			String batchSize = batchTextField.getText();
			if(batchSize.equals(""))
				BulkUtil.setBatchSize(10);
			else
				BulkUtil.setBatchSize(Integer.parseInt(batchSize));
		}
		else
			BulkUtil.setBatchSize(10);
		if (includeCheckBox.isSelected())
			includeSubDirectories = true;
		BulkUtil.calculateBatchCount(selectedFolderPath,includeSubDirectories);
		getProgressBar().setMinimum(0);
		getProgressBar().setMaximum(BulkUtil.getBatchCount());
		getProgressBar().setValue(0);
		getDialogTextArea().setText("");
		final JDialog jd = createLoadDialog();
		jd.setVisible(true);
		Thread t = new Thread(new Runnable()
		{
			public void run()
			{
				try
				{
					UserContext uc = UserContext.get();
					uc.pushSubject(ce.getSubject());
					loadDocuments(selectedFolderPath, selectedOSFolderPath);
					getDialogTextArea().append("Done with all the batches.");
					includeSubDirectories = false;
					batchNumber = 0;
					BulkUtil.setBatchCount(0);
				} 
				catch (EngineRuntimeException e)
				{
					exceptionTextArea.setText(e.getMessage());
					e.printStackTrace();
					jd.dispose();
				}
			}
		});
		t.start();
	}
	
	/*
	 * Action performed on clicking Close button.
	 * Closes this LoaderFrame.
	 */
	private void closeButtonActionPerformed(java.awt.event.ActionEvent e)
	{
		LoaderFrame.this.dispose();
	}
}
