/**
	IBM grants you a nonexclusive copyright license to use all programming code 
	examples from which you can generate similar function tailored to your own 
	specific needs.

	All sample code is provided by IBM for illustrative purposes only.
	These examples have not been thoroughly tested under all conditions.  IBM, 
	therefore cannot guarantee or imply reliability, serviceability, or function of 
	these programs.

	All Programs or code component contained herein are provided to you AS IS  
	without any warranties of any kind.
	The implied warranties of non-infringement, merchantability and fitness for a 
	particular purpose are expressly disclaimed.

	 Copyright IBM Corporation 2007, ALL RIGHTS RESERVED.
 */

package bulkloader;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import com.filenet.api.collection.ContentElementList;
import com.filenet.api.collection.FolderSet;
import com.filenet.api.collection.ReferentialContainmentRelationshipSet;
import com.filenet.api.constants.AutoClassify;
import com.filenet.api.constants.AutoUniqueName;
import com.filenet.api.constants.CheckinType;
import com.filenet.api.constants.DefineSecurityParentage;
import com.filenet.api.constants.RefreshMode;
import com.filenet.api.core.ContentTransfer;
import com.filenet.api.core.Document;
import com.filenet.api.core.Domain;
import com.filenet.api.core.Factory;
import com.filenet.api.core.Folder;
import com.filenet.api.core.IndependentlyPersistableObject;
import com.filenet.api.core.ObjectStore;
import com.filenet.api.core.ReferentialContainmentRelationship;
import com.filenet.api.core.UpdatingBatch;

/**
 * This class provides static methods to make API calls
 * to Content Engine and also some utility methods.
 */
public class BulkUtil
{
	private static ObjectStore os;
	private static int batchSize;
	private static int batchCount;
		
	/*
	 * Sets the user-selected object store from which
	 * folders will be retrieved.
	 */
	public static void setOs(ObjectStore os)
	{
		BulkUtil.os = os;
	}
	
	/*
	 * Returns the current Object Store.
	 */
	public static ObjectStore getOs()
	{
		return os;
	}
	
	/*
	 * API call to Content Engine to fetch Folder instance from
	 * given Object Store at given folder path.
	 */
	public static Folder fetchFolder(ObjectStore os, String folderPath)
	{
		Folder f = Factory.Folder.fetchInstance(os, folderPath, null);
		return f;
	}
	
    /*
     * Creates the Folder object at specified path using specified name.
     */
	public static Folder createFolder(ObjectStore os, String fPath, String fName)
    {
        Folder f = Factory.Folder.fetchInstance(os, fPath, null);
        Folder nf = f.createSubFolder(fName);
        return nf;
    }
	
	/*
	 * API call to Content Engine to retrieve sub folders of given 
	 * Folder instance.
	 */
	public static FolderSet getSubFolders(Folder f)
	{
		FolderSet fs = f.get_SubFolders();
		return fs;
	}
	
	/*
	 * API call to Content Engine to retrieve FolderName property of
	 * given Folder instance.
	 */
	public static String getFolderName(Folder f)
	{
		String name = f.get_FolderName();
		return name;
	}
	
	/*
	 * API call to Content Engine to retrieve containees of given Folder instance. 
	 * It returns ReferentialContainmentRelationshipSet which can be iterated through 
	 * to get ReferentialContainmentRelationship objects from which Documents,
	 * CustomObjects can be retrieved. 
	 */
	public static ReferentialContainmentRelationshipSet getFolderContainees(Folder f)
	{
		ReferentialContainmentRelationshipSet rcrSet = f.get_Containees();
		return rcrSet;
	}
	
	/*
	 * API call to Content Engine to retrieve ContainmentName property of 
	 * ReferentialContainmentRelationship object.
	 */
	public static String getContainmentName(ReferentialContainmentRelationship rcr)
	{
		String name = rcr.get_ContainmentName();
		return name;
	}
	
    /*
     * Reads the content from a file and stores it
     * in byte array. The byte array will later be
     * used to create ContentTransfer object.
     */
	public static byte[] readDocContentFromFile(File f)
    {
        FileInputStream is;
        byte[] b = null;
        int fileLength = (int)f.length();
        if(fileLength != 0)
        {
        	try
        	{
        		is = new FileInputStream(f);
        		b = new byte[fileLength];
        		is.read(b);
        		is.close();
        	}
        	catch (FileNotFoundException e)
        	{
        		e.printStackTrace();
        	}
        	catch (IOException e)
        	{
        		e.printStackTrace();
        	}
        }
        return b;
    }
	
    /*
     * Creates the ContentTransfer object from supplied file's
     * content.
     */
	public static ContentTransfer createContentTransfer(File f)
    {
        ContentTransfer ctNew = null;
        if(readDocContentFromFile(f) != null)
        {
        	ctNew = Factory.ContentTransfer.createInstance();
            ByteArrayInputStream is = new ByteArrayInputStream(readDocContentFromFile(f));
            ctNew.setCaptureSource(is);
            ctNew.set_RetrievalName(f.getName());
        }
        return ctNew;
    }
    
    /*
     * Creates the ContentElementList from ContentTransfer object.
     */
	public static ContentElementList createContentElements(File f)
    {
        ContentElementList cel = null;
        if(createContentTransfer(f) != null)
        {
        	cel = Factory.ContentElement.createList();
            ContentTransfer ctNew = createContentTransfer(f);
            cel.add(ctNew);
        }
        return cel;
    }
    
    /*
     * Creates the Document with content from supplied file.
     */
	public static Document createDocument(File f, ObjectStore os, String docName)
    {
        Document doc = Factory.Document.createInstance(os, null);
        doc.getProperties().putValue("DocumentTitle", docName);
        doc.checkin(AutoClassify.DO_NOT_AUTO_CLASSIFY, CheckinType.MAJOR_VERSION);
        ContentElementList cel = BulkUtil.createContentElements(f);
        if (cel != null)
        	doc.set_ContentElements(cel);
        return doc;
    }
	
    /*
     * Files the Containable object (i.e. Document,CustomObject) in
     * specified folder.
     */
	public static ReferentialContainmentRelationship fileDocument(Document d, String docName, Folder fo)
    {
    	ReferentialContainmentRelationship rcr;
    	rcr = fo.file(d, AutoUniqueName.AUTO_UNIQUE,
                    docName, DefineSecurityParentage.DO_NOT_DEFINE_SECURITY_PARENTAGE);
    	return rcr;
    }
	
	/*
	 * Creates the UpdatingBatch object.
	 */
	public static UpdatingBatch createBatch(Domain dom)
	{
		UpdatingBatch ub = UpdatingBatch.createUpdatingBatchInstance(dom, RefreshMode.REFRESH);
		return ub;
	}
	
	/*
	 * Populates the UpdatingBatch object with the supplied 
	 * IndependentlyPersistableObject instance.
	 */
	public static void populateBatch(UpdatingBatch ub, IndependentlyPersistableObject value)
	{
		ub.add(value, null);
	}
	
	/*
	 * Returns the array of the files contained in the 
	 * supplied directory.
	 */
	public static File[] getFileList(File dir)
	{
		File[] files = dir.listFiles(new FileFilter()
		{
			public boolean accept(File dir)
			{
				if (dir.isDirectory())
					return false;
				else
					return true;
			}
		});
		return files;
	}
	
	/*
	 * Returns the array of the directories contained in the 
	 * supplied directory.
	 */
	public static File[] getDirList(File dir)
	{
		File[] dirs = dir.listFiles(new FileFilter()
		{
			public boolean accept(File dir)
			{
				if (dir.isDirectory())
					return true;
				else
					return false;
			}
		});
		return dirs;
	}
	
	/*
	 * Returns the batch size.
	 */
	public static int getBatchSize()
	{
		return batchSize;
	}

	/*
	 * Sets the batch size.
	 */
	public static void setBatchSize(int batchSize)
	{
		BulkUtil.batchSize = batchSize;
	} 
	
	/*
	 * Calculates the batch count based on the count of files in supplied
	 * file system, whether subdirectories and batch size are included or not.
	 */
	public static void calculateBatchCount(String path, boolean includeSubDirectories)
	{
		File f = new File(path);
		File[] files = BulkUtil.getFileList(f);
		File[] dirs = BulkUtil.getDirList(f);
		int j = 0;
		j++;
		for(int i = 0; i < files.length; i++)
		{
			if (j <= getBatchSize())
			{
				j++;
			}
			if (j == getBatchSize() || i == files.length - 1)
			{
				++batchCount;
				j = 0;
			}
		}
		if(includeSubDirectories)
		{
			for(int k = 0; k < dirs.length; k++)
			{
				calculateBatchCount(dirs[k].getPath(),includeSubDirectories);
			}
		}
	}
	
	/*
	 * Returns the batch count.
	 */
	public static int getBatchCount()
	{
		return batchCount;
	}

	/*
	 * Sets the batch count.
	 */
	public static void setBatchCount(int batchCount)
	{
		BulkUtil.batchCount = batchCount;
	}
}
