# Sample program 
#  * (C) COPYRIGHT International Business Machines Corp., 2008, 2009
#  * All Rights Reserved * Licensed Materials - Property of IBM
#  *
#  * This sample program is provided AS IS and may be used, executed,
#  * copied and modified without royalty payment by customer
#  *
#  * (a) for its own instruction and study,
#  * (b) in order to develop applications designed to run with an IBM
#  *     WebSphere product for the customer's own internal use.
# 
# Version: 1.02
# 
# Information:
#  This script prints an XML summary of the runtime information available
#  for all queue points, publication points, mediation points, SIBus
#  links and WMQ links.
#  The script should be run against the deployment manager of the cell,
#  so that output is available from all running messaging engines in all buses.
#  The script collects the same information that can be displayed in via the
#  administrative console.
#  The following output is only available when the script is executed against
#  a WebSphere Application Server Version 7 environment:
#  - Service integration bus link information
#  - WebSphere MQ link information
# 
# Usage:
#  UNIX, Linux, z/OS:
#    <DMGR_PROFILE>/bin/wsadmin.sh -lang jython -f printSIBusSummary.py <OUTFILE.XML>
#  Windows:
#    <DMGR_PROFILE>\bin\wsadmin.bat -lang jython -f printSIBusSummary.py <OUTFILE.XML>

# Class to print an error out in full (with stack) to STDERR, as well as a short summary
# within the XML output. An integer is assigned to each error, to help match up the two.
class ErrorTracker:
  errorCount = 0
  # Helper method to print exception details as an error attribute in a tag
  def printErrorAndCloseTag(self, exception_tuple, oneLine):
    self.errorCount += 1
    # Print the error to STDERR
    sys.stderr.write("ERROR [" + str(self.errorCount) + "]:\n")
    sys.excepthook(exception_tuple[0],exception_tuple[1],exception_tuple[2])
    # Print a summary of the error to the output file
    out.write(' error="[' + str(self.errorCount) + ']: ')
    out.write(str(exception_tuple[0]).strip())
    if exception_tuple[1] != None: out.write(': ' + str(exception_tuple[1]).strip())
    if oneLine == 1: out.write('"/>\n')
    else: out.write('">\n')
# Single global instance of error tracker
et = ErrorTracker()

# Helper method to get a JMX attribute in string form
def getStrAttr(mBean, attr):
  val = AdminControl.getAttribute_jmx(mBean, attr)
  if val == None: val = ''
  else: val = str(val)
  return val

# Helper method to get the return value of JMX method in string form
def getStrReturnVal(mBean, method):
  val = AdminControl.invoke_jmx(mBean, method, [], [])
  if val == None: val = ''
  else: val = str(val)
  return val

# Helper toString method to handle None values as empty strings
def toStr(val):
  if val == None: val = ''
  else: val = str(val)
  return val

# Messaging engine class wraps an ME, maps its name to/from a UUID,
# and contains all the methods we used to print out the runtime state
# of that messaging engines
class MessagingEngine:
  "A class for printing a runtime summary of a messaging engine"
  # Constructor
  def __init__(self, bus, name, uuid):
    self.bus = bus
    self.name = name
    self.uuid = uuid

  # Method to print a summary of the runtime state of this messaging engine
  # - this is a the entry point into the class (other methods are logically private)
  def printRuntimeStateXML(self, indent):
    # Print the start of the tag (leaving room for additional properties)
    out.write(indent + '<MessagingEngine name="' + self.name + '" uuid="' + self.uuid + '"')
    meMBean = None
    try:
      # First lookup our MBean
      meLookupName = AdminControl.makeObjectName('WebSphere:type=SIBMessagingEngine,name=' + self.name + ',*')
      meMBeans = AdminControl.queryNames_jmx(meLookupName, None)
      if (meMBeans == None) or (meMBeans.size() == 0):
        # Just an empty messaging engine
        out.write(' state="Unknown (no MBean found)">\n')
      elif meMBeans.size() == 1:
        # Save a ref to the MBean
        meMBean = meMBeans[0]
        # Complete the entry tag
        out.write(' state="' + AdminControl.invoke_jmx(meMBean, "state", [], []) + '"')
        out.write(' activeServer="' + meMBean.getKeyProperty("process") + '"')
        out.write('>\n')
      else:
        # We only expect to zero/one MBean
        raise Exception, "Found " + str(len(meMBeans)) + " MBeans for messaging engine. Expected 1"
    except:
       et.printErrorAndCloseTag(sys.exc_info(), 0)
    # Iterate through the contents
    if (meMBean != None):
      self.printQueuePointsXML(meMBean, indent + " ")
      self.printRemoteQueuePointsXML(meMBean, indent + " ")
      self.printMediationPointsXML(meMBean, indent + " ")
      self.printRemoteMediationPointsXML(meMBean, indent + " ")
      self.printPublicationPointsXML(meMBean, indent + " ")
      self.printRemotePublicationPointsXML(meMBean, indent + " ")
      self.printRemoteSubscriptionPointsXML(meMBean, indent + " ")
      self.printBusLinks(meMBean, indent + " ")
      self.printWMQLinks(meMBean, indent + " ")
    # Complete our tag
    out.write(indent + "</MessagingEngine>\n")

  # Print a summary of all the queue points for the messaging engine,
  # using the supplied ME MBean looked up by the caller
  def printQueuePointsXML(self, meMBean, indent):  
    qpMBeans = None
    out.write(indent + "<QueuePoints")
    try:
      # Get a list of queue point MBeans
      qpLookupName = AdminControl.makeObjectName('WebSphere:type=SIBQueuePoint,SIBMessagingEngine=' + self.name + ',*')
      qpMBeans = AdminControl.queryNames_jmx(qpLookupName, None)
      out.write('>\n') # Complete the tag as lookup was successful
    except:
      et.printErrorAndCloseTag(sys.exc_info(), 0)
    # Run through each one
    for qpMBean in qpMBeans:
      rqps = {} # Directory of all RQPs for this queue point
      oneLineTag = 0
      out.write(indent + " <QueuePoint")
      try:
        out.write(' name="'  +  qpMBean.getKeyProperty("name") + '@' + self.name + '"')
        qpState = getStrAttr(qpMBean, "state")
        out.write(' state="' +  qpState + '"')
        out.write(' depth="' +  getStrAttr(qpMBean, "depth") + '"')
        hmt = AdminControl.getAttribute_jmx(qpMBean, "highMessageThreshold")
        if (hmt != None) and (hmt == java.lang.Long.MAX_VALUE): hmt = "MAX_VALUE"
        out.write(' highMessageThreshold="' +  str(hmt) + '"')
        out.write(' sendAllowed="' +  getStrAttr(qpMBean, "sendAllowed") + '"')
        # Only attempt to get additional details for active queue points
        if qpState == 'ACTIVE':
          # Get a list of inbound receivers, for remote queue points
          inboundReceivers = AdminControl.invoke_jmx(qpMBean, "listInboundReceivers", [], [])
          # Get a list of remote consumer transmitters, for remote queue points
          consumerTransmitters = AdminControl.invoke_jmx(qpMBean, "listRemoteConsumerTransmitters", [], [])
          # Add RQPs for all inbound receivers
          for ir in inboundReceivers:
            uuid = ir.getRemoteEngineUuid()
            if rqps.has_key(uuid): rqp = rqps[uuid]
            else: rqp = KnownRemoteQueuePoint(qpMBean, uuid)
            rqps[uuid] = rqp
            rqp.inboundReceiver = ir
          # Add RQPs for all consumer transmitters
          for ct in consumerTransmitters:
            uuid = ct.getRemoteEngineUuid()
            if rqps.has_key(uuid): rqp = rqps[uuid]
            else: rqp = KnownRemoteQueuePoint(qpMBean, uuid)
            rqps[uuid] = rqp
            rqp.consumerTransmitter = ct
        # If we do not have any RQPs then we can terminate the tag on this line
        if len(rqps.keys()) == 0:
          out.write('/>\n')
          oneLineTag = 1
        else:
          out.write('>\n') # We need a full tag
      except:
        et.printErrorAndCloseTag(sys.exc_info(), 0)
      # Process each RQP we found
      for rqpUuid in rqps.keys():
        rqp = rqps[rqpUuid]
        rqp.printSummaryXML(indent + "  ")
      # Complete our QueuePoint tag
      if oneLineTag == 0: out.write(indent + " </QueuePoint>\n")
    # Complete our QueuePoints tag
    out.write(indent + "</QueuePoints>\n")

  # Print a summary of all the remote queue points for the messaging engine,
  # using the supplied ME MBean looked up by the caller
  def printRemoteQueuePointsXML(self, meMBean, indent):  
    rqpMBeans = None
    out.write(indent + "<RemoteQueuePoints")
    try:
      # Get a list of remote queue point MBeans
      rqpLookupName = AdminControl.makeObjectName('WebSphere:type=SIBRemoteQueuePoint,SIBMessagingEngine=' + self.name + ',*')
      rqpMBeans = AdminControl.queryNames_jmx(rqpLookupName, None)
      out.write('>\n') # Complete the tag as lookup was successful
    except:
      et.printErrorAndCloseTag(sys.exc_info(), 0)
    # Run through each one
    for rqpMBean in rqpMBeans:
      oneLineTag = 0
      out.write(indent + " <RemoteQueuePoint")
      try:
        out.write(' name="'  +  rqpMBean.getKeyProperty("name") + '@' + self.name + '"')
        remoteMEuuid = getStrAttr(rqpMBean, "remoteMessagingEngineUuid")
        if (mesByUUID.has_key(remoteMEuuid)): remoteMEName = mesByUUID[remoteMEuuid].name
        else: remoteMEName = "Unknown"
        out.write(' remoteME="' +  remoteMEName + '"')
        out.write(' remoteMEUUID="' +  remoteMEuuid + '"')

        # Get outbound transmitter details, if one exists
        currentOutboundMessages = 0
        outboundMessagesSent = 0
        outboundTransmitter = AdminControl.invoke_jmx(rqpMBean, "getOutboundTransmitter", [], [])
        if (outboundTransmitter != None):
          currentOutboundMessages = outboundTransmitter.getDepth()
          outboundMessagesSent = outboundTransmitter.getNumberOfMessagesSent()
        out.write(' currentOutboundMessages="' + str(currentOutboundMessages) + '"')
        out.write(' outboundMessageSent="' + str(outboundMessagesSent) + '"')

        # Get remote consumer receiver, if one exists
        remoteConsumerReceiver = AdminControl.invoke_jmx(rqpMBean, "getRemoteConsumerReceiver", [], [])
        currentMessageRequests = 0
        completedMessageRequests = 0
        messageRequestsIssued = 0
        if (remoteConsumerReceiver != None):
          currentMessageRequests = remoteConsumerReceiver.getNumberOfActiveRequests()
          completedMessageRequests = remoteConsumerReceiver.getNumberOfCompletedRequests()
          messageRequestsIssued = remoteConsumerReceiver.getNumberOfRequestsIssued()
        out.write(' currentMessageRequests="' + str(currentMessageRequests) + '"')
        out.write(' completedMessageRequests="' + str(completedMessageRequests) + '"')
        out.write(' messageRequestsIssued="' + str(messageRequestsIssued) + '"')

        # Always one line for remote queue points
        out.write('/>\n')
      except:
        et.printErrorAndCloseTag(sys.exc_info(), 1)
    # Complete our QueuePoints tag
    out.write(indent + "</RemoteQueuePoints>\n")

  # Print a summary of all the remote publication points for the messaging engine,
  # using the supplied ME MBean looked up by the caller
  def printRemotePublicationPointsXML(self, meMBean, indent):  
    rppMBeans = None
    out.write(indent + "<RemotePublicationPoints")
    try:
      # Get a list of remote queue point MBeans
      rppLookupName = AdminControl.makeObjectName('WebSphere:type=SIBRemotePublicationPoint,SIBMessagingEngine=' + self.name + ',*')
      rppMBeans = AdminControl.queryNames_jmx(rppLookupName, None)
      out.write('>\n') # Complete the tag as lookup was successful
    except:
      et.printErrorAndCloseTag(sys.exc_info(), 0)
    # Run through each one
    for rppMBean in rppMBeans:
      oneLineTag = 0
      out.write(indent + " <RemotePublicationPoint")
      try:
        out.write(' name="'  +  rppMBean.getKeyProperty("name") + '@' + self.name + '"')
        remoteMEuuid = getStrAttr(rppMBean, "remoteMessagingEngineUuid")
        if (mesByUUID.has_key(remoteMEuuid)): remoteMEName = mesByUUID[remoteMEuuid].name
        else: remoteMEName = "Unknown"
        out.write(' remoteME="' +  remoteMEName + '"')
        out.write(' remoteMEUUID="' +  remoteMEuuid + '"')

        # Get outbound transmitter details, if one exists
        currentOutboundMessages = 0
        outboundMessagesSent = 0
        outboundTransmitters = AdminControl.invoke_jmx(rppMBean, "listOutboundTransmitters", [], [])
        if (outboundTransmitters != None):
          for outboundTransmitter in outboundTransmitters:
            currentOutboundMessages += outboundTransmitter.getDepth()
            outboundMessagesSent += outboundTransmitter.getNumberOfMessagesSent()
        out.write(' currentOutboundMessages="' + str(currentOutboundMessages) + '"')
        out.write(' outboundMessageSent="' + str(outboundMessagesSent) + '"')

        # Always one line for remote queue points
        out.write('/>\n')
      except:
        et.printErrorAndCloseTag(sys.exc_info(), 1)
    # Complete our RemotePublicationPoints tag
    out.write(indent + "</RemotePublicationPoints>\n")

  # Print a summary of all the remote subscription points for the messaging engine,
  # using the supplied ME MBean looked up by the caller
  def printRemoteSubscriptionPointsXML(self, meMBean, indent):  
    rspMBeans = None
    out.write(indent + "<RemoteSubscriptionPoints")
    try:
      # Get a list of remote queue point MBeans
      rspLookupName = AdminControl.makeObjectName('WebSphere:type=SIBRemoteSubscriptionPoint,SIBMessagingEngine=' + self.name + ',*')
      rspMBeans = AdminControl.queryNames_jmx(rspLookupName, None)
      out.write('>\n') # Complete the tag as lookup was successful
    except:
      et.printErrorAndCloseTag(sys.exc_info(), 0)
    # Run through each one
    for rspMBean in rspMBeans:
      oneLineTag = 0
      out.write(indent + " <RemoteSubscriptionPoint")
      try:
        out.write(' name="'  +  rspMBean.getKeyProperty("name") + '"')
        out.write(' topicSpace="' + getStrReturnVal(rspMBean, "getTopicSpace") + '"')
        remoteMEuuid = getStrAttr(rspMBean, "remoteMessagingEngineUuid")
        if (mesByUUID.has_key(remoteMEuuid)): remoteMEName = mesByUUID[remoteMEuuid].name
        else: remoteMEName = "Unknown"
        out.write(' remoteME="' +  remoteMEName + '"')
        out.write(' remoteMEUUID="' +  remoteMEuuid + '"')

        # Get remote consumer receiver, if one exists
        remoteConsumerReceiver = AdminControl.invoke_jmx(rspMBean, "getRemoteConsumerReceiver", [], [])
        currentMessageRequests = 0
        completedMessageRequests = 0
        messageRequestsIssued = 0
        if (remoteConsumerReceiver != None):
          currentMessageRequests = remoteConsumerReceiver.getNumberOfActiveRequests()
          completedMessageRequests = remoteConsumerReceiver.getNumberOfCompletedRequests()
          messageRequestsIssued = remoteConsumerReceiver.getNumberOfRequestsIssued()
        out.write(' currentMessageRequests="' + str(currentMessageRequests) + '"')
        out.write(' completedMessageRequests="' + str(completedMessageRequests) + '"')
        out.write(' messageRequestsIssued="' + str(messageRequestsIssued) + '"')

        # Always one line for remote queue points
        out.write('/>\n')
      except:
        et.printErrorAndCloseTag(sys.exc_info(), 1)
    # Complete our RemotePublicationPoints tag
    out.write(indent + "</RemoteSubscriptionPoints>\n")

  # Print a summary of all the mediation points for the messaging engine,
  # using the supplied ME MBean looked up by the caller
  def printMediationPointsXML(self, meMBean, indent):  
    mpMBeans = None
    out.write(indent + "<MediationPoints")
    try:
      # Get a list of mediation point MBeans
      mpLookupName = AdminControl.makeObjectName('WebSphere:type=SIBMediationPoint,SIBMessagingEngine=' + self.name + ',*')
      mpMBeans = AdminControl.queryNames_jmx(mpLookupName, None)
      out.write('>\n') # Complete the tag as lookup was successful
    except:
      et.printErrorAndCloseTag(sys.exc_info(), 0)
    # Run through each one
    for mpMBean in mpMBeans:
      out.write(indent + " <MediationPoint")
      try:
        out.write(' name="'  +  mpMBean.getKeyProperty("name") + '@' + self.name + '"')
        mpState = getStrAttr(mpMBean, "currentState")
        out.write(' status="' +  mpState + '"')
        out.write(' depth="' +  getStrAttr(mpMBean, "depth") + '"')
        hmt = AdminControl.getAttribute_jmx(mpMBean, "highMessageThreshold")
        if (hmt != None) and (hmt == java.lang.Long.MAX_VALUE): hmt = "MAX_VALUE"
        out.write(' highMessageThreshold="' +  str(hmt) + '"')
        out.write(' sendAllowed="' +  getStrAttr(mpMBean, "sendAllowed") + '"')
        out.write('/>\n')
      except:
        et.printErrorAndCloseTag(sys.exc_info(), 1)
    # Complete our QueuePoints tag
    out.write(indent + "</MediationPoints>\n")

  # Print a summary of all the remote mediation points for the messaging engine,
  # using the supplied ME MBean looked up by the caller
  def printRemoteMediationPointsXML(self, meMBean, indent):  
    rmpMBeans = None
    out.write(indent + "<RemoteMediationPoints")
    try:
      # Get a list of remote mediation point MBeans
      rmpLookupName = AdminControl.makeObjectName('WebSphere:type=SIBRemoteMediationPoint,SIBMessagingEngine=' + self.name + ',*')
      rmpMBeans = AdminControl.queryNames_jmx(rmpLookupName, None)
      out.write('>\n') # Complete the tag as lookup was successful
    except:
      et.printErrorAndCloseTag(sys.exc_info(), 0)
    # Run through each one
    for rmpMBean in rmpMBeans:
      oneLineTag = 0
      out.write(indent + " <RemoteMediationPoint")
      try:
        out.write(' name="'  +  rmpMBean.getKeyProperty("name") + '@' + self.name + '"')
        remoteMEuuid = getStrAttr(rmpMBean, "remoteMessagingEngineUuid")
        if (mesByUUID.has_key(remoteMEuuid)): remoteMEName = mesByUUID[remoteMEuuid].name
        else: remoteMEName = "Unknown"
        out.write(' remoteME="' +  remoteMEName + '"')
        out.write(' remoteMEUUID="' +  remoteMEuuid + '"')

        # Get outbound transmitter details, if one exists
        currentOutboundMessages = 0
        outboundMessagesSent = 0
        outboundTransmitter = AdminControl.invoke_jmx(rmpMBean, "getOutboundTransmitter", [], [])
        if (outboundTransmitter != None):
          currentOutboundMessages = outboundTransmitter.getDepth()
          outboundMessagesSent = outboundTransmitter.getNumberOfMessagesSent()
        out.write(' currentOutboundMessages="' + str(currentOutboundMessages) + '"')
        out.write(' outboundMessageSent="' + str(outboundMessagesSent) + '"')

        # Always one line for remote mediation points
        out.write('/>\n')
      except:
        et.printErrorAndCloseTag(sys.exc_info(), 1)
    # Complete our MediationPoints tag
    out.write(indent + "</RemoteMediationPoints>\n")

  # Print a summary of all the publication points for the messaging engine,
  # using the supplied ME MBean looked up by the caller
  def printPublicationPointsXML(self, meMBean, indent):  
    ppMBeans = None
    out.write(indent + "<PublicationPoints")
    try:
      # Get a list of queue point MBeans
      ppLookupName = AdminControl.makeObjectName('WebSphere:type=SIBPublicationPoint,SIBMessagingEngine=' + self.name + ',*')
      ppMBeans = AdminControl.queryNames_jmx(ppLookupName, None)
      out.write('>\n') # Complete the tag as lookup was successful
    except:
      et.printErrorAndCloseTag(sys.exc_info(), 0)
    # Run through each one
    for ppMBean in ppMBeans:
      inboundReceivers = []
      subscriptions = []
      out.write(indent + " <PublicationPoint")
      depth = None
      try:
        depth = getStrAttr(ppMBean, "depth")
      except:
        # Attribute does not exist
        pass
      try:
        out.write(' name="'  +  ppMBean.getKeyProperty("name") + '@' + self.name + '"')
        if depth != None: out.write(' depth="' +  depth + '"')
        hmt = AdminControl.getAttribute_jmx(ppMBean, "highMessageThreshold")
        if (hmt != None) and (hmt == java.lang.Long.MAX_VALUE): hmt = "MAX_VALUE"
        out.write(' highMessageThreshold="' +  str(hmt) + '"')
        out.write(' sendAllowed="' +  getStrAttr(ppMBean, "sendAllowed") + '"')
        # Get a list of inbound receivers (remote publication points)
        inboundReceivers = AdminControl.invoke_jmx(ppMBean, "listInboundReceivers", [], [])
        # Get a list of subscriptions
        subscriptions = AdminControl.invoke_jmx(ppMBean, "getSubscriptions", [], [])
        # Complete the tag
        out.write('>\n')
      except:
        et.printErrorAndCloseTag(sys.exc_info(), 0)
      # Run through each remote publication point (inbound receiver)
      for ir in inboundReceivers:
        out.write(indent + '  <KnownRemotePublicationPoint')
        try:
          remoteMEuuid = ir.getRemoteEngineUuid()
          if mesByUUID.has_key(remoteMEuuid): remoteMEname = mesByUUID[remoteMEuuid].name
          else: remoteMEname = "Unknown"
          out.write(' me="' + remoteMEname + '"')
          out.write(' meUUID="' + remoteMEuuid + '"')
          out.write(' currentInboundMessages="' + str(ir.getDepth()) + '"')
          out.write(' inboundMessagesReceived="' + str(ir.getNumberOfMessagesReceived()) + '"')
          out.write('/>\n')
        except:
          et.printErrorAndCloseTag(sys.exc_info(), 1)
      # Run through each subscription
      depthTally = 0
      for sub in subscriptions:
        oneLineTag = 0
        rsps = [] # remote subscription points
        out.write(indent + '  <Subscription')
        try:
          out.write(' subscriberId="' + toStr(sub.getSubscriberId()) + '"')
          subDepth = sub.getDepth()
          depthTally = depthTally + subDepth
          out.write(' depth="' + str(subDepth) + '"')
          # Write any selector
          selector = sub.getSelector()
          if selector != None: out.write(' selector="' + sub.getSelector() + '"')
          # Write topics
          topics = sub.getTopics()
          if (topics != None):
            out.write(' topics="')
            sep = ''
            for topic in topics:
              if (topic == None): topic = ''
              out.write(topic + sep)
              sep = ','
            out.write('"')
          # Get a list of remote subscription points
          rsps = AdminControl.invoke_jmx(ppMBean, "listRemoteConsumerTransmitters", [sub], ['com.ibm.websphere.sib.admin.SIBSubscription'])
          # Check if we have children, or can just close the tag here
          if len(rsps) == 0:
            out.write('/>\n')
            oneLineTag = 1
          else: out.write('>\n')
        except:
          et.printErrorAndCloseTag(sys.exc_info(), 0)
        # Do we have remote sub points?
        for rsp in rsps:
          out.write(indent + '   <KnownRemoteSubscriptionPoint')
          try:
            remoteMEuuid = rsp.getRemoteEngineUuid()
            if mesByUUID.has_key(remoteMEuuid): remoteMEname = mesByUUID[remoteMEuuid].name
            else: remoteMEname = "Unknown"
            out.write(' me="' + remoteMEname + '"')
            out.write(' meUUID="' + remoteMEuuid + '"')
            out.write(' currentMessageRequests="' + str(rsp.getDepth()) + '"')
            out.write(' completedMessageRequests="' + str(rsp.getNumberOfCompletedRequests()) + '"')
            out.write(' messageRequestsReceived="' + str(rsp.getNumberOfRequestsReceived()) + '"')
            out.write('/>\n')
          except:
            et.printErrorAndCloseTag(sys.exc_info(), 1)
        # Close the subscription tag if required
        if oneLineTag == 0: out.write(indent + '  </Subscription>\n')
      # Complete our PublicationPoint tag

      # give ourselves 50 messages breathing room to try minimise false positives
      depthCheck = int(depth) - 50
      out.write(indent + "  <DepthTally>" + str(depthTally))
      if depthTally < depthCheck: out.write(' - WARNING</DepthTally>\n')
      else: out.write(' - OK</DepthTally>\n')
      out.write(indent + " </PublicationPoint>\n")
    # Complete our PublicationPoints tag
    out.write(indent + "</PublicationPoints>\n")

  # Print a summary of all the SIBus links hosted on this messaging engine.
  # using the supplied ME MBean looked up by the caller
  def printBusLinks(self, meMBean, indent):  
    glMBeans = None
    out.write(indent + "<BusLinks")
    try:
      # Get a list of link transmitter MBeans - will be empty for <V7.0 MEs
      glLookupName = AdminControl.makeObjectName('WebSphere:type=SIBGatewayLink,SIBMessagingEngine=' + self.name + ',*')
      glMBeans = AdminControl.queryNames_jmx(glLookupName, None)
      out.write('>\n') # Complete the tag as lookup was successful
    except:
      et.printErrorAndCloseTag(sys.exc_info(), 0)
    # Keep track of all link target UUIDs we've seen as local links, to exclude from the remote transmitter list
    localLinkUuids = {}
    # Run through each transmitter
    for glMBean in glMBeans:
      oneLineTag = 0
      linkReceivers = []
      targetUuid = glMBean.getKeyProperty("targetUuid")
      localLinkUuids[targetUuid] = 1
      out.write(indent + " <BusLink")
      # First check we can query the foreign bus name... if this fails we are
      # talking to a < V7 messaging engine
      foreignBusName = None
      oldVersion = 0
      try:
        foreignBusName = getStrReturnVal(glMBean, "getForeignBusName")
      except:
        oldVersion = 1
      # Get the name (sometimes the returned name includes quotes)
      virtualLinkName = toStr(glMBean.getKeyProperty("name"))
      if not (virtualLinkName.find('"') == 0): virtualLinkName = '"' + virtualLinkName + '"'
      # Print the correct information based on the version
      if oldVersion:
        try:
          out.write(' name=' + virtualLinkName)
          out.write(' state="' +  getStrReturnVal(glMBean, "getStatus") + '"')
          out.write('>\n')
        except:
          et.printErrorAndCloseTag(sys.exc_info(), 0)
      else:
        try:
          out.write(' name=' + virtualLinkName )
          out.write(' foreignBus="' + foreignBusName + '"')
          stateString = AdminControl.invoke_jmx(glMBean, "getStatus", [], [])
          if stateString != None:
            out.write(' state="' +  stateString + '"')
          linkReceivers = AdminControl.invoke_jmx(glMBean, "listLinkReceivers", [], [])
          if linkReceivers == None: linkReceivers = []
          out.write('>\n')
        except:
          et.printErrorAndCloseTag(sys.exc_info(), 0)
        # Print out the link receivers for this link
        for lr in linkReceivers:
          out.write(indent + "  <LinkReceiver")
          try:
            out.write(' state="' +  lr.getState() + '"')
            receiverType = lr.getReceiverType()
            remoteMEuuid = lr.getForeignEngineUuid()
            if mesByUUID.has_key(remoteMEuuid): remoteMEname = mesByUUID[remoteMEuuid].name
            else: remoteMEname = "Unknown"
            out.write(' me="' +  remoteMEname + '"')
            out.write(' meUUID="' +  remoteMEuuid + '"')
            out.write(' receiverType="' +  receiverType + '"')
            if receiverType == "PUBLICATION":
              out.write(' topicSpace="' +  toStr(lr.getTargetDestination()) + '"')
            out.write(' currentInboundMessages="' +  str(lr.getDepth()) + '"')        
            out.write(' messagesReceived="' +  str(lr.getNumberOfMessagesReceived()) + '"')
            timeSinceLastMessageReceived = lr.getTimeSinceLastMessageReceived()
            if timeSinceLastMessageReceived > 0:
              out.write(' timeSinceLastMessageReceived="' +  str(timeSinceLastMessageReceived) + 'ms"')
            out.write(indent + "/>\n")
          except:
            et.printErrorAndCloseTag(sys.exc_info(), 1)
        # Print out the link transmitters for this link
        self.printLinkTransmittersXML(meMBean, targetUuid, {}, indent + '  ')
      # End the link tag
      out.write(indent + ' </BusLink>\n')
    # Just in case we have any orphaned link mBeans, print these out here
    self.printLinkTransmittersXML(meMBean, None, localLinkUuids, indent + ' ')
    # Complete our SIBLinks tag
    out.write(indent + "</BusLinks>\n")

  # Print a summary of SIBus link transmitters for the messaging engine,
  # using the supplied ME MBean looked up by the caller.
  # Either prints all transmitters with a particular target UUID, or
  # prints all transmitters excluding keys that exist in the excludeUuids hash.
  def printLinkTransmittersXML(self, meMBean, targetUuid, excludeUuids, indent):  
    ltMBeans = None
    try:
      # Get a list of link transmitter MBeans - will be empty for <V7.0 MEs
      lookupString = 'WebSphere:type=SIBLinkTransmitter,SIBMessagingEngine=' + self.name
      if targetUuid != None: lookupString += ',targetUuid=' + targetUuid
      lookupString += ',*'
      ltLookupName = AdminControl.makeObjectName(lookupString)
      ltMBeans = AdminControl.queryNames_jmx(ltLookupName, None)
    except:
      out.write('<SIBLinkTransmitters')
      et.printErrorAndCloseTag(sys.exc_info(), 1)
    # Run through each transmitter
    for ltMBean in ltMBeans:
      # Check this one shouldn't be excluded
      transmitterTargetUuid = ltMBean.getKeyProperty("targetUuid")
      if not excludeUuids.has_key(transmitterTargetUuid):
        out.write(indent + "<LinkTransmitter")
        try:
          if (targetUuid == None): out.write(' foreignBus="' + getStrReturnVal(ltMBean, "getForeignBusName") + '"')
          out.write(' state="' +  getStrReturnVal(ltMBean, "getState") + '"')
          out.write(' linkType="' + getStrReturnVal(ltMBean, "getLinkType") + '"')
          transmitterType = getStrReturnVal(ltMBean, "getTransmitterType")
          out.write(' transmitterType="' +  transmitterType + '"')
          if transmitterType == "PUBLICATION":
            out.write(' topicSpace="' +  getStrReturnVal(ltMBean, "getTargetDestination") + '"')
          putInhibited = AdminControl.invoke_jmx(ltMBean, "isPutInhibited", [], [])
          if putInhibited == 0: sendAllowed = 1
          else: sendAllowed = 0
          out.write(' sendAllowed="' + str(sendAllowed) + '"')
          out.write(' currentOutboundMessages="' +  getStrReturnVal(ltMBean, "getDepth") + '"')        
          out.write(' messagesSent="' +  getStrReturnVal(ltMBean, "getNumberOfMessagesSent") + '"')
          timeSinceLastMessageSent = AdminControl.invoke_jmx(ltMBean, "getTimeSinceLastMessageSent", [], [])
          if timeSinceLastMessageSent > 0:
            out.write(' timeSinceLastMessageSent="' +  str(timeSinceLastMessageSent) + 'ms"')
  
          # Always one line for link transmitters
          out.write('/>\n')
        except:
          et.printErrorAndCloseTag(sys.exc_info(), 1)

  # Print a summary of all the WMQ links hosted on this messaging engine
  # using the supplied ME MBean looked up by the caller
  def printWMQLinks(self, meMBean, indent):  
    mqlMBeans = None
    out.write(indent + "<WMQLinks")
    try:
      # Get a list of WMQ Link MBeans - will be empty for <V7.0 MEs
      mqlLookupName = AdminControl.makeObjectName('WebSphere:type=SIBMQLink,SIBMessagingEngine=' + self.name + ',*')
      mqlMBeans = AdminControl.queryNames_jmx(mqlLookupName, None)
      out.write('>\n') # Complete the tag as lookup was successful
    except:
      et.printErrorAndCloseTag(sys.exc_info(), 0)
    # Run through each WMQ Link MBean found
    for mqlMBean in mqlMBeans:
      schlMBeans = []
      rchlMBeans = []
      # We may not be able to query msgs received as introduced at v7.0
      msgsReceived = None
      try:
        msgsReceived = getStrReturnVal(mqlMBean, "getTotalLinkMessagesReceived")
      except:
        pass
      out.write(indent + " <WMQLink")
      try:
        # Get the targetUuid
        linkName = mqlMBean.getKeyProperty("name")
        out.write(' name="' +  linkName + '"')
        out.write(' state="' +  getStrReturnVal(mqlMBean, "getOverallStatus") + '"')
        if msgsReceived != None: out.write(' messagesReceived="' +  msgsReceived + '"')
        # Get a list of WMQ link sender channel instances
        schlLookupName = AdminControl.makeObjectName('WebSphere:type=SIBMQLinkSenderChannel,SIBMessagingEngine=' + self.name + ',name=' + linkName + 'SNDR,*')
        schlMBeans = AdminControl.queryNames_jmx(schlLookupName, None)
        # Get a list of WMQ link receiver channel instances
        rchlLookupName = AdminControl.makeObjectName('WebSphere:type=SIBMQLinkReceiverChannel,SIBMessagingEngine=' + self.name + ',name=' + linkName + 'RCVR,*')
        rchlMBeans = AdminControl.queryNames_jmx(rchlLookupName, None)
        out.write('>\n') # Complete the opening tag
      except:
          et.printErrorAndCloseTag(sys.exc_info(), 0)
      # Iterate through the sender channels
      for schlMBean in schlMBeans:
        sxmitMBeans = []
        out.write(indent + "  <SenderChannel")
        try:
          stateObject = AdminControl.invoke_jmx(schlMBean, "getCurrentStatus", [], [])
          channelName = stateObject.getChannelName()
          out.write(' channelName="' + channelName + '"')
          out.write(' state="' + toStr(stateObject.getState()) + '"')
          out.write(' virtualQmgr="' + stateObject.getQueueManager() + '"')
          out.write(' ipAddress="' + toStr(stateObject.getIpAddress()) + '"')
          out.write(' messagesSent="' + toStr(stateObject.getNumberOfMessagesSent()) + '"')
          out.write(' currentLUWID="' + toStr(stateObject.getCurrentLUWID()) + '"')
          out.write(' currentSequenceNo="' + toStr(stateObject.getCurrentSequenceNumber()) + '"')
          out.write(' inDoubt="' + toStr(stateObject.getInDoubt()) + '"')
          # Get a list of WMQ link sender channel transmitter instances
          sxmitLookupName = AdminControl.makeObjectName('WebSphere:type=SIBMQLinkSenderChannelTransmitter,SIBMessagingEngine=' + self.name + ',name=' + channelName + ',*')
          sxmitMBeans = AdminControl.queryNames_jmx(sxmitLookupName, None)
          out.write('>\n') # Complete the opening tag
        except:
          et.printErrorAndCloseTag(sys.exc_info(), 0)
        # Iterate through the sender channel transmitters
        for sxmitMBean in sxmitMBeans:
          out.write(indent + "   <SenderChannelTransmitter")
          # List of the known link transmitters
          knownLinkTransmitters = []
          oneLineTag = 0
          try:
            out.write(' status="' +  getStrReturnVal(sxmitMBean, "getState") + '"')
            out.write(' currentOutboundMessages="' +  getStrReturnVal(sxmitMBean, "getDepth") + '"')
            out.write(' messagesSent="' +  getStrReturnVal(sxmitMBean, "getNumberOfMessagesSent") + '"')
            timeSinceLastMessageSent = AdminControl.invoke_jmx(sxmitMBean, "getTimeSinceLastMessageSent", [], [])
            if timeSinceLastMessageSent > 0: 
              out.write(' timeSinceLastMessageSent="' + str(timeSinceLastMessageSent) + 'ms"')
            # List the known link transmitters
            knownLinkTransmitters = AdminControl.invoke_jmx(sxmitMBean, "listInboundReceivers", [], [])
            # Complete the tag
            if (knownLinkTransmitters == None) or (len(knownLinkTransmitters)) == 0: 
              out.write('/>\n') # Complete the one-line tag
              oneLineTag = 1
            else: out.write('>\n')
          except:
            et.printErrorAndCloseTag(sys.exc_info(), 1)
          # Print the known link transmitters
          if (oneLineTag == 0):
            for ir in knownLinkTransmitters:
              out.write(indent + '    <KnownRemoteSenderChannelTransmitter')
              try:
                remoteMEuuid = ir.getRemoteEngineUuid()
                if mesByUUID.has_key(remoteMEuuid): remoteMEname = mesByUUID[remoteMEuuid].name
                else: remoteMEname = "Unknown"
                out.write(' me="' + remoteMEname + '"')
                out.write(' meUUID="' + remoteMEuuid + '"')
                out.write(' currentInboundMessages="' + str(ir.getDepth()) + '"')
                out.write(' inboundMessagesReceived="' + str(ir.getNumberOfMessagesReceived()) + '"')
                out.write('/>\n')
              except:
                et.printErrorAndCloseTag(sys.exc_info(), 1)
            # Close the transmitter tag
            out.write(indent + "   </SenderChannelTransmitter>\n")
        # Close the sender tag
        out.write(indent + "  </SenderChannel>\n")
      # Iterate through the receiver channels
      for rchlMBean in rchlMBeans:
        out.write(indent + "  <ReceiverChannel")
        statusEntries = []
        try:
          out.write(' state="' +  getStrReturnVal(rchlMBean,"getOverallStatus") + '"')
          # List the status of each instance
          statusEntries = AdminControl.invoke_jmx(rchlMBean, "getCurrentStatus", [], [])
          if (statusEntries == None): statusEntries = []
          out.write('>\n') # Complete the opening tag
        except:
          et.printErrorAndCloseTag(sys.exc_info(), 0)
        # Print each entry
        for statusEntry in statusEntries:
          out.write(indent + "   <ReceiverChannelStatus")
          try:
            out.write(' channelName="' +  toStr(statusEntry.getChannelName()) + '"')
            out.write(' state="' +  toStr(statusEntry.getState()) + '"')
            out.write(' qmgr="' +  toStr(statusEntry.getQueueManager()) + '"')
            out.write(' ipAddress="' + toStr(statusEntry.getIpAddress()) + '"')
            out.write(' messagesReceived="' + toStr(statusEntry.getNumberOfMessagesReceived()) + '"')
            out.write(' currentLUWID="' + toStr(statusEntry.getCurrentLUWID()) + '"')
            out.write(' currentSequenceNo="' + toStr(statusEntry.getCurrentSequenceNumber()) + '"')
            out.write('/>\n') # Complete the tag
          except:
            et.printErrorAndCloseTag(sys.exc_info(), 1)
        # Close the tag
        out.write(indent + "  </ReceiverChannel>\n")
      # Complete our WMQLink tag
      out.write(indent + " </WMQLink>\n")
    # Complete our WMQLinks tag
    out.write(indent + "</WMQLinks>\n")

# A small class to help us agregate consumer-transmitter and
# inbound-receiver information on a queue point, and hence build a
# "known remote queue point" for each remote messaging engine
# UUID that we have remote put/get state for.
class KnownRemoteQueuePoint:
  "A class for printing a runtime summary of a known remote queue point"
  consumerTransmitter = None
  inboundReceiver = None
  def __init__ (self, qpMBean, uuid):
    self.uuid = uuid
    if mesByUUID.has_key(uuid): self.name = mesByUUID[uuid].name
    else: self.name = "Unknown"
    self.qpMBean = qpMBean
  # Print an XML tag summarising this KnownRemoteQueuePoint
  def printSummaryXML(self, indent):
    out.write(indent + '<KnownRemoteQueuePoint me="' + self.name + '" meUUID="' + self.uuid + '"')
    try:
      currentInboundMessages = 0
      inboundMessagesReceived = 0
      if self.inboundReceiver != None:
        currentInboundMessages = self.inboundReceiver.getDepth()
        inboundMessagesReceived = self.inboundReceiver.getNumberOfMessagesReceived()
      out.write(' currentInboundMessages="' + str(currentInboundMessages) + '"')
      out.write(' inboundMessagesReceived="' + str(inboundMessagesReceived) + '"')
      currentMessageRequests = 0
      completedMessageRequests = 0
      messageRequestsReceived = 0
      if self.consumerTransmitter != None:
        currentMessageRequests = self.consumerTransmitter.getDepth()
        completedMessageRequests = self.consumerTransmitter.getNumberOfCompletedRequests()
        messageRequestsReceived = self.consumerTransmitter.getNumberOfRequestsReceived()
      out.write(' currentMessageRequests="' + str(currentMessageRequests) + '"')
      out.write(' completedMessageRequests="' + str(completedMessageRequests) + '"')
      out.write(' messageRequestsReceived="' + str(messageRequestsReceived) + '"')
      # Only need one line for this
      out.write('/>\n')
    except:
      et.printErrorAndCloseTag(sys.exc_info(), 1)

# Script execution starts here ...

# The only input parameter is the name of the output file.
# Default to STDOUT if not specified
if len(sys.argv) > 0: 
  print "Writing output to", sys.argv[0]
  out = open(sys.argv[0], 'w')
  fileOpened = 1
else:
  out = sys.stdout
  fileOpened = 0

# Lookup all MEs in the configuration and build a ME UUID to name dictionary
mesByUUID = {}
mesByBus = {}
mes = AdminConfig.list("SIBMessagingEngine").split("\n")
for me in mes:
  # Remove carriage returns on Windows
  me = me.strip()
  # Construct an ME object
  meBus  = AdminConfig.showAttribute(me, "busName")
  meName = AdminConfig.showAttribute(me, "name")
  meUUID = AdminConfig.showAttribute(me, "uuid")
  meObject = MessagingEngine(meBus, meName, meUUID)
  # Place the ME into the map by UUID
  mesByUUID[meUUID] = meObject
  # Place the ME into the list for this bus
  if (mesByBus.has_key(meBus)): meList = mesByBus[meBus]
  else: meList = []
  meList.append(meObject)
  mesByBus[meBus] = meList

# Iterate through the buses, and MEs within each bus
try:
  out.write('<?xml version="1.0"?>\n')
  out.write('<SIBusSummary>\n')
  for busName in mesByBus.keys():
    indent = ' '
    out.write(indent + '<Bus name="' + busName + '">\n')
    meList = mesByBus[busName]
    for me in meList:
      me.printRuntimeStateXML(indent + ' ')
    out.write(indent + '</Bus>\n')
  out.write('</SIBusSummary>\n')
finally:
  if (fileOpened): out.close()